/* -*- Mode: C++ -*- */

/* trainer/TMbreakaway.C
 * CMUnited99 (code for off-line trainer)
 * Patrick Riley <pfr+@cs.cmu.edu>
 * Computer Science Department
 * Carnegie Mellon University
 * Copyright (C) 1999 Patrick Riley
 *
 * CMUnited-99 was created by Peter Stone, Patrick Riley, and Manuela Veloso
 *
 * You may copy and distribute this program freely as long as you retain this notice.
 * If you make any changes or have any comments we would appreciate a message.
 */

/* this is a sample training module. See the README for details about this program */

/* this is the training module for the breakaway behavior */

#include <unistd.h>
#include <iostream.h>
#include <fstream.h>
#include <iomanip.h>
#include "client.h"
#include "utils.h"
#include "TMbreakaway.h"
#include "MemPosition.h"


const char* TMBreakaway::Name = "Breakaway";

Bool TMBreakaway::Initialize()
{
  Ear(TRUE);

  num_breakaways = num_goals = num_saves = num_misses = 0;

  starting_dist = Mem->TP_breakaway_min_dist;

  should_reset = TRUE;
  
  cout << "Breakaway Training module" << endl;
  
  return TRUE;
}

/* ranges to check:
   */
void TMBreakaway::InitializeEpochController(EpochController* pEC)
{
  pEC->num_vars = 0;
  pEC->num_epochs = 0;
}


/* Does the following:
   */
void TMBreakaway::NewSightHandler()
{
  
  pPlayerInfo pPIgoalie = Mem->GetPlayer(TS_Left, 1);
  pPlayerInfo pPIstriker = Mem->GetPlayer(TS_Right, 1);
  if (pPIgoalie == NULL || pPIstriker == NULL) {
    cout << "Trainer: Waiting for players to connect" << endl;
    return; /* no player yet */
  }

  if (should_reset) {    
    /* if we're not using epochs, pick a ranodm dist */
    if (!Mem->TP_use_epochs)
      starting_dist = range_random(Mem->TP_breakaway_min_dist, Mem->TP_breakaway_max_dist);
    
    Say(""); /* to have the breaking player idle */
    float ang = range_random(-Mem->TP_breakaway_max_ang, Mem->TP_breakaway_max_ang);
    Vector player_pos = Vector(-Mem->SP_pitch_length / 2, 0) +
      Polar2Vector(starting_dist, ang);
    MovePlayer(TS_Right, 1, player_pos);
    MoveBall(player_pos + Polar2Vector(.6, 60));
    MovePlayer(TS_Left, 1, -Mem->SP_pitch_length/2 + 3, 0, ang);

/*    Vector def_disp = Polar2Vector(Mem->TP_breakaway_def_behind_dist,
				   range_random(-Mem->TP_breakaway_def_max_ang,
						Mem->TP_breakaway_def_max_ang));*/
    Vector def_disp = Polar2Vector(15, signf(player_pos.y)*90);
    
    MovePlayer(TS_Left, 2, player_pos + def_disp);
    
    if (Mem->GetPlayMode() != PM_Play_On)
      ChangeMode(PM_Play_On);

    time_not_kickable = 0;
    curr_breakaway_time = 0;
    num_breakaways++;

    should_reset = FALSE;
  } else {

    pBallInfo pBI = Mem->GetBall();

    if (pBI->pos.dist(pPIstriker->pos) > Mem->SP_kickable_area) {
      if (time_not_kickable++ == 0) {
	if (!Mem->TP_use_epochs)
	  cout << "Shot vel: " << pBI->vel.mod() << endl;
      }
    }
    
    switch (Mem->GetPlayMode()) {
    case PM_Play_On:
      //if (!Mem->TP_use_epochs) cout << "In play_on mode" << endl;
      if (++curr_breakaway_time > Mem->TP_breakaway_max_time) {
	if (!Mem->TP_use_epochs) cout << "Time: " << Mem->GetTime() << "\tFailure" << endl;
	num_failed++;
	should_reset = TRUE;
      }
      break;
    case PM_After_Goal_Right:
      if (!Mem->TP_use_epochs) cout << "Time: " << Mem->GetTime() << "\tGoal!" << endl;
      num_goals++;
      should_reset = TRUE;
      break;
    case PM_Goal_Kick_Right:
      if (!Mem->TP_use_epochs) cout << "Time: " << Mem->GetTime() << "\tMissed Shot!" << endl;
      num_misses++;
      should_reset = TRUE;
      break;
    default:
      /* looks like the goalie saved it */
      if (!Mem->TP_use_epochs) cout << "Time: " << Mem->GetTime() << "\tSaved!" << endl;
      num_saves++;
      should_reset = TRUE;
      break;
    }
  
  }
  
}

void TMBreakaway::LogHeader(EpochController* pEC)
{
  ofstream outfile;

  outfile.open(Mem->TP_training_log_fn, ios::out);
  if (!outfile) {
    my_error("Could not open data file: %s", Mem->TP_training_log_fn);
    return;
  }
    
  /* write header */
  outfile << "Training Info for Breakaway behavior" << endl;
  outfile << "Format: ";
  pEC->WriteOptionNames(outfile);
  outfile << "breakaway_dist ";
  outfile << "num_breakaways num_goals num_saves num_misses num_failed ";
  outfile << "perc_goals perc_saves perc_misses perc_fails" << endl;;
    
  outfile << "Epoch Length is: " << Mem->TP_epoch_length << endl;
  outfile.close();
}


void TMBreakaway::LogPerformance(EpochController* pEC)
{
  ofstream outfile;

  outfile.open(Mem->TP_training_log_fn, ios::app);
  if (!outfile) {
    my_error("Could not open data file: %s", Mem->TP_training_log_fn);
    return;
  }

  pEC->WriteCurrentOptionValues(outfile);
  outfile << setprecision(2) << starting_dist << '\t';
  outfile << num_breakaways << '\t'
	  << num_goals << '\t'
	  << num_saves << '\t'
	  << num_misses << '\t'
	  << num_failed << '\t';
  outfile << ((float)num_goals) / num_breakaways << '\t'
	  << ((float)num_saves) / num_breakaways << '\t'
	  << ((float)num_misses) / num_breakaways << '\t'
	  << ((float)num_failed) / num_breakaways << '\t';
  outfile << endl;
  outfile.close();

}

void TMBreakaway::ResetForEpoch()
{
  /* reset the counting values */
  num_breakaways = num_goals = num_saves = num_failed = num_misses =0;

  /* reset the ball */
  should_reset = TRUE;
}






