#!/usr/local/bin/perl
#
# generates statistics about the quality of a load trace replay and a plot
# comparing the measured load during replay with the desired load and the 
# load in the trace file
#
# the caller can restrict the range of the replay that is plotted and
# analyzed by using the optional range parameter
#
#
#

$tempfile = "_gnuplotin";

if ($#ARGV<1 || $#ARGV>2) { 
    print STDERR "usage: plot_replay.pl ascii_trace_file replay_output_file [rangemin:rangemax]\n";
    exit;
}

$trace = $ARGV[0];
$replay = $ARGV[1];

if ($#ARGV==2) { 
    $restrictrange = 1;
    ($rangemin,$rangemax) = split(/:/,$ARGV[2]);
} else {
    $restrictrange = 0;
}

$sumtrace=0;
$sum2trace=0;
$sumdesired=0;
$sum2desired=0;
$summeasured=0;
$sum2measured=0;
$sumerror=0;
$sum2error=0;

open(TRACE,$trace);
@temp = <TRACE>;
close(TRACE);
$numtrace = $#temp+1;
for ($i=0;$i<$numtrace;$i++) { 
    ($trace_time[$i],$trace_val[$i]) = split(/\s+/,$temp[$i]);
    if ($i==0) { 
	$trace_first = $trace_time[$i];
    }
    $trace_time[$i]-=$trace_first;
}
open(REPLAY,$replay);
$junk=<REPLAY>;
@temp=<REPLAY>;
close(REPLAY);
$numreplay = $#temp+1;
for ($i=0;$i<$numreplay;$i++) { 
    ($replay_time[$i],$replay_desired[$i],$replay_measured[$i]) = split(/\s+/,$temp[$i]);
    if ($i==0) { 
	$replay_first = $replay_time[$i];
    }
    $replay_time[$i]-=$replay_first;
}


if ($restrictrange) {
    if ($rangemin>MIN($numtrace-1,$numreplay-1)) {
	$rangemin=0;
	print STDERR "Reset rangemin to zero since one or both traces doesn't have enough samples\n";
    }
    if ($rangemax>MIN($numtrace-1,$numreplay-1)) { 
	$rangemax=MIN($numtrace-1,$numreplay-1);
	print STDERR "Reset rangemax to $rangemax since one or both traces doesn't have enough samples\n";
    }
} else {
    $rangemin = 0;
    $rangemax = MIN($numtrace-1,$numreplay-1);
}

for ($i=$rangemin;$i<=$rangemax;$i++) { 
    $sumtrace+=$trace_val[$i];
    $sum2trace+=$trace_val[$i]**2;
    $summeasured+=$replay_measured[$i];
    $sum2measured+=$replay_measured[$i]**2;
    $sumdesired+=$replay_desired[$i];
    $sum2desired+=$replay_desired[$i]**2;
    $sumerror+=$replay_desired[$i]-$replay_measured[$i];
    $sum2error+=($replay_desired[$i]-$replay_measured[$i])**2;
}

$numsamples=$rangemax-$rangemin+1;

$meantrace = $sumtrace/$numsamples;
$stdtrace = sqrt(($sum2trace - ($sumtrace**2)/$numsamples)/($numsamples-1));
@meantraceci = ($meantrace - 1.96*$stdtrace/sqrt($numsamples),
                $meantrace + 1.96*$stdtrace/sqrt($numsamples));

$meandesired = $sumdesired/$numsamples;
$stddesired = sqrt(($sum2desired - ($sumdesired**2)/$numsamples)/($numsamples-1));
@meandesiredci = ($meandesired - 1.96*$stddesired/sqrt($numsamples),
		  $meandesired + 1.96*$stddesired/sqrt($numsamples));

$meanmeasured = $summeasured/$numsamples;
$stdmeasured= sqrt(($sum2measured - ($summeasured**2)/$numsamples)/($numsamples-1));
@meanmeasuredci = ($meanmeasured - 1.96*$stdmeasured/sqrt($numsamples),
		   $meanmeasured + 1.96*$stdmeasured/sqrt($numsamples));

$meanerror = $sumerror/$numsamples;
$stderror= sqrt(($sum2error - ($sumerror**2)/$numsamples)/($numsamples-1));
@meanerrorci = ($meanerror - 1.96*$stderror/sqrt($numsamples),
                $meanerror + 1.96*$stderror/sqrt($numsamples));


$plotfile = "$trace.$replay.eps";

print
"number of samples in trace file:    $numtrace\n".
"number of samples in replay file:   $numreplay\n".
"sample range for stats and plot:    [$rangemin, $rangemax] ($numsamples samples)\n".
"trace mean:                         $meantrace\n".
"trace std:                          $stdtrace\n".
"trace mean ci:                      [$meantraceci[0] , $meantraceci[1]]\n".
"replay desired mean:                $meandesired\n".
"replay desired std:                 $stddesired\n".
"replay desired mean ci:             [$meandesiredci[0] , $meandesiredci[1]]\n".
"replay measured mean:               $meanmeasured\n".
"replay measured std:                $stdmeasured\n".
"replay measured mean ci:            [$meanmeasuredci[0] , $meanmeasuredci[1]]\n".
"replay error mean:                  $meanerror\n".
"replay error std:                   $stderror\n".
"replay error mean ci:               [$meanerrorci[0] , $meanerrorci[1]]\n";

print "\nGenerating plot in '$plotfile'...";

open(TEMP,">$tempfile");
for ($i=$rangemin;$i<=$rangemax;$i++) {
    print TEMP "$trace_time[$i]\t$trace_val[$i]\t$replay_time[$i]\t$replay_desired[$i]\t$replay_measured[$i]\n";
}
close(TEMP);

open(GNUPLOT,"|gnuplot");
print GNUPLOT "set terminal postscript eps color 18\n";
print GNUPLOT "set output \"$plotfile\"\n";
print GNUPLOT "set title \"trace: $trace replay: $replay\"\n";
print GNUPLOT "set xlabel \"Time (seconds)\"\n";
print GNUPLOT "set ylabel \"Load\"\n";
print GNUPLOT "plot \"$tempfile\" using 1:2 title 'Original Trace' with lines,".
              "\"$tempfile\" using 3:4 title 'Desired Replay' with lines,".
              "\"$tempfile\" using 3:5 title 'Measured Replay' with lines\n";
print GNUPLOT "quit\n";
close(GNUPLOT);

print "Done.\n";

sub MIN {
    return $_[0] < $_[1] ? $_[0] : $_[1];
}

