#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <assert.h>
#include <math.h>
#include <stdarg.h>
#include <string.h>

#include "LoadTrace.h"
#include "timing.h"
#include "getloadavg.h"
#include "apply.h"

/*#define DEBUG 1*/
#define EVEN_DIVISION_OF_LABOR   0  /* Divide work evenly or not */

#define PRINT_STATS 0  /* Print running stats to stderr */
#define PRINT_OPS   1  /* Print operational info to stdout */


#ifndef MAX
#define MAX(x,y) ((x)>(y) ? (x) : (y))
#endif

int numworkers;
int *inpipes;
int *outpipes;
int *pids;


int dfprintf(FILE *where, char *format, ...)
{
#if DEBUG
  va_list list;
  va_start(list,format);
  return vfprintf(where,format,list);
#else 
  return 0;
#endif
}

/* Used internally */
#define EPSILON  0.05


void usage()
{
  fprintf(stderr,"usage: playload tauoftrace tauofhost feedback [time|work] [nonperiodic|Hz] [alpha|network|text] tracefile\n");
}



double Max(double *seq, int len)
{
  int i;
  double max;

  assert(len>0);

  max=seq[0];
  
  for (i=1;i<len;i++) {
    if (seq[i]>max) {
      max=seq[i];
    }
  }
  return max;
}



void DeconvolvePeriodicLoadTrace(double tau,
				 double *tracein, 
				 double sample_interval,
				 int numsamples, 
				 double *traceout)
{
  int i;
  double Beta;

  Beta = exp(-sample_interval/tau);

  for (i=0;i<numsamples-1;i++) {
    traceout[i] = (tracein[i+1] - Beta*tracein[i])/(1-Beta);
  }
}


void ConvolvePeriodicLoadTrace(double tau,
			       double *tracein,
			       double sample_interval,
			       int numsamples,
			       double *traceout)
{
  int i;
  double Beta;

  Beta = exp(-sample_interval/tau);

  traceout[0]=0;

  for (i=1;i<numsamples;i++) {
    traceout[i] = (1-Beta)*tracein[i-1] + Beta*traceout[i-1];
  }
}


void DeconvolveNonperiodicLoadTrace(double tau,
				    double *tracein, 
				    double *timestamp,
				    int numsamples, 
				    double *traceout)
{
  int i;
  double Beta;


  for (i=0;i<numsamples-1;i++) {
    Beta = exp(-(timestamp[i+1]-timestamp[i])/tau);
    traceout[i] = (tracein[i+1] - Beta*tracein[i])/(1-Beta);
  }
}


void ConvolveNonperiodicLoadTrace(double tau,
				  double *tracein,
				  double *timestamp,
				  int numsamples,
				  double *traceout)
{
  int i;
  double Beta;


  traceout[0]=0;

  for (i=1;i<numsamples;i++) {
    Beta = exp(-(timestamp[i]-timestamp[i-1])/tau);
    traceout[i] = (1-Beta)*tracein[i-1] + Beta*traceout[i-1];
  }
}

#define CHK_DEL(x) { if (x) { free(x); x=0;} }


int main(int argc, char *argv[])
{
  int pipetemp1[2], pipetemp2[2];
  int myinpipe;
  int myoutpipe;
  int myid;
  int i,j;
  int i_am_master;
  int rc;
  double sample_interval;
  double level, dur, work;
  double *trace;
  double *timestamps;
  int     numsamples;
  double  max;
  double *dtrace;   /* Deconvolved trace */
  double *ctrace;   /* Convolved trace appearence on this host */
  double tautrace, tauhost, feedbacklevel;
  double betahost, betatrace;
  double sum, sum2;
  char *tracerate, *tracetype, *tracefile;
  char *applyarg;
  char *sampleint;

  ApplySemantics applysem;

  if (argc!=8) {
    usage();
    exit(0);
  }

  tautrace=atof(argv[1]);
  tauhost=atof(argv[2]);
  feedbacklevel=atof(argv[3]);
  applyarg=argv[4];
  tracerate=argv[5];
  tracetype=argv[6];
  tracefile=argv[7];

  
  dfprintf(stderr,"Loading Trace %s...",tracefile);
  
  if (!strcasecmp(applyarg,"work")) { 
    applysem=WORK_BASED;
  } else if (!strcasecmp(applyarg,"time")) { 
    applysem=TIME_BASED;
  } else {
    fprintf(stderr,"Unknown apply semantics '%s'\n",applyarg);
    exit(-1);
  }

  if (!strcasecmp(tracerate,"nonperiodic")) { 
    sample_interval=-1;
  } else {
    sample_interval=1.0/atof(tracerate);
  }

  if (!strcasecmp(tracetype,"alpha")) { 
    numsamples=LoadAlphaBinaryTraceFile(tracefile,&timestamps,&trace);
  } else if (!strcasecmp(tracetype,"network")) {
    numsamples=LoadNetworkBinaryTraceFile(tracefile,&timestamps,&trace);
  } else if (!strcasecmp(tracetype,"text")) {
    numsamples=LoadAsciiTraceFile(tracefile,&timestamps,&trace);
  } else {
    fprintf(stderr,"Unknown trace format '%s'\n",tracetype);
    exit(-1);
  }

  if (numsamples<0) {
    fprintf(stderr,"FAILED to load trace\n");
    exit(-1);
  }
  max=Max(trace,numsamples);
  dfprintf(stderr,"done (%d samples, max=%lf)\n",numsamples,max);

  dfprintf(stderr,"Deconvolving Trace...");
  dtrace = (double*)malloc(sizeof(double)*(numsamples));
  ctrace = (double*)malloc(sizeof(double)*(numsamples));
  if (!dtrace || !ctrace) { 
    fprintf(stderr,"Out of memory\n");
    goto finish;
  }
  if (sample_interval>0) { 
    DeconvolvePeriodicLoadTrace(tautrace,trace,sample_interval,numsamples,dtrace);
    ConvolvePeriodicLoadTrace(tauhost,dtrace,sample_interval,numsamples,ctrace);
  } else {
    DeconvolveNonperiodicLoadTrace(tautrace,trace,timestamps,numsamples,dtrace);
    ConvolveNonperiodicLoadTrace(tauhost,dtrace,timestamps,numsamples,ctrace);
  }

  dfprintf(stderr,"done\n");

  dfprintf(stderr,"Calibrating Timing...");
  CalibrateTiming(EPSILON);
  dfprintf(stderr,"done\n");

  dfprintf(stderr,"Calibrating Loop...");
  CalibrateLoop(EPSILON);
  dfprintf(stderr,"done\n");


  numworkers=(int)(Max(dtrace,numsamples-1)+1);
  inpipes = (int *)malloc(sizeof(int)*numworkers);
  outpipes = (int *)malloc(sizeof(int)*numworkers);
  pids = (int *)malloc(sizeof(int)*(numworkers));

  if (!inpipes || !outpipes || !pids) { 
    goto finish;
  }

  dfprintf(stderr,"Firing up %d workers...",numworkers); fflush(stdout);

  for (i=0;i<numworkers;i++) {
    pipe(pipetemp1);
    pipe(pipetemp2);
    if ((pids[i]=fork())!=0) {
      /* master */
      i_am_master=1;
      outpipes[i]= pipetemp1[1];
      close(pipetemp1[0]);
      inpipes[i] = pipetemp2[0];
      close(pipetemp2[1]);
      write(outpipes[i],&i,sizeof(int));
    } else {
      /* slave */
      i_am_master=0;
      myinpipe = pipetemp1[0];
      close(pipetemp1[1]);
      myoutpipe = pipetemp2[1];
      close(pipetemp2[0]);
      read(myinpipe,&myid,sizeof(int));
      break;
    }
  }
  
#if 0
  sleep(60); /* Wait for startup noise to die down */
#endif

  if (i_am_master) {
    double curloadavg, error;
    double feedback,targetlevel,requestlevel,desiredlevel;
    double prevmeasure;
    double dmeasure;
    int    numfull, numrun;
    double workfull,workpartial, levelfull, levelpartial;
    TimeValue ts;
    TimeValue now, interval;

    sum=sum2=0;

    dfprintf(stderr,"done\n");
    fprintf(stdout,"#Time\tDesired\tMeasured\tTarget\tRequest\tDMeasured\tError\tPercent\n");
    prevmeasure=0;
    feedback=0;


    for (i=0;i<numsamples-1;i++) {
      GetCurrentTime(&ts);

      if (sample_interval>0) { 
	dur=sample_interval;
      } else {
	dur= (i==0) ? 1.0 : timestamps[i]-timestamps[i-1];
      }

      betatrace = exp(-dur/tautrace);
      betahost = exp(-dur/tauhost);

      /* The load we would ideally apply is the deconvolved trace value */
      targetlevel=dtrace[i];


      /* The load value we expect to measure is the deconvolved level
	 convolved according to the tau of the host system */
      desiredlevel=ctrace[i+1]; 

      /* The load we will apply is the ideal summed with an error feedback */
      requestlevel=targetlevel+feedback;
      requestlevel=MAX(requestlevel,0);

      numfull = (int)requestlevel;


      /* workfull is the work each "full up" subprocess will do */
      /* workpartial is the work the one partially full subprocess will do */
      if (numfull==0) { 
	workpartial=requestlevel*dur;
	levelpartial=requestlevel;
	workfull=0;
	levelfull=0;
      } else {
	if ((double)numfull==requestlevel) {
	  workfull=dur/numfull;
	  levelfull=1;
	  workpartial=0;
	  levelpartial=0;
	} else {
#if EVEN_DIVISION_OF_LABOR
	  workfull=workpartial=dur/(numfull+1);
	  levelfull=levelpartial=requestlevel/(numfull+1);
#else
	  workpartial=dur*(requestlevel-(double)numfull)/requestlevel;
	  levelpartial = requestlevel-numfull;
	  workfull=(dur-workpartial)/numfull;
	  levelfull=1;
#endif
	}
      }

      dfprintf(stderr,"request=%lf=%lf+%lf\n",requestlevel,targetlevel,feedback);
      dfprintf(stderr,"workfull=%lf, workpartial=%lf\n",workfull,workpartial);
      dfprintf(stderr,"levelfull=%lf, levelpartial=%lf\n",levelfull,levelpartial);

      /* Tell child processes how much load to generate */
      
      for (j=0;j<numfull;j++) {
	write(outpipes[j],&levelfull,sizeof(double));
	write(outpipes[j],&dur,sizeof(double));
	write(outpipes[j],&workfull,sizeof(double));
      }
      write(outpipes[numfull],&levelpartial,sizeof(double));
      write(outpipes[numfull],&dur,sizeof(double));
      write(outpipes[numfull],&workpartial,sizeof(double));


      /* Wait for child processes to complete this round */
      for (j=0;j<(numfull+1);j++) {
	read(inpipes[j],&rc,1);
      }

      /* Centralize all remaining sleep here */
      GetCurrentTime(&now);
      DiffTimes(&ts,&now,&interval);

      if (TimeIntervalToSeconds(&interval)<dur) {
	/* fprintf(stderr,"Sleep remaing %lf\n",dur-TimeIntervalToSeconds(&interval));*/
	Sleep(dur-TimeIntervalToSeconds(&interval));
      }  

      GetCurrentTime(&now);
      DiffTimes(&ts,&now,&interval);

      /* get the load average */
      getloadavg(&curloadavg,1);

      /* deconvolve to see how closely the load we saw tracked
         the load we applied */
      dmeasure = (curloadavg - betahost * prevmeasure)/(1-betahost);
      prevmeasure = curloadavg;


      dfprintf(stderr,"dur=%lf, betahost=%lf, targetlevel=%lf\n",
	       dur, betahost, targetlevel);

      /* the error is with respect to the desired level, *not*
	 with respect to target level (deconvolved desired level)  
         or the request level (desired+feedback) we applied 
         The goal is to get the load average onthis machine to
         track what the load average on the trace machine would have
         looked like given this machine's tau */
      error=curloadavg-desiredlevel;
      

      /* feed back the error in reproducing the load */
#if 0
      /* This version feeds back the error in reproducing this cycle */
      feedback = feedbacklevel*(dmeasure-requestlevel);
#else
      /* This version feeds back the error in the overall load signal */
      feedback = feedbacklevel*error;
#endif
      /* Gather stats */
      sum+=error;
      sum2+=(error*error);

#if PRINT_STATS
      /* Spew error stats to stderr */
      if (i>0) {
	fprintf(stderr,"iter: %d\tmeanerr: %lf\trmserr: %lf\tsum: %lf\tsum2: %lf\r",i+1,
		MEANFROMSUMS(i+1,sum),
		STDFROMSUMS(i+1,sum,sum2),sum,sum2 );
      }
#endif

#if PRINT_OPS
      /* Spew operational info  to stdout */
      fprintf(stdout,"%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\t%lf\n",
	      TimeIntervalToSeconds(&ts),
	      desiredlevel,curloadavg,targetlevel,requestlevel,dmeasure,
	      error, desiredlevel!=0 ? 100.0*(error)/desiredlevel : 0.0);
      fflush(stdout);
#endif
    }
    level=-99e99;
    dur=-99e99;
    work=-99e99;
    for (i=0;i<numworkers;i++) {
      write(outpipes[i],&level,sizeof(double));
      write(outpipes[i],&dur,sizeof(double));
      write(outpipes[i],&work,sizeof(double));

      waitpid(pids[i],&rc,0);
      dfprintf(stderr,"%d: exit(%d)\n",i,WEXITSTATUS(rc));
    }
  } else {
    /* Until done, do what master says */
    while (1) {
      read(myinpipe,&level,sizeof(double));	
      read(myinpipe,&dur,sizeof(double));
      read(myinpipe,&work,sizeof(double));
      dfprintf(stderr,"%d: Applying %lf load for %lf seconds for %lf work\n",myid,level,dur,work);
      if (level<0 || dur<0 || work<0) {
	exit(0);
      } else {
	ApplyLoad(level,dur,work,applysem);
	rc=0;
	write(myoutpipe,&rc,1);
      }
    }
  }

 finish:
  CHK_DEL(dtrace);
  CHK_DEL(ctrace);
  CHK_DEL(trace);
  CHK_DEL(timestamps);
  CHK_DEL(inpipes);
  CHK_DEL(outpipes);
  CHK_DEL(pids);
  exit(0);
}
