#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <assert.h>
#include <sys/resource.h>
#include <math.h>

#include "timing.h"
#include "apply.h"

#define DEBUG_APPLY   0
#define CHECKUSAGE    1     
#define MONITOR_ISSUE 1     /* Set to zero to monitor completion */

#if defined(__osf__)
const double cycles_per_second=123;
#else
const double cycles_per_second=31;
#endif

const double max_issue_error=0.05;



#define MIN(x,y) ((x)<(y) ? (x) : (y))
#define MAX(x,y) ((x)>(y) ? (x) : (y))

static volatile double x,y;
static int iters_per_second;

#define LB_ITERS 100000
#define UB_ITERS 1000000

void SpinInternal(int iters)
{
  int i;
#if DEBUG_APPLY
  fprintf(stderr,"Run %d iters\n",iters);
#endif
  for (i=0;i<iters;i++) {
    y+=x;
  }
}

double Spin(double interval)
{
  double spintotal;
#if 0

  double dur;
  double wait;
  int iters;


  wait=0;
  while (interval>0) {
#if DEBUG_APPLY
    fprintf(stderr,"Attempt Spin for %lf secs\n",interval);
#endif
    iters = (int) (interval*iters_per_second);
    dur=TimeOnce(SpinInternal,iters);
    interval-=dur;
    wait+=dur;
  }
  spintotal=wait;
#if DEBUG_APPLY
  fprintf(stderr,"Spun for total of %lf seconds\n",wait);
#endif
#else
#if CHECKUSAGE
  struct rusage ru;
  double initcpu, curcpu;

  getrusage(RUSAGE_SELF,&ru);
  initcpu = (double)ru.ru_utime.tv_sec + 1e-6*(double)ru.ru_utime.tv_usec +
            (double)ru.ru_stime.tv_sec + 1e-6*(double)ru.ru_stime.tv_usec;
  curcpu=initcpu;

  while (curcpu-initcpu < interval) {
#if DEBUG_APPLY
    fprintf(stderr,"Attempt Spin for %lf secs\n",interval-(curcpu-initcpu));
#endif
    SpinInternal((int)((interval-(curcpu-initcpu))*iters_per_second));
    getrusage(RUSAGE_SELF,&ru);
    curcpu = (double)ru.ru_utime.tv_sec + 1e-6*(double)ru.ru_utime.tv_usec +
             (double)ru.ru_stime.tv_sec + 1e-6*(double)ru.ru_stime.tv_usec; 

  }
  spintotal=curcpu-initcpu;
#if DEBUG_APPLY
  fprintf(stderr,"Spun for total of %lf seconds\n",curcpu-initcpu);
#endif

#else 
  SpinInternal((int)(interval*iters_per_second));
  spintotal=interval;
#endif
#endif
  return spintotal;
}  

void PortableSleepInternal(double second) 
{
#if 0
  struct timeval timeout;
    
  timeout.tv_sec = (int)second;
  timeout.tv_usec = (int)(1e6*(second - ((int)second)));

#if DEBUG_APPLY
  fprintf(stderr,"Sleep for %d secs, %d usecs\n",timeout.tv_sec,timeout.tv_usec);
#endif

  select(FD_SETSIZE, NULL, NULL, NULL, &timeout);
  
#else
  unsigned usec = (unsigned) (second*1e6);
#if DEBUG_APPLY
  fprintf(stderr,"Sleep for %d usecs\n",usec);
#endif
  usleep(usec);
#endif

}
  
double Sleep(double interval)
{
  double sleeptotal;
#if CHECKUSAGE
  double dur;
  double wait;

  wait=0;
  while (interval>0) {
#if DEBUG_APPLY
    fprintf(stderr,"Attempt Sleep for %lf secs\n",interval);
#endif
    dur=TimeOnceDouble(PortableSleepInternal,interval);
    interval-=dur;
    wait+=dur;
  }
  sleeptotal=wait;
#if DEBUG_APPLY
  fprintf(stderr,"Slept for total of %lf seconds\n",wait);
#endif
#else
  sleeptotal=TimeOnceDouble(PortableSleepInternal,interval);
#endif
  return sleeptotal;
}

double UniformRandom(double low, double high)
{
  static int inited=0;
  double val;

  if (!inited) { 
    srand48(getpid());
    inited=1;
  }

  val=drand48();

  return low+val*(high-low);
}





void CalibrateLoop(double epsilon)
{
  double lbdur, ubdur;
  int lbiters=LB_ITERS;
  int ubiters=UB_ITERS;

  lbdur=TimeIt(SpinInternal,lbiters,epsilon);
  ubdur=TimeIt(SpinInternal,ubiters,epsilon);

  iters_per_second = (int)((ubiters-lbiters)/(ubdur-lbdur));

}



#define MONITOR_ISSUE 1

#if MONITOR_ISSUE
#define FLUIDMODEL_ACTUAL_RATE fluidmodel_actualissuerate
#define FLUIDMODEL_IDEAL_RATE  fluidmodel_idealissuerate
#else
#define FLUIDMODEL_ACTUAL_RATE fluidmodel_actualcompletionrate
#define FLUIDMODEL_IDEAL_RATE  fluidmodel_idealcompletionrate
#endif



/* level is 0..1 duration is in seconds */ 
void ApplyLoad(double level, double duration, double work, ApplySemantics asem)
{
  TimeValue start, end, interval;
  double cycle_time;
  double consumed_time;
  double thiscompute, thissleep;
  double totalcompute, totalsleep;
  double totalcomputeissue, totalsleepissue;
  double computeleft, sleepleft;
  double fluidmodel_idealissuerate, fluidmodel_idealcompletionrate;
  double fluidmodel_actualissuerate;
  double fluidmodel_actualcompletionrate;
  double numiters;
  int dowork;

#if DEBUG_APPLY
  fprintf(stderr,"applyload: level=%lf, dur=%lf, work=%lf\n",
	  level,duration,work);
#endif

  computeleft=work;
  sleepleft=duration-work;
  
  numiters = duration*cycles_per_second;
  
  /*  fluidmodel_idealissuerate=duration>0.0 ? work/duration : 0.0; */
  
  fluidmodel_idealissuerate=level;
  fluidmodel_idealcompletionrate = fluidmodel_idealissuerate;

  totalcompute=totalsleep=0.0;
  totalcomputeissue=totalsleepissue=0.0;

  cycle_time=1.0/cycles_per_second;

  GetCurrentTime(&start);
  GetCurrentTime(&end);
  DiffTimes(&start,&end,&interval);
  consumed_time=TimeIntervalToSeconds(&interval);

  while ( ((consumed_time<duration) && (asem==TIME_BASED)) || 
	  ((computeleft>0.0) && (asem==WORK_BASED)) ) { 
    if (computeleft<=0.0) { 
      /* if we've done all the work, just sleep */
      dowork=0;
    } else {
      /* determine what the actual issue rate is given fluid model */
      fluidmodel_actualissuerate = (totalcomputeissue+totalsleepissue) > 0.0 ?
	(totalcomputeissue)/(totalcomputeissue+totalsleepissue) : 0.0;
      /* and also the ideal completion rate */
      fluidmodel_actualcompletionrate = (totalcompute+totalsleep) > 0.0 ?
	(totalcompute)/(totalcompute+totalsleep) : 0.0;
      /* if the fluid issue rate is far behind the ideal, we'll force work */
      if (  FLUIDMODEL_ACTUAL_RATE < 
	    (1-max_issue_error)*FLUIDMODEL_IDEAL_RATE) {
	dowork=1;
#if DEBUG_APPLY
	fprintf(stderr,"%d: Force work! fluid_ideal=%lf fluid_actual=%lf\n",
		getpid(),FLUIDMODEL_IDEAL_RATE,FLUIDMODEL_ACTUAL_RATE); 
#endif
      } else if (FLUIDMODEL_ACTUAL_RATE >
		 (1+max_issue_error)*FLUIDMODEL_IDEAL_RATE) {
	/* if we're far ahead, we'll force sleep */
	dowork=0;
#if DEBUG_APPLY
	fprintf(stderr,"%d: Force sleep! fluid_ideal=%lf fluid_actual=%lf\n",
		getpid(),FLUIDMODEL_IDEAL_RATE,FLUIDMODEL_ACTUAL_RATE);
#endif
      } else {
	/* if we're on track, we'll decide randomly whether do work or not
	   depending on the load we're asked to replicate */
	dowork = (UniformRandom(0.0,1.0) <= level);
#if DEBUG_APPLY
	fprintf(stderr,"%d: Randomly chose %s level=%lf fluid_ideal=%lf fluid_actual=%lf\n",
		getpid(),dowork ? "work" : "sleep", level,
		FLUIDMODEL_IDEAL_RATE,FLUIDMODEL_ACTUAL_RATE);
#endif
      }
    }
    if (dowork) { 
      thiscompute=Spin(cycle_time);
      totalcompute+=thiscompute;
      computeleft-=thiscompute;
      totalcomputeissue+=cycle_time;
    } else {
      thissleep=Sleep(cycle_time);
      totalsleep+=thissleep;
      sleepleft-=thissleep;
      totalsleepissue+=cycle_time;
    }
    GetCurrentTime(&end);
    DiffTimes(&start,&end,&interval);
    consumed_time=TimeIntervalToSeconds(&interval);
  }
#if DEBUG_APPLY
  fprintf(stderr,"consumedtime/totalcompute/totalsleep = %lf/%lf/%lf - level/workreq/workissued/sleepreq/sleepissued=%lf/%lf/%lf/%lf/%lf - fluidideal/fluidactual=%lf/%lf\n", 
	  consumed_time,totalcompute,totalsleep, level, work, totalcomputeissue,
	  duration-work,totalsleepissue,FLUIDMODEL_IDEAL_RATE,FLUIDMODEL_ACTUAL_RATE);
#endif
}

