/*
 * ps2eps
 *
 * Author: Michael Hemy     (mhemy@cs.cmu.edu)
 * Date:   August 18, 1995
 *
 * ps2eps will convert a ps file to an eps file and also remove
 * junk from an msdos generated psfile.
 * also there is the ability to clip an image to its real bounding box
 * with the -clip argument
 *
 * v 1.0: Aug 16, 1995             (mhemy@cs.cmu.edu)
 *           - initial release
 * v 1.2: Aug 18, 1995             (mhemy@cs.cmu.edu)
 *           - added support for multiple files
 *           - added clipping borders
 *           - cleaned up code
 * v 1.3: Aug 23, 1995             (mhemy@cs.cmu.edu)
 *           - made border size default to 0 when no clipping requested 
 *           - insert tags can be set by user if everything else fails
 *           - multiple tags searched
 *           - made it use ps2epsi.ps since it exists everywhere, is
 *             stable and does not require to install ps2eps.ps
 *
 * v 1.4: Sep 7, 1995
 *           - added -i flag to generate epsi files
 *           - added -u -d -l -r flags to generate assymetrical borders.
 *
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>
#include <getopt.h>

#define BUFFER_SIZE 1024
#define DEFAULT_BORDER 8
#define VERSION "1.4"
#define ADOBE_EPSF_TAG "%!PS-Adobe-2.0 EPSF-1.2\n"
#define END_PREVIEW_TAG "%%EndPreview\n"
#define MAX_TAGS 10

char *insert_tags[MAX_TAGS] = {
    "%%EndSetup\n",
    "SS\n",
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL
};    
    
int num_tags = 2;   
      
char *adobe_epsf_tag = ADOBE_EPSF_TAG;

/* not all systems have mktemp in their include files */
char *mktemp(char *template);

static void 
usage(char *arg0)
{
    fprintf(stderr, 
	    "usage:\n"
	    "        %s [...] [-o epsfile] psfile\n"
	    "        %s [...] psfile1 psfile2 psfile3 ...\n\n"
	    "    [-c [-t tag1] [-t tag2] ...]\n"
            "            clips image at bounding box. Additional tags can be\n"
	    "            used for searching insertion point of clipping\n"
	    "            command.\n"
	    "    [-v]\n"
	    "            Verbose output\n"
	    "    [-i]\n"
	    "            Generate epsi style file with preview image. No \n"
	    "            borders can be specified with this flag.\n"
	    "    [-b points]\n"
	    "            Specifies symetrical border around image in pixel\n"
	    "            points: 72 points = 1 inch. If clipping is used the\n"
	    "            default border is 8 points. If -i is specified, a \n"
	    "            border of 0 points is implied\n"
	    "    [-u points]\n"
	    "    [-d points]\n"
	    "    [-l points]\n"
	    "    [-r points]\n"
	    "            Specific non-symetric border designation. Can not\n"
	    "            combined with -b.\n"
	    "    [-V]\n"
	    "            Reports version number of %s\n"
	    "    [-h]\n"
	    "            Prints this help message\n",
	    arg0, arg0, arg0);
}


int main(int argc, char *argv[])
{
    char buffer[1024];
    int  opt;
    int  border_up_size = 0;
    int  border_down_size = 0;
    int  border_left_size = 0;
    int  border_right_size = 0;
    int  border = 0;
    int  i, error;
    int  tag;
    int  clip = 0;
    int  didclip;
    char *infile;
    char *basename;
    char *endname;
    char *startline;
    char *token;
    char outfile[256];
    char infilenew[256];
    int  filename_len;
    int  linelen;
    int  verbose = 0;
    int  onefile = 0;
    int  newlen;
    int  pssize;
    int  psnewsize;
    int  missing = 0;
    int  create_preview = 0;
    int  xlo,ylo,xhi,yhi;
    int  found_end_preview = 0;
    struct stat file_stat;
    FILE *psin;
    FILE *psout;
    FILE *epsout;

    /* initialize */
    outfile[0] = '\0';
    xlo = ylo = xhi = yhi = 0;
    
    /* get operating parameters */
    while( (opt = getopt( argc, argv, "it:b:u:d:l:r:chvVo:") ) != -1 ) {
	switch (opt) {
	  case 'i':
	    create_preview = 1;
	    break;
	  case 'c': 
	    clip = 1;
	    break;
	  case 'b': 
	    border_up_size = atoi(optarg);
	    border_down_size = border_up_size;
	    border_left_size = border_up_size;
	    border_right_size = border_up_size;
	    border = 1;
	    break;
	  case 'u': 
	    border_up_size = atoi(optarg);
	    border = 1;
	    break;
	  case 'd': 
	    border_down_size = atoi(optarg);
	    border = 1;
	    break;
	  case 'l': 
	    border_left_size = atoi(optarg);
	    border = 1;
	    break;
	  case 'r': 
	    border_right_size = atoi(optarg);
	    border = 1;
	    break;
	  case 't':
	    if (num_tags == MAX_TAGS) {
		fprintf(stderr, "Ignoring tag %s: reached max of %d tags\n", 
			optarg, MAX_TAGS);
	    } else {
		if (optarg[0] != '\0') {
		    insert_tags[num_tags] = optarg;
		    num_tags++;
		}
	    }
	  case 'v': 
	    verbose = 1;
	    break;
	  case 'o':
            onefile = 1;
	    strcpy(outfile, optarg);
	    break;
	  case 'V':
	    printf("%s version %s -- Michael Hemy (mhemy@cs.cmu.edu)\n",
		   argv[0], VERSION);
	    exit(1);
	  case 'h':
	  default:
	    usage(argv[0]);
	    exit(1);
	}
    }
    if ((optind >= argc) ||
	(onefile && ((argc-optind) > 1))) {
	usage(argv[0]);
	exit(1);
    }
    
    if (border & create_preview) {
	printf("WARNING: border specified but not used\n");
	border_up_size = 0;
	border_down_size = 0;
	border_left_size = 0;
	border_right_size = 0;
    }

    if (clip && verbose) {
	printf("%s> searching with tag:\n", argv[0]);
	for (tag=0;tag<num_tags;tag++) {
	    puts(insert_tags[tag]);
	}
	printf("\n Total of %d tags used in search.\n", num_tags);
    }

    for (i=optind; i<argc; i++) {
        error = 0;
	infile = argv[i];
        didclip = 0;
	found_end_preview = 0;
	
	if (stat(infile, &file_stat) == -1) {
	    fprintf(stderr, "File %s does not exist\n", infile);
	    continue;
	}
	
	if (!S_ISREG(file_stat.st_mode)) {
	    fprintf(stderr, 
		    "Sorry: %s can only operate on regular files\n", argv[0]);
	    continue;
	}
	
	/* get size of psfile */
	pssize = file_stat.st_size;

	/* get the basename of the input file */
	basename=strrchr(infile, (int)'/');
	if (basename) {
	    basename++;
	} else {
	    basename = infile;
	}
	endname = strrchr(basename, (int)'.');
	if (!endname)
	  endname = basename + strlen(basename);
	filename_len = endname - basename;
	
	/* now figure out the name of the eps output file */
	if (!onefile) {
	    strncpy(outfile, basename, filename_len);
	    outfile[filename_len] = '\0';
	    strcat(outfile, (create_preview ? ".epsi" : ".eps") );
	}
	
	/* generate a temporary file for removing junk from the ps file */    
	strncpy(infilenew, basename, filename_len);
	infilenew[filename_len] = '\0';
	strcat(infilenew, ".ps.XXXXXX");
	if (!mktemp(infilenew)) {
	    fprintf(stderr, "Error creating temporary file %s\n", infilenew);
	    fprintf(stderr, "Fatal error, exiting\n");
	    exit(1);
	}

	/* parse the ps file and remove all  
 and  */
	if (!(psin = fopen(infile, "r"))) {
	    fprintf(stderr, "Error opening file %s\n", infile);
	    continue;
	}
	if (!(psout = fopen(infilenew, "w"))) {
	    fprintf(stderr, "Error truncating file %s\n", infilenew);
	    fclose(psin);
	    continue;
	}

	while (!feof(psin)) {
	    if (fgets(buffer, BUFFER_SIZE, psin)) {
		linelen = strlen(buffer);
		if (!linelen)
		  continue;

		if (linelen >= (BUFFER_SIZE -1)) {
		    fprintf(stderr, 
			    "Postscript files with lines longer then %d\n"
			    "are unsupported for this release of %s\n",
			    BUFFER_SIZE, argv[0]);
		    error = 1;
		    break;
		}

		/* if line starts with an escape sequence omit it */
		if (buffer[0] == '')
		  continue;

		/* remove escape sequences */
		newlen = linelen;
		if (buffer[linelen-1] == '\n') {
		    if (buffer[linelen-2] == '
') {
			buffer[linelen-2] = '\n';
			buffer[linelen-1] = '\0';
			newlen--;
		    } 
		} else {
		    if (buffer[linelen-1] == '
') {
			buffer[linelen-1] = '\0';
			newlen--;
		    } 
		}		
		if (buffer[0] == '') {
		    startline = &buffer[1];
		    newlen--;
		} else {
		    startline = buffer;
		}
		
		/* copy the line to psout */
		if (newlen > 0) {
		    fputs(startline, psout);
		}
	    }
	}
	fclose(psin);
	fclose(psout);


	/* see if the psfile has to be saved removed or deleted */
	if (error) {
	    unlink(infilenew);
	    continue;
	} else {
	    stat(infilenew, &file_stat);
	    psnewsize = file_stat.st_size;
	    if (psnewsize == pssize) {
		unlink(infilenew);
	    } else {
		rename(infilenew, infile);
		if (verbose) {
		    printf("%s> escape sequences from %s removed\n",
			   argv[0], infile);
		}
	    }
	}
	
	/* calculate bounding box */
	sprintf(buffer, 
		"outfile=%s; "
		"export outfile; "
		"gs -q -sDEVICE=bit -dNOPAUSE ps2epsi.ps < %s 1>&2",
		outfile, infile);
	system(buffer);
	
	if ((stat(outfile, &file_stat) == -1) ||
	    (file_stat.st_size == 0)) {
	    fprintf(stderr, 
		    "Error while calling gs to calculate boundingbox on %s\n",
		    infile);
	    continue;
	}

	if (!(epsout = fopen(outfile, "r"))) {
	    fprintf(stderr, "Error opening file %s\n", outfile);
	    continue;
	}

	if (!(psin = fopen(infile, "r"))) {
	    fprintf(stderr, "Error opening file %s\n", infile);
	    fclose(epsout);
	    continue;
	}

	if (fgets(buffer, BUFFER_SIZE, epsout)) {
	    if (strcmp(buffer, adobe_epsf_tag)) {
		fprintf(stderr, "Error: generated invalid %s\n", outfile);
		fclose(psin);
		fclose(epsout);
		continue;
	    }
	}
	if (fgets(buffer, BUFFER_SIZE, epsout)) {
	    token = strtok(buffer, " \t\n");
	    if (strcmp(token, "%%BoundingBox:")) {
		fprintf(stderr, "Error: generated invalid %s\n", outfile);
		fclose(psin);
		fclose(epsout);
		continue;
	    }
	}
	if ((token = strtok(NULL, " \t\n"))) {
	    xlo = atoi(token);
	    if ((token = strtok(NULL, " \t\n"))) {
		ylo = atoi(token);
		if ((token = strtok(NULL, " \t\n"))) {
		    xhi = atoi(token);
		    if ((token = strtok(NULL, " \t\n"))) {
			yhi = atoi(token);
		    } else {
			missing = 1;
		    }
		} else {
		    missing = 2;
		}
	    } else {
		missing = 3;
	    }
	} else {
	    missing = 4;
	}
	
	if (missing) {
	    fprintf(stderr, 
		    " Error: BoundingBox is missing %d args in %s\n", 
		    missing, outfile);
	    fclose(psin);
	    fclose(epsout);
	    continue;
	}

	if (clip && (!border)) {
	    if (create_preview) {
		border_up_size = 0;
		border_down_size = 0;
		border_left_size = 0;
		border_right_size = 0;
	    } else {
		border_up_size = DEFAULT_BORDER;
		border_down_size = DEFAULT_BORDER;
		border_left_size = DEFAULT_BORDER;
		border_right_size = DEFAULT_BORDER;
	    }
	}

	/* update bounding box */
	xlo -= border_left_size;
	ylo -= border_down_size;
	xhi += border_right_size;
	yhi += border_up_size;

	if (verbose){
	    printf("%s> using BoundingBox %d %d %d %d for %s\n",
		   argv[0], xlo, ylo, xhi, yhi, outfile);
	}
	fclose(epsout);

	if (create_preview) {
	    /* use gs file header and preview image */
	    if (!(epsout = fopen(outfile, "r+"))) {
		fprintf(stderr, 
			"Error opening file %s for updating\n", outfile);
		continue;
	    }
	    while (!feof(epsout)) {
		if (fgets(buffer, BUFFER_SIZE, epsout)) {
		    if (strcmp(buffer, END_PREVIEW_TAG))
		      continue;
		    else {
			found_end_preview = 1;
			fflush(epsout);
			break;
		    }
		}
	    }
	    if (!found_end_preview) {
		fprintf(stderr, "Preview image not generated\n");
		fclose(epsout);
		/* generate a new eps file header (do the minimal thing) */
		if (!(epsout = fopen(outfile, "w"))) {
		    fprintf(stderr, 
			    "Error opening file %s for writing\n", outfile);
		    continue;
		}
		fputs(adobe_epsf_tag, epsout);
		fprintf(epsout, "%%%%BoundingBox: %d %d %d %d\n",
			xlo, ylo, xhi, yhi);
	    }
	} else {
	    /* generate a new eps file header */
	    if (!(epsout = fopen(outfile, "w"))) {
		fprintf(stderr, 
			"Error opening file %s for writing\n", outfile);
		continue;
	    }
	    fputs(adobe_epsf_tag, epsout);
	    fprintf(epsout, "%%%%BoundingBox: %d %d %d %d\n",
		    xlo, ylo, xhi, yhi);

	}
	/* add standard header */
	fprintf(epsout, 
		"%% Generated by ps2eps - %s (mhemy@cs.cmu.edu) from %s\n"
		"%% ps2eps: %s clipping and borders of u:%d d:%d l:%d r:%d\n",
		VERSION, infile, (clip? "with" : "without"), 
		border_up_size, border_down_size, 
		border_left_size, border_right_size);
	fprintf(epsout, 
		"/InitDictCount countdictstack def\n"
		"gsave\n"
		"save\n"
		"mark\n"
		"newpath\n"
		"/showpage {} def\n");
	
	/* copy over the first part of the psfile if clipping */
	if (clip) {
	    while (!feof(psin) && !didclip) {
		if (fgets(buffer, BUFFER_SIZE, psin)) {
		    for (tag=0; tag<num_tags;tag++) {
			if (!strncmp(buffer, insert_tags[tag], 
				     strlen(insert_tags[tag]))) {
			    /* insert clipping around the image */
			    fprintf(epsout, 
				    "%d %d moveto %d %d lineto %d %d "
				    "lineto %d %d lineto %d %d "
				    "clip newpath\n",
				    xlo, ylo, xhi, ylo, xhi, yhi, 
				    xlo, yhi, xlo, ylo);
			    didclip = 1;
			    break;
			}
		    }
		    fputs(buffer, epsout);
		}
	    }
	    if (verbose) {
		if (!didclip) {
		    printf("%s> unable to generate clip box in %s\n",
			   argv[0], outfile);
		} else {
		    printf("%s> generated clip box in %s\n",
			   argv[0], outfile);
		}
	    }
	}
	
	/* copy over the rest of the psfile */
	while (!feof(psin)) {
	    if (fgets(buffer, BUFFER_SIZE, psin)) {
		fputs(buffer, epsout);
	    }
	}    
	
	/* generate the eps file trailer */
	fprintf(epsout, 
		"countdictstack InitDictCount sub { end } repeat\n"
		"cleartomark\n"
		"restore\n"
		"grestore\n");
	
	/* nicely exit */
	fclose(psin);
	fclose(epsout);
    }
    exit(0);
}

