structure MarkAndSweepGC : GC = 
struct

  (* Raised to indicate that an allocation or dereference has failed (due to
  lack of resources (or breaking of another invariant) and a collection should
  be invoked. *) 
  exception OutOfMemory
  exception Error of string

  type location = int

  datatype sVal =
      Int of int
    | True 
    | False 
    | Nil 
    | Loc of location
    | Unit
  and lVal = 
      Closure of Env * DBMinML.exp
    | Cons of sVal * sVal
    | Pair of sVal * sVal
  and Bind =
      ValBind of sVal 
    | ExpBind of sVal
  
  withtype Env = Bind list
  
  datatype Frame =
      FPrimopN of P.primop * sVal list * DBMinML.exp list
    | If1 of DBMinML.exp * DBMinML.exp
    | App1 of DBMinML.exp 
    | App2 of sVal
    | Cons1 of DBMinML.exp 
    | Cons2 of sVal 
    | FPair1 of DBMinML.exp
    | FPair2 of sVal
    | FFst
    | FSnd
    | Case1 of DBMinML.exp * (DBMinML.bind * DBMinML.bind * DBMinML.exp)
    | Let1 of DBMinML.bind * DBMinML.exp
  
  datatype Cont =
      Frame of Frame
    | Env of Env
  
  type stack = Cont list
  
  datatype HeapElement = Free of int | Unmarked of lVal | Marked of lVal

  val (heap, next) = (ref (Array.tabulate (0,fn i => Free(i+1))), ref 0)

  val allocs = ref 0
  val reads = ref 0
  val collects = ref 0
  val accesses = ref 0

  (* Initialize the GC heap. *)
  fun init heap_size = 
    let in
      heap := Array.tabulate (heap_size,fn i => Free(i+1));
      next := 0
    end
  
  fun sweep (0) = ()
    | sweep (i) = sweep' (i-1)

  and sweep' (i) =
      (case Array.sub(!heap,i)
         of Unmarked(w) => (Array.update(!heap,i,Free(!next));
                            next := i; 
                            sweep (i))
          | Marked(w) => (Array.update(!heap,i,Unmarked(w));
                          sweep (i))
          | Free(w) => sweep (i))
  
  fun markStack (nil) = ()
    | markStack (Env(eta)::k) = (markEnv eta; markStack k)
    | markStack (Frame(FPrimopN(po,svl,el))::k) = markStack k
    | markStack (Frame(If1(e2,e3))::k) = markStack k
    | markStack (Frame(App1(e2))::k) = markStack k
    | markStack (Frame(App2(Loc(l)))::k) = (markPtr l; markStack k)
    | markStack (Frame(Cons1(e2))::k) = markStack k
    | markStack (Frame(Cons2(v1))::k) = (marksVal v1; markStack k)
    | markStack (Frame(Case1(e2,(x,y,e3)))::k) = markStack k
    | markStack (Frame(Let1(x,e2))::k) = markStack k
    | markStack (Frame(FPair1 e) :: k) = markStack k
    | markStack (Frame(FPair2 v) :: k) = (marksVal v; markStack k)
    | markStack (Frame(FFst) :: k) = markStack k
    | markStack (Frame(FSnd) :: k) = markStack k
    | markStack _ = raise Error "markStack reached a bad state"

  and markEnv nil = ()
    | markEnv (ExpBind(v)::eta) = (marksVal v; markEnv eta)
    | markEnv (ValBind(v)::eta) = (marksVal v; markEnv eta)

  and markPtr l =
      (case Array.sub(!heap,l)
         of Unmarked(w) => (Array.update(!heap,l,Marked(w));
                            marklVal w)
          | Marked(w) => ()
          | _ => raise Error "Reached a Free cell while marking")

  and marksVal (Loc(l)) = markPtr l
    | marksVal _ = ()

  and marklVal (Cons(v1,v2)) = (marksVal v1; marksVal v2)
    | marklVal (Closure(eta,e)) = markEnv eta
    | marklVal (Pair(v1,v2)) = (marksVal v1; marksVal v2)

  fun gc (k,w) = 
    let in
       (*print "Calling GC\n";*)
       marklVal w;
       markStack k;
       sweep (Array.length (!heap));
       if !next >= (Array.length (!heap))
          then raise OutOfMemory
       else ();
       (*print ("Returning from GC (collected: "^Int.toString (Array.foldr (fn (Free _,i) => i+1 | (_,i) => i) 0 heap)^")\n");*)
       ()
    end

  fun alloc k w =
      let
        val _ = if !next >= (Array.length (!heap)) then gc (k,w) else () (* could raise *)
        val l = !next
        val i = case Array.sub(!heap,l) of Free i => i | _ => raise Error "Free List corrupted."
        val _ = next := i
        val _ = Array.update(!heap, l, Unmarked(w))
      in l end
  
  fun extract(Unmarked(w)) = w
    | extract _ = raise Error "extracting marked or free cell"
  
  fun read l = 
    let in
      extract (Array.sub(!heap, l))
    end

  fun numAllocs () = !allocs

  fun numReads () = !reads

  fun numCollects () = !collects

  fun numAccesses () = !accesses

end;



