functor Eval (structure G : GC) =
struct

exception Error of string

structure DB = DBMinML

(* this should be split into two types, as
   indicated by the following datasort decl
*)
open G;

exception Lookup

fun toStringsVal (Int(n)) = Int.toString(n)
  | toStringsVal (True) = "True"
  | toStringsVal (False) = "False"
  | toStringsVal (Nil) = "Nil"
  | toStringsVal (Loc(r)) = "^" ^ toStringlVal (read r)
  | toStringsVal Unit = "()"

and toStringlVal (Cons(v1,v2)) = "Cons(" ^ toStringsVal v1 ^ "," ^ toStringsVal v2 ^ ")"
  | toStringlVal (Closure _) = "-"
  | toStringlVal (Pair(v1,v2)) = "("^toStringsVal v1^","^toStringsVal v2^")"

fun lookupEnv eta x = List.nth(eta,x-1) handle _ => raise Lookup

(* combine all the bindings into an environment *)
fun getEta (Frame _ :: k) = getEta k
  | getEta (Env eta :: k) = eta@(getEta k)
  | getEta [] = []

fun lookupStack (Frame _::k) x = lookupStack k x
  | lookupStack (Env(eta)::k) x = (lookupEnv eta x
                                  handle Lookup => (lookupStack k (x - length(eta))))
  | lookupStack _ _ = raise Error "no environment found"

fun lookupVal k x = case lookupStack k x of 
                      ValBind(v) => v
                    | _ => raise Error "lookupVal return non ValBind"

fun lookupExp k u = case lookupStack k u of 
                      ExpBind(e) => e
                    | _ => raise Error "lookupExp return non ExpBind"

fun closure (k, e) = Closure(DBUtil.trim(getEta k, e))

fun expOf (Int i) = DB.Int i
  | expOf True = DB.Bool true
  | expOf False = DB.Bool false
  | expOf Nil = DB.Nil(T.UNIT)
  | expOf Unit = DB.Unit
  | expOf _ = raise Error "Taking expOf Location"

fun printEnv eta = 
  let in
    case eta of
      ValBind v :: t => 
      	let in
      	  print "ValBind: ";
      	  print (toStringsVal v ^ "\n");
      	  printEnv t
      	end
    | ExpBind v :: t =>
      	let in
      	  print "ExpBind: ";
      	  print (toStringsVal v ^ "\n");
      	  printEnv t
      	end
    | [] => 
      	let in
      	  ()
      	end
  end

fun eval k (DB.Var(x)) = 
     let in
       case lookupStack k x of
         ValBind(v) => return k v
       | ExpBind(Loc l) =>
          let in 
            case read l of
               Closure(eta,e) => (eval (Env eta ::k) e handle err => raise err)
             | _ => raise Error "Exp Variable bound to non-Closure"
          end
       | _ => raise Error "Exp Variable doesn't contain a location"
     end
  | eval k (DB.Int(n)) = return k (Int(n))
  | eval k (DB.Primop(po,elist)) = eval (Frame(FPrimopN(po,[],tl elist))::k) (hd elist)
  | eval k (DB.Bool b) = return k (if b then True else False)
  | eval k (DB.If(e1,e2,e3)) = eval (Frame(If1(e2,e3))::k) e1
    (* implement for asst8 *)
  | eval k (DB.Fn(x,(_,e))) = raise Error "Unimplemented"
  | eval k (DB.Apply(e1,e2)) = eval (Frame(App1(e2))::k) e1
    (* implement for asst8 *)
  | eval k (DB.Nil(t)) = raise Error "Unimplemented"
  | eval k (DB.Cons(e1,e2)) = eval (Frame(Cons1(e2))::k) e1
  | eval k (DB.Case(e1,e2,(x,y,e3))) = eval (Frame(Case1(e2, (x,y,e3)))::k) e1
  | eval k (DB.Let(e1,(x,e2))) = eval (Frame(Let1(x,e2))::k) e1
  | eval k (DB.Rec(x,((),e))) = 
     let in
       eval (Env(
                 ExpBind(Loc(alloc k (closure(k,DB.Rec(x,((),e))))))::nil
               )::k) e
     end
  | eval k (DB.Unit) = return k Unit
  | eval k (DB.Pair(e1,e2)) = eval (Frame(FPair1 e2) :: k) e1
  | eval k (DB.Fst e) = eval (Frame(FFst) :: k) e
  | eval k (DB.Snd e) = eval (Frame(FSnd) :: k) e

and return nil v = v
  | return (Env(eta)::k) v = return k v
  | return (Frame(FPrimopN(po,svl,e::el))::k) v = 
     eval (Frame(FPrimopN(po,v::svl,el))::k) e
  | return (Frame(FPrimopN(po,svl,[]))::k) v = 
     return k (case DB.evalPrimop(po,rev(map expOf (v::svl))) of
              SOME(DB.Int i) => Int i
            | SOME(DB.Bool b) => if b then True else False
            | _ => raise Error "evalPrimop returned non-Int/Bool")
  | return (Frame(If1(e2,e3))::k) (True) = eval k e2
  | return (Frame(If1(e2,e3))::k) (False) = eval k e3
  | return (Frame(App1(e2))::k) v1 = eval (Frame(App2(v1))::k) e2
    (* implement for asst8 *)
  | return (Frame(App2(Loc(l)))::k) v2 = raise Error "Unimplemented"
  | return (Frame(Cons1(e2))::k) v1 = eval (Frame(Cons2(v1))::k) e2
    (* implement for asst8 *)
  | return (Frame(Cons2(v1))::k) v2 = raise Error "Unimplemented"
    (* implement for asst8 *)
  | return (Frame(Case1(e2,(x,y,e3)))::k) (Nil) = raise Error "Unimplemented"
    (* implement for asst8 *)
  | return (Frame(Case1(e2,(x,y,e3)))::k) (Loc l) = raise Error "Unimplemented"
  | return (Frame(Let1(x,e2))::k) v =
      eval (Env(ValBind(v)::nil)::k) e2
  | return (Frame(Case1(e2,(x,y,e3)))::k) v = raise Error (toStringsVal v)
  | return (Frame(FPair1 e2) :: k) v = eval (Frame (FPair2 v) :: k) e2
  | return (Frame(FPair2 v1) :: k) v2 = return k (Loc (alloc k (Pair(v1,v2))))
  | return (Frame(FFst) :: k) (Loc l) = 
    let in
      case read l of
        Pair(v1,v2) => return k v1
      | v => raise Error ("attempting to project from non-pair: "^toStringlVal v)
    end
  | return (Frame(FSnd) :: k) (Loc l) = 
    let in
      case read l of
        Pair(v1,v2) => return k v2
      | v => raise Error ("attempting to project from non-pair: "^toStringlVal v)
    end
  | return (Frame(App2(v))::k) v2 = raise Error ("sval: "^toStringsVal v)
  | return (Frame(FFst) :: k) v = raise Error ("sval: "^toStringsVal v)

fun evaluate heapSize e = 
  let in
    init heapSize;
    eval [Env []] e
  end

end
