(* Type checking MinML *)
(* Operates on de Bruijn form of expressions *)

signature TYPING =
sig

  type typ = DBMinML.T.typ

  exception Error of string

  val typecheck : DBMinML.exp -> typ * DBMinML.exp

  val typeOpt : DBMinML.exp -> typ option

end;  (* signature TYPING *)

structure Typing :> TYPING =
struct

  exception Error of string
  structure M = DBMinML
  structure T = M.T
  structure P = M.P
  type typ = T.typ

  (* walks through a type and increments the free variables by n
     (number of bindings walked through.
     shift 0 n t : is the typical way this should be called *)
  fun shift thr n (T.VAR i) = if i > thr then T.VAR(i+n) else T.VAR(i)
    | shift thr n (T.CROSS(t1,t2)) = T.CROSS(shift thr n t1, shift thr n t2)
    | shift thr n (T.SUM(t1,t2)) = T.SUM(shift thr n t1, shift thr n t2)
    | shift thr n (T.ARROW(t1,t2)) = T.ARROW(shift thr n t1, shift thr n t2)
    | shift thr n (T.MU(b,t)) = T.MU(b, shift (thr+1) (n+1) t)
    | shift thr n (T.ALL(b,t)) = T.ALL(b, shift (thr+1) (n+1) t)
    | shift thr n (T.EXISTS(b,t)) = T.EXISTS(b, shift (thr+1) (n+1) t)
    | shift thr n t = t

  (* substitutes t1 for x in t2 (having been through n binders),
     typically called as: tsub t1 tvar t2 0 *)
  fun tsub t1 x t2 n = 
    let in
      case t2 of
        T.CROSS(s1,s2) => T.CROSS(tsub t1 x s1 n, tsub t1 x s2 n)
      | T.SUM(s1,s2) => T.SUM(tsub t1 x s1 n, tsub t1 x s2 n)
      | T.ARROW(s1,s2) => T.ARROW(tsub t1 x s1 n, tsub t1 x s2 n)
      | T.MU(b,s) => T.MU(b,tsub t1 (x+1) s (n+1))
      | T.ALL(b,s) => T.ALL(b,tsub t1 (x+1) s (n+1))
      | T.EXISTS(b,s) => T.EXISTS(b,tsub t1 (x+1) s (n+1))
      | T.VAR y => if x = y then shift 0 n t1 else T.VAR y
      | t => t
    end

  (* substitutes the type v for x in the DBMinML.exp, having seen
     n binders.  This should only be needed for lettype, also be
     careful, this should be called as: subst_tp typ tvar exp 1
     Notice this has a 1, because we are inside the scope of the
     lettype binder now.
     *)
  fun subst_tp v x (M.If (e, e1, e2)) n = M.If (subst_tp v x e n,
                                       subst_tp v x e1 n,
                                       subst_tp v x e2 n)
    | subst_tp v x (M.Primop (primop, elist)) n =
      M.Primop (primop, map (fn y => (subst_tp v x y n)) elist)
    | subst_tp v x (M.Fn (b, e)) n =
      M.Fn (b, subst_tp v (x + 1) e (n+1))
    | subst_tp v x (M.Rec (b, e)) n =
      M.Rec (b, subst_tp v (x + 1) e (n+1))
    | subst_tp v x (M.Let (e1, (b1, e2))) n =
      M.Let (subst_tp v x e1 n, (b1, subst_tp v (x + 1) e2 (n+1)))
    | subst_tp v x (M.LetType (t, (b1, e))) n =
      M.LetType (tsub v x t n, (b1, subst_tp v (x + 1) e (n+1)))
    | subst_tp v x (M.Apply (e1, e2)) n = M.Apply (subst_tp v x e1 n, subst_tp v x e2 n)
    | subst_tp v x (M.Inl e) n = M.Inl(subst_tp v x e n)
    | subst_tp v x (M.Inr e) n = M.Inr(subst_tp v x e n)
    | subst_tp v x (M.Roll e) n = M.Roll(subst_tp v x e n)
    | subst_tp v x (M.Pack(t,e)) n = M.Pack(tsub v x t n, subst_tp v x e n)
    | subst_tp v x (M.Pair(e1,e2)) n = M.Pair(subst_tp v x e1 n, subst_tp v x e2 n)
    | subst_tp v x (M.TFn(b,e)) n = M.TFn(b,subst_tp v (x+1) e (n+1))
    | subst_tp v x (M.Case(e,(b1,e1),(b2,e2))) n = M.Case(subst_tp v x e n, (b1, subst_tp v (x+1) e1 (n+1)), (b2, subst_tp v (x+1) e2 (n+1)))
    | subst_tp v x (M.Open(e,(b1,b2,e1))) n = M.Open(subst_tp v x e n, (b1, b2, subst_tp v (x+2) e1 (n+2)))
    | subst_tp v x (M.Inst (e,t)) n = M.Inst(subst_tp v x e n, tsub v x t n)
    | subst_tp v x (M.Unroll e) n = M.Unroll(subst_tp v x e n)
    | subst_tp v x (M.Fst e) n = M.Fst (subst_tp v x e n)
    | subst_tp v x (M.Itof e) n = M.Itof (subst_tp v x e n)
    | subst_tp v x (M.Snd e) n = M.Snd (subst_tp v x e n)
    | subst_tp v x (M.Abort e) n = M.Abort (subst_tp v x e n)
    | subst_tp v x (M.Annotate (e,t)) n = M.Annotate (subst_tp v x e n, tsub v x t n)
    | subst_tp v x e n = e

  infix 1 ++
  fun (ctx ++ x') = x'::ctx

  fun lookup x ctx = List.nth(ctx,x-1)

  fun typecheck e = raise Error "Unimplemented typecheck"

  fun typeOpt e = raise Error "Unimplemented typeOpt"

end;  (* structure TypingSolution *)
