(* Parser for MinML *)
(* Generates abstract syntax in named form *)
(* Written as a transducer from token streams to expression streams *)

signature PARSE =
sig
  val parse : Lexer.token Stream.stream -> MinML.exp Stream.stream

  exception Error of string
end;  (* signature PARSE *)

structure Parse :> PARSE =
struct
    
    structure S = Stream
    structure M = MinML
    structure L = Lexer

    exception Error of string

    (* next s = (x,s'), where x is the head of s, s' the tail of s raises
       Error if stream is empty *)
    fun next s =
        case S.force s
          of S.Nil => raise Error "Unexpected end of stream"
           | S.Cons result => result

    fun printStrm st = 
      (case next st of
        (tok, s) => (print ((L.tokenToString tok) ^ "\n"); printStrm s) handle _ => ())

    (* match tok s = s', s' is the tail of s raises Error if head of s does
       not match tok *)
    fun match tok s =
        let
	  val (n, s') = next s
        in
	  if L.eq(tok,n) then s'
	  else raise Error ("Expected " ^ Lexer.tokenToString tok ^ " token")
        end

    fun build_primop ((primop, e), e') =
          M.Primop(primop, [e', e])

    (* build_primops: exp -> (primop * exp) list -> exp *)
    fun build_primops exp op_exps =
           foldl build_primop exp op_exps

    (* parse_program r = (e,s') where e is the result of parsing the beginning
       of r and s' the unprocessed tail of r *)
    fun parse_program r =
        let
            val (e, s) = parse_exp (S.delay (fn () => S.Cons r))
        in
            (e, match L.SEMICOLON s)
        end

    (* parse_factors: Recursively consume adjacent atomic factors
       (parse_factora), forming them into a chain of applications. *)
    and parse_factors s eo =
	case parse_factor_maybe s of
	    SOME (e, s) => (case eo of
				NONE => parse_factors s (SOME e)
			      | SOME f => parse_factors s (SOME (M.Apply(f, e))))
	  | NONE => (case eo of
			 NONE => 
			   let in
			     (*printStrm s;*)
			     raise Error ("Expected expression")
			   end
		       | SOME e => (e, s))

    and parse_factor es = parse_factors es NONE

    (* parse_factora (t,s) attempts to find an atomic expression (no
       applications) starting with the token t, perhaps continuing through the
       stream.  Returns SOME(e, s) if the exp e was successfully recognized,
       with s the stream remaining after it.  Returns NONE if the token cannot
       begin any exp.  May raise exception Error if the input stream does not
       represent any valid MinML program. *)

    (* parse_factora : L.token * L.token S.stream -> 
                       (M.exp * L.token S.stream) option *)
    and parse_factora (L.TRUE, s) = SOME (M.Bool true, s)
      | parse_factora (L.FALSE, s) = SOME (M.Bool false, s)
      | parse_factora (L.NUMBER n, s) = SOME (M.Int n, s)
      | parse_factora (L.FLT r, s) = SOME (M.Float r, s)
      | parse_factora (L.VAR v, s) = SOME (M.Var v, s)
      | parse_factora (L.IF, s) =
          let
              val (ec, s) = parse_exp s
              val s = match L.THEN s
              val (et, s) = parse_exp s
              val s = match L.ELSE s
              val (ef, s) = parse_exp s
              val s = match L.FI s
          in
              SOME (M.If(ec, et, ef), s)
          end
      | parse_factora (L.LPAREN, s) =
        (case next s of
          (L.RPAREN, s') => SOME(M.Unit, s')
        | _ => 
          let
              val (e, s) = parse_exp s
          in
             case next s of
               (L.COLON,s') => 
                 let
                   val (tp,s'') = parse_type (next s')
                 in
                   SOME(M.Annotate(e,tp), match L.RPAREN s'')
                 end
             | (L.COMMA,s') => 
                 let
                   val (e',s'') = parse_exp s'
                 in
                   SOME(M.Pair(e,e'), match L.RPAREN s'')
                 end
             | _ => SOME (e, match L.RPAREN s)
          end)
      | parse_factora (L.LET, s) =
          let
              val (x, s) = parse_var (next s)
              val s = match L.EQUALS s
              val (e1, s) = parse_exp s
              val s = match L.IN s
              val (e2, s) = parse_exp s
          in
              SOME (M.Let(e1, (x,e2)), match L.END s)
          end
      | parse_factora (L.LETTYPE, s) =
          let
              val (x, s) = parse_var (next s)
              val s = match L.EQUALS s
              val (t, s) = parse_type (next s)
              val s = match L.IN s
              val (e, s) = parse_exp s
          in
              SOME (M.LetType(t, (x,e)), match L.END s)
          end
      | parse_factora (L.ROLL, s) =
          let
              val (e,s) = parse_exp s
          in
            SOME(M.Roll e,s)
          end

      | parse_factora (L.PACK, s) =
          let
             val s = match L.LPAREN s
             val (t,s) = parse_type (next s)
             val s = match L.COMMA s
             val (e,s) = parse_exp s
             val s = match L.RPAREN s
          in
            SOME(M.Pack(t,e), s)
          end

      | parse_factora (L.UNROLL, s) =
          let
             val (e,s) = parse_exp s
          in
            SOME(M.Unroll e, s)
          end

      | parse_factora (L.INL, s) =
          let
             val (e,s) = parse_exp s
          in
            SOME(M.Inl e, s)
          end

      | parse_factora (L.INR, s) =
          let
             val (e,s) = parse_exp s
          in
            SOME(M.Inr e, s)
          end

      | parse_factora (L.AFN, s) =
          let
            val (x,s) = parse_var (next s)
            val s = match L.DARROW s
            val (e,s) = parse_exp s
          in
            SOME(M.TFn(x,e),s)
          end

      | parse_factora (L.CASE, s) =
          let
            val (e,s) = parse_exp s
            val s = match L.OF s
            val s = match L.INL s
            val s = match L.LPAREN s
            val (x,s) = parse_var (next s)
            val s = match L.RPAREN s
            val s = match L.DARROW s
            val (e1,s) = parse_exp s
            val s = match L.BAR s
            val s = match L.INR s
            val s = match L.LPAREN s
            val (y,s) = parse_var (next s)
            val s = match L.RPAREN s
            val s = match L.DARROW s
            val (e2,s) = parse_exp s
          in
            SOME(M.Case(e,(x,e1),(y,e2)), s)
          end

      | parse_factora (L.SQRT, s) =
          let in
            case parse_factora (next s) of
              SOME(e,s) =>
                 SOME(M.Primop(P.Sqrt, [e]), s)
            | NONE => raise Error "Expected Factor"
          end

      | parse_factora (L.FST, s) =
          let
             val (e,s) = parse_exp s
          in
            SOME(M.Fst e, s)
          end

      | parse_factora (L.ITOF, s) =
          let
             val (e,s) = parse_exp s
          in
            SOME(M.Itof e, s)
          end

      | parse_factora (L.SND, s) =
          let
             val (e,s) = parse_exp s
          in
            SOME(M.Snd e, s)
          end

      | parse_factora (L.ABORT, s) =
          let
             val (e,s) = parse_exp s
          in
            SOME(M.Abort e, s)
          end

      | parse_factora (L.OPEN, s) =
          let
            val s = match L.LPAREN s
            val (x,s) = parse_var (next s)
            val s = match L.COMMA s
            val (y,s) = parse_var (next s)
            val s = match L.RPAREN s
            val s = match L.EQUALS s
            val (e1,s) = parse_exp s
            val s = match L.IN s
            val (e2,s) = parse_exp s
          in
            SOME(M.Open(e1,(x,y,e2)), s)
          end

      | parse_factora (L.FN, s) =
          let
              val (x, s) = parse_var (next s)
              val s = match L.DARROW s
              val (e, s) = parse_exp s
          in
              SOME (M.Fn(x, e), s)
          end
      | parse_factora (L.REC, s) =
          let
              val (x, s) = parse_var (next s)
              val s = match L.DARROW s
              val (e, s) = parse_exp s
          in
              SOME (M.Rec(x, e), s)
          end
      | parse_factora (L.FUN, s) =
          let
              val (f, s) = parse_var (next s)
              val s = match L.LPAREN s
              val (x, s) = parse_var (next s)
              val s = match L.RPAREN s
              val s = match L.DARROW s
              val (e, s) = parse_exp s
          in
            SOME (M.Rec(f, M.Fn(x, e)), s)
          end
      | parse_factora (p, s) =
          case p of
              L.NEGATE =>
		 (case parse_factora (next s) of
		      SOME (operand, s) => SOME (M.Primop (P.Negate, 
                                                           [operand]), s)
		    | NONE => NONE)
            | _ => NONE
         
    (* XXX necessary? *)
    and parse_factor_maybe s = case S.force s of
        S.Nil => NONE
      | S.Cons res => parse_factora res
    
    (* parse_exp_aux : (primop * exp) list -> stream -> 
                       (primop * exp) list * stream *)
    and parse_exp_aux acc s =
	let val relop =
	    case next s of
		(L.EQUALS, s) => SOME (P.Equals, s)
	      | (L.LESSTHAN, s) => SOME (P.LessThan, s)
	      | _  => NONE
	in
	    case relop of
		SOME (relop, s) => 
		    let val (e, s) = parse_exp' s
		    in
			parse_exp_aux (acc @ [(relop, e)]) s
		    end
	      | NONE => (acc, s)  (* No more exp-primes; return what we have 
                                     so far. *)
	end
    
    and parse_exp es =
	let val (e, s) = parse_exp' es
	    val (exp's, s) = parse_exp_aux [] s
        in
            case next s of
              (L.LBRACK, s) =>
                let
                  val (t,s) = parse_type (next s)
                in
                  (M.Inst(build_primops e exp's, t), match L.RBRACK s)
                end
            | _ => (build_primops e exp's, s)
        end

    (* parse_exp'_aux : (primop * exp) list -> stream -> 
                        (primop * exp) list * stream *)
    and parse_exp'_aux acc s =
	let val addop =
	    case next s of
		(L.PLUS, s) => 
		  let in
		    case next s of
		      (L.DOT, s) => SOME(P.FPlus, s)
		    | _          => SOME(P.Plus, s)
		  end
	      | (L.MINUS, s) => SOME (P.Minus, s)
	      | _  => NONE
	in
	    case addop of
		SOME (addop, s) => 
		    let val (e, s) = parse_term s
		    in
			parse_exp'_aux (acc @ [(addop, e)]) s
		    end
	      | NONE => (acc, s)  (* No more terms; return what we have 
                                     so far *)
	end

    and parse_exp' es =
	let val (e, s) = parse_term es
	    val (terms, s) = parse_exp'_aux [] s
        in
	    (build_primops e terms, s)
        end

    (* parse_term_aux : (primop * exp) list -> stream -> 
                        (primop * exp) list * stream *)
    and parse_term_aux acc s =
	let val mulop =
	    case next s of
		(L.TIMES, s) => 
		  let in
		    case next s of
		      (L.DOT, s) => SOME(P.FTimes, s)
		    | _          => SOME(P.Times, s)
		  end
              | (L.PERCENT, s) => SOME (P.Mod, s)
	      | _  => NONE
	in
	    case mulop of
		SOME (mulop, s) => 
		    let val (e, s) = parse_factor s
		    in
			parse_term_aux (acc @ [(mulop, e)]) s
		    end
	      | NONE => (acc, s)  (* No more factors; return what we have 
                                     so far. *)
	end

    and parse_term es =
	let val (e, s) = parse_factor es
	    val (factors, s) = parse_term_aux [] s
        in
	    (build_primops e factors, s)
        end

    and parse_var (L.VAR v, s) = (v, s)
      | parse_var _ = raise Error "Expected var"

    and parse_basetype (L.INT, s) = (T.INT, s)
      | parse_basetype (L.FLOAT, s) = (T.FLOAT, s)
      | parse_basetype (L.UNIT, s) = (T.UNIT, s)
      | parse_basetype (L.VOID, s) = (T.VOID, s)
      | parse_basetype (L.BOOL, s) = (T.BOOL, s)
      | parse_basetype (L.VAR(x), s) = (T.VAR(x), s)
      | parse_basetype (L.LPAREN, s) =
	let
	  val (t, s) = parse_type (next s)
	  val s = match L.RPAREN s
	in
	  (t, s)
	end
      | parse_basetype (_,s) = ((*printStrm s;*)raise Error "Expected type")

    and parse_type (L.MU, s) =
        let
          val (x, s) = parse_var (next s)
          val s = match L.DOT s
          val (tp, s) = parse_type (next s)
        in
          (T.MU(x,tp), s)
        end
      | parse_type (L.ALL, s) = 
        let
          val (x, s) = parse_var (next s)
          val s = match L.DOT s
          val (tp, s) = parse_type (next s)
        in
          (T.ALL(x,tp), s)
        end
      | parse_type (L.EXISTS, s) = 
        let
          val (x, s) = parse_var (next s)
          val s = match L.DOT s
          val (tp, s) = parse_type (next s)
        in
          (T.EXISTS(x,tp), s)
        end
      | parse_type (tok, s) = 
        let
	  val (domain, s) = parse_basetype (tok, s)
        in
	  case next s of 
	     (L.ARROW, s) =>
	       let 
		 val (range, s') = parse_type (next s)
	       in
		 (T.ARROW(domain, range), s')
	       end
	   | (L.PLUS, s) =>
	       let 
		 val (range, s') = parse_type (next s)
	       in
		 (T.SUM(domain, range), s')
	       end
	   | (L.TIMES, s) =>
	       let 
		 val (range, s') = parse_type (next s)
	       in
		 (T.CROSS(domain, range), s')
	       end
	   | _ => (domain, s)
        end

    (* exported parsing function *)
    fun parse (s : L.token S.stream) : M.exp S.stream = 
        S.delay (fn () => parse' (S.force s))

    and parse' S.Nil = S.Nil
      | parse' (S.Cons r) = 
        let 
            val (e, s) = parse_program r
        in 
            S.Cons (e, parse s)
        end

end;  (* structure Parse *)
