(* MinML Abstract Syntax *)
(* Given in named form and deBruijn form without variable names *)

(* Types and primitive operations *)
signature TYPE =
sig
  type var
  type bind

  (* MinML types *)
  datatype typ =
      INT
    | FLOAT
    | VAR of var
    | UNIT
    | VOID
    | BOOL
    | CROSS of typ * typ
    | SUM of typ * typ
    | MU of bind * typ
    | ALL of bind * typ
    | EXISTS of bind * typ
    | ARROW of typ * typ

  val eq : typ * typ -> bool
end;  (* signature TP *)

signature PRIMOP =
sig  
  type typ

  (* Primitive operations *)
  datatype primop = Equals | LessThan | Plus | Minus | Times | Negate | Mod | Sqrt | FPlus | FTimes

  (* given a primop, get its argument and result types *)
  val typeOfPrimop : primop -> typ list * typ

end;  (* signature PRIMOP *)

(* MinML *)
(* Shared between representations *)
signature MINML =
sig

  (* val varEqual : var * var -> bool *)

  type bind
  type var

  structure T : TYPE
  structure P : PRIMOP

  sharing type T.bind = bind
  and type T.var = var
  and type P.typ = T.typ

  (* Expression *)
  (* Binders are grouped with their scope *)
  datatype exp =
      Int of int			(* k *)
    | Bool of bool			(* true or false *)
    | If of exp * exp * exp		(* if e then e1 else e2 fi *)
    | Primop of P.primop * exp list	(* e1 <op> e2  or  <op> e *)
    | Fn of bind * exp                  (* fn x:t = e *)
    | Rec of bind * exp                 (* rec x:t = e *)
    | Let of exp * (bind * exp)		(* let x = e1 in e2 end *)
    | LetType of T.typ * (bind * exp)	(* lettype t = typ in e end *)
    | Apply of exp * exp		(* e1 e2 *)
    | Var of var			(* x *)

    | Float of real			(* k *)
    | Unit
    | Inl of exp
    | Inr of exp
    | Roll of exp
    | Pack of T.typ * exp
    | Pair of exp * exp
    | TFn of bind * exp
    | Itof of exp

    | Case of exp * (bind * exp) * (bind * exp)
    | Open of exp * (bind * bind * exp)
    | Inst of exp * T.typ
    | Unroll of exp
    | Fst of exp
    | Snd of exp
    | Abort of exp

    | Annotate of exp * T.typ		(* e : t *)


  (* given a primop and some arguments, try to apply it *)
  val evalPrimop : P.primop * exp list -> exp option

end;  (* signature MINML *)

(* Named representation *)
signature NAMEDMINML =
  MINML where type bind = string and type var = string;

(* de Bruijn representation *)
signature DBMINML =
  MINML where type bind = unit and type var = int;


(* MinML *)
(* Shared between representations *)
functor MinML
   (eqtype bind
    eqtype var)
 :> MINML where type bind = bind and type var = var
=
struct

  (* Variables *)
  type bind = bind
  type var = var

  structure T : TYPE = 
   struct

    type bind = bind
    type var = var

    (* MinML types *)
    datatype typ =
      INT
    | FLOAT
    | VAR of var
    | UNIT
    | VOID
    | BOOL
    | CROSS of typ * typ
    | SUM of typ * typ
    | MU of bind * typ
    | ALL of bind * typ
    | EXISTS of bind * typ
    | ARROW of typ * typ
    
    fun eq (t1,t2) = t1=t2
   end;  (* structure T *)

  structure P : PRIMOP = 
   struct

    type typ = T.typ

    (* Primitive operations *)
    datatype primop = Equals | LessThan | Plus | Minus | Times | Negate | Mod | Sqrt | FPlus | FTimes

    fun typeOfPrimop Equals  = ([T.INT, T.INT], T.BOOL)
      | typeOfPrimop LessThan  = ([T.INT, T.INT], T.BOOL)
      | typeOfPrimop Plus   = ([T.INT, T.INT], T.INT)
      | typeOfPrimop Minus  = ([T.INT, T.INT], T.INT)
      | typeOfPrimop Times  = ([T.INT, T.INT], T.INT)
      | typeOfPrimop Negate = ([T.INT], T.INT)
      | typeOfPrimop Mod = ([T.INT, T.INT], T.INT)
      | typeOfPrimop Sqrt = ([T.FLOAT], T.FLOAT)
      | typeOfPrimop FPlus = ([T.FLOAT, T.FLOAT], T.FLOAT)
      | typeOfPrimop FTimes = ([T.FLOAT, T.FLOAT], T.FLOAT)
   end;  (* structure P *)


  (* Expressions *)
  (* Binders are grouped with their scope *)
  datatype exp =
      Int of int			(* k *)
    | Bool of bool			(* true or false *)
    | If of exp * exp * exp		(* if e then e1 else e2 fi *)
    | Primop of P.primop * exp list	(* e1 <op> e2  or  <op> e *)
    | Fn of bind * exp                  (* fn x:t => e *)
    | Rec of bind * exp                 (* rec x:t => e *)
    | Let of exp * (bind * exp)		(* let x = e1 in e2 end *)
    | LetType of T.typ * (bind * exp)	(* lettype t = typ in e end *)
    | Apply of exp * exp		(* e1 e2 *)
    | Var of var			(* x *)

    | Float of real			(* k *)
    | Unit
    | Inl of exp
    | Inr of exp
    | Roll of exp
    | Pack of T.typ * exp
    | Pair of exp * exp
    | TFn of bind * exp
    | Itof of exp

    | Case of exp * (bind * exp) * (bind * exp)
    | Open of exp * (bind * bind * exp)
    | Inst of exp * T.typ
    | Unroll of exp
    | Fst of exp
    | Snd of exp
    | Abort of exp

    | Annotate of exp * T.typ		(* e : t *)


  (* Evaluation of primops on evaluated arguments *)
  fun evalPrimop (P.Equals, [Int i, Int i']) = SOME (Bool (i = i'))
    | evalPrimop (P.LessThan, [Int i, Int i']) = SOME (Bool (i < i'))
    | evalPrimop (P.Plus, [Int i, Int i']) = SOME (Int (i + i'))
    | evalPrimop (P.Minus, [Int i, Int i']) = SOME (Int (i - i'))
    | evalPrimop (P.Times, [Int i, Int i']) = SOME (Int (i * i'))
    | evalPrimop (P.Negate, [Int i]) = SOME (Int (~i))
    | evalPrimop (P.Mod, [Int i, Int i']) = SOME (Int (i mod i'))
    | evalPrimop (P.Sqrt, [Float r]) = SOME (Float (Math.sqrt r))
    | evalPrimop (P.FPlus, [Float r, Float r']) = SOME (Float (r + r'))
    | evalPrimop (P.FTimes, [Float r, Float r']) = SOME (Float (r * r'))
    | evalPrimop _ = NONE

end;  (* functor MinML *)

structure MinML = MinML (type bind = string
                         type var = string);
structure DBMinML = MinML (type bind = unit
                           type var = int);

structure T = MinML.T

structure P = MinML.P


