(* Evaluation of MinML Expression *)
(* Uses de Bruijn representation *)

signature EVAL =
sig
  exception Stuck of string

  val eval : DBMinML.exp -> DBMinML.exp

end;  (* signature EVAL *)

structure Eval :> EVAL =
struct

  structure T = DBMinML.T

  open T
  open P
  open DBMinML
  structure S = Stream

  exception Stuck of string
  exception Done

  (* walks through a type and increments the free variables by n
     (number of bindings walked through.
     shift 0 n t : is the typical way this should be called *)
  fun shift thr n (T.VAR i) = if i > thr then T.VAR(i+n) else T.VAR(i)
    | shift thr n (T.CROSS(t1,t2)) = T.CROSS(shift thr n t1, shift thr n t2)
    | shift thr n (T.SUM(t1,t2)) = T.SUM(shift thr n t1, shift thr n t2)
    | shift thr n (T.ARROW(t1,t2)) = T.ARROW(shift thr n t1, shift thr n t2)
    | shift thr n (T.MU(b,t)) = T.MU(b, shift (thr+1) (n+1) t)
    | shift thr n (T.ALL(b,t)) = T.ALL(b, shift (thr+1) (n+1) t)
    | shift thr n (T.EXISTS(b,t)) = T.EXISTS(b, shift (thr+1) (n+1) t)
    | shift thr n t = t

  (* substitutes t1 for x in t2 (having been through n binders),
     typically called as: tsub t1 tvar t2 0 *)
  fun tsub t1 x t2 n = 
    let in
      case t2 of
        T.CROSS(s1,s2) => T.CROSS(tsub t1 x s1 n, tsub t1 x s2 n)
      | T.SUM(s1,s2) => T.SUM(tsub t1 x s1 n, tsub t1 x s2 n)
      | T.ARROW(s1,s2) => T.ARROW(tsub t1 x s1 n, tsub t1 x s2 n)
      | T.MU(b,s) => T.MU(b,tsub t1 (x+1) s (n+1))
      | T.ALL(b,s) => T.ALL(b,tsub t1 (x+1) s (n+1))
      | T.EXISTS(b,s) => T.EXISTS(b,tsub t1 (x+1) s (n+1))
      | T.VAR y => if x = y then shift 0 n t1 else T.VAR y
      | t => t
    end

  (* substitutes the type v for x in the DBMinML.exp, having seen
     n binders.  This should only be needed for lettype, also be
     careful, this should be called as: subst_tp typ tvar exp 1
     Notice this has a 1, because we are inside the scope of the
     lettype binder now.
     *)
  fun subst_tp v x (If (e, e1, e2)) n = If (subst_tp v x e n,
                                       subst_tp v x e1 n,
                                       subst_tp v x e2 n)
    | subst_tp v x (Primop (primop, elist)) n =
      Primop (primop, map (fn y => (subst_tp v x y n)) elist)
    | subst_tp v x (Fn (b, e)) n =
      Fn (b, subst_tp v (x + 1) e (n+1))
    | subst_tp v x (Rec (b, e)) n =
      Rec (b, subst_tp v (x + 1) e (n+1))
    | subst_tp v x (Let (e1, (b1, e2))) n =
      Let (subst_tp v x e1 n, (b1, subst_tp v (x + 1) e2 (n+1)))
    | subst_tp v x (LetType (t, (b1, e))) n =
      LetType (tsub v x t n, (b1, subst_tp v (x + 1) e (n+1)))
    | subst_tp v x (Apply (e1, e2)) n = Apply (subst_tp v x e1 n, subst_tp v x e2 n)
    | subst_tp v x (Inl e) n = Inl(subst_tp v x e n)
    | subst_tp v x (Inr e) n = Inr(subst_tp v x e n)
    | subst_tp v x (Roll e) n = Roll(subst_tp v x e n)
    | subst_tp v x (Pack(t,e)) n = Pack(tsub v x t n, subst_tp v x e n)
    | subst_tp v x (Pair(e1,e2)) n = Pair(subst_tp v x e1 n, subst_tp v x e2 n)
    | subst_tp v x (TFn(b,e)) n = TFn(b,subst_tp v (x+1) e (n+1))
    | subst_tp v x (Case(e,(b1,e1),(b2,e2))) n = Case(subst_tp v x e n, (b1, subst_tp v (x+1) e1 (n+1)), (b2, subst_tp v (x+1) e2 (n+1)))
    | subst_tp v x (Open(e,(b1,b2,e1))) n = Open(subst_tp v x e n, (b1, b2, subst_tp v (x+2) e1 (n+2)))
    | subst_tp v x (Inst (e,t)) n = Inst(subst_tp v x e n, tsub v x t n)
    | subst_tp v x (Unroll e) n = Unroll(subst_tp v x e n)
    | subst_tp v x (Fst e) n = Fst (subst_tp v x e n)
    | subst_tp v x (Itof e) n = Itof (subst_tp v x e n)
    | subst_tp v x (Snd e) n = Snd (subst_tp v x e n)
    | subst_tp v x (Abort e) n = Abort (subst_tp v x e n)
    | subst_tp v x (Annotate (e,t)) n = Annotate (subst_tp v x e n, tsub v x t n)
    | subst_tp v x e n = e

  (* Substitute v for x in e. *)
  (* subst : exp -> var -> exp -> exp *)
  fun subst v x (If (e, e1, e2)) = If (subst v x e,
                                       subst v x e1,
                                       subst v x e2)
    | subst v x (Var x1) = if x = x1 then v else Var x1
    | subst v x (Primop (primop, elist)) =
      Primop (primop, map (subst v x) elist)
    | subst v x (Fn (b, e)) =
      Fn (b, subst v (x + 1) e)
    | subst v x (Rec (b, e)) =
      Rec (b, subst v (x + 1) e)
    | subst v x (Let (e1, (b1, e2))) =
      Let (subst v x e1, (b1, subst v (x + 1) e2))
    | subst v x (LetType (t, (b1, e))) =
      LetType (t, (b1, subst v (x + 1) e))
    | subst v x (Apply (e1, e2)) = Apply (subst v x e1, subst v x e2)
    | subst v x (Inl e) = Inl(subst v x e)
    | subst v x (Inr e) = Inr(subst v x e)
    | subst v x (Roll e) = Roll(subst v x e)
    | subst v x (Pack(t,e)) = Pack(t,subst v x e)
    | subst v x (Pair(e1,e2)) = Pair(subst v x e1, subst v x e2)
    | subst v x (TFn(b,e)) = TFn(b,subst v (x+1) e)
    | subst v x (Case(e,(b1,e1),(b2,e2))) = Case(subst v x e, (b1, subst v (x+1) e1), (b2, subst v (x+1) e2))
    | subst v x (Open(e,(b1,b2,e1))) = Open(subst v x e, (b1, b2, subst v (x+2) e1))
    | subst v x (Inst (e,t)) = Inst(subst v x e, t)
    | subst v x (Unroll e) = Unroll(subst v x e)
    | subst v x (Fst e) = Fst (subst v x e)
    | subst v x (Itof e) = Itof (subst v x e)
    | subst v x (Snd e) = Snd (subst v x e)
    | subst v x (Abort e) = Abort (subst v x e)
    | subst v x (Annotate (e,t)) = Annotate (subst v x e, t)
    | subst v x e = e

  fun eval (Int i) = Int i
    | eval (Bool b) = Bool b
    | eval (If (e, e1, e2)) =
        (case eval e
          of Bool b => if b then eval e1 else eval e2
           | _ => raise Stuck "If condition not a bool")
    | eval (Primop (primop, elist)) =
      let 
        val elist' = map eval elist
        val e = Primop(primop,elist)
      in
        case evalPrimop (primop, elist') of 
          SOME e => e
        | NONE => raise Stuck ("Primop: " ^ DBPrint.expToString e ^ " failed to evaluate")
      end 
    | eval (Fn x) = (Fn x)
    | eval (Rec (b, e)) = eval (subst (Rec (b, e)) 1 e)
    | eval (Let (e1, (b, e2))) =
       eval(subst (eval e1) 1 e2)
    | eval (LetType (t, (b, e))) =
       eval(subst_tp t 1 e 1)
    | eval (Apply (e1, e2)) =
       (case eval e1 
         of Fn (_, e11) => eval (subst (eval e2) 1 e11)
             | _ => raise Stuck "Application of non-function")
    | eval (Var x) = raise Stuck "Variables should have all been replaced"
    | eval (Float r) = Float r
    | eval (Unit) = Unit
    | eval (Inl e) = Inl(eval e)
    | eval (Inr e) = Inr(eval e)
    | eval (Roll e) = Roll(eval e)
    | eval (Pack(t,e)) = Pack(t,eval e)
    | eval (Pair(e1,e2)) = Pair(eval e1, eval e2)
    | eval (TFn p) = TFn p
    | eval (Annotate(e,t)) = eval e
    | eval (Case(e,(_,e1),(_,e2))) = 
       (case eval e of
          Inl v => eval(subst v 1 e1)
        | Inr v => eval(subst v 1 e2)
        | _ => raise Stuck "Attempting to case over non-injection")
    | eval (Open(e,(_,_,e'))) = 
       (case eval e of
          Pack(t,v) => eval(subst v 1 (subst_tp t 2 e' 2))
        | _ => raise Stuck "Attempting to open non-existential")
    | eval (Inst(e,t)) = 
       (case eval e of
          TFn(_,e) => eval (subst_tp t 1 e 1)
        | _ => raise Stuck "Attempting to instantiate non-universal")
    | eval (Unroll e) = eval e
    | eval (Itof e) = 
       (case eval e of
          Int i => Float(real i)
        | _ => raise Stuck "Attempting to apply itof to non-int")
    | eval (Fst e) = 
       (case eval e of
          Pair(v1,_) => v1
        | _ => raise Stuck "Attempting to project from non-pair")
    | eval (Snd e) = 
       (case eval e of
          Pair(_,v2) => v2
        | _ => raise Stuck "Attempting to project from non-pair")
    | eval (Abort e) = 
      (eval e; raise Stuck "Something of void type returned")

end;  (* structure EvalSolution *)
