/***************************************************************************
                          uitab.cpp  -  description
                             -------------------
    begin                : Wed Apr 2 2003
    copyright            : (C) 2003 by Chris Colohan
    email                : colohan+@cs.cmu.edu
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <keditlistbox.h>
#include <klineedit.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kurl.h>
#include <kwin.h>
#include <qcheckbox.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qregexp.h>
#include <qwhatsthis.h>
#include "uitab.h"
#include "kzephyr.h"
#include "tabsettingsdialog.h"


UiTab::UiTab(QString name,
	     Kzephyr *kzephyr,
	     QTabWidget *parent) :
    name_(name),
    kzephyr_(kzephyr),
    parent_(parent),
    removed_(false)
{
    splitter = new QSplitter(this, "Splitter");
    splitter->setOrientation(QSplitter::Vertical);

    // Above the splitter bar we have the RecvPage, which displays the
    // zephyrs:
    rcvPage = new RecvPage(splitter);
    
    // Below the splitter we have a whole collection of things which
    // we embed in a layout widget so we can add it as a single object
    // to the splitter:
    QWidget* layoutWidget = new QWidget(splitter);
    QVBoxLayout* VLayout1 = new QVBoxLayout(layoutWidget, 5, 5);

    QHBoxLayout* HLayout1 = new QHBoxLayout(0, 0, 5);
    VLayout1->addLayout(HLayout1);

    // Who we are sending the zephyr to:
    QLabel* TextLabel1 = new QLabel(layoutWidget);
    TextLabel1->setText(tr2i18n("To:"));
    HLayout1->addWidget(TextLabel1);
    toEdit = new KLineEdit(layoutWidget);
    HLayout1->addWidget(toEdit);

    // The message typed by the user:
    zephyrEdit = new ZephTextEdit(layoutWidget);
    VLayout1->addWidget(zephyrEdit);
    zephyrEdit->setTextFormat(PlainText);

    QHBoxLayout* HLayout2 = new QHBoxLayout(0, 0, 5);
    VLayout1->addLayout(HLayout2);

    // This button sends the zephyr:
    sendButton = new QPushButton(layoutWidget);
    sendButton->setText(tr2i18n("&Send"));
    HLayout2->addWidget(sendButton);

    // This button determines whether the zephyr is broadcast or is
    // private:
    personalCheck = new QCheckBox(layoutWidget);
    personalCheck->setText(tr2i18n("&Personal"));
    HLayout2->addWidget(personalCheck);

    // Cause the whole mess to fill the parent widget completely:
    QHBoxLayout* HLayout3 = new QHBoxLayout(this, 5, 0);
    HLayout3->addWidget(splitter);

    // Add this tab to the tab widget:
    parent->addTab(this, name);

    // Read in the configuration for this tab:
    UiTab *curPage = (UiTab *)parent->currentPage();
    kzephyr_->config()->setGroup(QString("Tab ") + name);
    instanceFiltersExclusive_ =
	kzephyr_->config()->readBoolEntry("Instance Filters Exclusive", true);
    frontOnNewZephyr_ =
	kzephyr_->config()->readBoolEntry("Bring to Front on New Zephyrs",
					  false);
    beepOnNewZephyr_ =
	kzephyr_->config()->readBoolEntry("Beep on New Zephyrs",
					  false);
    toEdit->setText(kzephyr_->config()->readEntry("Recipient",
						  QString("")));

    // FIXME: Temporary code, erase me later...
    if(kzephyr_->config()->readEntry("Instance Filters") != QString::null ||
       kzephyr_->config()->readEntry("Show Personal Zephyrs") != QString::null ||
       kzephyr_->config()->readEntry("Show Public Zephyrs") != QString::null) {
	KMessageBox::sorry(0,
			   "This new version of kzephyr doesn't support the\n"
			   "same instance filtering mechanism used by older\n"
			   "versions of kzephyr.  Please set up your filters\n"
			   "again in the Tabs->Settings... dialog.");
	kzephyr_->config()->deleteEntry("Instance Filters");
	kzephyr_->config()->deleteEntry("Show Personal Zephyrs");
	kzephyr_->config()->deleteEntry("Show Public Zephyrs");
    }
    if(kzephyr_->config()->readEntry("Private Checked") != QString::null) {
	// FIXME:  patch things up...
	bool oldsetting = kzephyr_->config()->readBoolEntry("Private Checked",
							    true);
	kzephyr_->config()->writeEntry("Personal Checked",
				       oldsetting);
	kzephyr_->config()->deleteEntry("Private Checked");
    }

    QStringList filterStringList =
	kzephyr_->config()->readListEntry("Filters");
    while(filterStringList.size() >= 3) {
	QString type = filterStringList.first();
	filterStringList.pop_front();
	QString accept = filterStringList.first();
	filterStringList.pop_front();
	QString re = filterStringList.first();
	filterStringList.pop_front();
	addFilter(Filter(type, accept, re));
    }

    bool defaultPersonal = true;
    if(curPage) {
	defaultPersonal = curPage->personalCheck->isChecked();
    }
    personalCheck->setChecked(kzephyr_->config()->
			      readBoolEntry("Personal Checked",
					    defaultPersonal));

    QValueList<int> sizes = kzephyr_->config()->readIntListEntry("Pane Sizes");
    if(sizes.empty()) {
	if(curPage != NULL) {
	    // Copy the proportions of the current page:
	    sizes = curPage->splitter->sizes();

	} else {
	    // Just make the panes 3/4 to 1/4 in size:
	    int h = height();
	    sizes.append(h * 3 / 4);
	    sizes.append(h / 4);
	}
    }
    splitter->setSizes(sizes);

    connect(zephyrEdit, SIGNAL(ctrlJPressed(QKeyEvent *)), kzephyr_,
	    SLOT(keyPressEvent(QKeyEvent *)));
    connect(zephyrEdit, SIGNAL(pageUpPressed(QKeyEvent *)), kzephyr_,
	    SLOT(keyPressEvent(QKeyEvent *)));
    connect(zephyrEdit, SIGNAL(pageDownPressed(QKeyEvent *)), kzephyr_,
	    SLOT(keyPressEvent(QKeyEvent *)));
    connect(sendButton, SIGNAL(clicked()), kzephyr_,
	    SLOT(sendZephyr()));
    connect(rcvPage, SIGNAL(urlClick(const QString &)), this,
	    SLOT(linkClicked(const QString &)));


    QWhatsThis::add(rcvPage,
		    i18n("<qt>Incoming zephyrs are displayed here.  A link "
			 "on an instance will open a new tab that shows just "
			 "that instance.  A link on an <b>X</b> eliminates "
			 "that instance from this tab.  Look in the "
			 "<b>Tab Settings...</b> menu to change the filters "
			 "which apply to this tab.</qt>"));
    QWhatsThis::add(toEdit,
		    i18n("<qt>If you are sending a personal zephyr then you "
			 "can type a list of recipients here, separated by "
			 "commas or spaces.  If you are sending a public "
			 "zephyr you type the name of the <b>instance</b> "
			 "here.  Right click for useful options.</qt>"));
    QWhatsThis::add(zephyrEdit,
		    i18n("<qt>Type your message to send here.  Either hit the "
			 "send button below or type control-j to send "
			 "it.</qt>"));
    QWhatsThis::add(sendButton,
		    i18n("<qt>Clicking this button will send the message you "
			 "have typed above.  For speed freaks, control-j will "
			 "also send the message.</qt>"));
    QWhatsThis::add(personalCheck,
		    i18n("<qt>If you check this box then any message you send "
			 "will be <b>personal</b>, meaning that only the "
			 "people you specify will receive it.  Otherwise, "
			 "everyone will be able to read your message, and it "
			 "will be sent on the specified "
			 "<b>instance</b>.</qt>"));
    

    // Set up auto-completion:
    toEdit->setCompletionMode(KGlobalSettings::CompletionAuto);
    toEdit->completionObject()->setIgnoreCase(true);
}

UiTab::~UiTab()
{
    if(!removed_) {
	saveOptions();
    }
}

void
UiTab::gotZephyr(ZephyrMsgPtr msg)
{
    if(showZephyr(msg)) {

	// Allow completion:
	toEdit->completionObject()->addItem(msg->instance());
	toEdit->completionObject()->addItem(msg->sender());
    
	if(msg->personal() &&
	   QRegExp("^\\[(\\w+\\s*)+\\]").search(msg->message()) != -1) {
	    // A list of recipients is at the top of this zephyr.  Parse
	    // out that list and add it to our completion list.
	    
	    int bindex = msg->message().find(QRegExp("\\]"));
	    assert(bindex != -1);
	    QString toLine = msg->message().left(bindex + 1);
	    
	    // Strip off left and right brackets:
	    toLine = toLine.left(toLine.length() - 1);
	    toLine = toLine.right(toLine.length() - 1);
	    
	    // Find recipients:
	    QStringList recipients = QStringList::split(" ", toLine);
	    
	    // Remember these ids:
	    for(QStringList::Iterator r = recipients.begin();
		r != recipients.end(); ++r) {
		kzephyr_->useridList.insert(*r);
		toEdit->completionObject()->addItem(*r);
	    }
	    
	    // Now build that list into a send string:
	    QString sendString = msg->sender() + "," + recipients.join(",");
	    
	    // Remove my own name from this list:
	    QString userid = kzephyr_->tzc->userid;
	    userid.replace(QRegExp("@.*$"), "");
	    sendString.replace(QRegExp(userid), "");
	    sendString.replace(QRegExp(",,"), ",");
	    sendString.replace(QRegExp("^,"), "");
	    sendString.replace(QRegExp(",$"), "");
	    
	    toEdit->completionObject()->addItem(sendString);
	}

	appendZephyr(msg);

	if(!msg->fake()) {
	    if(frontOnNewZephyr_) {
		// Bring this page to the front:
		parent_->setCurrentPage(parent_->indexOf(this));
		KWin::deIconifyWindow(kzephyr_->winId());
		KWin::setActiveWindow(kzephyr_->winId());
	    } else {
		if(parent_->currentPage() != this) {
		    parent_->setTabLabel(this, name_ + QString("*"));
		}
	    }
	    if(beepOnNewZephyr_) {
		QApplication::beep();
		msg->setNotified();
	    }
	}
    }
}

bool
UiTab::showZephyr(ZephyrMsgPtr msg)
{
    if(msg->fake()) {
	return true;
    }

    for(list<Filter>::iterator i = filterList_.begin();
	i != filterList_.end(); i++) {
	Filter::Status status = i->apply(msg);
	if(status == Filter::ACCEPT) {
	    return true;
	} else if(status == Filter::REJECT) {
	    return false;
	}
    }
    return instanceFiltersExclusive_;
}

void
UiTab::appendZephyr(ZephyrMsgPtr msg)
{
    bool atBottom = rcvPage->verticalScrollBar()->value() ==
	rcvPage->verticalScrollBar()->maxValue();

    zephyrs_.push_back(msg);

    rcvPage->append(toHtml(msg));

    if(atBottom) {
	// We don't remove old zephyrs otherwise since this:
	//   a) causes the window to scroll to the top, and
	//   b) could remove something the user is currently looking at.

	// 2 is an arbitrary constant here:
	int numZephyrs = (int)zephyrs_.size();
	if(numZephyrs > kzephyr_->history_size * 2) {
	    while(numZephyrs-- > kzephyr_->history_size) {
		zephyrs_.pop_front();
	    }

	    // Repopulate this window
	    repopulate();
	}

	// Only scroll if the user is not looking back at a previous
	// thing in their buffer:
	rcvPage->scrollToBottom();
    }
}

void
UiTab::editInstanceFilters()
{
    TabSettingsDialog tsd(filterList_, kzephyr_, 0, true);

    tsd.tabName->setText(name_);
    tsd.beepNew->setChecked(beepOnNewZephyr_);
    tsd.frontNew->setChecked(frontOnNewZephyr_);
    tsd.inclusive->setCurrentItem((int)instanceFiltersExclusive_);

    if(tsd.exec() == QDialog::Accepted) {
	// Rename the tab:
	if(name_ != tsd.tabName->text()) {
	    QString newName = kzephyr_->makeTabNameUnique(tsd.tabName->text());
	    kzephyr_->config()->deleteGroup(QString("Tab ") + name_);
	    name_ = newName;
	    parent_->setTabLabel(this, name_);
	}
	
	tsd.getFilters(&filterList_);
	beepOnNewZephyr_ = tsd.beepNew->isChecked();
	frontOnNewZephyr_ = tsd.frontNew->isChecked();
	instanceFiltersExclusive_ = (bool)tsd.inclusive->currentItem();

	saveOptions();

	refilter();
	repopulate();

    }
}

void
UiTab::focusOnInstance(QString instance)
{
    filterList_.clear();
    addFilter(Filter(Filter::INSTANCE, Filter::ACCEPT, QRegExp(instance)));
    instanceFiltersExclusive_ = false;
    toEdit->setText(instance);

    saveOptions();
}

void
UiTab::filterOutInstance(QString instance)
{
    filterList_.push_front(Filter(Filter::INSTANCE, Filter::REJECT,
				  QRegExp(instance)));
    refilter();
    saveOptions();
}

void
UiTab::clearZephyrs()
{
    zephyrs_.clear();
    repopulate();
}

void
UiTab::addFilter(Filter filt)
{
    filterList_.push_back(filt);
}

void
UiTab::remove()
{
    kzephyr_->config()->deleteGroup(QString("Tab ") + name_);
    removed_ = true;
}

void
UiTab::saveOptions()
{
    kzephyr_->config()->setGroup(QString("Tab ") + name_);
    kzephyr_->config()->writeEntry("Instance Filters Exclusive",
				   instanceFiltersExclusive_);
    kzephyr_->config()->writeEntry("Personal Checked",
				   personalCheck->isChecked());
    kzephyr_->config()->writeEntry("Pane Sizes", splitter->sizes());
    kzephyr_->config()->writeEntry("Bring to Front on New Zephyrs",
				   frontOnNewZephyr_);
    kzephyr_->config()->writeEntry("Beep on New Zephyrs",
				   beepOnNewZephyr_);
    kzephyr_->config()->writeEntry("Recipient",
				   toEdit->text());
    QStringList filterStringList;
    for(list<Filter>::iterator i = filterList_.begin();
	i != filterList_.end(); i++) {
	filterStringList += (*i).typeStr();
	filterStringList += (*i).acceptStr();
	filterStringList += (*i).reStr();
    }
    kzephyr_->config()->writeEntry("Filters", filterStringList);
}

QString
UiTab::toHtml(ZephyrMsgPtr msg)
{
    QString sender = msg->sender();
    QString signature = msg->signature();
    QString instance = msg->instance();
    QString time = msg->time();
    bool    auth = msg->auth();
    QString message = msg->message();
    QString result;
    
    if(sender.length() != 0) {
	result += "<table width=100% cellspacing=0 cellpadding=0 border=0>";
	result += "<tr><td width=10%><font color=\"#ff0000\">";


	QString userid = sender;
	userid.replace(QRegExp("@.*"), "");
	QString domain = "";
	if(sender.find(QRegExp("@")) != -1) {
	    domain = sender;
	    domain.replace(QRegExp("^.*@"), "");
	    domain = domain.lower();
	}
	
	result += userid;

	if(domain.length() != 0) {
	    QString domain_root = domain;
	    domain_root.replace(QRegExp("\\..*$"), "");
	    result += "<p>@" + domain_root;
	}
	    
	if(kzephyr_->show_faces) {
	    if(domain.length() == 0) {
		domain = kzephyr_->tzc->userid;
		domain.replace(QRegExp(".*@"), "");
		domain = domain.lower();
	    }
	    // Get rid of suffixes on userids, since they are usually
	    // just remnants of kerberos sub-ids:
	    userid.replace(QRegExp("\\..*"), "");

	    QString fname = "/afs/" + domain + "/user/" + userid +"/.";
	    struct stat buf;
	    if(stat((fname + "zface.gif").latin1(),&buf) == 0) {
		result += "<p><img src=\"" + fname + "zface.gif\" height=30>";
	    } else if(stat((fname + "zface.xpm").latin1(),&buf) == 0) {
		result += "<p><img src=\"" + fname + "zface.xpm\" height=30>";
	    } else if(stat((fname + "face.gif").latin1(),&buf) == 0) {
		result += "<p><img src=\"" + fname + "face.gif\" height=30>";
	    } else if(stat((fname + "face").latin1(),&buf) == 0) {
		result += "<p><img src=\"" + fname + "face\" height=30>";
	    } else if(stat((fname + "face.xpm").latin1(),&buf) == 0) {
		result += "<p><img src=\"" + fname + "face.xpm\" height=30>";
	    } else if(stat((fname + "face.jpg").latin1(),&buf) == 0) {
		result += "<p><img src=\"" + fname + "face.jpg\" height=30>";
	    } else if(stat((fname + "zface.jpg").latin1(),&buf) == 0) {
		result += "<p><img src=\"" + fname + "zface.jpg\" height=30>";
	    }
	}

	result += "</font></td><td width=90%><p>";
	if(!msg->personal() ||
	   (msg->personal() && instance.compare("PERSONAL") != 0)) {
	    if(msg->fake()) {
		result += "(" + instance + ") ";
	    } else {
		result += "(<A href=\"instance://" + makesafe(instance) + "\">";
		result += makesafe(instance);
		result += "</A>)";
		result += " <A href=\"remove://" + makesafe(instance) + "\">[X]</A> ";
	    }
	}
	if(kzephyr_->show_sigs && signature.length() != 0) {
	    result += "\"" + makesafe(signature) + "\" ";
	}
	
	result += " <font color=\"#0000ff\">[";

	// Extract time from whole date expression:
	QRegExp just_time("^.*(\\d+:\\d+:\\d+).*$");
	int pos = just_time.search(time);
	if(pos > -1) {
	    time = just_time.cap(1);
	}
	//time.replace(QRegExp("^\\w+ \\w+ \\d+ "), "");
	//time.replace(QRegExp(" \\d+$"), "");
	result += time;
	result += "]</font>";
	if(!auth) {
	    result += " <b><i><font color=\"#00ff00\">(not authenticated)</font></b></i>";
	}
	result += "</p><p>";
	result += munge(msg);
	result += "</p></td></tr></table>\n";
    }

    return result;
}

QString
UiTab::makesafe(QString message)
{
    // Escape all angle brackets to avoid html in zephyrs:
    int pos = 0;
    while(pos >= 0) {
	pos = message.find(QRegExp("<"), pos);
	if(pos >= 0) {
	    message.replace(pos, 1, "&lt;");
	    pos += 4;
	}
    }
    pos = 0;
    while(pos >= 0) {
	pos = message.find(QRegExp(">"), pos);
	if(pos >= 0) {
	    message.replace(pos, 1, "&gt;");
	    pos += 4;
	}
    }
    return message;
}

QString
UiTab::munge(ZephyrMsgPtr msg)
{
    QString message = makesafe(msg->message());
    
    // Replace all URLs with hot links:
    int pos = 0;
    while(pos >= 0) {
	pos = message.find(QRegExp("\\b(s?https?|ftp|file|gopher|news|telnet|wais|mailto|):(//[-a-aA-Z0-9_.]+:[0-9]*)?([-a-zA-Z0-9_=!?#$@~`%&*+|\\/:;.,]|\\w)+([-a-zA-Z0-9_=#$@~`%&*+|\\/]|\\w)"), pos);
	if(pos >= 0) {
	    int end = message.find(QRegExp("\\s"), pos);
	    if(end == -1) {
		end = message.length();
	    }

	    QString url = message.mid(pos, end - pos);

#define CMU_SCS
#ifdef CMU_SCS
	    if(kzephyr_->url_archive &&
	       !msg->personal()) {
		url = KURL::encode_string_no_slash(url);

		url = QString("http://zarchive.srv.cs.cmu.edu/prog/fetch_url?") +
#define NOFRAME
#ifdef NOFRAME
		    QString("noframe=1&") +
#endif
		    QString("url=") + url +
		    QString("&timesecs=%1.").arg(msg->time_secs());
	    }
#endif

	    QString prefix = QString("<A href=\"") + url + QString("\">");
	    QString suffix("</A>");
	    message.insert(end, suffix);
	    message.insert(pos, prefix);

	    pos = end + prefix.length() + suffix.length();
	}
    }

    // Replace all newlines with <br> tags:
    pos = 0;
    while(pos >= 0) {
	pos = message.find(QRegExp("\\n"), pos);
	if(pos >= 0) {
	    message.remove(pos, 1);
	    message.insert(pos, "<br>");
	    // 4 = strlen("<br>")
	    pos += 4;
	}
    }

    return message;
}

void
UiTab::repopulate()
{
    rcvPage->clear();
    for(list<ZephyrMsgPtr>::iterator i = zephyrs_.begin();
	i != zephyrs_.end(); i++) {
	rcvPage->append(toHtml(*i));
    }
}

void
UiTab::refilter()
{
    list<ZephyrMsgPtr>::iterator i = zephyrs_.begin();
    while(i != zephyrs_.end()) {
	if(!showZephyr(*i)) {
	    list<ZephyrMsgPtr>::iterator dead = i;
	    i++;
	    zephyrs_.erase(dead);
	} else {
	    i++;
	}
    }
}

void
UiTab::linkClicked(const QString &link)
{
    if(link.find(QRegExp("^instance:")) != -1) {
	QString instance = link;
	instance.replace(QRegExp("^instance://"), "");
	instance.replace(QRegExp("/$"), "");
	UiTab *ut = kzephyr_->addTabInstance(instance);

	// Populate the new tab with any appropriate zephyrs from this
	// buffer:
	for(list<ZephyrMsgPtr>::iterator i = zephyrs_.begin();
	    i != zephyrs_.end(); i++) {
	    ut->gotZephyr(*i);
	}

	// Bring the new tab to the front:
	parent_->setCurrentPage(parent_->indexOf(ut));

    } else if(link.find(QRegExp("^remove:")) != -1) {
	QString instance = link;
	instance.replace(QRegExp("^remove://"), "");
	instance.replace(QRegExp("/$"), "");
	filterOutInstance(instance);
	repopulate();

    } else {
	kapp->invokeBrowser(link);
    }
}

