/***************************************************************************
                          tzc.cpp  -  description
                             -------------------
    begin                : Mon Sep 30 2002
    copyright            : (C) 2002 by Chris Colohan
    email                : colohan+@cs.cmu.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <sys/types.h>
#include <sys/wait.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <qapplication.h>
#include <kstatusbar.h>
#include <klocale.h>
#include <qregexp.h>

#include "tzc.h"
#include "kzephyr.h"
#include "parsetree.h"

Tzc::Tzc(Kzephyr *kzephyr,
	 KConfig *config) :
    kzephyr_(kzephyr),
    config_(config),
    exiting_(false)
{
    readOptions();
    
    start_tzc();

    QThread::start();
}

Tzc::~Tzc()
{
    exiting_ = true;

    // Kill the tzc process:
    kill(tzc_pid_, SIGTERM);

    // Wait for the child to exit:
    QThread::wait();
    
    saveOptions();
}

// Child thread:  starts up tzc and handles input from tzc.
void
Tzc::run()
{
    bool in_quote = false;
    bool escape_next = false;
    while(1) {

//#define TEST
#ifdef TEST
	sleep(5);
	qApp->lock();
	// kzephyr_->recvZephyr(ZephyrMsgPtr(new ZephyrMsg(sender, instance, time, auth, personal,
	// 			msg2HTML(message))));
	kzephyr_->recvZephyr(
	    ZephyrMsgPtr(new ZephyrMsg("testuser", "Test", "", "Noon", 0,
				       true, true,
				       msg2HTML("[user1 user2 user3]\nThis is a test."))));
	kzephyr_->recvZephyr(
	    ZephyrMsgPtr(new ZephyrMsg("testuser", "Test", "PERSONAL", "Noon", 0,
				       true, true,
				       msg2HTML("[user2 user3 user4]\nThis is a test too."))));
	kzephyr_->recvZephyr(
	    ZephyrMsgPtr(new ZephyrMsg("testuser", "Test", "foobar", "Noon", 0,
				       true, false,
				       msg2HTML("Whee."))));
	kzephyr_->recvZephyr(
	    ZephyrMsgPtr(new ZephyrMsg("testuser", "Test", "foobar", "Noon", 0,
				       false, false,
				       msg2HTML("Multiple.\nLines."))));
	qApp->unlock();
#endif

	
	int rc = read(from_tzc_[0], input_buff_, sizeof(input_buff_));
	if(rc <= 0) {
	    // Oh no, tzc is gone!

	    // Try to stop anyone from writing to this pipe when we close it:
	    qApp->lock();

	    if(close(to_tzc_[1]) == -1) {
		perror("Close of input of pipe to tzc failed");
	    }
	    
	    if(close(from_tzc_[0]) == -1) {
		perror("Close of output of pipe from tzc failed");
	    }
	    
	    if(exiting_) {
		// This is a normal program exit...
		qApp->unlock();
		return;

	    } else {
		kzephyr_->statusMsg(
		    QString("Tzc died unexpectedly, restarting..."));

		// Reap the child if it is really gone:
		waitpid(tzc_pid_, NULL, WNOHANG);

		// If tzc died after we sent a zephyr but before we
		// got an ack, abort the send:
		kzephyr_->zephyrSent(false);

		start_tzc();

		qApp->unlock();

		// Put a delay in here so we don't chew up all cpu
		// time if our restart fails:
		QThread::sleep(10);
	    }	    

	} else {
	    for(int p = 0; p < rc; p++) {
		if(!escape_next) {
		    if(input_buff_[p] == '\\') {
			escape_next = true;
		    }
		    if(input_buff_[p] == '"') {
			in_quote = !in_quote;
		    }
		    if(!in_quote &&
		       input_buff_[p] == '\n') {
			process_line();
			input_line_ = "";
		    }
		} else {
		    escape_next = false;
		}
		input_line_.append(input_buff_[p]);
	    }
	}
    }
}

void
Tzc::start_tzc()
{
#ifdef DEBUG_TZC
    printf("Creating pipes...\n");
#endif // DEBUG_TZC
    if(pipe(to_tzc_) != 0) {
	perror("Connecting pipe to tzc failed");
	::exit(8);
    }
    if(pipe(from_tzc_) != 0) {
	perror("Connecting pipe from tzc failed");
	::exit(8);
    }

#ifdef DEBUG_TZC
    printf("Forking tzc process...\n");
#endif // DEBUG_TZC

    tzc_pid_ = fork();
    if(tzc_pid_ == -1) {
	perror("Fork of tzc failed");
	::exit(8);
    }
    if(tzc_pid_ == 0) {
	// Set up a pipe from tzc to kzephyr:
	if(close(1) == -1) {
	    perror("Close of stdout failed");
	    ::exit(8);
	}
	if(dup2(from_tzc_[1], 1) == -1) {
	    perror("dup2 of stdout failed");
	    ::exit(8);
	}
	if(close(from_tzc_[1]) == -1) {
	    perror("Close of piped stdout failed");
	    ::exit(8);
	}

	// Set up a pipe from kzephyr to tzc:
	if(close(0) == -1) {
	    perror("Close of stdin failed");
	    ::exit(8);
	}
	if(dup2(to_tzc_[0], 0) == -1) {
	    perror("dup2 of stdin failed");
	    ::exit(8);
	}
	if(close(to_tzc_[0]) == -1) {
	    perror("Close of piped stdin failed");
	    ::exit(8);
	}

#ifdef DEBUG_TZC
	printf("Starting tzc...\n");
#endif // DEBUG_TZC
	
	execl("/bin/sh", "/bin/sh", "-c", tzc_command.latin1(), NULL);
	perror("Failed to exec tzc");
	::exit(8);
    }

    if(close(to_tzc_[0]) == -1) {
	perror("Close of output of pipe to tzc failed");
	::exit(8);
    }

    if(close(from_tzc_[1]) == -1) {
	perror("Close of input of pipe from tzc failed");
	::exit(8);
    }
}

void
Tzc::process_line()
{
    ParseTree parse(input_line_);

#ifdef DEBUG_TZC
    qWarning("Input from tzc: %s", input_line_.latin1());
#endif

    if(parse.findNode("tzcspew")->child(2)->contents().compare("message")
       == 0) {

	if(parse.findNode("opcode")->child(2)->contents().compare("PING")
	   == 0) {
	    qApp->lock();
	    kzephyr_->ping(parse.findNode("sender")->child(2)->contents());
	    qApp->unlock();

	} else if(parse.findNode("opcode")->child(2)->contents().compare("nil")
		  == 0) {
	    QString sender =  msg2HTML(parse.findNode("sender")->
				       child(2)->contents());
	    QString instance= msg2HTML(parse.findNode("instance")->
				       child(2)->contents());
	    QString time =    parse.findNode("time")->child(2)->contents();
	    bool    auth =    parse.findNode("auth")->child(2)->contents()
		.compare("no") != 0;
	    bool personal =   parse.findNode("recipient")->child(2)->contents()
		.compare("") != 0;
	    QString signature = parse.findNode("message")->
		child(2)->child(0)->contents();
	    QString message = msg2HTML(parse.findNode("message")->
				       child(2)->child(1)->contents());
	    long time_secs = (parse.findNode("time-secs")->child(2)->
			      child(0)->contents().toLong() << 16) +
		parse.findNode("time-secs")->child(2)->child(1)->contents().
		toLong();
	    qApp->lock();
	    kzephyr_->recvZephyr(
		ZephyrMsgPtr(new ZephyrMsg(sender, signature,instance, time,
					   time_secs, auth, personal,
					   message)));
	    qApp->unlock();
	} else {

	    qWarning("Unhandled input from tzc: %s", input_line_.latin1());
	}
	
    } else if(parse.findNode("tzcspew")->child(2)->contents().compare("error")
	      == 0) {
	qApp->lock();
	kzephyr_->statusMsg(QString("Error from tzc: ") + 
			    parse.findNode("message")->child(2)->
			    contents());
	QApplication::beep();

	// If tzc died after we sent a zephyr but before we
	// got an ack, abort the send:
	kzephyr_->zephyrSent(false);
	
	qApp->unlock();

    } else if(parse.findNode("tzcspew")->child(2)->contents().compare("sent")
	      == 0) {
	qApp->lock();
	kzephyr_->zephyrSent(true);
	qApp->unlock();

    } else if(parse.findNode("tzcspew")->child(2)->contents().compare("not-sent")
	      == 0) {
	qApp->lock();
	QString err = "Send of zephyr to " + 
	    parse.findNode("to")->child(3)->contents() + " failed.";
	kzephyr_->statusMsg(err);
	
	kzephyr_->zephyrSent(false);
	qApp->unlock();

    } else if(parse.findNode("tzcspew")->child(2)->contents().compare("start")
	      == 0) {
	userid = parse.findNode("zephyrid")->child(2)->contents();
	qApp->lock();
	kzephyr_->statusMsg("Tzc started successfully.");
	qApp->unlock();

    } else if(parse.numChildren() > 1) {
	qWarning("Unhandled input from tzc: %s", input_line_.latin1());
    }
}

void
Tzc::writeZephyr(QString message)
{
    if(write(to_tzc_[1], message.latin1(), message.length()) == -1) {
	perror("Pipe to tzc failed");
	::exit(8);
    }
}

void
Tzc::readOptions()
{
    config_->setGroup("Tzc Options");
    tzc_command = config_->readEntry("Tzc Command", "tzc");
    userid = config_->readEntry("Userid", "nobody");
    zsig = config_->readEntry("Signature", "KZephyr User");
}

void
Tzc::saveOptions()
{
    config_->setGroup("Tzc Options");
    config_->writeEntry("Tzc Command", tzc_command);
    config_->writeEntry("Userid", userid);
    config_->writeEntry("Signature", zsig);
}

void
Tzc::sendZephyr(QStringList recipients,
		QStringList instances,
		QString message)
{
    assert(recipients.count() == instances.count());

    QString tzc_message;
    tzc_message =
	QString("((tzcfodder . send) (class . \"MESSAGE\") (auth . t)"
		" (recipients ");
    
    QStringList::Iterator r, i;
    for(r = recipients.begin(), i = instances.begin();
	r != recipients.end() && i != instances.end();
	++r, ++i) {
	tzc_message += QString("(\"") + armourZephyr(*i) +
	    QString("\" . \"") + armourZephyr(*r) +
	    QString("\") ");
    }
    tzc_message += QString(") (sender . \"") +
	armourZephyr(QString(userid)) +
	QString("\") (message . (\"") +
	armourZephyr(zsig) +
	QString("\" \"") +
	armourZephyr(message) + QString("\")))\n");
    
#ifdef DEBUG_TZC
    qWarning(tzc_message);
#endif

    writeZephyr(tzc_message);
}

// Munge the outgoing zephyr so that any quotes or backslashes are
// escaped and do not confuse tzc:
QString
Tzc::armourZephyr(QString message)
{
    int pos = 0;
    while(pos >= 0) {
	pos = message.find(QRegExp("\\\\"), pos);
	if(pos >= 0) {
	    message.insert(pos, "\\");
	    pos += 2;
	}
    }
    pos = 0;
    while(pos >= 0) {
	pos = message.find(QRegExp("\""), pos);
	if(pos >= 0) {
	    message.insert(pos, "\\");
	    pos += 2;
	}
    }

    return message;
}

// Take a raw message from tzc and format it for display in HTML:
QString
Tzc::msg2HTML(QString message)
{
    // Replace all escaped thingees with thingees:
    int pos = 0;
    while(pos >= 0) {
	pos = message.find(QRegExp("\\\\"), pos);
	if(pos >= 0) {
	    message.remove(pos, 1);
	    pos++;
	}
    }

    return message;
}
