/***************************************************************************
                          kzephyr.cpp  -  description
                             -------------------
    begin                : Sat Sep 28 14:51:54 EDT 2002
    copyright            : (C) 2002 by Chris Colohan
    email                : colohan+@cs.cmu.edu
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <klocale.h>
#include <kaction.h>
#include <kstdaction.h>
#include <kaboutdialog.h>
#include <kglobalsettings.h>
#include <klineedit.h>
#include <kmessagebox.h>
#include <krestrictedline.h>
#include <kstatusbar.h>
#include <qaction.h>
#include <qcheckbox.h>
#include <qlayout.h>
#include <qpushbutton.h>
#include <qregexp.h>
#include <qsplitter.h>
#include <qtabwidget.h>
#include <qtextedit.h>
#include "kzephyr.h"
#include "preferences.h"


#ifdef BOOST_NO_EXCEPTIONS
void boost::throw_exception(std::exception const & e)
{
    assert(0);
}
#endif

Kzephyr::Kzephyr(KConfig *config, QWidget *parent, const char *name) :
    KMainWindow(parent, name),
    config_(config),
    sentZephyrTab_(NULL)
{
    // Create the UI:
    ui_ = new KzephyrUI(this, name);
    setCentralWidget(ui_);
    //layout_ = new QHBoxLayout(this, 11, 6, "MainLayout"); 
    //layout_->addWidget(ui_);

    // Set up the menu:
    (new KAction(i18n("&Preferences..."), 0, 0,
		 this, SLOT(preferences()), actionCollection(),
		 "preferencesAction"))->setStatusText(
		     i18n("Changes global preferences"));
    KStdAction::quit(this, SLOT(close()), actionCollection());
    (new KAction(i18n("&Clear"), 0, 0,
		 this, SLOT(clearZephyrs()), actionCollection(),
		 "clearTabAction"))->setStatusText(
		     i18n("Clears all messages from the current tab"));
    (new KAction(i18n("&Add"), 0, 0,
		 this, SLOT(addTab()), actionCollection(),
		 "addTabAction"))->setStatusText(
		     i18n("Add a new tab to the zephyr view"));
    (removeTabAction_ =
     new KAction(i18n("&Remove"), 0, 0,
		 this, SLOT(removeTab()), actionCollection(),
		 "removeTabAction"))->setStatusText(
		     i18n("Remove the current tab from the zephyr view"));
    (new KAction(i18n("&Settings..."), 0, 0,
		 this, SLOT(editInstanceFilters()), actionCollection(),
		 "editInstanceFiltersAction"))->setStatusText(
		     i18n("Change settings for this tab"));

    statusMsg("Starting...");

    // Load the GUI description from the XML file kzephyr.rc:
    createGUI();
    
    // Restore the geometry of kzephyr from the saved configuration:
    config_->setGroup("Kzephyr Options");
    QSize size = config_->readSizeEntry("Geometry");
    if(!size.isEmpty()) {
	resize(size);
    }

    // Restore other prefs:
    history_size = config_->readNumEntry("History Size", 500);
    show_sigs = config_->readBoolEntry("Show Signatures", true);
    show_faces = config_->readBoolEntry("Show Faces", true);
    url_archive = config_->readBoolEntry("Use URL Archive", true);

    // Restore the saved tab configuration:
    QStringList tabnames = config_->readListEntry("Tabs");
    for(QStringList::Iterator i = tabnames.begin();
	i != tabnames.end(); i++) {
	uiTabs_.push_back(new UiTab(*i, this, ui_->TabWidget));
    }

    if(uiTabs_.size() == 0) {
	// No tabs!  This must be the first time kzephyr has been run.
	// Give the user a reasonable default configuration:

	UiTab *pub = new UiTab("Public", this, ui_->TabWidget);
	uiTabs_.push_back(pub);
	UiTab *personal = new UiTab("Personal", this, ui_->TabWidget);
	uiTabs_.push_back(personal);

	// Discard personal zephyrs in the public tab:
	pub->addFilter(Filter(Filter::PERSONAL, false, true));

	// Show emergency zephyrs as personal zephyrs:
	personal->addFilter(Filter(Filter::INSTANCE, true,
				  QRegExp("EMERGENCY")));
				 
	// Also show facilities.announce zephyrs:
	personal->addFilter(Filter(Filter::INSTANCE, true,
				   QRegExp("$facilities\\.announce")));
				 
	// Discard all other non-personal zephyrs:
	personal->addFilter(Filter(Filter::PERSONAL, false, false));
				 
	// Grab the user's attention when new zephyrs arrive:
	personal->setBeepOnNew(true);
	personal->frontOnNew(true);

	// Set the default state for the personal checkbox:
	pub->setPersonalChecked(false);
	personal->setPersonalChecked(true);
    }

    if(uiTabs_.size() <= 1) {
	removeTabAction_->setEnabled(false);
    }
}

Kzephyr::~Kzephyr()
{
    saveOptions();
}


void
Kzephyr::sendZephyr()
{
    QStringList recipients, instances;
    UiTab* curPg = curPage();
    QString message = formatMessage(curPg->zephyrEdit->text());
    bool personalChecked = curPg->personalCheck->isChecked();
    QString toField = curPg->toEdit->text();

    if(sentZephyrTab_ != NULL) {
	QApplication::beep();
	statusMsg(QString("Zephyr send already in progress, be patient..."));
	return;
    }

    if(message.length() == 0) {
	int answer = KMessageBox::warningYesNo(this,
		    "Really send an empty message?",
		    "Kzephyr Warning",
		    KStdGuiItem::yes(), KStdGuiItem::no(),
		    "Send empty message");
	if(answer == KMessageBox::No) {
	    return;
	}
    }
	
    if(!personalChecked) {
	set<QString>::iterator i = useridList.find(toField);
	if(i != useridList.end() ||
	   toField.find(QRegExp(",")) != -1) {
	    int answer = KMessageBox::warningYesNoCancel(this,
		    "You are sending a public zephyr, but the instance\n"
		    "is the same as a user's name or contains commas.\n"
		    "Change this to be a personal zephyr?",
		    "Kzephyr Warning",
		    KStdGuiItem::yes(), KStdGuiItem::no(),
		    "Warn about public zephyr missends");
	    if(answer == KMessageBox::Yes) {
		curPg->personalCheck->setChecked(true);
		personalChecked = true;

	    } else if(answer == KMessageBox::Cancel) {
		return;
	    }
	}
    }


    if(personalChecked) {
	// Allow multiple personal zephyrs to be sent at once:
	toField.replace(QRegExp("\\s"), ",");
	recipients = QStringList::split(",", toField);

	for(QStringList::Iterator r = recipients.begin();
	    r != recipients.end(); ++r) {
	    instances.push_front("PERSONAL");
	}

	QString userid = tzc->userid;
	userid.replace(QRegExp("@.*$"), "");

	char buff[26];
	time_t t = time(NULL);
	QString date = QString(ctime_r(&t, buff));

	// If this is to multiple people, put a header on the message
	// saying who it was sent to:
	if(recipients.count() > 1) {
	    message = "[" + recipients.join(" ") + "]\n" + message;

	    sentZephyr_ = ZephyrMsgPtr(
		new ZephyrMsg(userid, "", "Sent", date, 0, true, true,
			      message, true));
	} else {
	    sentZephyr_ = ZephyrMsgPtr(
		new ZephyrMsg(userid, "", "Sent to " + *(recipients.begin()),
			      date, 0, true, true, message, true));
	}
    } else {
	recipients.push_front("");
	instances.push_front(curPg->toEdit->text());
    }
    
    sentZephyrTab_ = curPg;
    tzc->sendZephyr(recipients, instances, message);
}

void
Kzephyr::recvZephyr(ZephyrMsgPtr msg)
    
{
    useridList.insert(msg->sender());

    // Forward this zephyr to all of the receive pages:
    for(list<UiTab *>::iterator i = uiTabs_.begin();
	i != uiTabs_.end(); i++) {
	(*i)->gotZephyr(msg);
    }
}

void
Kzephyr::zephyrSent(bool success)
{
    if(success) {
	if(sentZephyrTab_) {
	    if(sentZephyr_) {
		sentZephyrTab_->gotZephyr(sentZephyr_);
		sentZephyr_.reset();
	    }	

	    sentZephyrTab_->zephyrEdit->setText("");
	}

    } else {
	sentZephyr_.reset();
    }	
    sentZephyrTab_ = NULL;
}

void
Kzephyr::ping(QString user)
{
    statusMsg(QString("Received ping from user ") +
	      user + QString("."));
}

void
Kzephyr::clearZephyrs()
{
    UiTab *cur = curPage();
    cur->clearZephyrs();
}

void
Kzephyr::addTab()
{
    uiTabs_.push_back(new UiTab(makeTabNameUnique("Tab"),
				this, ui_->TabWidget));
    removeTabAction_->setEnabled(true);
    ui_->TabWidget->setCurrentPage(ui_->TabWidget->count()-1);
}

UiTab *
Kzephyr::addTabInstance(QString instance)
{
    UiTab *rp = new UiTab(makeTabNameUnique(instance),
			  this, ui_->TabWidget);
    uiTabs_.push_back(rp);
    rp->focusOnInstance(instance);

    removeTabAction_->setEnabled(true);
    ui_->TabWidget->setCurrentPage(ui_->TabWidget->count()-1);

    return rp;
}

void
Kzephyr::removeTab()
{
    UiTab *cur = curPage();

    if(cur) {
	int answer = KMessageBox::warningContinueCancel(this,
	     "Are you sure you want to remove the tab\n"
	     "\"" + cur->name() + "\"?", 
	     "Kzephyr Warning",
	     KStdGuiItem::cont(),
	     "Warn about removing tabs");

	if(answer == KMessageBox::Continue) {    
	    uiTabs_.remove(cur);
	    cur->remove();
	    delete cur;
	    
	    if(uiTabs_.size() <= 1) {
		removeTabAction_->setEnabled(false);
	    }
	}
    }
}

QString
Kzephyr::makeTabNameUnique(QString name)
{
    bool done = false;
    while(!done) {
	done = true;
	for(list<UiTab *>::iterator i = uiTabs_.begin();
	    i != uiTabs_.end(); i++) {
	    if(name == (*i)->name()) {
		done = false;
	    }
	}
	if(!done) {
	    // Cheesy way of making each tab name unique:
	    name += "1";
	}
    }

    return name;
}

void
Kzephyr::editInstanceFilters()
{
    curPage()->editInstanceFilters();
}

void
Kzephyr::preferences()
{
    Preferences pref(this, 0, true);
    pref.Signature->setText(tzc->zsig);
    pref.TzcCommand->setText(tzc->tzc_command);
    pref.HistorySize->setValidChars("1234567890");
    pref.HistorySize->setText(QString("%1").arg(history_size));
    pref.ShowSignature->setChecked(show_sigs);
    pref.ShowFaces->setChecked(show_faces);
    pref.URLArchive->setChecked(url_archive);
    if(pref.exec() == QDialog::Accepted) {
	tzc->zsig = pref.Signature->text();
	tzc->tzc_command = pref.TzcCommand->text();
	history_size = pref.HistorySize->text().toInt();
	show_sigs = pref.ShowSignature->isChecked();
	show_faces = pref.ShowFaces->isChecked();
	url_archive = pref.URLArchive->isChecked();
    }
}

void
Kzephyr::saveOptions()
{
    config_->setGroup("Kzephyr Options");
    config_->writeEntry("Geometry", size());
    config_->writeEntry("History Size", history_size);
    config_->writeEntry("Show Signatures", show_sigs);
    config_->writeEntry("Show Faces", show_faces);
    config_->writeEntry("Use URL Archive", url_archive);

    QStringList tabnames;
    for(list<UiTab *>::iterator i = uiTabs_.begin();
	i != uiTabs_.end(); i++) {
	tabnames += (*i)->name();
    }
    config_->writeEntry("Tabs", tabnames);
}

void
Kzephyr::statusMsg(const QString &text)
{
    statusBar()->message(text, 5000);
}

QString
Kzephyr::formatMessage(QString message)
{
    // Make sure the message is not longer than 79 columns wide:
    QString output;

    while(message.length() > 79) {
	int pos;

	// First check to see if there is already a good line break to
	// use:
	pos = message.find(QRegExp("\\n"));
	if(pos == -1 ||
	   pos > 79) {
	    // Nope.  We must insert a line break if we can.

	    pos = message.findRev(QRegExp("\\s"), 79);

	    if(pos == -1) {
		// No space we can use in the first 79 characters.
		// Give up and try to find our next space or line
		// break and use it.
		pos = message.find(QRegExp("[\\s\\n]"));

		// No spaces or line breaks in this message, can't fill:
		if(pos == -1) {
		    break;
		}
	    }
	}

	// Break at the specified position:
	output += message.left(pos) + "\n";
	message = message.right(message.length() - pos - 1);
    }

    
    // Append what we have left to our message:
    output += message;

    return output;
}

void
Kzephyr::keyPressEvent(QKeyEvent *e)
{
    // Send the current zephyr if Ctrl-J is pressed in the interface:
    if((e->state() & Qt::ControlButton) != 0 &&
       e->key() == Qt::Key_J) {
	sendZephyr();

    } else if(e->key() == Qt::Key_PageUp ||
	      e->key() == Qt::Key_PageDown) {
	curPage()->rcvPage->scroll(e);

    } else {
	e->ignore();
    }
}
