#ifndef DFT_h
#define DFT_h

/***********************************************************
 dft.h
 Description: Data structures used on DFT algorithm 
 Author: Ying
*************************************************************/
 
#define LONG long
#define BYTE char

#define BUFFER_SIZE 5
#define FRAME_COUNT 4
#define EDI_COUNT   3
#define NRULES      5

#define DFT_THRESHOLD (604800 * 1000)  /* 168 hours * 60 = 10080 minutes = 604800 sec */
#define TUPLE_THRESHOLD (168 * 1000)  /* 2.8 minutes * 60 = 168 sec */


/* Day and hour defined in seconds */
#define ONE_DAY     (86400*1000)   /* 1440 minutes */
#define ONE_HOUR    (3600*1000)    /* 60 minutes */
#define INTERVAL    (300*1000)     /* 5 minutes */


/* 
   The event (error) categories defined in the binary error log are:
   Print = 0
   Eventlog = 1
   Rdr = 2
   Service Control Manager = 3
   BROWSER = 4
   ( AFS Daemon = 5 ) (this is not applicable cz it is what appears in the "Application" log, which has nothing to do with the system device drivers
   
	 ...

   These events are system dependent 
*/

#define NDEVICES 10

enum FRU_Type { Print, Eventlog, Rdr, ServiceControlManager, BROWSER, aci78xx, SaveDump, Mouclass, Kbdclass, Sparrow };

static enum FRU_Type fruType[NDEVICES] = { 
     Print, Eventlog, Rdr, ServiceControlManager, BROWSER, aci78xx, SaveDump, Mouclass, Kbdclass, Sparrow};

static char *fruName[NDEVICES] = { 
     "Print", "Eventlog", "Rdr", "ServiceControlManager", "BROWSER", "aci78xx", "Save Dump", "Mouclass", "Kbdclass", "Sparrow"};

struct Element {
  LONG  arrivalTime;
  int   frameSize;  /* interarrival time */
};
 
struct FRU {
  enum FRU_Type type;
  char     *name;
  struct   Element buffer[BUFFER_SIZE];  /* circular buffer */
  int      top;  /* index to last element in buffer */
  int      counter; /* number of filled entries in the buffer */
  int      edi[FRAME_COUNT][FRAME_COUNT-1];  /* EDIs per frame */
  LONG     lastArrivalTime;  /* Time of arrival of most recent error */
  int      rulesFired[NRULES];
  LONG     errorCount;
};

/* 
   Define an FRU (Field Replacement Unit) for every device (error category)
   monitored - System dependent
*/


/* Interface */
#if defined(__cplusplus)
extern "C"{
#endif /* C++ */

void  resetBuffer(struct FRU deviceId);
void  addBufferEntry(struct FRU deviceId, struct Element element);
int recordError(LONG arrivalTime, struct FRU deviceId);
int applyDFTRules(struct FRU deviceId);
int test4in1Rule(struct FRU deviceId);
int test3_3Rule(struct FRU deviceId);
int test2_2Rule(struct FRU deviceId);
int test4DecreasingRule(struct FRU deviceId);
void computeEDIs(struct FRU deviceId);
int frame(int top, int number);
void initFRU(struct FRU deviceId, enum FRU_Type type, char *name);

#if defined(__cplusplus)
}
#endif /* C++ */


#endif  /* of DFT_h */
  
