/* **************************************************************
  crashMonitor.cpp
  Author: Ying 

  This program monitor system crashes by writing time stamps every
  INTERVAL seconds on a file.
  
  If records the time when the program was run for the first time
  and computes the downtime after crashes. It estimates the time of
  the crash within INTERVAL seconds (currently 5 minutes - 300 sec) 
  of the actual crash, by using the time when the time stamp file 
  was last modified.

  Files created/used/modified by program
  timeStamp.avail: daily time stamps written every INTERVAL/(60*1000) minutes
                   to monitor system crashes
  timeRef.avail:   records time when program was started by the first time.
                   Time reference to compute availability
  downTime.avail:  preserves the cumulative down time across crashes.
  crashHistory.out: report of outages (date/time)

*************************************************************** */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <string.h>
#include <stdlib.h>
#include <process.h>
#include <windows.h>
#include <winbase.h>
#include "availability.h"


struct stat file_st;
time_t refTime;     /* Time program was run for the first time */
time_t last;        /* timeStamp File last modification time */
time_t n_td, o_td;
time_t upTime, totalTime;
float availability;


void periodic_task()
{
  time_t  tt = time(NULL);

  fopen(timeStampFile, "a");
  fprintf(timeStamp,"%ld\n", tt);
  fclose(timeStamp);
}

void writeTimeStamp()
{
  time_t  tt1=time(NULL); 
  time_t  tt0=time(NULL);
  
  timeStamp = fopen(timeStampFile, "w");
  if (!timeStamp) {
     fprintf(stderr, "Could not create time stamp file: %s\n", timeStampFile);
     exit(1);
  }
  fclose(timeStamp);


  while(1)
  {
  periodic_task();
  Sleep(INTERVAL);
  }

}

int checkFile(char *fileName)
{
   FILE *fd;

   fd = fopen(fileName, "r");
   if (!fd) return 0;   /* File does not exist */
   fscanf(fd, "%d", &refTime);
   fclose(fd);
   return -1;           /* File did exist */
}

void crashMon()
{
  char    *strTime;
  time_t  tt;

  tt = time(NULL);
  if (checkFile(timeRefFile) >= 0) {  /* First time program is run */
      /* Create availability files */
      timeRef = fopen(timeRefFile, "w");
      if (!timeRef) {
	 fprintf(stderr, "Could not create time reference file: %s\n", 
		         timeRefFile);
	 exit(1);
      }
      fprintf(timeRef, "%d\n", tt);
      fclose(timeRef);

      downTime = fopen(downTimeFile, "w");
      if (!downTime) {
	 fprintf(stderr, "Could not create down time file: %s\n", 
		         downTimeFile);
	 exit(1);
      }
      fprintf(downTime, "0\n");
      fclose(downTime);

      history = fopen(historyFile, "w");
      if (!history) {
	 fprintf(stderr, "Could not create outage history file: %s\n", 
		         historyFile);
	 exit(1);
      }
      fclose(history);
       
      writeTimeStamp();  /* Infinite Loop - exited only if system crashes */  
  }
  else {  /* There was a system crash */
    tt = time(NULL);   /* Get current time */

    /* check time of stamp file */
    timeStamp = fopen(timeStampFile, "r");
    if (!timeStamp) {
	 fprintf(stderr, "Could not open time stamp file: %s\n", 
		         timeStampFile);
	 exit(1);
    }
    stat(timeStampFile, &file_st);
    last = file_st.st_mtime;
    n_td = tt - last; /* down time = current time - timestamp last mod. time */
    /* get previous down time */
    downTime = fopen(downTimeFile, "r+");
    if (!downTime) {
	 fprintf(stderr, "Could not open down time file: %s\n", 
		         downTimeFile);
	 exit(1);
    }
    fscanf(downTime, "%d", &o_td);  
    o_td += n_td;

    /* Udate down time file */
    fseek(downTime, 0, SEEK_SET);
    fprintf(downTime, "%d\n", o_td);
    fclose(downTime);

    /* Update outage history file */
    history = fopen(historyFile, "a");
    if (!history) {
	 fprintf(stderr, "Could not open outage history file: %s\n", 
		         historyFile);
	 exit(1);
    }
    strTime = ctime(&last);
    strTime[strlen(strTime)-1]='\0';
    fprintf(history, "%s\tdown time=%d (sec)\n", 
                      strTime, n_td);
    fclose(history);
        
    writeTimeStamp();  /* Infinite Loop - exited only if system crashes */     
  }
}



