/* ******************************************************************  
  availability.cpp
  Author: Ying 

  This program is run everyday at midnight to create a daily report
  of the availability and status of the machine.

  Availability is computed using as time reference (t=0) the time when the
  program that monitors machine crashes (crashMonitor) is executed for the
  first time. This time is preserved in the file timeRef.avail.

  The status of the machine gives a snapshot of the machine's "health" by 
  computing the time intervals between the timeStamps created by the program
  crashMonitor and reporting their offsets (in seconds) from the time 
  interval specified. Currently, this time interval is set to 5 minutes 
  (300 seconds). 

  The availability and status reports are written to the file availability.out
  and status.out respectively.

  Files read by this program:
  - timeStamp.avail: daily time stamps used to monitor system crashes.
  - timeRef.avail:   time reference to compute availability.
  - downTime.avail:  preserves the cumulative down time across crashes. 

  Files created/updated by this program:
  - availability.out
  - status.out

  The format of the files is as follows:
  Status:
  **** Date ****
  offset1\toffset2\toffset3\toffset4\toffset5\offset6\offset7\toffset8\n
  ...
  **** Date ****
  offset1\toffset2\toffset3\toffset4\toffset5\offset6\offset7\toffset8\n
  ...

  Availability:
  Date\tAvailability%\n
  Date\tAvailability%\n
  ...
*/

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>
#include <string.h>
#include <stdlib.h>
#include <process.h>


#include "availability.h"

FILE * openFile(char *fileName)
{
   FILE *fd;
   fd = fopen(fileName, "r");
   if (!fd) {   /* File does not exist */
      fd = fopen(fileName, "w");
      if (!fd) {
	  fprintf(stderr, "Could not create file: %s\n", fileName);
	  exit(1);
      }
   }
   else { /* File exists */
      fclose(fd);
      fd = fopen(fileName, "a");
      if (!fd) {
	  fprintf(stderr, "Could not open file for append: %s\n", fileName);
	  exit(1);
      }
   }
   return fd;
}
float computeAvailability(time_t current_time)
{
   time_t  r_time, d_time, totalTime;
   float   availability;
   /* Get Time Reference */
   timeRef = fopen(timeRefFile, "r");
   if (!timeRef) {
       fprintf(stderr, "Could not open timeRef file: %s\n", timeRefFile);
       exit(1);
   }   
   fscanf(timeRef, "%d", &r_time);
   fclose(timeRef);
   /* Get Down Time */
   downTime = fopen(downTimeFile, "r");
   if (!downTime) {
       fprintf(stderr, "Could not open down time file: %s\n", downTimeFile);
       exit(1);
   }
   fscanf(downTime, "%d", &d_time);  
   fclose(downTime);
   totalTime = current_time - r_time;  
   /* availability = upTime or (total-down) / total */
   availability = (float) (totalTime - d_time)/(float)(totalTime);
   return(availability);
}
void AvailCalc()
{
  time_t t1, t2, t3;
  char   *date;
  int    tmp;
  int    counter=0;
  float availability;

  timeStamp = fopen(timeStampFile, "r");
  if (!timeStamp) {
     fprintf(stderr, "Could not read time stamp file: %s\n", timeStampFile);
     exit(1);
  }
  status = openFile(statusFile);
  avail =  openFile(availFile);
  t1 = time(NULL);
  date = ctime(&t1);
  date[strlen(date)-1]='\0';
  /* Write Availability Report */
  availability = computeAvailability(t1);
  fprintf(avail, "%s\t%2.2f\n", date, availability*100);
  fclose(avail);
  /* Write Status Report */ 
  fprintf(status, "**** %s ****\n", date);
  while (!feof(timeStamp)) {
     tmp = fscanf(timeStamp, "%d", &t1);
     tmp = fscanf(timeStamp, "%d", &t2);
     if (tmp != EOF) {
        t3 = t2 - t1 - INTERVAL;
	fprintf(status, "%d\t", t3);   /* Print time difference in seconds */
	counter++;
	if (counter == 8) {
	   fprintf(status, "\n");
	   counter=0;
	}
     }     
     else break;
  }
  fprintf(status, "\n");
  fclose(status);

  /* Delete "contents" (not the file) of timeStamp */
  fclose(timeStamp);
  timeStamp = fopen(timeStampFile, "w");
  fclose(timeStamp);
}// end AvailCalc()

void main()
{
	AvailCalc();
}// end of main

