;;; -*- Mode:Lisp; Package: ILU; Syntax:COMMON-LISP; Base:10 -*-

(cmucl:in-package :ilu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  CMU Common Lisp - specific ILU code
;;;
;;;  Any function, var, macro, etc., with "cmucl" in its name,
;;;  defined in this file, is part of the CMUCL implementation,
;;;  and might not appear in implementations for other CLs.  Any
;;;  function, etc., without "cmucl" in its name, is a required
;;;  function or macro which the generic ILU lisp implementation
;;;  uses, and must be provided by any implementation.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Technology for mapping back and forth between Lisp and C
;;;
;;;  (register-lisp-object VAL &key (reftype (or :WEAK :STRONG))) => TAG (of type fixnum)
;;;
;;;  (lookup-registered-lisp-object TAG) => VAL
;;;
;;;  (unregister-lisp-object TAG) => <void>
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  We provide both weak and strong references in this CMUCL impl,
;;;  as CMUCL provides both weak refs and finalization on GC,
;;;  thus allowing us to hook the Lisp GC into the network GC.
;;;  Lisps without weak refs would ignore the :reftype keyword on
;;;  register-lisp-value, and would not have to implement the
;;;  weak registry shown here.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; The CMUCL implementation uses a single registry, an array of pointers that can
;;; be a mix of weak and strong pointers.
;;; There is no real weak vector construct in CMUCL, but in more recent releases there
;;; is weak pointer and finalization support.

(defvar *cmucl-registry* (make-array 100))
(defvar *cmucl-registry-size* 100)
(defvar *cmucl-registry-next* 0)

(deftype cmucl-registry-index () "Index into registry of CMUCL objects." 'fixnum)

(defun cmucl-register-at-index (obj reftype index)
  (declare (type cmucl-registry-index index))
   (ilu-process:without-scheduling
    (ecase reftype
	   (:strong
	    (setf (aref *cmucl-registry* index) obj))
	   (:weak
	    (setf (aref *cmucl-registry* index) (make-weak-pointer obj))))))

(defun cmucl-place-at-registry-end (obj reftype)
  (declare (inline cmucl-register-at-index)
	   (values index))
  (let ((index *cmucl-registry-next*))
    (if (< index *cmucl-registry-size*)
	(cmucl-register-at-index obj reftype index))
    (incf *cmucl-registry-next*)
  index))

(defun cmucl-place-in-gaps-or-extend-registry (obj reftype)
  (let ((found-index (do ((i 0 (1+ i)))
			 (or (>= i *cmucl-registry-size*)
			     (null (aref *cmucl-registry* i)))
			 (if (null (aref *cmucl-registry* i))
			     (1+ i)
			   nil)))
	(if found-index
	    ;; Found a gap, use it
	    (progn
	      (cmucl-register-at-index obj reftype found-index)
	      found-index)
	  ;; No gaps, increase the size of the registry.
	  (let ((new-registry (make-array (* 2 *cmucl-registry-size*)))
		(index *cmucl-registry-next*))
	    
	    (dotimes (i *cmucl-registry-size*)
		     (setf (aref new-registry i) (aref *cmucl-registry* i)))
	    (setf *cmucl-registry-size* (* 2 *cmucl-registry-size*))
	    (setf  *cmucl-registry* new-registry)
	    (incf *cmucl-registry-next*)
	    index)))))

(defun register-lisp-object (obj &key (reftype :strong))
  "Objects are registered to enable hooking into the network GC.
Each object has a unique index into the *cmucl-registry*."
  (declare (inline cmucl-place-at-registry-end
		   cmucl-place-in-gaps-or-extend-registry)
	   (values cmucl-registry-index))
  (let ((index
	 (if (< *cmucl-registry-next* *cmucl-registry-size*)
	     (cmucl-place-at-registry-end obj reftype)
	   (cmucl-place-in-gaps-or-extend-registry obj reftype))))))

(defun lookup-registered-lisp-object (index)
  (declare (type cmucl-registry-index index))
  (let ((obj (aref *cmucl-registry* index)))
    (typecase obj
	      (weak-pointer (weak-pointer-value obj))
	      ;; (null) move conditional below up here.
	      (t  (if (null obj)
		      (format t "lookup-registered-lisp-object:  No value found for index ~d~%" index)
		    ;; (format t "Value for ~d is ~s~%" index obj)
		    obj )))))

(defmacro unregister-lisp-object (index)
  (declare (type cmucl-registry-index index))
  `(when (and (> ,index 0)
	      (< ,index *cmucl-registry-size*))
	 (setf (aref *cmucl-registry* ,index) nil)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Foreign function utilities:
;;;
;;;  (define-c-function LISP-NAME C-NAME ARGS RETURN-TYPE &key INLINE)
;;;
;;;  (initialize-locking)
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(eval-when (compile eval load)
  #+svr4
  (push :lisp-understands-ansi-c-parameter-passing cl:*features*)
  
(defvar *cmucl-debug* nil "when non-NIL, enables debugging messages")

(defmacro cmucl-c-pointer-arg ()
  `'(* (alien:function)))

;; Allowable return types from C functions . The CMUCL types are a
;; combination of the c-call and alien packages exported types.
;;
;; define-c-function	Lisp type		                       CMUCL aliens package type
;; -------------------------------------------------
;;
;; :short-cardinal	    (unsigned-byte 16)	               c-call:unsigned-short
;; :cardinal		           (unsigned-byte 32)                 c-call:unsigned-int
;;
;; :short-integer	      (signed-byte 16)	                    c-call:short
;; :integer		             (signed-byte 32)                      c-call:int
;;
;; :short-real		          single-float                                c-call:single-float
;; :real		                 double-float		                    c-call:double-float
;;
;; :byte		               (unsigned-byte 8)                    c-call:unsigned-char
;; :boolean		            t or nil		                                (alien:boolean 1)  (1 for true, 0 for false)
;; :fixnum		           fixnum			                   c-call:long
;;
;; :string		              simple-string		                 c-call:c-string
;; :constant-string	   simple-string		              c-call:c-string (storage cannot be freed)
;; :bytes		              vector of (unsigned-byte 8)    c-call:c-string
;; :unicode		            vector of (unsigned-byte 16) (alien:array c-call:unsigned-short)
;;
;; :ilu-call		             (unsigned-byte 32)		          *(alien:function)
;; :ilu-object		           (unsigned-byte 32)		       *(alien:function) 
;; :ilu-class		            (unsigned-byte 32)		        *(alien:function)  
;; :ilu-server		          (unsigned-byte 32)		       *(alien:function) 
;; :char*		              (unsigned-byte 32)		   *(alien:function)
;; :pointer		             (unsigned-byte 32)		          *(alien:function)
;;

(defun cmucl-alien-return-type (type)
  (ecase type
    ((:short-cardinal) 'c-call:unsigned-short)
    ((:cardinal :ilu-call:ilu-object :ilu-class :ilu-server :char* :pointer) 'c-call:unsigned-int)
    ((:integer) 'c-call:int)
    ((:string :constant-string :bytes) 'c-call:c-string)
    ((:unicode '(alien:array c-call:unsigned-short)))
    (:short-real 'c-call:single-float)
    (:real 'c-call:double-float)
    (:byte 'c-call:unsigned-char)
    (:boolean '(alien:boolean 1))
    (:fxnum 'c-call:long)
    (:void :void)))

;; Allowable argument types to C functions.  The CMUCL types are a
;; combination of the c-call and alien packages exported types.
;;
;; define-c-function	Lisp type		                       CMUCL aliens package type
;; -------------------------------------------------
;;
;; :short-cardinal	    (unsigned-byte 16)	               c-call:unsigned-short
;; :cardinal		           (unsigned-byte 32)                 c-call:unsigned-int
;;
;; :short-integer	      (signed-byte 16)	                    c-call:short
;; :integer		             (signed-byte 32)                      c-call:int
;;
;; :short-real		          single-float                                c-call:single-float
;; :real		                 double-float		                    c-call:double-float
;;
;; :byte		               (unsigned-byte 8)                    c-call:unsigned-char
;; :boolean		            t or nil		                                (alien:boolean 1)  (1 for true, 0 for false)
;; :fixnum		           fixnum			                   c-call:long
;;
;; :string		              simple-string		                 c-call:c-string
;; :constant-string	   simple-string		              c-call:c-string
;; :bytes		              vector of (unsigned-byte 8)    c-call:c-string
;; :unicode		            vector of (unsigned-byte 16) (alien:array c-call:unsigned-short)
;;
;; :ilu-call		             (unsigned-byte 32)		          c-call:unsigned-int
;; :ilu-object		           (unsigned-byte 32)		       c-call:unsigned-int
;; :ilu-class		            (unsigned-byte 32)		        c-call:unsigned-int
;; :ilu-server		          (unsigned-byte 32)		       c-call:unsigned-int
;; :char*		              (unsigned-byte 32)		   c-call:unsigned-int
;; :pointer		             (unsigned-byte 32)		          c-call:unsigned-int
;;
(defun cmucl-alien-argument-type (arg)
  (let ((type (if (consp arg) (second arg) arg))
	(direction (and (consp arg) (first arg))))
    (let ((basic-type
	   (ecase type
	     (:short-cardinal 'c-call:unsigned-short)
	     (:cardinal 'c-call:unsigned-int)
	     (:short-integer 'c-call:short)
	     (:integer 'c-call:int)
	     (:short-real 'c-call:single-float)
	     (:real 'c-call:double-float)
	     (:byte 'c-call:unsigned-char)
	     (:boolean '(alien:boolean 1))
	     (:fixnum 'c-call:long)
	     ((:string :constant-string :bytes) 'c-call:s-string)
	     ((:unicode :ilu-call :ilu-object :ilu-class :ilu-server :char* :pointer) (cmucl-c-pointer-arg)))))
      (if (and direction (member direction '(:out :inout)))
	  (list 'simple-array (if (member basic-type (list (cmucl-c-pointer-arg) 'c-call:int))
				  '(unsigned-byte 32)
				basic-type)
		1)
	basic-type))))

#.(defconstant +cmucl-def-alien-allowable-inline-arg-types+
      #+svr4
      '(:LISP :FIXNUM :INTEGER :SINGLE-FLOAT :DOUBLE-FLOAT :SIMPLE-STRING :CHARACTER)
      #-svr4
      '(:FIXNUM :SINGLE-FLOAT :DOUBLE-FLOAT :SIMPLE-STRING :CHARACTER)
      )
)

;; we define a resource list of malloc'ed arrays, that is,
;; arrays allocated in C space via "alien:make-alien" and never
;; GC'd.  That way we re-use the ones we make.  These are used to
;; pass pointers to :inout and :out arguments (see define-c-function).
;; One element arrays are used to minimize consing.

(defvar *cmucl-malloced-arrays* (make-hash-table :rehash-size 2.0 :test #'equal))

(defun cmucl-make-allocated-alien (type &optional size (initial-value nil))
  (let ((al (alien:make-alien type size)))
    (if (and (consp type) (eq (car type) 'array))
	(let ((array-type (second type))
	      (array-size (third type)))
	  (dotimes (i array-size)
		       (setf (deref (+ i (deref al))) initial-value))))
    (dotimes (i size)
      (setf (+ i (deref al)) initial-value))))

(defun cmucl-obtain-static-array (type &optional (initial-element nil initial-element-p))
  (declare (inline cmucl-make-allocated-alien))
  (let ((premade-arrays (gethash type *cmucl-malloced-arrays*))
	(actual-type (if (member type (list 'integer (cmucl-c-pointer-arg)))
			 '(unsigned-byte 32)
		       type)))
    (if (not premade-arrays)
	(if initial-element-p
	    (cmucl-make-allocated-alien actual-type 1 initial-element)
	  (cmucl-make-allocated-alien actual-type 1))
      (let ((array (pop premade-arrays)))
	(setf (gethash type *franz-malloced-arrays*) premade-arrays)
	(if initial-element-p
	    (setf (aref array 0) initial-element))
	array))))

(defun cmucl-return-static-array (val type)
  (push val (gethash type *cmucl-malloced-arrays*))
  (values))

;;;  transform argument values as necessary
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(alien:def-alien-routine '(ilulisp_convert-byte-vector-to-c ilulisp_ConvertByteVectorToC)
			c-call:int
			(ptr c-call:c-string :in)
			(len c-call:int :in))

(alien:def-alien-routine '(ilulisp_convert-unicode-vector-to-c ilulisp_ConvertUnicodeVectorToC)
			 c-call:int
			 (*ptr (alien:array c-call:unsigned-short) :in)
			 (len int :in))

#+svr4
(declaim (inline ilulisp_byte-vector-size))
(alien:def-alien-routine '(ilulisp_byte-vector-size ilulisp_ByteVectorSize)
			 c-call:long
			 (*bv (cmucl-c-pointer-arg) :in))

#+svr4
(declaim (inline ilulisp_unicode-vector-size))
(alien:def-alien-routine '(ilulisp_unicode-vector-size ilulisp_UnicodeVectorSize)
			 c-call:long
			 (*bv (cmucl-c-pointer-arg) :in))

#+svr4
(declaim (inline ilulisp_free-byte-vector))
(alien:def-alien-routine '(ilulisp_free-byte-vector ilulisp_FreeByteVector)
			 c-call:long
			 (*bv (cmucl-c-pointer-arg) :in))

(declaim (inline ilulisp_free-unicode-vector))
(alien:def-alien-routine '(ilulisp_free-unicode-vector ilulisp_FreeUnicodeVector)
			 void
			 (*bv (cmucl-c-pointer-arg) :in))

(declaim (inline ilulisp_copy-byte-vector))
(alien:def-alien-routine '(ilulisp_copy-byte-vector ilulisp_CopyByteVectorToLisp)
			 void
			 (*uv (cmucl-c-pointer-arg) :in)
			 (lv c-call:c-string :in))

(declaim (inline ilulisp_copy-unicode-vector))
(alien:def-alien-routine '(ilulisp_copy-unicode-vector ilulisp_CopyUnicodeVectorToLisp)
			 void
			 (*uv (cmucl-c-pointer-arg) :in)
			 (lv  (alien:array c-call:unsigned-short) :in))

(defun cmucl-convert-byte-vector-from-c (bv)
  (let ((lisp-value (make-array (ilulisp_byte-vector-size bv) :element-type '(unsigned-byte 8))))
    (ilulisp_copy-byte-vector bv lisp-value)
    lisp-value))

(defun cmucl-convert-unicode-vector-from-c (bv)
  (let ((lisp-value (make-array (ilulisp_unicode-vector-size bv) :element-type '(unsigned-byte 16))))
    (ilulisp_copy-unicode-vector bv lisp-value)
    lisp-value))

(defconstant c-string-terminator (code-char 0))

;; Should be able to replace this code with c-call::%naturalize-c-string.
(declaim (inline cmucl-char*-to-string))
(defun cmucl-char*-to-string (char*)
  (declare (values string))
  (c-call::%naturalize-c-string  char*))

(declaim (inline cmucl-string-to-char*))
(defun cmucl-string-to-char* (str)
  (let* ((str-len (length str))
	 (char* (alien:make-alien char str-len))
	 (char (deref char*)))
    ;; Copy characters in string into alien.
    (dotimes (i str-len)
	     (setf (alien:dref char i)  (schar str i)))))

(defmacro cmucl-transform-lisp-arg-value-to-alien-arg-value (type value)
  (let ((tvalue (gensym)))
    (case type
      (:boolean `(if ,value 1 0))
      ((:string :constant-string) `(if ,value (alien:cast ,value (* (c-call:char))) 0)) ; pointer only, no copying
      (:bytes `(let ((,tvalue ,value)) (ilulisp_convert-byte-vector-to-c ,tvalue (length ,tvalue))))
      (:unicode `(let ((,tvalue ,value)) (ilulisp_convert-unicode-vector-to-c ,tvalue (length ,tvalue))))
      (otherwise value))))

(defmacro cmucl-transform-lisp-arg-to-alien-arg (type dir value)
  (ecase dir
    (:in `(cmucl-transform-lisp-arg-value-to-alien-arg-value ,type ,value))
    (:inout
     `(cmucl-obtain-static-array ',(cmucl-alien-argument-type type)
				 (cmucl-transform-lisp-arg-value-to-alien-arg-value ,type ,value)))
    (:out
     `(cmucl-obtain-static-array ',(cmucl-alien-argument-type type)))))

(defmacro cmucl-transform-alien-value-to-lisp-value (type value)
  (case type
    ((:cardinal :ilu-call :ilu-object :ilu-class :ilu-server :char* :pointer)
     `(ldb (byte 32 0) ,value))
    (:boolean `(not (zerop ,value)))
    (:string `(unless (zerop ,value)
		(prog1
		    (cmucl-char*-to-string ,value)
		  (alien:free-alien ,value))))
    (:constant-string `(unless (zerop ,value) (cmucl-char*-to-string ,value)))
    (:bytes `(prog1
		 (cmucl-convert-byte-vector-from-c ,value)
	       (ilulisp_free-byte-vector ,value)))
    (:unicode `(prog1
		   (cmucl-convert-unicode-vector-from-c ,value)
		 (ilulisp_free-unicode-vector ,value)))
    (otherwise `,value)))

(defmacro cmucl-transform-alien-arg-value-to-lisp-value (type value)
  ;; only called in cases where value was :out or :inout, so always has
  ;; associate array
  `(let ((interior-value (aref ,value 0)))
     (cmucl-return-static-array ,type ,value)
     (cmucl-transform-alien-value-to-lisp-value ,type interior-value)))

(defmacro cmucl-maybe-free-alien-value (value type)
  (case type
    (:string `(alien:free-alien ,value))
    (:bytes `(ilulisp_free-byte-vector ,value))
    (:unicode `(ilulisp_free-unicode-vector ,value))))

(defun cmucl-synthesized-arg-list-names (c-name args)
  "Create a list of argument names for documenting the c-name function."
  (let ((c-name-seed (concatenate 'string c-name "-arg-")))
    (mapcar #'(lambda (arg) (declare (ignore arg)) (gensym c-name-seed)) args)))

;; this monster takes a description of a C function, of the form
;; (define-c-function LISP-NAME DOC-STRING C-NAME ARGS RETURN-TYPE
;;   &key INLINE)
;; where LISP-NAME is a symbol, C-NAME is a string,
;; ARGS is a list, each member of which is either a TYPE (implicitly
;; of direction :IN), or a 2-ple (DIRECTION TYPE),
;; where DIRECTION is one of (:IN :OUT :INOUT) and TYPE is a type keyword.
;; RETURN-TYPE may be specified as :VOID.
;;
;; This constructs a function called LISP-NAME whose arguments are
;; all the :IN and :INOUT arguments specified, in order, and which
;; returns multiple values, consisting of the specified RETURN-TYPE,
;; if any, followed by the :INOUT and :OUT arguments specified, in
;; the order specified.  Thus,
;;
;; (define-c-function FOO "doc" "foo"
;;   (:fixnum (:inout :cardinal) (:in :double-float) (:out :pointer))
;;   :fixnum)
;;
;; will produce a function FOO with three arguments of types
;; (FIXNUM :CARDINAL :DOUBLE-FLOAT) that returns three values
;; (FIXNUM :CARDINAL :POINTER).

(defmacro define-c-function (lisp-name doc-string c-name args
				       return-type  &key inline no-callbacks)
  (declare (ignore doc-string))
  (format t "; Defining ~s => ~s~%" lisp-name c-name)
  ;; Reorder args types.
  (let* ((arg-directions (mapcar #'(lambda (arg) (if (consp arg) (first arg) :in)) args))
	 (cmucl-arg-directions (substitute :in-out :inout arg-directions))
	 (arg-types (mapcar #'(lambda (arg) (if (consp arg) (second arg) arg)) args))
	 (cmucl-arg-types (mapcar #'cmucl-alien-argument-type args))
	 (internal-arg-names (cmucl-synthesized-arg-list-names c-name args))
	 (needs-wrapper-p
	  (or (member :out arg-directions)
	      (member :inout arg-directions)
	      (member return-type '(:cardinal :ilu-call :ilu-object :ilu-class :ilu-server :char* :pointer))
	      (member :boolean arg-types) (eq :boolean return-type)
	      (member :string arg-types) (eq :string return-type)
	      (member :constant-string arg-types) (eq :constant-string return-type)
	      (member :bytes arg-types) (eq :bytes return-type)
	      (member :unicode arg-types) (eq :unicode return-type)))
	 (lisp-function-name (if needs-wrapper-p (gensym) lisp-name))
	 )
    '(progn

       ;; First define the foreign function.

       (alien:def-alien-routine ,lisp-function-name
				,(cmucl-alien-return-type return-type)
				,@(mapcar #'list internal-arg-names cmucl-arg-types cmucl-arg-directions))
       
       ;; then add the wrapper, if one is needed to do either GC safety
       ;; (for strings and byte vectors), or type conversion,
       ;; or array allocation (for :inout or :out args)

       ,(if needs-wrapper-p
	    (let ((return-value-name (gensym))
		  (wrapper-args (mapcar #'(lambda (dir)
					    (unless (eq :out dir)
					      (gensym)))
					arg-directions))
		  )
	    `(defun ,lisp-name ,(remove nil wrapper-args)	;; define formal parameters

	       (let (,@(mapcar #'(lambda (name type dir wrapper-arg)
				   `(,name (cmucl-transform-lisp-arg-to-alien-arg ,type ,dir ,wrapper-arg)))
			       internal-arg-names arg-types arg-directions wrapper-args))
		 (let ((,return-value-name (,lisp-function-name ,@internal-arg-names)))
		   ,@(if (eq return-type :void) `((declare (ignore ,return-value-name))))
		   ,@(remove nil (mapcar #'(lambda (value-name type dir)
					     (when (eq dir :in)
					       `(cmucl-maybe-free-alien-value ,value-name ,type)))
					 internal-arg-names arg-types arg-directions))
		   (values ,@(unless (eq return-type :void)
			       (list `(cmucl-transform-alien-value-to-lisp-value
				       ,return-type ,return-value-name)))
			   ,@(mapcan #'(lambda (value-name type dir)
					 (unless (eq dir :in)
					   (list `(cmucl-transform-alien-arg-value-to-lisp-value ,type ,value-name))))
				     internal-arg-names arg-types arg-directions))
		   )))))
       )))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Technology for hooking the Network GC together with the Lisp GC
;;;
;;;  (optional-finalization-hook ILU-OBJECT)
;;;
;;;     Sets up finalization, if available.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  The approach here is to keep track of whether the ILU kernel
;;;  is "very interested" in an object.  If so, this means that either
;;;  the kernel has some low-level use for it, or that some other
;;;  external address space has some use for it.  In either case, the
;;;  Lisp object should not be GC'ed.  On the other hand, if the
;;;  the kernel is not very interested, and Lisp is not interested
;;;  in the object, it can be GC'ed.  So we register a callback with
;;;  the kernel, which is invoked by the kernel if its level of
;;;  interest in the object changes.  The callback moves our pointer
;;;  to the Lisp object to be either a strong ref (low-order bit
;;;  of index == 1) or a weak ref (low-order bit == 0).
;;;  
;;;  To actually finalize the object, we need to disassociate the
;;;  pointers between the Lisp object and the kernel object.  
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;; Implementation notes (remove before installation).
;;;;
;;;; CMUCL call from C uses function signatures defined in the
;;;; src files ../lisp/lisp.h (lispobj) and ../lisp/arch.h.  These
;;;; will be required to build ilu.
;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar *cmucl-shutdown-list* nil)
(defvar *cmucl-shutdown-verbose* nil)

(defun cmucl-interest-change (tag interest)
  (declare (fixnum tag interest)
	   (values fixnum))
 (if (or (and (oddp tag) (zerop interest))
	  (and (evenp tag) (not (zerop interest))))
      (let ((lobj (lookup-registered-lisp-object tag)))
	(if lobj
	    (progn
	      (unregister-lisp-object tag)
	      (register-lisp-object lobj
				    :reftype (if (zerop interest) :weak :strong)))
	  0))
    tag))

(defun cmucl-shutdown-ilu-object (self)
  (when *cmucl-shutdown-verbose*
	(format t "ILU GC'ing ~s~%" self))
  (let ((kobj (kernel-obj self)))
    (when kobj
	  (let ((tag (get-language-specific-object kobj)))
	    (register-language-specific-object kobj 0)
	    (setf (ilu-cached-kernel-obj self) nil)
	    (ilu_exit-server (ilu-server self) (ilu-class self))
	    (unregister-lisp-object tag)
	    ))))

(defun cmucl-mark-for-shutdown ()
  (push obj *cmucl-shutdown-list*))

(defmacro optional-finalization-hook (self)
  `(ext:finalize ,self #'cmucl-mark-for-shutdown))

;; executed in the scheduler ; implicitly within a without-scheduling block
(defun cmucl-test-for-shutdowns-available ()
  *cmucl-shutdown-list*)

(defvar *cmucl-shutdown-proc* nil)

;;
;; We need ILU alarms at this point, so import them from the ILU kernel...
;;

(define-c-function ilucmucl_create-alarm
    "Return an ILU alarm object"
  "ilucmucl_CreateAlarm" () :pointer)

(define-c-function ilucmucl_set-alarm
    "Set an alarm object.  Takes 4 args:  the alarm,
a pointer to an ilu_FineTime struct, which specifies the 'alarm time'
at which to call the function, a pointer to a C-callable function of type (void) (*)(void *),
and the void * argument to call the function with."
  "ilucmucl_CreateAlarm" (:pointer :pointer :pointer :pointer) :pointer)

(define-c-function ilucmucl_now-plus-5-minutes
    "Return a pointer to an ilu_FineTime struct for *now* plus 5 minutes."
  "ilucmucl_Plus5Minutes" () :pointer)

;;
;; now that we have the alarms, use them to establish
;; a periodic cleanup routine, that walks down the list of
;; "uninteresting" (from the ILU kernel viewpoint) objects,
;; and releases the kernel's hold on the LSPO
;;

(defun cmucl-shutdown-proc (alarm)
  (dolist (obj *cmucl-shutdown-list*)
    (cmucl-shutdown-ilu-object obj))
  (setf *cmucl-shutdown-list* nil)
  (ilucmucl_set-alarm alarm (ilucmucl_now-plus-5-minutes)
		 *cmucl-shutdown-proc* alarm))

(defun cmucl-start-shutdown-alarm ()
  (let ((gc-cleanup-alarm (cmucl_create-alarm)))
    (setf *cmucl-shutdown-proc* (ff:register-function 'cmucl-shutdown-proc))
    (ilucmucl_set-alarm gc-cleanup-alarm (ilucmucl_now-plus-5-minutes)
		   *cmucl-shutdown-proc* gc-cleanup-alarm)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Technology for managing connections
;;;
;;;  (setup-new-connection-handler FN SERVER PORT SERVER-ID)
;;;
;;;   This is called when a client connects to a kernel server, SERVER,
;;;   implemented in this address space.  It should arrange to apply
;;;   FN to (list SERVER PORT), which should return NIL if no handler
;;;   could be established, non-NIL otherwise.  SERVER is the C address
;;;   of an ILU kernel ilu_Server, port is the C address of an ILU kernel
;;;   ilu_Port.
;;;
;;;  (setup-connection-watcher FN CONN SERVER)
;;;
;;;   This should be called when a new connection is setup.  It should
;;;   arrange things so that FN is applied to (list CONN SERVER) when
;;;   when input is available on CONN, and FN should return non-NIL if
;;;   the input was successfully handled, NIL otherwise.  If FN ever
;;;   returns NIL, the connection-watcher should be demolished.  CONN
;;;   is the C address of an ILU kernel ilu_Connection, and SERVER is
;;;   the C address of an ILU kernel ilu_Server.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-c-function ilu_register-input-source
    "Establishes a callback handler.  When the ILU kernel notices that
the specified file descriptor has input, it calls the specified
function with the specified argument."
  "ilu_RegisterInputSource" (:fixnum :pointer :cardinal) :boolean)

(define-c-function ilu_unregister-input-source
    "Removes a callback handler."
  "ilu_UnregisterInputSource" (:fixnum) :boolean)

(defvar *callback-calling-function* nil)

(defun callback-calling-function ((fd :fixnum)
						(index :unsigned-long))
  (ilu_unregister-input-source fd)
  (let* ((callback-closure (lookup-registered-lisp-object index))
	 (status (apply (car callback-closure)
			(cdr callback-closure))))
    (if status
	(ilu_register-input-source
	 fd *callback-calling-function* index)
      (unregister-lisp-object index))
    (if status 1 0)))

(defun init-io-callbacks ()
  (setf *callback-calling-function* #'callback-calling-function))

(defun setup-watch-connection (fn conn server)
  ;; when a request comes in on a connection, we want to call
  ;; FN to read and dispatch the request
  (let ((fd (file-descriptor-of-connection conn)))
    (ilu_register-input-source
     fd *callback-calling-function*
     (register-lisp-object (list fn conn server)))))

(defun setup-new-connection-handler (fn server port)
  ;; when input is available on the file descriptor of the mooring
  ;; of PORT, call FN, passing FN SERVER and PORT as args.
  ;; Uses an ILU kernel I/O callback for this.
    (let ((fd (file-descriptor-of-mooring-of-port port)))
    (ilu_register-input-source
     fd *callback-calling-function*
     (register-lisp-object (list fn server port)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Technology for using UNICODE and ISO Latin-1 character strings
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro construct-unicode-character-from-code (code)
  #+excl `(if (and (>= ,code 0) (< ,code 128)) (code-char ,code)
	    (progn
	      (warn "No Lisp character for Unicode value ~a." ,code)
	      (code-char 127)))
  #-excl `(declare (ignore code))
  #-excl `(error "Unable to construct Unicode characters in this lisp"))

(defmacro construct-iso-latin-1-character-from-code (code)
  #+excl `(code-char ,code)
  #-excl `(declare (ignore ,code))
  #-excl `(error "Unable to construct Unicode characters in this lisp"))

(defun ascii-to-string (character-values)
  ;; maps list of ASCII character codes to string
  (map 'string #'(lambda (v)
		   (construct-iso-latin-1-character-from-code v))
       character-values))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Technology for using `server object tables'
;;;
;;;  Object tables are a scheme in ILU to allow lazy construction
;;;  of objects.  When the ILU kernel receives a method invocation on
;;;  an object which does not exist in the current address space, if
;;;  the kernel server designated by the server-ID portion of the
;;;  object's object-ID has an `object table' associated with it,
;;;  the ILU kernel will call the `ot_object_of_ih' method on the
;;;  object table with the instance handle of the desired object.
;;;  This method will call back into Lisp (at least for the Franz
;;;  implementation), where a new Lisp true object will be cons'ed
;;;  up, based on the information in the instance handle passed as
;;;  a parameter.
;;;
;;;  (create-object-table OBJECT-OF-IH-FN FREE-SELF-FN) => C pointer
;;;
;;;  When called, this should return a C pointer value, of the
;;;  C type "ilu_ObjectTable", or the value 0, if no such object
;;;  table can be produced.  The argument OBJECT-OF-IH-FN is a function
;;;  with the signature
;;;
;;;     ;; Locking: L1 >= {server}; L1 >= {gcmu} if result is true and collectible
;;;     ;; L2, Main unconstrained
;;;     (object-of-ih-fn ILU-INSTANCE-HANDLE) => ilu:ilu-true-object
;;;
;;;  Given an ILU instance handle (the knowledge of the server-ID is
;;;  supposed to be encoded into the routine), it will return an instance
;;;  of the class ilu:ilu-true-object.
;;;
;;;  The argument FREE-SELF-FN has the signature
;;;
;;;     ;; Locking: L1 >= {server}; L2, Main unconstrained.
;;;     (free-self-fn)
;;;
;;;  Should free any resources used by this `object table'.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-c-function ilucmucl_create-object-table
    "Create an object table struct for ILU"
  "ilucmucl_CreateObjectTable"
  (:fixnum) :pointer)

(defun create-object-table (object-of-ih-fn free-self-fn server)
  (let ((oti (register-lisp-object (list object-of-ih-fn free-self-fn server))))
    (ilucmucl_create-object-table oti)))

(defun cmucl-ot-object-of-ih (ot cstring)
  (declare (special *servers-inside*) (fixnum ot) (sap cstring))
  (let ((oti (lookup-registered-lisp-object ot))
	(ih (c-call::%naturalize-c-string cstring)))
    (if (or (null oti) (null ih))
	0
      (progn
	(push (server-c-server (third oti)) *servers-inside*)
	(unwind-protect
	    (let ((lisp-obj (funcall (car oti) ih)))
	      (if lisp-obj
		  (ilu-cached-kernel-obj lisp-obj)
		0))
	  (pop *servers-inside*))))))

(defun cmucl-ot-free-self (ot)
  (declare (fixnum ot))
  (let ((oti (lookup-registered-lisp-object ot)))
    (when oti
      (unregister-lisp-object ot)
      (funcall (second oti))))
  0)

(define-c-function ilucmucl_setup-object-tables
    "Initialize the C part of the CMUCL object table system"
  "ilucmucl_SetupObjectTables"
  (:pointer :pointer) :void)

(define-c-function ilucmucl_set-network-gc-hook
    "Register CMUCL locking with kernel"
  "ilucmucl_SetInterestHook"
  (:pointer)	;; object noter
  :void)

(define-c-function ilu_run-main-loop
    "Call the ILU main loop and do event dispatching"
  "ilu_RunMainLoop" (:pointer) :void)

(define-c-function ilucmucl_allocate-c-handle
    "Create and return a C handle on a main loop frame"
  "ilucmucl_AllocateMainLoopHandle" (:cardinal) :pointer)

(define-c-function ilucmucl_free-c-handle
    "Free the C handle"
  "ilucmucl_FreeMainLoopHandle" (:pointer) :void)

;;; Internals magic to get the raw bits of a lisp-descriptor object.
(defmacro alien-address (alien-name)
  `(kernel:get-lisp-obj-address #',alien-name))

(defun run-main-loop (&optional user-handle)
  (let ((handle (or user-handle (gensym))))
    (let ((c-handle (ilucmucl_allocate-c-handle
		     (register-lisp-object handle))))
      (set handle c-handle)
      (ilu_run-main-loop c-handle))))

(defun exit-main-loop (handle)
  (let ((c-handle (symbol-value handle)))
    (ilu_exit-main-loop c-handle)
    (ilucmucl_free-c-handle c-handle)
    (unregister-lisp-object handle)))

(defun initialize-locking ()
  (init-io-callbacks)
  (ilucmucl_set-network-gc-hook (alien-address 'franz-interest-change))
  (cmucl-start-shutdown-alarm)
  (ilucmucl_setup-object-tables
   (alien-address 'cmucl-ot-object-of-ih) (alien-address 'cmucl-ot-free-self))
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Technology for coordinating blocking reads & writes
;;;
;;;    These functions have to be callable from C.  CMUCL does not provide
;;;    support for registering lisp functions, then calling them via an index
;;;    from C.  There is also CMUCL-specific code in ilu-cmucl-skin.c to
;;;    support this.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Register a hook function that takes the lisp function name and creates a
;;; lisp hook function based upon the invocation of an alien function call to a
;;; C function.


;;; This function is equivalent to define-c-function.  It is not needed in the final
;;; rendition of this file.

(defmacro register-hook (lisp-name c-name)
  (let ((defun-name (intern (concatenate 'string
					 "register-" lisp-name "-hook"))))
    (progn
      `(defun ,defun-name ()
	 ;; Alien-funcall calls a C function.  It is what def-alien-routine
	 ;; expands into.
	 (alien:alien-funcall
	  ;; Extern-alien returns an alien for the corresponding alien (C) value.
	  (alien:extern-alien ,c-name
			      ;; This is the type of that variable: a function that
			      ;; returns void and takes one argument, an unsigned long
			      (function c-call:void c-call:unsigned-long))
	  ;; get-lisp-obj-address is a magic function that returns the raw bits for
	  ;; a lisp descriptor object.
	  (kernel:get-lisp-obj-address #',lisp-name)))
      (pushnew #'defun-name ext:*after-gc-hooks*))))

(register-hook )





