;;; -*-Scheme-*-
;;;
;;; $Id: sos.scm,v 1.28 1993/06/13 21:49:44 cph Exp $
;;;
;;; Copyright (c) 1993 Massachusetts Institute of Technology
;;;
;;; This material was developed by the Scheme project at the
;;; Massachusetts Institute of Technology, Department of Electrical
;;; Engineering and Computer Science.  Permission to copy this
;;; software, to redistribute it, and to use it for any purpose is
;;; granted, subject to the following restrictions and understandings.
;;;
;;; 1. Any copy made of this software must include this copyright
;;; notice in full.
;;;
;;; 2. Users of this software agree to make their best efforts (a) to
;;; return to the MIT Scheme project any improvements or extensions
;;; that they make, so that these may be included in future releases;
;;; and (b) to inform MIT of noteworthy uses of this software.
;;;
;;; 3. All materials developed as a consequence of the use of this
;;; software shall duly acknowledge such use, in accordance with the
;;; usual standards of acknowledging credit in academic research.
;;;
;;; 4. MIT has made no warrantee or representation that the operation
;;; of this software will be error-free, and MIT is under no
;;; obligation to provide any services, by way of maintenance, update,
;;; or otherwise.
;;;
;;; 5. In conjunction with products arising from the use of this
;;; material, there shall be no use of the name of the Massachusetts
;;; Institute of Technology nor of any adaptation thereof in any
;;; advertising, promotional, or sales literature without prior
;;; written consent from MIT in each case.

;;;; Scheme Object System
;;;  Loosely based on "tiny-clos" version 1.2 by Gregor Kiczales.  The
;;;  ideas and some of the design are the same, but most of the code
;;;  has been rewritten from scratch.

;;; Need hooks to generate descriptions for instances.

(declare (usual-integrations))
(declare (integrate-external "wrapper"))

;;;; Porting Aids

(define-integrable (weak-cons car cdr)
  (system-pair-cons (ucode-type weak-cons) car cdr))

(define-syntax define-smacro
  (lambda body
    `(DEFINE-MACRO ,@body)))

(define-smacro (serror procedure message . objects)
  `(ERROR ,message ,@objects))

(define (write-hash object port)
  (write-char #\space port)
  (write (hash object) port))

(define-integrable (get-slot-key-property name)
  (hash-table/get slot-key-database name #f))

(define-integrable (put-slot-key-property name index)
  (hash-table/put! slot-key-database name index))

(define slot-key-database
  (make-symbol-hash-table))

(define-integrable (make-procedure-records-table)
  (make-object-hash-table))

(define-integrable (get-procedure-record table procedure)
  (hash-table/get table procedure #f))

(define-integrable (put-procedure-record table procedure record)
  (hash-table/put! table procedure record))

;;; The following runtime system hooks must be reset in order to
;;; prevent lossage when this file is re-loaded.

(let ((environment (->environment '(RUNTIME RECORD))))
  (set! (access record-type-initialization-hook environment) #f))

(let ((environment (->environment '(RUNTIME UNPARSER))))
  (set! (access hook/unparse-record environment) #f)
  (set! (access hook/procedure-unparser environment) #f))

;;;; Memory Substrate

(define-structure (%instance (constructor %%make-instance))
  wrapper
  slots)

(define-integrable (%make-instance class nslots)
  (%%make-instance (class-wrapper class) (make-vector nslots)))

(define-integrable (%instance-class instance)
  (wrapper-class (%instance-wrapper instance)))

(define-integrable (%instance-ref instance index)
  (vector-ref (%instance-slots instance) index))

(define-integrable (%instance-set! instance index new-value)
  (vector-set! (%instance-slots instance) index new-value))

(define-integrable (instance? object)
  (and (%record? object)
       (eq? %instance (%record-ref object 0))))

(define-integrable (guarantee-instance instance procedure)
  (if (not (instance? instance))
      (error:wrong-type-argument instance "instance" procedure)))

(define-integrable (%class<=? c1 c2)
  (memq c2 (class-precedence-list c1)))

(define-integrable (%instance-of? i c)
  (%class<=? (%instance-class i) c))

(define (class? object)
  (and (instance? object)
       (%instance-of? object <class>)
       #t))

(define-integrable (guarantee-class class procedure)
  (if (not (class? class))
      (error:wrong-type-argument class "class" procedure)))

(define (instance-class instance)
  (guarantee-instance instance 'INSTANCE-CLASS)
  (%instance-class instance))

(define (instance-of? object class)
  (and (%class<=? (object-class object) class)
       #t))

(define-integrable (guarantee-symbol-or-false name procedure)
  (if (not (or (not name) (symbol? name)))
      (error:wrong-type-argument name "symbol or name" procedure)))

(define (object-class object)
  (cond ((not (%record? object))
	 (built-in-object-class object))
	((eq? %instance (%record-ref object 0))
	 (%instance-class object))
	((and (%record? (%record-ref object 0))
	      (eq? record-type-type (%record-ref (%record-ref object 0) 0)))
	 (%record-type-class (%record-ref object 0)))
	(else
	 (built-in-object-class object))))

(define-integrable (object-wrapper object)
  (cond ((not (%record? object))
	 (class-wrapper (built-in-object-class object)))
	((eq? %instance (%record-ref object 0))
	 (%instance-wrapper object))
	((and (%record? (%record-ref object 0))
	      (eq? record-type-type (%record-ref (%record-ref object 0) 0)))
	 (%record-type-class-wrapper (%record-ref object 0)))
	(else
	 (class-wrapper (built-in-object-class object)))))

(define (built-in-object-class object)
  ;; This temporary definition is replaced below.
  object
  <object>)

;;;; Slot Descriptors

(define-structure (slot (constructor %make-slot))
  name
  class
  allocation
  initializer
  plist
  key
  %accessor
  %modifier
  %initpred)

(define (make-slot name class allocation initializer plist)
  (%make-slot name class allocation initializer plist
	      (get-slot-key name) #f #f #f))

(define-integrable (guarantee-slot slot procedure)
  (if (not (slot? slot))
      (error:wrong-type-argument slot "slot descriptor" procedure)))

(define-integrable (%instance-slot? slot)
  (eq? (slot-allocation slot) 'INSTANCE))

(define (slot-plist-lookup plist key default)
  (let loop ((plist* plist))
    (if (pair? plist*)
	(if (pair? (cdr plist*))
	    (if (eq? key (car plist*))
		(cadr plist*)
		(loop (cddr plist*)))
	    (error:bad-plist plist 'SLOT-PLIST-LOOKUP))
	(if (null? plist*)
	    default
	    (error:bad-plist plist 'SLOT-PLIST-LOOKUP)))))

(define (error:bad-plist plist procedure)
  (error:wrong-type-argument plist "property list" procedure))

(define (slot-property slot key #!optional default)
  (if (default-object? default)
      (let ((default (list 'DEFAULT)))
	(let ((value (slot-plist-lookup (slot-plist slot) key default)))
	  (if (eq? value default)
	      (serror 'SLOT-PROPERTY "No such key in property list:" key)
	      value)))
      (slot-plist-lookup (slot-plist slot) key default)))

(define-integrable (slot-name? object)
  (symbol? object))

(define-integrable (guarantee-slot-name name procedure)
  (if (not (slot-name? name))
      (error:wrong-type-argument name "slot name" procedure)))

(define (get-slot-key name)
  (or (get-slot-key-property name)
      (let ((index current-slot-key))
	(set! current-slot-key (fix:+ current-slot-key 1))
	(put-slot-key-property name index)
	index)))

(define current-slot-key 0)

(define (sort-slot-descriptors descriptors)
  (list->vector
   (sort descriptors (lambda (x y) (fix:< (slot-key x) (slot-key y))))))

(define (find-slot-descriptor descriptor-vector key)
  (let loop ((low 0) (high (fix:- (vector-length descriptor-vector) 1)))
    (if (fix:< high low)
	#f
	(let ((probe (fix:quotient (fix:+ high low) 2)))
	  (let ((descriptor (vector-ref descriptor-vector probe)))
	    (let ((key* (slot-key descriptor)))
	      (cond ((fix:= key key*) descriptor)
		    ((fix:< key key*) (loop low (fix:- probe 1)))
		    (else (loop (fix:+ probe 1) high)))))))))

;;;; Slot Accessors

(define-integrable (instance->slot instance name v1 v2 cache procedure)
  (guarantee-instance instance procedure)
  (cond ((eq? (%instance-wrapper instance) (system-pair-car v1))
	 (system-pair-cdr v1))
	((eq? (%instance-wrapper instance) (system-pair-car v2))
	 (system-pair-cdr v2))
	((car cache)
	 (or (probe-cache-1 (car cache) (%instance-wrapper instance))
	     (slot-accessor-computer (%instance-wrapper instance) name
				     v1 v2 cache procedure)))
	(else
	 (slot-accessor-computer (%instance-wrapper instance) name
				 v1 v2 cache procedure))))

(define (slot-accessor name)
  (let ((v1 (weak-cons #f #f))
	(v2 (weak-cons #f #f))
	(cache (list #f)))
    (lambda (instance)
      ((slot-%accessor
	(instance->slot instance name v1 v2 cache 'SLOT-ACCESSOR))
       instance))))

(define (slot-modifier name)
  (let ((v1 (weak-cons #f #f))
	(v2 (weak-cons #f #f))
	(cache (list #f)))
    (lambda (instance value)
      ((slot-%modifier
	(instance->slot instance name v1 v2 cache 'SLOT-MODIFIER))
       instance value))))

(define (slot-initpred name)
  (let ((v1 (weak-cons #f #f))
	(v2 (weak-cons #f #f))
	(cache (list #f)))
    (lambda (instance)
      ((slot-%initpred
	(instance->slot instance name v1 v2 cache 'SLOT-INITPRED))
       instance))))

(define (slot-accessor-computer wrapper name v1 v2 cache procedure)
  (let ((value (%slot-descriptor (wrapper-class wrapper) name procedure)))
    (let ((w1 (system-pair-car v1))
	  (w2 (system-pair-car v2)))
      (cond ((or (car cache) (and w1 w2))
	     (if (not (car cache))
		 (let ((cache*
			(fill-cache (fill-cache (new-cache)
						(list w1)
						(system-pair-cdr v1))
				    (list w2)
				    (system-pair-cdr v2))))

		   (without-interrupts
		    (lambda ()
		      (set-car! cache cache*)
		      (system-pair-set-car! v1 #f)
		      (system-pair-set-cdr! v1 #f)
		      (system-pair-set-car! v2 #f)
		      (system-pair-set-cdr! v2 #f)))))
	     (set-car! cache
		       (fill-cache (car cache) (list wrapper) value)))
	    ((not w1)
	     (without-interrupts
	      (lambda ()
		(system-pair-set-car! v1 wrapper)
		(system-pair-set-cdr! v1 value))))
	    (else
	     (without-interrupts
	      (lambda ()
		(system-pair-set-car! v2 wrapper)
		(system-pair-set-cdr! v2 value))))))
    value))

(define (slot-value instance name)
  ((slot-%accessor
    (%slot-descriptor (instance-class instance) name 'SLOT-VALUE))
   instance))

(define (set-slot-value! instance name value)
  ((slot-%modifier
    (%slot-descriptor (instance-class instance) name 'SET-SLOT-VALUE!))
   instance value))

(define (slot-initialized? instance name)
  ((slot-%initpred
    (%slot-descriptor (instance-class instance) name 'SLOT-INITIALIZED?))
   instance))

(define (slot-descriptor class name)
  (guarantee-class class 'SLOT-DESCRIPTOR)
  (%slot-descriptor class name 'SLOT-DESCRIPTOR))

(define (%slot-descriptor class name procedure)
  (if (slot? name)
      (if (eq? class (slot-class name))
	  name
	  (%%slot-descriptor class (slot-name name) (slot-key name)))
      (begin
	(guarantee-slot-name name procedure)
	(%%slot-descriptor class name (get-slot-key name)))))

(define (%%slot-descriptor class name key)
  (or (find-slot-descriptor (class-instance-slots class) key)
      (find-slot-descriptor (class-virtual-slots class) key)
      (error:unbound-slot class name)))

(define (initialize-instance-slot-accessors! slots)
  (do ((slots slots (cdr slots))
       (i 0 (fix:+ i 1)))
      ((null? slots))
    (let ((slot (car slots)))
      (set-slot-%accessor! slot (%instance-slot-accessor slot i))
      (set-slot-%modifier! slot (%instance-slot-modifier slot i))
      (set-slot-%initpred! slot (%instance-slot-initpred slot i)))))

(define-smacro (generate-index-cases slot index limit procedure)
  `(CASE ,index
     ,@(let loop ((i 0))
	 (if (= i limit)
	     `((ELSE (,procedure ,slot ,index)))
	     `(((,i) (,procedure ,slot ,i)) ,@(loop (+ i 1)))))))

(define-integrable (%%instance-slot-accessor slot index)
  (lambda (instance)
    (let ((value (%instance-ref instance index)))
      (if (eq? instance-slot-uninitialized value)
	  (error:uninitialized-instance-slot instance slot)
	  value))))

(define-integrable (%%instance-slot-modifier slot index)
  slot
  (lambda (instance value)
    (%instance-set! instance index value)))

(define-integrable (%%instance-slot-initpred slot index)
  slot
  (lambda (instance)
    (not (eq? instance-slot-uninitialized (%instance-ref instance index)))))

(define (%instance-slot-accessor slot index)
  (generate-index-cases slot index 16 %%instance-slot-accessor))

(define (%instance-slot-modifier slot index)
  (generate-index-cases slot index 16 %%instance-slot-modifier))

(define (%instance-slot-initpred slot index)
  (generate-index-cases slot index 16 %%instance-slot-initpred))

(define instance-slot-uninitialized
  (list 'INSTANCE-SLOT-UNINITIALIZED))

;;;; Classes

(define <class>
  (let ((slot-names
	 '(NAME
	   DIRECT-SUPERCLASSES
	   DIRECT-SLOTS
	   PRECEDENCE-LIST
	   INSTANCE-SLOTS
	   VIRTUAL-SLOTS
	   WRAPPER)))
    (let ((<class> (%%make-instance #f (make-vector (length slot-names)))))
      (let ((wrapper (make-wrapper <class>)))
	(set-%instance-wrapper! <class> wrapper)
	(%instance-set! <class> 6 wrapper))
      (%instance-set! <class> 0 '<CLASS>)
      (%instance-set! <class> 2 (map list slot-names))
      (let ((slots
	     (map (lambda (name)
		    (make-slot name <class> 'INSTANCE #f '()))
		  slot-names)))
	(initialize-instance-slot-accessors! slots)
	(%instance-set! <class> 4 (sort-slot-descriptors slots)))
      (%instance-set! <class> 5 (sort-slot-descriptors '()))
      <class>)))

;;; <CLASS> is now mostly defined -- all that is missing is its direct
;;; superclasses and precedence list.  These will be filled in as soon
;;; as the superclasses are defined.  First, we must bootstrap the
;;; instance variable lookup mechanim:

(define class-instance-slots #f)
(define class-virtual-slots #f)

(set! class-instance-slots (lambda (class) (%instance-ref class 4)))
(set! class-virtual-slots (lambda (class) (%instance-ref class 5)))

(let ((accessor (slot-accessor 'INSTANCE-SLOTS)))
  (accessor <class>)
  (set! class-instance-slots accessor))

(let ((accessor (slot-accessor 'VIRTUAL-SLOTS)))
  (accessor <class>)
  (set! class-virtual-slots accessor))

(define set-class-instance-slots! (slot-modifier 'INSTANCE-SLOTS))
(define set-class-virtual-slots! (slot-modifier 'VIRTUAL-SLOTS))

(define class-name (slot-accessor 'NAME))
(define set-class-name! (slot-modifier 'NAME))

(define class-direct-superclasses (slot-accessor 'DIRECT-SUPERCLASSES))
(define set-class-direct-superclasses! (slot-modifier 'DIRECT-SUPERCLASSES))

(define class-direct-slots (slot-accessor 'DIRECT-SLOTS))
(define set-class-direct-slots! (slot-modifier 'DIRECT-SLOTS))

(define class-precedence-list (slot-accessor 'PRECEDENCE-LIST))
(define set-class-precedence-list! (slot-modifier 'PRECEDENCE-LIST))

(define class-wrapper (slot-accessor 'WRAPPER))
(define set-class-wrapper! (slot-modifier 'WRAPPER))

(define (class-nslots class)
  (vector-length (class-instance-slots class)))

(define (slot-descriptors class)
  (append! (vector->list (class-instance-slots class))
	   (vector->list (class-virtual-slots class))))

(define (t-make-class name direct-superclasses direct-slots)
  (let ((class (%make-instance <class> (class-nslots <class>)))
	(direct-slots (map list direct-slots)))
    (set-class-name! class name)
    (set-class-direct-superclasses! class direct-superclasses)
    (set-class-direct-slots! class direct-slots)
    (let ((precedence-list
	   (cons
	    class
	    (let loop ((classes direct-superclasses))
	      (if (null? classes)
		  '()
		  (cons (car classes)
			(loop (class-direct-superclasses (car classes)))))))))
      (set-class-precedence-list! class precedence-list)
      (let ((slots
	     (map (lambda (slot)
		    (make-slot (car slot) class 'INSTANCE #f '()))
		  (append-map class-direct-slots precedence-list))))
	(initialize-instance-slot-accessors! slots)
	(set-class-instance-slots! class (sort-slot-descriptors slots))))
    (set-class-virtual-slots! class (sort-slot-descriptors '()))
    (set-class-wrapper! class (make-wrapper class))
    class))

(define <object>
  (t-make-class '<OBJECT> '() '()))

(define <instance>
  (t-make-class '<INSTANCE> (list <object>) '()))

(set-class-direct-superclasses! <class> (list <instance>))
(set-class-precedence-list! <class>
			    (cons <class> (class-precedence-list <instance>)))

(define <primitive-class>
  (t-make-class '<PRIMITIVE-CLASS> (list <class>) '()))

;;; Change the metaclass of <OBJECT> to be <PRIMITIVE-CLASS>.  This
;;; works because sorting of instance slots guarantees that the
;;; instance slots of <PRIMITIVE-CLASS> are in the same order as those
;;; of <CLASS>.
(set-%instance-wrapper! <object> (class-wrapper <primitive-class>))

(define <generic>
  (t-make-class '<GENERIC>
		(list <instance>)
		'(NAME MIN-ARITY MAX-ARITY METHODS PROCEDURE RECORD)))

(define <method>
  (t-make-class '<METHOD>
		(list <instance>)
		'(SPECIALIZERS PROCEDURE)))

(define <leaf-method>
  (t-make-class '<LEAF-METHOD> (list <method>) '(LEAF-PROCEDURE)))

;;;; Generic Procedures

(define generic-name (slot-accessor 'NAME))
(define set-generic-name! (slot-modifier 'NAME))

(define generic-min-arity (slot-accessor 'MIN-ARITY))
(define set-generic-min-arity! (slot-modifier 'MIN-ARITY))

(define generic-max-arity (slot-accessor 'MAX-ARITY))
(define set-generic-max-arity! (slot-modifier 'MAX-ARITY))

(define generic-methods (slot-accessor 'METHODS))
(define set-generic-methods! (slot-modifier 'METHODS))

(define generic->procedure (slot-accessor 'PROCEDURE))
(define set-generic->procedure! (slot-modifier 'PROCEDURE))

(define generic-record (slot-accessor 'RECORD))
(define set-generic-record! (slot-modifier 'RECORD))

(define (t-make-generic-procedure name min-arity max-arity)
  (generic->procedure
   (let ((generic (%make-instance <generic> (class-nslots <generic>))))
     (set-generic-name! generic name)
     (set-generic-min-arity! generic min-arity)
     (set-generic-max-arity! generic max-arity)
     (set-generic-methods! generic '())
     (set-generic->procedure! generic (%make-generic-procedure generic))
     generic)))

(define (%make-generic-procedure generic)
  (let ((record (cons (null-emp generic) generic)))
    (set-generic-record! generic record)
    (let ((procedure
	   (let ((min-arity (generic-min-arity generic))
		 (max-arity (generic-max-arity generic)))
	     (if (and (eqv? min-arity max-arity)
		      (<= 1 min-arity 4))
		 (cond ((= 1 min-arity)
			(lambda (arg1)
			  ((car record) arg1)))
		       ((= 2 min-arity)
			(lambda (arg1 arg2)
			  ((car record) arg1 arg2)))
		       ((= 3 min-arity)
			(lambda (arg1 arg2 arg3)
			  ((car record) arg1 arg2 arg3)))
		       (else
			(lambda (arg1 arg2 arg3 arg4)
			  ((car record) arg1 arg2 arg3 arg4))))
		 (lambda args
		   (apply (car record) args))))))
      (put-procedure-record generic-procedure-records procedure record)
      procedure)))

(define (generic-handler generic)
  (car (generic-record generic)))

(define (set-generic-handler! generic handler)
  (set-car! (generic-record generic) handler))

(define (%procedure->generic procedure name)
  (let ((record (get-procedure-record generic-procedure-records procedure)))
    (if (not record)
	(error:wrong-type-argument procedure "generic procedure" name))
    (cdr record)))

(define (procedure->generic object)
  (let ((record (get-procedure-record generic-procedure-records object)))
    (and record
	 (cdr record))))

(define (generic-procedure? object)
  (if (get-procedure-record generic-procedure-records object) #t #f))

(define generic-procedure-records
  (make-procedure-records-table))

;;;; Methods

(define method-specializers (slot-accessor 'SPECIALIZERS))
(define set-method-specializers! (slot-modifier 'SPECIALIZERS))

(define method-procedure (slot-accessor 'PROCEDURE))
(define set-method-procedure! (slot-modifier 'PROCEDURE))

(define leaf-method-procedure (slot-accessor 'LEAF-PROCEDURE))
(define set-leaf-method-procedure! (slot-modifier 'LEAF-PROCEDURE))

(define (t-make-method specializers procedure)
  (let ((method (%make-instance <method> (class-nslots <method>))))
    (set-method-specializers! method specializers)
    (set-method-procedure! method procedure)
    method))

(define (t-make-leaf-method specializers procedure)
  (let ((method (%make-instance <leaf-method> (class-nslots <leaf-method>))))
    (set-method-specializers! method specializers)
    (set-method-procedure! method (lmp->mp procedure))
    (set-leaf-method-procedure! method procedure)
    method))

(define (lmp->mp procedure)
  (lambda (call-next-method . args)
    call-next-method
    (apply procedure args)))

;;;; Generic Invocation Subprotocol

(define (add-method generic-procedure method)
  (let ((generic (%procedure->generic generic-procedure 'ADD-METHOD)))
    (for-each
     (lambda (method)
       (let ((methods (generic-methods generic))
	     (specializers (method-specializers method)))
	 (let loop ((tail methods))
	   (cond ((null? tail)
		  (set-generic-methods! generic (cons method methods)))
		 ((specializers=? specializers
				  (method-specializers (car tail)))
		  (set-car! tail method))
		 (else
		  (loop (cdr tail)))))))
     (let ((specializers (method-specializers method)))
       ;; Assumes that method instantiation has guaranteed that there
       ;; is at least one specializer.
       (if (> (length specializers) (generic-min-arity generic))
	   (serror 'ADD-METHOD
		   "Method has too many specializers:" method generic))
       (if (let loop ((specializers specializers))
	     (and (not (null? specializers))
		  (or (union-spec? (car specializers))
		      (loop (cdr specializers)))))
	   (map (if (%leaf-method? method)
		    (let ((procedure (leaf-method-procedure method)))
		      (lambda (specializers)
			(make-leaf-method specializers procedure)))
		    (let ((procedure (method-procedure method)))
		      (lambda (specializers)
			(make-method specializers procedure))))
		(let loop ((specializers specializers))
		  (let ((classes
			 (let ((specializer (car specializers)))
			   (if (union-spec? specializer)
			       (union-spec-classes specializer)
			       (list specializer)))))
		    (if (null? (cdr specializers))
			(map (lambda (class) (list class)) classes)
			(let ((tails (loop (cdr specializers))))
			  (append-map (lambda (class)
					(map (lambda (tail)
					       (cons class tail))
					     tails))
				      classes))))))
	   (list method))))
    (set-generic-handler! generic (compute-apply-generic generic))))

(define (reset-generic-procedure-cache procedure)
  (let ((generic
	 (%procedure->generic procedure 'RESET-GENERIC-PROCEDURE-CACHE)))
    (set-generic-handler! generic (compute-apply-generic generic))))
  
(define (specializers=? s1 s2)
  (cond ((null? s1)
	 (let loop ((s2 s2))
	   (or (null? s2)
	       (and (eq? <object> (car s2))
		    (loop (cdr s2))))))
	((null? s2)
	 (let loop ((s1 s1))
	   (and (eq? <object> (car s1))
		(or (null? (cdr s1))
		    (loop (cdr s1))))))
	(else
	 (and (eq? (car s1) (car s2))
	      (specializers=? (cdr s1) (cdr s2))))))

(define-structure union-spec
  classes)

(define (union-specializer . specializers)
  (make-union-spec
   (append-map (lambda (specializer)
		 (cond ((union-spec? specializer)
			(union-spec-classes specializer))
		       ((class? specializer)
			(list specializer))
		       ((record-type? specializer)
			(list (record-type-class specializer)))
		       (else
			(error:wrong-type-argument specializer
						   "method specializer"
						   'UNION-SPECIALIZER))))
	       specializers)))

(define (specializer? object)
  (or (class? object)
      (record-type? object)
      (union-spec? object)))

(define (null-emp generic)
  (lambda args
    (error:no-applicable-methods generic args)))

(define compute-apply-generic
  (t-make-generic-procedure 'COMPUTE-APPLY-GENERIC 1 1))

(define compute-effective-method-procedure
  (t-make-generic-procedure 'COMPUTE-EFFECTIVE-METHOD-PROCEDURE 2 2))

(define compute-methods
  (t-make-generic-procedure 'COMPUTE-METHODS 2 2))

(define compute-method-more-specific?
  (t-make-generic-procedure 'COMPUTE-METHOD-MORE-SPECIFIC? 1 1))

;; In order to add our first method to COMPUTE-APPLY-GENERIC, we need
;; something sitting there, so it can be called:
(set-generic-handler!
 (%procedure->generic compute-apply-generic #f)
 (lambda (generic)
   (leaf-method-procedure (car (generic-methods generic)))))

(define-smacro (define-unimethod name class bvl . body)
  ;; This macro simplifies the descriptions of the methods described
  ;; in this file, and furthermore makes it easy to name the lambda
  ;; expressions.
  (let ((leaf? (eq? 'CALL-NEXT-METHOD (car body))))
    (let ((bvl (if leaf? bvl `(CALL-NEXT-METHOD ,@bvl)))
	  (body (if leaf? (cdr body) body)))
      `(ADD-METHOD ,name
	 (,(if leaf? 'T-MAKE-LEAF-METHOD 'T-MAKE-METHOD)
	  (LIST ,class)
	  ,(if (and (symbol? name) (symbol? class))
	       `(NAMED-LAMBDA (,(symbol-append name ': class) ,@bvl) ,@body)
	       `(LAMBDA ,bvl ,@body)))))))

(define-unimethod compute-apply-generic <generic> (generic)
  call-next-method
  (let ((min-arity (generic-min-arity generic))
	(max-arity (generic-max-arity generic))
	(cache (list (new-cache))))
    (cond ((memq generic ground-generics)
	   ;; This ground case stops the infinite regression when
	   ;; adding a method to one of the generic procedures
	   ;; that is called when adding a method to a generic
	   ;; procedure.
	   (let ((procedure
		  (leaf-method-procedure
		   (car (last-pair (generic-methods generic)))))
		 (default (apply-generic generic cache)))
	     (lambda (generic . args)
	       (apply (if (memq generic ground-generics) procedure default)
		      generic
		      args))))
	  ((not (eqv? min-arity max-arity))
	   (apply-generic generic cache))
	  ((= 1 min-arity) (apply-generic-1 generic cache))
	  ((= 2 min-arity) (apply-generic-2 generic cache))
	  ((= 3 min-arity) (apply-generic-3 generic cache))
	  ((= 4 min-arity) (apply-generic-4 generic cache))
	  (else (apply-generic generic cache)))))

(define ground-generics
  (map procedure->generic
       (list compute-apply-generic
	     compute-effective-method-procedure
	     compute-methods
	     compute-method-more-specific?)))

(define (apply-generic generic cache)
  (let ((min-arity (generic-min-arity generic))
	(max-arity (generic-max-arity generic)))
    (let ((extra (and max-arity (- max-arity min-arity)))
	  (wna
	   (lambda (args)
	     (error:wrong-number-of-arguments generic
					      (if (= min-arity max-arity)
						  max-arity
						  (cons min-arity max-arity))
					      args))))
      (lambda args
	(apply (let ((wrappers
		      (let loop ((args* args) (n min-arity))
			(if (fix:= n 0)
			    (begin
			      (if (and extra
				       (let loop ((args* args*) (n extra))
					 (and (not (null? args*))
					      (or (fix:= n 0)
						  (loop (cdr args*)
							(fix:- n 1))))))
				  (wna args))
			      '())
			    (begin
			      (if (null? args*)
				  (wna args))
			      (cons (object-wrapper (car args*))
				    (loop (cdr args*) (fix:- n 1))))))))
		 (or (probe-cache (car cache) wrappers)
		     (compute-emp-and-store generic cache wrappers)))
	       args)))))

(define (apply-generic-1 generic cache)
  (lambda (arg1)
    (let ((w1 (object-wrapper arg1)))
      (let ((procedure (probe-cache-1 (car cache) w1)))
	(if procedure
	    (procedure arg1)
	    ((compute-emp-and-store generic cache (list w1)) arg1))))))

(define (apply-generic-2 generic cache)
  (lambda (arg1 arg2)
    (let ((w1 (object-wrapper arg1))
	  (w2 (object-wrapper arg2)))
      (let ((procedure (probe-cache-2 (car cache) w1 w2)))
	(if procedure
	    (procedure arg1 arg2)
	    ((compute-emp-and-store generic cache (list w1 w2)) arg1 arg2))))))

(define (apply-generic-3 generic cache)
  (lambda (arg1 arg2 arg3)
    (let ((w1 (object-wrapper arg1))
	  (w2 (object-wrapper arg2))
	  (w3 (object-wrapper arg3)))
      (let ((procedure (probe-cache-3 (car cache) w1 w2 w3)))
	(if procedure
	    (procedure arg1 arg2 arg3)
	    ((compute-emp-and-store generic cache (list w1 w2 w3))
	     arg1 arg2 arg3))))))

(define (apply-generic-4 generic cache)
  (lambda (arg1 arg2 arg3 arg4)
    (let ((w1 (object-wrapper arg1))
	  (w2 (object-wrapper arg2))
	  (w3 (object-wrapper arg3))
	  (w4 (object-wrapper arg4)))
      (let ((procedure (probe-cache-4 (car cache) w1 w2 w3 w4)))
	(if procedure
	    (procedure arg1 arg2 arg3 arg4)
	    ((compute-emp-and-store generic cache (list w1 w2 w3 w4))
	     arg1 arg2 arg3 arg4))))))

(define (compute-emp-and-store generic cache wrappers)
  (let ((procedure (compute-emp generic (map wrapper-class wrappers))))
    (set-car! cache (fill-cache (car cache) wrappers procedure))
    procedure))

(define (compute-emp generic classes)
  (let ((methods (compute-methods generic classes)))
    (if (null? methods)
	(null-emp generic)
	(or (compute-slot-emp generic classes methods)
	    (compute-effective-method-procedure generic methods)))))

(define-integrable (%leaf-method? method)
  (%instance-of? method <leaf-method>))

(define-unimethod compute-effective-method-procedure <generic>
  (generic methods)
  call-next-method
  (cond ((null? methods)
	 (null-emp generic))
	((%leaf-method? (car methods))
	 (leaf-method-procedure (car methods)))
	(else
	 (call-with-values (lambda () (methods->procedures methods))
	   (lambda (procedures last-is-leaf?)
	     (compute-emp-helper
	      generic
	      (car procedures)
	      (let loop ((procedures (cdr procedures)))
		(cond ((null? procedures)
		       (null-emp generic))
		      ((and (null? (cdr procedures)) last-is-leaf?)
		       (car procedures))
		      (else
		       (lambda args
			 (apply (car procedures)
				(loop (cdr procedures))
				args)))))))))))

(define (methods->procedures methods)
  (let loop ((methods methods) (result '()))
    (cond ((null? methods)
	   (values (reverse! result) #f))
	  ((%leaf-method? (car methods))
	   (values (reverse!
		    (cons (leaf-method-procedure (car methods)) result))
		   #t))
	  (else
	   (loop (cdr methods)
		 (cons (method-procedure (car methods)) result))))))

(define (compute-emp-helper generic procedure call-next-method)
  (let ((min-arity (generic-min-arity generic)))
    (cond ((or (not (eqv? min-arity (generic-max-arity generic)))
	       (not (<= 1 min-arity 4)))
	   (lambda args
	     (apply procedure call-next-method args)))
	  ((= 1 min-arity)
	   (lambda (arg1)
	     (procedure call-next-method arg1)))
	  ((= 2 min-arity)
	   (lambda (arg1 arg2)
	     (procedure call-next-method arg1 arg2)))
	  ((= 3 min-arity)
	   (lambda (arg1 arg2 arg3)
	     (procedure call-next-method arg1 arg2 arg3)))
	  (else
	   (lambda (arg1 arg2 arg3 arg4)
	     (procedure call-next-method arg1 arg2 arg3 arg4))))))

(define-unimethod compute-methods <generic> (generic classes)
  call-next-method
  (sort (let loop ((methods (generic-methods generic)))
	  (cond ((null? methods)
		 methods)
		((let loop
		     ((specializers (method-specializers (car methods)))
		      (classes classes))
		   (or (null? specializers)
		       (and (%class<=? (car classes) (car specializers))
			    (loop (cdr specializers) (cdr classes)))))
		 (cons (car methods) (loop (cdr methods))))
		(else
		 (loop (cdr methods)))))
	(let ((method-more-specific?
	       (compute-method-more-specific? generic)))
	  (lambda (m1 m2)
	    (method-more-specific? m1 m2 classes)))))

(define-integrable (class-more-specific? c1 c2 c3)
  (memq c2 (cdr (memq c1 (class-precedence-list c3)))))

(define-unimethod compute-method-more-specific? <generic> (generic)
  call-next-method generic
  (lambda (m1 m2 classes)
    (let loop
	((specializers1 (method-specializers m1))
	 (specializers2 (method-specializers m2))
	 (classes classes))
      (and (not (null? specializers1))
	   (or (null? specializers2)
	       (if (eq? (car specializers1) (car specializers2))
		   (loop (cdr specializers1) (cdr specializers2) (cdr classes))
		   (class-more-specific? (car specializers1)
					 (car specializers2)
					 (car classes))))))))

;;;; Instantiation Subprotocol

(define (make-instance class . initargs)
  (let ((instance (allocate-instance class)))
    (apply initialize-instance instance initargs)
    instance))

(define allocate-instance
  (t-make-generic-procedure 'ALLOCATE-INSTANCE 1 1))

(define-unimethod allocate-instance <class> (class)
  call-next-method
  (let ((slots (class-instance-slots class)))
    (if (not (%class<=? class <instance>))
	(serror 'ALLOCATE-INSTANCE
		"Can't allocate an instance of this class:" class))
    (let ((nslots (vector-length slots)))
      (let ((instance (%make-instance class nslots)))
	(do ((i 0 (fix:+ i 1)))
	    ((fix:= i nslots))
	  ((slot-%modifier (vector-ref slots i))
	   instance
	   (let ((initializer (slot-initializer (vector-ref slots i))))
	     (if initializer
		 (initializer)
		 instance-slot-uninitialized))))
	instance))))

(define initialize-instance
  (t-make-generic-procedure 'INITIALIZE-INSTANCE 1 #f))

(define-unimethod initialize-instance <instance> (instance)
  call-next-method instance
  unspecific)

(define-unimethod initialize-instance <generic>
  (generic name min-arity max-arity)
  (guarantee-symbol-or-false name 'INITIALIZE-INSTANCE)
  (if (not (and (exact-integer? min-arity) (> min-arity 0)))
      (error:wrong-type-argument min-arity "exact positive integer"
				 'INITIALIZE-INSTANCE))
  (if (not (or (not max-arity)
	       (and (exact-integer? max-arity) (>= max-arity min-arity))))
      (error:wrong-type-argument max-arity "maximum arity"
				 'INITIALIZE-INSTANCE))
  (call-next-method generic)
  (set-generic-name! generic name)
  (set-generic-min-arity! generic min-arity)
  (set-generic-max-arity! generic max-arity)
  (set-generic-methods! generic '())
  (set-generic->procedure! generic (%make-generic-procedure generic)))

(define-unimethod initialize-instance <method> (method specializers procedure)
  (if (not (and (list? specializers)
		(for-all? specializers specializer?)))
      (error:wrong-type-argument specializers "list of method specializers"
				 'INITIALIZE-INSTANCE))
  (if (not (procedure? procedure))
      (error:wrong-type-argument procedure "procedure" 'INITIALIZE-INSTANCE))
  (call-next-method method)
  (set-method-specializers! method
			    (map (lambda (specializer)
				   (if (record-type? specializer)
				       (record-type-class specializer)
				       specializer))
				 specializers))
  (set-method-procedure! method procedure))

(define-unimethod initialize-instance <leaf-method>
  (method specializers procedure)
  (if (not (procedure? procedure))
      (error:wrong-type-argument procedure "procedure" 'INITIALIZE-INSTANCE))
  (call-next-method method specializers (lmp->mp procedure))
  (set-leaf-method-procedure! method procedure))

;;;; Class Initialization Subprotocol

(define-unimethod initialize-instance <class>
  (class name direct-superclasses direct-slots)
  (guarantee-symbol-or-false name 'INITIALIZE-INSTANCE)
  (if (not (and (list? direct-superclasses)
		(for-all? direct-superclasses class?)))
      (error:wrong-type-argument direct-superclasses "direct superclasses"
				 'INITIALIZE-INSTANCE))
  (if (not (list? direct-slots))
      (error:wrong-type-argument direct-slots "direct slots"
				 'INITIALIZE-INSTANCE))
  (call-next-method class)
  (set-class-name! class name)
  (set-class-direct-superclasses! class
				  (if (null? direct-superclasses)
				      (list <instance>)
				      direct-superclasses))
  (set-class-direct-slots! class (map canonicalize-slot-argument direct-slots))
  (set-class-precedence-list! class (compute-precedence-list class))
  (let loop
      ((slots (compute-slots class))
       (instance-slots '())
       (virtual-slots '()))
    (cond ((null? slots)
	   (initialize-instance-slot-accessors! instance-slots)
	   (set-class-instance-slots!
	    class
	    (sort-slot-descriptors instance-slots))
	   (set-class-virtual-slots!
	    class
	    (sort-slot-descriptors virtual-slots))
	   (for-each (lambda (slot)
		       (call-with-values
			   (lambda ()
			     (compute-virtual-slot-accessors class slot))
			 (lambda (accessor modifier initpred)
			   (set-slot-%accessor! slot accessor)
			   (set-slot-%modifier! slot modifier)
			   (set-slot-%initpred! slot initpred))))
		     virtual-slots)
	   (install-slot-accessor-methods class))
	  ((%instance-slot? (car slots))
	   (loop (cdr slots)
		 (cons (car slots) instance-slots)
		 virtual-slots))
	  (else
	   (loop (cdr slots)
		 instance-slots
		 (cons (car slots) virtual-slots)))))
  (set-class-wrapper! class (make-wrapper class)))

(define compute-precedence-list
  (t-make-generic-procedure 'COMPUTE-PRECEDENCE-LIST 1 1))

(define-unimethod compute-precedence-list <class> (class)
  call-next-method
  (let ((elements (build-transitive-closure class-direct-superclasses class)))
    (topological-sort
     elements
     (build-constraints class-direct-superclasses elements)
     (lambda (partial-cpl elements)
       (let loop ((partial-cpl (reverse partial-cpl)))
	 (if (null? partial-cpl)
	     (serror 'COMPUTE-PRECEDENCE-LIST "Nothing valid."))
	 (let ((ds-of-ce
		(class-direct-superclasses (car partial-cpl))))
	   (let find-common ((elements elements))
	     (cond ((null? elements) (loop (cdr partial-cpl)))
		   ((memq (car elements) ds-of-ce) (car elements))
		   (else (find-common (cdr elements)))))))))))

(define compute-slots
  (t-make-generic-procedure 'COMPUTE-SLOTS 1 1))

(define-unimethod compute-slots <class> (class)
  call-next-method
  (let loop
      ((slots
	(append-map class-direct-slots (class-precedence-list class))))
    (if (null? slots)
	'()
	(let ((slot (car slots)))
	  (let ((name (car slot)))
	    (let inner ((slots (cdr slots)) (same '()) (diff '()))
	      (cond ((null? slots)
		     (cons (compute-slot-descriptor
			    class
			    (cons slot (reverse! same)))
			   (loop (reverse! diff))))
		    ((eq? name (caar slots))
		     (inner (cdr slots)
			    (cons (car slots) same)
			    diff))
		    (else
		     (inner (cdr slots)
			    same
			    (cons (car slots) diff))))))))))

(define compute-slot-descriptor
  (t-make-generic-procedure 'COMPUTE-SLOT-DESCRIPTOR 2 2))

(define-unimethod compute-slot-descriptor <class> (class slots)
  call-next-method
  (call-with-values
      (lambda ()
	(parse-slot-argument (merge-slot-arguments slots)))
    (lambda (name allocation initializer plist)
      (make-slot name class allocation initializer plist))))

(define compute-virtual-slot-accessors
  (t-make-generic-procedure 'COMPUTE-VIRTUAL-SLOT-ACCESSORS 2 2))

(define-unimethod compute-virtual-slot-accessors <class> (class slot)
  call-next-method
  (serror 'COMPUTE-VIRTUAL-SLOT-ACCESSORS
	  "No virtual slot accessors definition:" class slot))

;;;; Slot Arguments

(define (canonicalize-slot-argument argument)
  (cond ((slot-name? argument)
	 (list argument))
	((and (pair? argument)
	      (slot-name? (car argument))
	      (slot-argument-plist? (cdr argument)))
	 argument)
	(else
	 (serror 'CANONICALIZE-SLOT-ARGUMENT
		 "Malformed slot argument:" argument))))

(define (slot-argument-plist? object)
  (let loop ((l1 object) (l2 object))
    (if (pair? l1)
	(and (not (eq? (cdr l1) l2))
	     (symbol? (car l1))
	     (pair? (cdr l1))
	     (loop (cddr l1) (cdr l2)))
	(null? l1))))

(define (merge-slot-arguments slots)
  (if (null? (cdr slots))
      (car slots)
      (let ((slots (reverse slots)))
	(let ((result (list-copy (car slots))))
	  (for-each (lambda (slot)
		      (merge-slot-arguments! slot result))
		    (cdr slots))
	  result))))

(define (merge-slot-arguments! x y)
  (do ((x (cdr x) (cddr x)))
      ((null? x))
    (let ((key (car x))
	  (value (cadr x)))
      (let loop ((z (cdr y)))
	(cond ((null? z) (set-cdr! y (cons* key value (cdr y))))
	      ((eq? key (car z)) (set-car! (cdr z) value))
	      (else (loop (cddr z))))))))

(define (parse-slot-argument argument)
  (let ((allocation 'INSTANCE)
	(initializer #f))
    (let ((name (car argument))
	  (plist
	   (let loop ((plist (cdr argument)))
	     (if (null? plist)
		 '()
		 (let loop ((rest (loop (cddr plist))))
		   (case (car plist)
		     ((INITIALIZER)
		      (set! initializer (cadr plist))
		      rest)
		     ((ALLOCATION)
		      (set! allocation (cadr plist))
		      rest)
		     ((ACCESSOR MODIFIER INITPRED)
		      rest)
		     (else
		      (cons* (car plist) (cadr plist) rest))))))))
      (values name allocation initializer plist))))

;;;; Topological Sort

;;; Topologically sort a list of ELEMENTS.  CONSTRAINTS is the partial
;;; order, expressed as a list of pairs (X . Y) where X precedes Y.
;;; TIE-BREAKER is a procedure that is called when it is necessary to
;;; choose from multiple minimal elements; it is called with the
;;; partial result and the set of minimal elements as its arguments.

(define (topological-sort elements original-constraints tie-breaker)
  (let ((result (cons '() '())))
    (let ((add-to-result
	   (lambda (element)
	     (let ((tail (list element)))
	       (if (null? (car result))
		   (set-car! result tail)
		   (set-cdr! (cdr result) tail))
	       (set-cdr! result tail)))))
      (let loop
	  ((elements (list-copy elements))
	   (constraints (list-copy original-constraints)))
	(if (null? elements)
	    (car result)
	    (let ((minimal
		   (remove-if (lambda (element)
				(let loop ((constraints constraints))
				  (and (not (null? constraints))
				       (or (eq? (cdar constraints) element)
					   (loop (cdr constraints))))))
			      elements)))
	      (if (null? minimal)
		  (serror 'TOPOLOGICAL-SORT
			  "Invalid constraints:" original-constraints))
	      (let ((elements
		     (remove-if! (lambda (element)
				   (memq element minimal))
				 elements))
		    (constraints
		     (remove-if! (lambda (constraint)
				   (or (memq (car constraint) minimal)
				       (memq (cdr constraint) minimal)))
				 constraints)))
		(let break-ties ((minimal minimal))
		  (if (null? (cdr minimal))
		      (let ((choice (car minimal)))
			(add-to-result choice)
			(loop elements constraints))
		      (let ((choice (tie-breaker (car result) minimal)))
			(add-to-result choice)
			(break-ties (remove-item! choice minimal))))))))))))

(define (build-transitive-closure get-follow-ons element)
  (let loop ((result '()) (pending (list element)))
    (cond ((null? pending)
	   result)
	  ((memq (car pending) result)
	   (loop result (cdr pending)))
	  (else
	   (loop (cons (car pending) result)
		 (append (get-follow-ons (car pending)) (cdr pending)))))))

(define (build-constraints get-follow-ons elements)
  (let loop ((elements elements) (result '()))
    (if (null? elements)
	result
	(loop (cdr elements)
	      (let loop
		  ((element (car elements))
		   (follow-ons (get-follow-ons (car elements))))
		(if (null? follow-ons)
		    result
		    (cons (cons element (car follow-ons))
			  (loop (car follow-ons) (cdr follow-ons)))))))))

(define (remove-if predicate items)
  (let loop ((items items))
    (if (pair? items)
	(if (predicate (car items))
	    (loop (cdr items))
	    (cons (car items) (loop (cdr items))))
	'())))

(define (remove-if! predicate items)
  (letrec ((trim-initial-segment
	    (lambda (items)
	      (if (pair? items)
		  (if (predicate (car items))
		      (trim-initial-segment (cdr items))
		      (begin
			(locate-initial-segment items (cdr items))
			items))
		  items)))
	   (locate-initial-segment
	    (lambda (last this)
	      (if (pair? this)
		  (if (predicate (car this))
		      (set-cdr! last (trim-initial-segment (cdr this)))
		      (locate-initial-segment this (cdr this)))
		  this))))
    (trim-initial-segment items)))

(define (remove-item! item items)
  (cond ((null? items)
	 items)
	((eq? item (car items))
	 (cdr items))
	(else
	 (let loop ((last items) (this (cdr items)))
	   (if (not (null? this))
	       (if (eq? item (car this))
		   (set-cdr! last (cdr this))
		   (loop this (cdr this)))))
	 items)))

;;;; Slot Accessor Methods

(define (install-slot-accessor-methods class)
  (for-each (lambda (slot-arg)
	      (let* ((slot (slot-descriptor class (car slot-arg)))
		     (install
		      (lambda (keyword maker)
			(let ((accessor
			       (slot-plist-lookup (cdr slot-arg) keyword #f)))
			  (if accessor
			      (add-method accessor (maker slot)))))))
		(install 'ACCESSOR slot-accessor-method)
		(install 'MODIFIER slot-modifier-method)
		(install 'INITPRED slot-initpred-method)))
	    (class-direct-slots class)))

(define (%slot-method-maker access-type procedure make-procedure)
  (lambda (slot)
    (guarantee-slot slot procedure)
    (let* ((name (slot-name slot))
	   (method
	    (make-instance <slot-method>
			   (list (slot-class slot))
			   (make-procedure name))))
      (set-slot-method-name! method name)
      (set-slot-method-access-type! method access-type)
      method)))

(define slot-accessor-method
  (%slot-method-maker 'ACCESSOR 'SLOT-ACCESSOR-METHOD slot-accessor))

(define slot-modifier-method
  (%slot-method-maker 'MODIFIER 'SLOT-MODIFIER-METHOD slot-modifier))

(define slot-initpred-method
  (%slot-method-maker 'INITPRED 'SLOT-INITPRED-METHOD slot-initpred))

(define (compute-slot-emp generic classes methods)
  (and (pair? classes)
       (%class<=? (car classes) <instance>)
       (%instance-of? (car methods) <slot-method>)
       (let ((slot
	      (%slot-descriptor (car classes)
				(slot-method-name (car methods))
				generic)))
	 (case (slot-method-access-type (car methods))
	   ((ACCESSOR) (and (null? (cdr classes)) (slot-%accessor slot)))
	   ((MODIFIER)
	    (and (pair? (cdr classes))
		 (null? (cddr classes))
		 (slot-%modifier slot)))
	   ((INITPRED) (and (null? (cdr classes)) (slot-%initpred slot)))
	   (else #f)))))

(define <slot-method>
  (t-make-class '<SLOT-METHOD> (list <leaf-method>) '(NAME ACCESS-TYPE)))

(define slot-method-name (slot-accessor 'NAME))
(define set-slot-method-name! (slot-modifier 'NAME))
(define slot-method-access-type (slot-accessor 'ACCESS-TYPE))
(define set-slot-method-access-type! (slot-modifier 'ACCESS-TYPE))

;;;; Special Instantiators

(define (make-class name direct-superclasses direct-slots)
  (let ((class (allocate-instance <class>)))
    (initialize-instance class name direct-superclasses direct-slots)
    class))

(define (make-generic-procedure name min-arity max-arity)
  (generic->procedure
   (let ((generic (allocate-instance <generic>)))
     (initialize-instance generic name min-arity max-arity)
     generic)))

(define (make-method specializers procedure)
  (let ((method (allocate-instance <method>)))
    (initialize-instance method specializers procedure)
    method))

(define (make-leaf-method specializers procedure)
  (let ((method (allocate-instance <leaf-method>)))
    (initialize-instance method specializers procedure)
    method))

;;;; Primitive Classes

(define-smacro (define-primitive-class name . superclasses)
  `(DEFINE ,name
     (MAKE-INSTANCE <PRIMITIVE-CLASS>
		    ',name
		    (LIST ,@superclasses)
		    '())))

(define-primitive-class <boolean> <object>)
(define-primitive-class <char> <object>)
(define-primitive-class <pair> <object>)
(define-primitive-class <string> <object>)
(define-primitive-class <symbol> <object>)
(define-primitive-class <vector> <object>)

(define-primitive-class <math-object> <object>)
(define-primitive-class <number> <math-object>)
(define-primitive-class <complex> <number>)
(define-primitive-class <real> <complex>)
(define-primitive-class <rational> <real>)
(define-primitive-class <integer> <rational>)

(define-primitive-class <exact> <number>)
(define-primitive-class <exact-complex> <complex> <exact>)
(define-primitive-class <exact-real> <real> <exact-complex>)
(define-primitive-class <exact-rational> <rational> <exact-real>)
(define-primitive-class <exact-integer> <integer> <exact-rational>)

(define-primitive-class <inexact> <number>)
(define-primitive-class <inexact-complex> <complex> <inexact>)
(define-primitive-class <inexact-real> <real> <inexact-complex>)
(define-primitive-class <inexact-rational> <rational> <inexact-real>)
(define-primitive-class <inexact-integer> <integer> <inexact-rational>)

(define <procedure-class>
  (make-class '<PROCEDURE-CLASS> (list <class>) '()))

(define-smacro (define-procedure-class name . superclasses)
  `(DEFINE ,name
     (MAKE-INSTANCE <PROCEDURE-CLASS>
		    ',name
		    (LIST ,@superclasses)
		    '())))

(define-procedure-class <procedure> <object>)
(define-procedure-class <generic-procedure> <procedure>)

;;;; Printer Support

(define write-instance
  (make-generic-procedure 'WRITE-INSTANCE 2 2))

(define-unimethod write-instance <instance> (instance port)
  call-next-method
  (write-instance-helper 'INSTANCE instance port
    (lambda ()
      (let ((name (class-name (instance-class instance))))
	(if name
	    (begin
	      (write-string " of " port)
	      (write name port)))))))

(define-unimethod write-instance <class> (class port)
  call-next-method
  (write-instance-helper 'CLASS class port
    (lambda ()
      (let ((name (class-name class)))
	(if name
	    (begin
	      (write-char #\space port)
	      (write name port)))))))

(define-unimethod write-instance <generic> (generic port)
  call-next-method
  (write-instance-helper 'GENERIC generic port
    (lambda ()
      (let ((name (generic-name generic)))
	(if name
	    (begin
	      (write-char #\space port)
	      (write name port)))))))

(define-unimethod write-instance <method> (method port)
  call-next-method
  (write-instance-helper 'METHOD method port #f))

(define-unimethod write-instance <leaf-method> (method port)
  call-next-method
  (write-instance-helper 'LEAF-METHOD method port #f))

(define (write-instance-helper name object port thunk)
  (write-string "#[" port)
  (write name port)
  (if object
      (write-hash object port))
  (if thunk
      (thunk))
  (write-char #\] port))

(define (ppi object)
  (if (instance? object)
      (begin
	(newline)
	(write object)
	(for-each pp (instance-description object)))
      (pp object)))

(define (instance-description instance)
  (map (lambda (slot)
	 (cons (slot-name slot)
	       (if (slot-initialized? instance slot)
		   (list (slot-value instance slot))
		   '())))
       (slot-descriptors (instance-class instance))))

(set-record-type-method! %instance 'DESCRIPTION instance-description)

;;;; Errors

(define condition-type:slot-error
  (make-condition-type 'SLOT-ERROR condition-type:cell-error
      '()
    (lambda (condition port)
      (write-string "Anonymous error associated with slot " port)
      (write (access-condition condition 'LOCATION) port)
      (write-string "." port))))

(define condition-type:uninitialized-slot
  (make-condition-type 'UNINITIALIZED-SLOT condition-type:slot-error
      '()
    (lambda (condition port)
      (write-string "Uninitialized slot " port)
      (write (access-condition condition 'LOCATION) port)
      (write-string "." port))))

(define condition-type:uninitialized-instance-slot
  (make-condition-type 'UNINITIALIZED-INSTANCE-SLOT
      condition-type:uninitialized-slot
      '(INSTANCE)
    (lambda (condition port)
      (write-string "Uninitialized slot " port)
      (write (access-condition condition 'LOCATION) port)
      (write-string " in " port)
      (write (access-condition condition 'INSTANCE) port)
      (write-string "." port))))

(define error:uninitialized-instance-slot
  (condition-signaller condition-type:uninitialized-instance-slot
		       '(INSTANCE LOCATION)
		       standard-error-handler))

(define condition-type:unbound-slot
  (make-condition-type 'UNBOUND-SLOT condition-type:slot-error '(CLASS)
    (lambda (condition port)
      (write-string "Unbound slot " port)
      (write (access-condition condition 'LOCATION) port)
      (write-string " in " port)
      (write (access-condition condition 'CLASS) port)
      (write-string "." port))))

(define error:unbound-slot
  (condition-signaller condition-type:unbound-slot
		       '(CLASS LOCATION)
		       standard-error-handler))

(define condition-type:no-applicable-methods
  (make-condition-type 'NO-APPLICABLE-METHODS condition-type:error
      '(OPERATOR OPERANDS)
    (lambda (condition port)
      (write-string "No applicable methods for " port)
      (write (access-condition condition 'OPERATOR) port)
      (write-string " with these arguments: " port)
      (write (access-condition condition 'OPERANDS) port)
      (write-string "." port))))

(define error:no-applicable-methods
  (condition-signaller condition-type:no-applicable-methods
		       '(OPERATOR OPERANDS)
		       standard-error-handler))

;;;; Implementation-Specific Primitive Classes

(define-primitive-class <record> <object>)
(define-primitive-class <fixnum> <exact-integer>)
(define-primitive-class <bignum> <exact-integer>)
(define-primitive-class <ratnum> <exact-rational>)
(define-primitive-class <recnum> <complex>)
(define-primitive-class <exact-recnum> <recnum> <exact-complex>)
(define-primitive-class <inexact-recnum> <recnum> <inexact-complex>)
(define-primitive-class <inexact-recnum-real> <inexact-recnum> <inexact-real>)
(define-primitive-class <flonum> <inexact-rational>)
(define-primitive-class <flonum-vector> <flonum>)
(define-primitive-class <flonum-integer> <flonum> <inexact-integer>)

(define-procedure-class <entity> <procedure>)

(define built-in-object-class
  (let ((class-table (make-vector (microcode-type/code-limit) <object>)))
    (let ((assign-type
	   (lambda (name class)
	     (vector-set! class-table
			  (microcode-type/name->code name)
			  class))))
      (let ((assign-simple-type
	     (lambda (name class)
	       (assign-type name (lambda (object) object class)))))
	(assign-simple-type 'INTERNED-SYMBOL <symbol>)
	(assign-simple-type 'UNINTERNED-SYMBOL <symbol>)
	(assign-simple-type 'CHARACTER <char>)
	(assign-simple-type 'VECTOR <vector>)
	(assign-simple-type 'PAIR <pair>)
	(assign-simple-type 'BIGNUM <bignum>)
	(assign-simple-type 'FIXNUM <fixnum>)
	(assign-simple-type 'RATNUM <ratnum>)
	(assign-simple-type 'STRING <string>)
	(assign-simple-type 'PRIMITIVE <procedure>)
	(assign-simple-type 'RECORD <record>)
	(assign-simple-type 'ENTITY <entity>))

      (assign-type 'FALSE
		   (lambda (object) (if (eq? #f object) <boolean> <object>)))
      (assign-type 'CONSTANT
		   (lambda (object) (if (eq? #t object) <boolean> <object>)))
      (assign-type 'FLONUM
		   (lambda (object)
		     (cond ((flo:= object (flo:round object)) <flonum-integer>)
			   ((fix:= 2 (system-vector-length object)) <flonum>)
			   (else <flonum-vector>))))
      (assign-type 'RECNUM
		   (lambda (object)
		     (cond ((exact? object) <exact-recnum>)
			   ((zero? (imag-part object)) <inexact-recnum-real>)
			   (else <inexact-recnum>))))
      (assign-type 'COMPILED-ENTRY
		   (lambda (object)
		     (cond ((not (procedure? object)) <object>)
			   ((generic-procedure? object) <generic-procedure>)
			   (else <procedure>))))
      (let ((procedure-type
	     (lambda (object)
	       (if (generic-procedure? object)
		   <generic-procedure>
		   <procedure>))))
	(assign-type 'EXTENDED-PROCEDURE procedure-type)
	(assign-type 'PROCEDURE procedure-type)))
    (lambda (object)
      ((vector-ref class-table (object-type object)) object))))

;;;; Record Classes

(define (record-type-class record-type)
  (if (not (record-type? record-type))
      (error:wrong-type-argument record-type "record type" 'RECORD-TYPE-CLASS))
  (%record-type-class record-type))

(define (record-class record)
  (%record-type-class (record-type-descriptor record)))

(define-integrable (%record-type-class-wrapper record-type)
  (%record-ref record-type 5))

(define-integrable (%set-record-type-class-wrapper! record-type wrapper)
  (%record-set! record-type 5 wrapper))

(define-integrable (%record-type-class record-type)
  (wrapper-class (%record-type-class-wrapper record-type)))

(define (record-type-initialization-hook record-type)
  (%set-record-type-class-wrapper!
   record-type
   (class-wrapper
    (make-instance <record-class>
		   (string->symbol (record-type-name record-type))
		   (list <record>)
		   '()))))

(define <record-class>
  (make-class '<RECORD-CLASS> (list <primitive-class>) '()))

(let ((environment (->environment '(RUNTIME RECORD))))
  (set! (access record-type-initialization-hook environment)
	record-type-initialization-hook)
  (map-over-population! (access record-type-population environment)
			record-type-initialization-hook))

;;;; Implementation-Specific Printer Support

(define (unparse-record state record)
  (unparse-instance record state))

(define (procedure-unparser procedure)
  (and (generic-procedure? procedure)
       (lambda (state procedure)
	 (unparse-instance procedure state))))

(let ((environment (->environment '(RUNTIME UNPARSER))))
  (set! (access hook/unparse-record environment) unparse-record)
  (set! (access hook/procedure-unparser environment) procedure-unparser))

(define unparse-instance
  (make-generic-procedure 'UNPARSE-INSTANCE 2 2))

(define-unimethod unparse-instance <object> (object state)
  call-next-method
  (let ((port (unparser-state/port state)))
    (if *unparse-with-maximum-readability?*
	(begin
	  (write-string "#@" port)
	  (write (hash object)))
	(write-instance object port))))

(define-unimethod write-instance <record> (record port)
  call-next-method
  (write-instance-helper (class-name (record-class record)) record port #f))

(define-unimethod write-instance <generic-procedure> (procedure port)
  call-next-method
  (write-instance-helper 'GENERIC-PROCEDURE procedure port
    (lambda ()
      (let ((generic (procedure->generic procedure)))
	(let ((name (generic-name generic)))
	  (if name
	      (begin
		(write-char #\space port)
		(write name port))))))))

(define-unimethod write-instance (record-type-class slot) (slot port)
  call-next-method
  (write-instance-helper 'SLOT slot port
    (lambda ()
      (write-char #\space port)
      (write (slot-name slot) port))))