/*
 * Cont.c -- Implementation of Scheme Continuations
 *
 * (C) m.b (Matthias Blume); Apr, 1992, HUB; Jan 1993 PU/CS
 *         Humboldt-University of Berlin
 *         Princeton University, Dept. of Computer Science
 *
 * ident "@(#) Cont.c (C) M.Blume, Princeton University, 2.10"
 */

# ident "@(#)Cont.c		(C) M.Blume, Princeton University, 2.10"

# include <assert.h>
# include <string.h>

# include "storext.h"
# include "Cont.h"
# include "identifier.h"
# include "stack-t.h"
# include "type.h"
# include "except.h"
# include "speccont.h"
# include "mode.h"

static MEM_cnt measure (void *vcont)
{
  ScmCont *cont = vcont;

  return MEM_UNITS (sizeof (ScmCont) + (cont->length - 1) * sizeof (void *));
}

static void s_iterator (void *vcont, MEM_visitor proc, void *cd)
{
  ScmCont *cont = (ScmCont *) vcont;
  unsigned i;

  (*proc) ((void *)&cont->father, cd);
  (* proc) ((void *)&cont->u.s.code, cd);
  (* proc) ((void *)&cont->u.s.constants, cd);
  for (i = cont->top; i-- > 0; )
    (* proc) ((void *) &CPOS (cont, i), cd);
}

static void c_iterator (void *vcont, MEM_visitor proc, void *cd)
{
  ScmCont *cont = (ScmCont *) vcont;
  unsigned i;

  (*proc) ((void *)&cont->father, cd);
  (* proc) ((void *)&cont->u.c.environ, cd);
  (* proc) ((void *)&cont->u.c.mode, cd);
  for (i = cont->top; i-- > 0; )
    (* proc) ((void *) &CPOS (cont, i), cd);
}

static void normal_result (void);
static void split_result (void);

static void s_dumper (void *vcont, FILE *file)
{
  ScmCont *cont = (ScmCont *)vcont;

  MEM_dump_ul (cont->length, file);
  MEM_dump_ul (cont->top, file);
  putc (cont->result == split_result ? 'y' : 'n', file);
  MEM_dump_ul (cont->u.s.pc, file);
}

static void c_dumper (void *vcont, FILE *file)
{
  ScmCont *cont = (ScmCont *)vcont;

  MEM_dump_ul (cont->length, file);
  MEM_dump_ul (cont->top, file);
  putc (cont->result == split_result ? 'y' : 'n', file);
  MEM_dump_ul (cont->u.c.mode_id, file);
  MEM_dump_ul (cont->u.c.prim_no, file);
}

static void *s_excavator (FILE *file)
{
  ScmCont *cont;
  unsigned length, top;

  length = MEM_restore_ul (file);
  top = MEM_restore_ul (file);
  SCM_VNEW (cont, Cont, length, void *);
  cont->length = length;
  cont->top = top;
  cont->result = (getc (file) == 'y') ? split_result : normal_result;
  cont->u.s.pc = MEM_restore_ul (file);
  return cont;
}

static void *c_excavator (FILE *file)
{
  ScmCont *cont;
  unsigned length, top;

  length = MEM_restore_ul (file);
  top = MEM_restore_ul (file);
  SCM_VNEW (cont, Cont, length, void *);
  cont->_ = ScmType (CCont);
  cont->length = length;
  cont->top = top;
  cont->result = (getc (file) == 'y') ? split_result : normal_result;
  cont->u.c.mode_id = MEM_restore_ul (file);
  cont->u.c.prim_no = MEM_restore_ul (file);
  return cont;
}

static void display (void *vcont, putc_proc pp, void *cd)
{
  char buf[64];

  sprintf (buf, "#<Continuation %p>", vcont);
  putc_string (buf, pp, cd);
}

static void *save_push_item = NULL;
static ScmCont *save_cont = NULL;

# ifndef SINGLE_FRAME_CACHE
static ScmCont *frame_caches [N_FRAME_CACHES + 1];
# else
static ScmCont *frame_cache = NULL;
# endif

static struct {
  int valid;
  void *item;
} mode_cache [SCM_MODE_CACHE_SIZE]; /* initially all zero! */

ScmCont *ScmCC = NULL;

static void after_gc (void)
{
# ifndef SINGLE_FRAME_CACHE
  int slot;
  for (slot = 0; slot <= N_FRAME_CACHES; slot++)
    frame_caches [slot] = NULL;
# else
  frame_cache = NULL;
# endif
}

/*ARGSUSED*/
static void mode_cache_iterator (void *ign, MEM_visitor proc, void *cd)
{
  int i;

  for (i = 0; i < SCM_MODE_CACHE_SIZE; i++)
    (* proc) (&mode_cache [i].item, cd);
}

static void module_init (void)
{
  MEM_root_var (save_push_item);
  MEM_root_var (save_cont);
  MEM_root_var (ScmCC);
  MEM_root (mode_cache, mode_cache_iterator);
  ScmDirtyModeCache (-1);	/* initialize mode cache -- see below */
}

MEM_VECTOR (Cont,
	    0, measure,
	    s_iterator, s_dumper, s_excavator, MEM_NULL_revisor,
	    module_init, MEM_NULL_task, after_gc,
	    EXT (SCM_NO_NUMBER,
		 cannot_cvt_real, display, display, NULL_eq, NULL_eq));

MEM_VECTOR (CCont,
	    0, measure,
	    c_iterator, c_dumper, c_excavator, MEM_NULL_revisor,
	    MEM_NULL_task, MEM_NULL_task, MEM_NULL_task,
	    EXT (SCM_NO_NUMBER,
		 cannot_cvt_real, display, display, NULL_eq, NULL_eq));

# define stack_round_up(s) \
  (((s)+STACK_SIZE_INCR-1)&~(unsigned)(STACK_SIZE_INCR-1))

# define CACHE_SLOT(s) ((s) / STACK_SIZE_INCR - 1)

static
ScmCont *get_cont (unsigned stackreq)
{
  ScmCont *cont;

# ifndef SINGLE_FRAME_CACHES

  int slot = CACHE_SLOT (stackreq);

  slot = (slot > N_FRAME_CACHES) ? N_FRAME_CACHES : slot;
# ifndef DONT_LOOK_FOR_BIGGER_STACKS
  if (frame_caches [slot] == NULL && slot < N_FRAME_CACHES)
    slot++;
# endif
  if (frame_caches [slot] == NULL ||
      (slot == N_FRAME_CACHES && frame_caches [slot]->length < stackreq))
    SCM_VNEW (cont, Cont, stackreq, void *);
  else {
    cont = frame_caches [slot];
    frame_caches [slot] = frame_caches [slot]->father;
  }

# else

  if ((cont = frame_cache) == NULL || cont->length < stackreq)
    SCM_VNEW (cont, Cont, stackreq, void *);
  else    
    frame_cache = cont->father;

# endif
  return cont;
}

void Push (void *item)
{
  ScmCont *nc;
  unsigned length;

  if (ScmCC->top < ScmCC->length) {
    PUSH (item);
    return;
  }
  save_push_item = item;
  length = ScmCC->length + STACK_SIZE_INCR;
  nc = get_cont (length);
  memcpy (nc, ScmCC, sizeof (ScmCont) + (ScmCC->top - 1) * sizeof (void *));
  nc->length = length;
  ScmDisposeCont (ScmCC);
  ScmCC = nc;
  PUSH (save_push_item);
  save_push_item = NULL;
}

ScmCont *ScmNewSCont (ScmCont *father, unsigned stackreq)
{
  unsigned length = stack_round_up (stackreq);
  ScmCont *cont;

  save_cont = father;
  cont = get_cont (length);
  cont->_ = ScmType (Cont);
  cont->length = length;
  cont->top = 0;
  cont->result = normal_result;
  cont->father = save_cont;
  save_cont = NULL;
  return cont;
}

void ScmPushContinuation (unsigned stackreq)
{
  unsigned length = stack_round_up (stackreq);
  ScmCont *cont = get_cont (length);

  cont->_ = ScmType (Cont);
  cont->length = length;
  cont->top = 0;
  cont->result = normal_result;
  cont->u.s.code = NULL;
  cont->u.s.constants = NULL;
  cont->u.s.pc = 0;
  cont->father = ScmCC;
  ScmCC = cont;
}

void ScmPushCContinuation (unsigned stackreq, unsigned prim_no, void *environ)
{
  unsigned length = stack_round_up (stackreq);
  ScmCont *cont;

  save_push_item = environ;
  cont = get_cont (length);
  cont->_ = ScmType (CCont);
  cont->length = length;
  cont->top = 0;
  cont->result = normal_result;
  cont->u.c.environ = save_push_item;
  cont->u.c.mode = NULL;
  cont->u.c.prim_no = prim_no;
  cont->father = ScmCC;
  ScmCC = cont;
  save_push_item = NULL;
}

static ScmCont *clone_cont (ScmCont *orig, unsigned int stackreq)
{
  ScmCont *copy;

  save_cont = orig;
  copy = get_cont (stackreq);
  orig = save_cont;
  save_cont = NULL;
  memcpy (copy, orig, sizeof (ScmCont) + (orig->top - 1) * sizeof (void *));
  copy->length = stackreq;
  copy->result = normal_result;
  return copy;
}

/* make a continuation ``current'' */
void ScmSetContinuation (ScmCont *cont, unsigned int stackcnt)
{
  unsigned length, siz;
  void **from, **to;

  /* all kinds of trouble due to multi-arg continuations... */
  if (stackcnt != 1 && ScmMultiCont (cont) == 0)
    error ("wrong number of args (%u) to escape procedure", stackcnt);
  /*
   * The ``shared'' property will be propagated to parent continuations
   * only when we go back to them;  therefore we cannot detect whether
   * parts of the old continuation are shared. This means that we cannot
   * put those records onto the free cache.  But it also makes call/cc
   * run in O(1) time now (if you ignore the stack issue).
   */
  length = cont->length;
  siz = cont->top + stackcnt;
  if (siz > cont->length)
    length = siz;
  /* It is always shared --> we have to make a private copy: */
  cont = clone_cont (cont, length);
  cont->father->result = split_result;
  from = &POS (ScmCC->top - stackcnt);
  to = &CPOS (cont, cont->top);
  cont->top += stackcnt;
  while (stackcnt-- > 0)
    *to++ = *from++;
  ScmCC = cont;
}

void ScmDisposeCont (ScmCont *cont)
{
# ifndef SINGLE_FRAME_CACHES

  unsigned slot = CACHE_SLOT (cont->length);
  if (slot > N_FRAME_CACHES)
    slot = N_FRAME_CACHES;
  cont->father = frame_caches [slot];
  frame_caches [slot] = cont;

# else

  cont->father = frame_cache;
  frame_cache = cont;

# endif
}

/*
 * Copy the topmost ``stackcnt'' elements from ScmCC's stack to the
 * stack of ScmCC->father; then revert to ScmCC->father;
 * The compiler pre-computes the expected size of each stack but it
 * cannot safely estimate the ``extra'' stack size that is needed to
 * perform tail recursion elimination.  Thus we  have to check for
 * stack overflow and provide more space if necessary.
 */
void ScmRevertToFatherContinuation (unsigned stackcnt)
{
  register unsigned siz;
  ScmCont *cont;
  void (* res) (void);

  siz = ScmCC->father->top + stackcnt;
  res = ScmCC->father->result;
  if (res == split_result || siz > ScmCC->father->length) {
    siz = stack_round_up (siz);
    cont = clone_cont (ScmCC->father, siz);
    assert (cont->father != NULL);
    cont->father->result = res;
  } else
    cont = ScmCC->father;

# ifndef MEMCPY_IS_REALLY_FAST
  {
    register int i;
    register void **frombase = &POS (ScmCC->top - stackcnt);
    register void **tobase = &CPOS (cont, cont->top);
    for (i = stackcnt; i-- > 0; )
      tobase [i] = frombase [i];
  }
# else
  memcpy (&CPOS (cont, cont->top),
	  &POS (ScmCC->top - stackcnt),
	  stackcnt * sizeof (void *));
# endif
  cont->top += stackcnt;
  ScmDisposeCont (ScmCC);
  ScmCC = cont;
}

static void normal_result (void)
{
  ScmCont *cont = ScmCC->father;
  void *item = PEEK ();
  ScmDisposeCont (ScmCC);
  ScmCC = cont;
  PUSH (item);
}

static void split_result (void)
{
  ScmCont *cont = clone_cont (ScmCC->father, ScmCC->father->length);
  void *item = PEEK ();
  ScmDisposeCont (ScmCC);
  ScmCC = cont;
  PUSH (item);
}

/*
 * Look for something like the current input/ouput file
 * the current error handler etc. (collectively known as ``mode''s).
 * Modes are referenced by integers, and are stored into the ``u.c.mode''
 * member of a C-continuation.  The identifying integer is stored into
 * ``mode_id''.
 * In order to avoid repeated traversals of the current continuation
 * we cache the result of a lookup in mode_cache.  The procedure
 * ScmDirtyModeCache is mark some value as being invalid.
 */

/* a negative argument dirties all entries */
void ScmDirtyModeCache (int mode)
{
  int i;

  if (mode >= 0) {
    mode_cache [mode].valid = 0;
    mode_cache [mode].item = NULL;
  } else
    for (i = 0; i < SCM_MODE_CACHE_SIZE; i++) {
      mode_cache [i].valid = 0;
      mode_cache [i].item = NULL;
    }
}

void *ScmMode (int mode)
{
  ScmCont *c;
  void *res;
  
  if (mode_cache [mode].valid)
    return mode_cache [mode].item;
  res = NULL;
  for (c = ScmCC; c != NULL; c = c->father)
    if (ScmTypeOf (c) == ScmType (CCont) &&
	c->u.c.mode != NULL && c->u.c.mode_id == mode) {
      res = c->u.c.mode;
      break;
    }
  mode_cache [mode].valid = 1;
  mode_cache [mode].item = res;
  return res;
}

void ScmSetMode (int mode, void *item)
{
  assert (ScmTypeOf (ScmCC) == ScmType (CCont) && ScmCC->u.c.mode == NULL);
  ScmCC->u.c.mode = item;
  ScmCC->u.c.mode_id = mode;
  mode_cache [mode].valid = 1;
  mode_cache [mode].item = item;
}

int ScmMultiCont (ScmCont *cont)
{
  return ScmTypeOf (cont) == ScmType (CCont) &&
         cont->u.c.prim_no == SCM_VM_CW_VALUES_CONT;
}

# ifdef BACKTRACE
/* include some extra stuff -- not reflected in Makefile !!!! */
# include "io.h"
# include "Primitive.h"
# include "Code.h"
# include "String.h"
# include "Vector.h"

# define PEEPHOLE_SIZE 15

extern
long ScmDisassemble
  (ScmCode *code, unsigned long pc, const char **opcode, int *length,
   unsigned short *op1, unsigned short *op2, unsigned short *op3);


static
void disass (ScmCont *cont)
{
  long pc, newpc;
  const char *opcode;
  int length;
  unsigned short op1, op2, op3;
  ScmCode *code = cont->u.s.code;

  if (code == NULL)
    return;
  fputs ("--- assembly code excerpt ---\n.\n.\n.\n", stderr);
  for (pc = 0; pc >= 0 && pc < cont->u.s.pc + PEEPHOLE_SIZE; pc = newpc) {
    newpc = ScmDisassemble (code, pc, &opcode, &length, &op1, &op2, &op3);
    if (newpc >= 0 && pc > cont->u.s.pc - PEEPHOLE_SIZE) {
      fprintf (stderr, "%8lu %c %20s",
	       pc,
	       pc == cont->u.s.pc ? '*' : ' ',
	       opcode);
      if (length > 1) {
	fprintf (stderr, " %3u", (unsigned) op1);
	if (length > 2) {
	  fprintf (stderr, ", %3u", (unsigned) op2);
	  if (length > 3) {
	    fprintf (stderr, ", %3u", (unsigned) op3);
	    if (length > 4)
	      fputs (", ...", stderr);
	  }
	}
      }
      putc ('\n', stderr);
    }
  }
  fputs (".\n.\n.\n--- end of assembly code excerpt ---\n", stderr);
}

static
void print_constants (ScmVector *vect)
{
  unsigned i;

  fprintf (stderr, "\tConstants (%u)...\n", (unsigned) vect->length);
  for (i = 0; i < vect->length; i++) {
    fprintf (stderr, "\t%3u: ", i);
    write_object (vect->array [i], file_putc, stderr);
    putc ('\n', stderr);
  }
}

static
void trace_stack (ScmCont *cont)
{
  unsigned i, j;
  void *item;
  ScmVector *vect;

  fprintf (stderr, "\tStack (%u/%u)...\n",
	   (unsigned) cont->top, (unsigned) cont->length);
  for (i = 0; i <  cont->top; i++) {
    fprintf (stderr, "\t %2u: ", i);
    item = CPOS (cont, i);
    if (ScmTypeOf (item) == ScmType (Vector)) {
      fputs ("Vector...\n", stderr);
      vect = item;
      for (j = 0; j < vect->length; j++) {
	fprintf (stderr, "\t\t%2u: ", j);
	write_object (vect->array [j], file_putc, stderr);
	putc ('\n', stderr);
      }
    } else {
      write_object (item, file_putc, stderr);
      putc ('\n', stderr);
    }
  }
}
/*
 * The following function definition, although having external linkage,
 * will not show up in the header file.  It is intended to be used when
 * debugging the system only.
 */
extern void ScmBackTrace (void);
void ScmBackTrace (void)
{
  ScmCont *cont;
  const char *name;
  int namlen;

  fputs ("*** VSCM BackTrace Listing ***\n", stderr);
  for (cont = ScmCC; cont; cont = cont->father) {
    if (ScmTypeOf (cont) == ScmType (CCont)) {
      ScmPrimitive *prim = GetScmPrimitive (cont->u.c.prim_no);
      name = prim ? prim->name : "<<not a primitive>>";
      namlen = strlen (name);
    } else {
      ScmString *strg;
      if (cont->u.s.code == NULL) {
	name = "<<no code>>";
	namlen = strlen (name);
      } else {
	strg = cont->u.s.code->proc_name;
	if (strg == NULL) {
	  name = "<<unnamed>>";
	  namlen = strlen (name);
	} else {
	  name = strg->array;
	  namlen = strg->length;
	}
      }
    }
    fprintf (stderr, "    %c-Cont(\"%.*s\")[ca: %d%s]\n",
	     ScmTypeOf (cont) == ScmType (CCont) ? 'C' : 'S',
	     namlen, name,
	     cont->call_again,
	     cont->result == split_result ? ", shared" : "");
    if (ScmTypeOf (cont) == ScmType (CCont)) {
      fputs ("\tenviron: ", stderr);
      write_object (cont->u.c.environ, file_putc, stderr);
      fprintf (stderr, "\n\tmode (id: %d): ", cont->u.c.mode_id);
      write_object (cont->u.c.mode, file_putc, stderr);
      fprintf (stderr, "\n\tprim_no: %u\n", (unsigned) cont->u.c.prim_no);
      trace_stack (cont);
    } else {
      fputs ("\tcode: ", stderr);
      write_object (cont->u.s.code, file_putc, stderr);
      fprintf (stderr, "\n\tpc: %lu\n", cont->u.s.pc);
      disass (cont);
      print_constants (cont->u.s.constants);
      trace_stack (cont);
    }
  }
}
# endif

int ScmContIsShared (ScmCont *cont)
{
  return cont->result == split_result;
}

void ScmContSetShared (ScmCont *cont)
{
  cont->result = split_result;
}
