/*
 *
 * s y n t a x . c			-- Syntaxic forms implementation
 *
 * Copyright (C) 1993, 1994 Erick Gallesio - I3S - CNRS / UNSA <eg@unice.fr>
 * 
 *
 * Permission to use, copy, and/or distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that both the above copyright notice and this permission notice appear in
 * all copies and derived works.  Fees for distribution or use of this
 * software or derived works may only be charged with express written
 * permission of the copyright holder.  
 * This software is provided ``as is'' without express or implied warranty.
 *
 * This software is a derivative work of other copyrighted softwares; the
 * copyright notices of these softwares are placed in the file COPYRIGHTS
 *
 *
 *           Author: Erick Gallesio [eg@kaolin.unice.fr]
 *    Creation date: 25-Oct-1993 23:39
 * Last file update:  2-Jun-1994 11:36
 */

/* Notes:
 * ------ 
 * C functions syntax_xxx implement the scheme syntax xxx. A syntax function 
 * returns its work in its first argument (SCM *pform). The function result
 * is a boolean. If false, it indicates to eval that this result is 
 * a final one (eval can return it unmodified). Otherwise, the eval function 
 * take the result stored in *pform and evaluates it again (in the same eval 
 * frame). This mechanaism permits to treat tail recursive calls as jump in
 * the eval function.
 *
 * Syntax function which returns EVALCAR(zzzz) are not tail recursive in debug 
 * mode (in normal mode only the first call is non tail recursive, since this 
 * first call will replace the original code by an equivalent code which is 
 * clean on tail recursive calls.
 */

#include "stk.h"


#ifdef COMPACT_SMALL_CST
#   define makecell(type) ((SCM) MAKE_SMALL_CST(0, type))
#else
static SCM makecell(int type)
{
  register SCM z;
  NEWCELL(z, type);
  return z;
}
#endif


static SCM define2lambda(SCM l)
{
  /* transform (define (f p) b) in (define f (lambda (p) b)) */
  if (CONSP(l) && CONSP(CAR(l))){
    if (llength(l) < 2) goto Error;
    return cons(CAR(CAR(l)), cons(cons(sym_lambda,
				       cons(CDR(CAR(l)), CDR(l))),
				  NIL));
  }
  else 
    if (llength(l) == 2) return l;
Error:
  err("define: bad parameter list", l);
}

PRIMITIVE syntax_quote(SCM *pform, SCM env)
{
  SCM args = CDR(*pform);

  if (llength(args) != 1) err("quote: Bad syntax", *pform);
  if (ModifyCode()) CAR(*pform) = makecell(tc_quote);
  SYNTAX_RETURN(CAR(args), ntruth);
}

PRIMITIVE syntax_lambda(SCM *pform, SCM env)
{
  register SCM z, args=CDR(*pform);

  if (llength(args) < 2) err("lambda: bad syntax", *pform);

  if(ModifyCode()) CAR(*pform) = makecell(tc_lambda);

  NEWCELL(z, tc_closure);
  z->storage_as.closure.env  = env;
  z->storage_as.closure.code = args;
  SYNTAX_RETURN(z, ntruth);
}

PRIMITIVE syntax_if(SCM *pform, SCM env)
{
  SCM args = CDR(*pform);
  
  switch (llength(args)) {
    case 2:  args = cons(CAR(args), cons(CAR(CDR(args)), cons(UNDEFINED, NIL)));
    case 3:  break;
    default: err("if: bad syntax", *pform);
  }

  if (ModifyCode()) {
    CAR(*pform) = makecell(tc_if);
    CDR(*pform) = args; /* will always contain a else part */
  }

  SYNTAX_RETURN(NEQ(EVALCAR(args), ntruth) ? CAR(CDR(args)):CAR(CDR(CDR(args))), 
		truth);
}

PRIMITIVE syntax_setq(SCM *pform, SCM env)
{
  SCM var, *tmp, args = CDR(*pform);

  if (llength(args) != 2)      err("set!: bad assignement", args);
  if (NSYMBOLP(var=CAR(args))) err("set!: first argument is not a symbol", var);

  tmp = varlookup(var, env, 0);
  if (*tmp == UNBOUND) err("set!: variable not defined", var);
  if (ModifyCode()) CAR(*pform) = makecell(tc_setq);
  *tmp = EVALCAR(CDR(args));
#ifdef USE_TK
  Tcl_ChangeValue(PNAME(var));
#endif

  SYNTAX_RETURN(UNDEFINED, ntruth);
}

PRIMITIVE syntax_cond(SCM *pform, SCM env)
{
  SCM l, tmp;
  
  for (l=CDR(*pform); CONSP(l); l=CDR(l)) {
    if (NCONSP(CAR(l))) err("cond: malformed clause", CAR(l));
    if (EQ(CAR(CAR(l)), sym_else) && NNULLP(CDR(l)))
      err("cond: else clause must be the last", *pform);
  }
  if (NNULLP(l)) err("cond: malformed clauses list", CDR(*pform));

  tmp = cons(makecell(tc_cond), CDR(*pform));
  if (ModifyCode()) CAR(*pform) = CAR(tmp);

  SYNTAX_RETURN(tmp, truth);
}

PRIMITIVE syntax_and(SCM *pform, SCM env)
{
  SCM l   = CDR(*pform);
  int len = llength(l);

  if (len < 0)  err("and: bad argument list", *pform);
  if (ModifyCode()) CAR(*pform) = makecell(tc_and);

  if (len == 0) SYNTAX_RETURN(truth, ntruth);

  for (--len; len; len--, l=CDR(l)) {
    if (EVALCAR(l) == ntruth) SYNTAX_RETURN(ntruth, ntruth);
  }
  SYNTAX_RETURN(EVALCAR(l), ntruth);
}

PRIMITIVE syntax_or(SCM *pform, SCM env)
{
  SCM l   = CDR(*pform);
  int len = llength(l);
  SCM val;

  if (len < 0)  err("or: bad argument list", *pform);
  if (ModifyCode()) CAR(*pform) = makecell(tc_or);

  if (len == 0) SYNTAX_RETURN(ntruth, ntruth);

  for (--len; len; len--, l=CDR(l)) {
    if ((val=EVALCAR(l)) != ntruth) SYNTAX_RETURN(val, ntruth);
  }
  SYNTAX_RETURN(EVALCAR(l), ntruth);
}

static SCM syntax_let_family(SCM *pform, SCM env, char *who, int type)
{
  register SCM p, tmp, fl=NIL, al=NIL;
  char buffer[100];
  int named_let = 0, len = llength(*pform);

  if (len < 3) goto Error;

  p = CAR(CDR(*pform));
  if (SYMBOLP(p) && type == tc_let) {
    /* It's a named let. Re-initialize to a correct value */ 
    if (len < 4) goto Error;
    named_let = 1;
    p = CAR(CDR(CDR(*pform)));
  }

  for(  ; NNULLP(p); p=CDR(p)) {
    if (llength(tmp=CAR(p)) != 2 || NSYMBOLP(CAR(tmp))) {
      sprintf(buffer, "%s: badly formed binding", who);
      err(buffer, CONSP(p)? tmp: p);
    }
    /* Verify that this binding doesn't already exist in fl */
    if (memv(CAR(tmp),fl)!=ntruth) {
      sprintf(buffer, "%s: duplicate binding", who);
      err(buffer, CAR(CDR(*pform)));
    }
    fl = cons(CAR(tmp),fl); 
    al = cons(CAR(CDR(tmp)),al);
  }

  tmp = named_let ?
    	   /* named let */
    	   LIST4(makecell(tc_letrec), 
		 LIST1(CAR(CDR(*pform))),
		 LIST1(cons(sym_lambda, cons(reverse(fl), CDR(CDR(CDR(*pform)))))),
		 cons(CAR(CDR(*pform)), reverse(al))) :
	   /* normal case */
	   cons(makecell(type), 
		cons(reverse(fl), 
		     cons(reverse(al), 
			  CDR(CDR(*pform)))));

  if (ModifyCode()) {
    CAR(*pform) = CAR(tmp); 
    CDR(*pform) = CDR(tmp); 
  }
  SYNTAX_RETURN(tmp, truth);

Error:
  sprintf(buffer, "%s: incorrect number of subforms", who);
  err(buffer, *pform);
}


PRIMITIVE syntax_let(SCM *pform, SCM env)
{
  return syntax_let_family(pform, env, "let", tc_let);
}

PRIMITIVE syntax_letstar(SCM *pform, SCM env)
{
  return syntax_let_family(pform, env, "let*", tc_letstar);
}

PRIMITIVE syntax_letrec(SCM *pform, SCM env)
{
  return syntax_let_family(pform, env, "letrec", tc_letrec);
}

PRIMITIVE syntax_begin(SCM *pform, SCM env)
{
  register SCM l = CDR(*pform);

  if (NULLP(l)) err("begin: no subform in sequence", l);
  for ( ; NNULLP(CDR(l));  l = CDR(l))
    EVALCAR(l);
  if (ModifyCode()) CAR(*pform) = makecell(tc_begin);
  SYNTAX_RETURN(EVALCAR(l), ntruth);;
}

PRIMITIVE syntax_delay(SCM *pform, SCM env)
{
  SCM z, tmp;

  if (llength(*pform) != 2) err("delay: Bad expression", *pform);

  /* Build (lambda () expr) in tmp */
  NEWCELL(tmp, tc_closure);
  tmp->storage_as.closure.env = env;
  tmp->storage_as.closure.code = cons(NIL, CDR(*pform));

  /* save this closure in the promise */
  NEWCELL(z, tc_promise);
  z->storage_as.promise.expr = tmp;
  z->storage_as.promise.resultknown = 0;
  SYNTAX_RETURN(z, ntruth);
}


static SCM backquotify(SCM x, SCM env, int level)
{
  if (VECTORP(x)) return list2vector(backquotify(vector2list(x), env, level));

  if (NCONSP(x)) return x;

  if (EQ(CAR(x), sym_quasiquote))
    return LIST2(sym_quasiquote,
		 backquotify(CAR(CDR(x)), env, level+1));

  if (EQ(CAR(x), sym_unquote))
    return (level == 1) ? EVALCAR(CDR(x))
      			: LIST2(sym_unquote,
				backquotify(CAR(CDR(x)), env, level-1));

  if (CONSP(CAR(x)) && EQ(CAR(CAR(x)), sym_unquote_splicing))
    return NULLP(CDR(x)) ? EVALCAR(CDR(CAR(x)))
	    		 : append(LIST2(EVALCAR(CDR(CAR(x))),
					backquotify(CDR(x), env, level)));
  /* Otherwise */
  return cons(backquotify(CAR(x), env, level), backquotify(CDR(x), env, level));
}


PRIMITIVE syntax_quasiquote(SCM *pform, SCM env)
{
  if (NULLP(CDR(*pform))) err("quasiquote: no form", NIL);
  SYNTAX_RETURN(backquotify(CAR(CDR(*pform)), env, 1), ntruth);
}

PRIMITIVE syntax_define(SCM *pform,SCM env)
{
  SCM *tmp, var, args;

  args = define2lambda(CDR(*pform));
  var  = CAR(args); if (NSYMBOLP(var)) err("define: bad variable name", var);

  if (NULLP(env)) {
    /* Global var */
    tmp  = varlookup(var, env, 0);
    *tmp = EVALCAR(CDR(args));
  }
  else {
    /* Extend current environment for that definition */
    SCM tmp  = CAR(env);

    tmp = cons(cons(var, 		CAR(tmp)),
	       cons(EVALCAR(CDR(args)), CDR(tmp)));
    CAR(CAR(env)) = CAR(tmp);
    CDR(CAR(env)) = CDR(tmp);
  }

#ifdef USE_TK
  Tcl_ChangeValue(PNAME(var));
#endif

  SYNTAX_RETURN(UNDEFINED, ntruth);
}
