/*
 *
 *  s t k l o s . c			-- STklos support
 *
 * Copyright (C) 1993 Erick Gallesio - I3S - CNRS / UNSA <eg@unice.fr>
 * 
 *
 * Permission to use, copy, and/or distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that both the above copyright notice and this permission notice appear in
 * all copies and derived works.  Fees for distribution or use of this
 * software or derived works may only be charged with express written
 * permission of the copyright holder.  
 * This software is provided ``as is'' without express or implied warranty.
 *
 * This software is a derivative work of other copyrighted softwares; the
 * copyright notices of these softwares are placed in the file COPYRIGHTS
 *
 *
 *            Author: Erick Gallesio [eg@unice.fr]
 *    Creation date:  9-Feb-1994 15:56
 * Last file update: 10-Jun-1994 23:05 
 */

#ifdef USE_STKLOS

#include <stk.h>
#include "stklos.h"
#include "gf.h"

int tc_instance;		/* The type field of an instance */

static SCM Top, Object, Class;
static SCM Boolean, Char, Pair, Procedure, String, Symbol, Vector, Number, 
	   Null, Real, Integer, Keyword, Unknown;
#ifdef USE_TK
static SCM Widget;
#endif

/******************************************************************************
 *
 * tc_instance type definition
 *
 ******************************************************************************/

SCM make_instance(SCM class, long  size, int type)
{
  SCM z;

  NEWCELL(z, tc_instance);
  EXTDATA(z) =  must_malloc(sizeof(Instance));

  CLASS_OF(z)     = class;
  SLOTS_OF(z) 	  = makevect(size, UNBOUND); 
  INST_TYPE(z)	  = type;
  ACCESSORS_OF(z) = class? THE_SLOT_OF(class, S_getters_n_setters) : NIL;
  return z;
}

static void mark_instance(SCM instance)
{
  gc_mark(CLASS_OF(instance));
  gc_mark(SLOTS_OF(instance));
  gc_mark(ACCESSORS_OF(instance));
}

static void free_instance(SCM instance)
{
  free(INST(instance));
}

static void display_instance(SCM instance, FILE *f, int type)
{
  char buffer[50];

  Puts("#[", f);
  lprint(THE_SLOT_OF(CLASS_OF(instance), S_name), f, type);
  sprintf(buffer, " %lx]",(unsigned long) instance);
  Puts(buffer, f);
}

static extended_scheme_type instance_type = {
  "instance",		/* name */
  EXT_EVALPARAM,	/* flags */
  mark_instance,	/* gc_mark_fct */
  free_instance,	/* gc_sweep_fct */
  apply_generic,	/* apply_fct   ---- defined in gf.c */		
  display_instance	/* display_fct */
};


/******************************************************************************
 *
 * compute-cpl
 *
 *   This version doesn't handle multiple-inheritance. It serves only for
 * booting classes and will be overaloaded in Scheme
 *
 ******************************************************************************/

static SCM compute_cpl(SCM supers, SCM res)
{
  return NULLP(supers)? reverse(res)
    		      : compute_cpl(THE_SLOT_OF(CAR(supers), S_direct_supers),
					 cons(CAR(supers), res));
}

/******************************************************************************
 *
 * compute-slots
 *
 ******************************************************************************/

static SCM remove_duplicate_slots(SCM l, SCM res, SCM slots_already_seen)
{
  SCM tmp;

  if (NULLP(l)) return res;

  tmp = CONSP(CAR(l)) ? CAR(CAR(l)) : CAR(l);
  if (NSYMBOLP(tmp)) err("%compute-slots: bad slot name", tmp);
  
  if (memq(tmp, slots_already_seen) == ntruth) {
    res 	       = cons(CAR(l), res);
    slots_already_seen = cons(tmp, slots_already_seen);
  }
  
  return remove_duplicate_slots(CDR(l), res, slots_already_seen);
}

static SCM build_slots_list(SCM dslots, SCM cpl)
{
  register SCM res = dslots;

  for (cpl = CDR(cpl); NNULLP(cpl); cpl = CDR(cpl))
    res = append(LIST2(THE_SLOT_OF(CAR(cpl), S_direct_slots), res));

  /* res contains a list of slots. Remove slots which appears more than once */
  return remove_duplicate_slots(reverse(res), NIL, NIL);
}


static PRIMITIVE compute_slots(SCM class)
{
  if (NCLASSP(class)) err("compute-class: bad class", class);
  return build_slots_list(THE_SLOT_OF(class, S_direct_slots),
			  THE_SLOT_OF(class, S_cpl));
}

/******************************************************************************
 *
 * compute-getters-n-setters
 *  
 *   This version doesn't handle slot options. It serves only for booting 
 * classes and will be overaloaded in Scheme.
 *
 ******************************************************************************/

static SCM compute_getters_n_setters(SCM slots)
{
  SCM get_n_set, alloc, tmp, res = NIL;
  long i=0;

  for (  ; NNULLP(slots); slots = CDR(slots)) 
    res = cons(cons(CAR(slots),makeinteger(i++)), res);

  return res;
}


/******************************************************************************
 *
 * compute-initializers
 *
 ******************************************************************************/

static SCM build_initializers(SCM slots)
{
  SCM initform, tmp, curr_slot, res = NIL;

  for ( ; NNULLP(slots); slots = CDR(slots)) {
    tmp       = NIL;
    curr_slot = CAR(slots);

    if (CONSP(curr_slot)) {	
      /* slot is a pair. See if an :initform is provided */
      if (llength(curr_slot) > 1) {
	if (initform = get_keyword(makekey(":initform"), CDR(curr_slot), NULL))
	  tmp = leval(LIST3(sym_lambda, NIL, initform), NIL);
      }
    }
    res = cons(tmp, res);
  }
  return reverse(res);
}


/******************************************************************************
 *
 * initialize-object
 *
 ******************************************************************************/

static PRIMITIVE initialize_object(SCM obj, SCM initargs)
{
  SCM tmp, initializers, slots;

  if (NINSTANCEP(obj)) 	       err("%initialize-object: bad instance", obj);
  if (NCONSP(obj)&&NULLP(obj)) err("%initialize-object: bad init list", initargs);
  
  initializers = THE_SLOT_OF(CLASS_OF(obj), S_initializers);
  slots        = THE_SLOT_OF(CLASS_OF(obj), S_slots);
  
  /* See for each slot how it must be initialized */
  for ( ; NNULLP(initializers); initializers=CDR(initializers), slots=CDR(slots)) {
    SCM slot_name  = CAR(slots);
    SCM slot_value = NULL;
    
    if (CONSP(slot_name)) {
      /* This slot admits (perhaps) to be initialized at creation time */
      tmp 	= get_keyword(makekey(":init-keyword"), CDR(slot_name), NULL);
      slot_name = CAR(slot_name);
      if (tmp) {
	/* an initarg was provided for this slot */
	if (NKEYWORDP(tmp)) 
	  err("%initialize-object: initarg must be a keyword. It was", tmp);
	slot_value = get_keyword(tmp, initargs, NULL);
      }
    }

    if (slot_value)
      /* set slot to provided value */
      slot_set(obj, slot_name, slot_value);
    else {
      /* set slot to its :initform if it exists */
      if (NNULLP(CAR(initializers)))
	slot_set(obj, slot_name, apply(CAR(initializers), NIL));
    }
  }

  return obj;
}

/******************************************************************************/

SCM basic_make_class(SCM class, SCM name, SCM dsupers, SCM dslots)
{
  SCM z, cpl, slots, g_n_s;

  /* Allocate one instance */
  z     = make_instance(class, NUMBER_OF_CLASS_SLOTS, TYPE_INSTANCE);

  /* Initialize its slots */
  cpl   = compute_cpl(dsupers, LIST1(z));
  slots = build_slots_list(dslots, cpl);
  g_n_s = compute_getters_n_setters(slots);

  THE_SLOT_OF(z, S_name)	      = name;
  THE_SLOT_OF(z, S_direct_supers)     = dsupers;
  THE_SLOT_OF(z, S_direct_slots)      = dslots;
  THE_SLOT_OF(z, S_cpl)		      = cpl;
  THE_SLOT_OF(z, S_slots)	      = slots;
  THE_SLOT_OF(z, S_nfields)	      = makeinteger(llength(slots));
  THE_SLOT_OF(z, S_getters_n_setters) = g_n_s;
  THE_SLOT_OF(z, S_initializers)      = build_initializers(slots);

  /* Don't forget to set the accessors list of the object */
  ACCESSORS_OF(z) = THE_SLOT_OF(class, S_getters_n_setters);
  
  return z;
}

/******************************************************************************/

static void create_Top_Object_Class(void)
{
  SCM tmp, slots_of_class = LIST8(intern("name"), 
				  intern("direct-supers"),
				  intern("direct-slots"),
				  intern("cpl"),
				  intern("slots"),
				  intern("nfields"),
				  intern("initializers"),
				  intern("getters-n-setters"));

  /**** <Class> ****/
  tmp 	= intern("<class>");
  Class = make_instance(NULL, NUMBER_OF_CLASS_SLOTS, TYPE_INSTANCE);

  CLASS_OF(Class)     = Class;
  ACCESSORS_OF(Class) = compute_getters_n_setters(slots_of_class);

  THE_SLOT_OF(Class, S_name) 		  = tmp;
  THE_SLOT_OF(Class, S_direct_supers)	  = NIL; /* will be changed */
  THE_SLOT_OF(Class, S_direct_slots)	  = slots_of_class;
  THE_SLOT_OF(Class, S_cpl)		  = NIL;  /* will be changed */
  THE_SLOT_OF(Class, S_slots)		  = slots_of_class;
  THE_SLOT_OF(Class, S_nfields)		  = makeinteger(NUMBER_OF_CLASS_SLOTS);
  THE_SLOT_OF(Class, S_initializers)      = build_initializers(slots_of_class);
  THE_SLOT_OF(Class, S_getters_n_setters) = ACCESSORS_OF(Class);

  VCELL(tmp) = Class;

  /**** <Top> ****/
  tmp = intern("<top>");
  Top = basic_make_class(Class, tmp, NIL, NIL);

  VCELL(tmp) = Top;
  
  /**** <Object> ****/
  tmp	 = intern("<object>");
  Object = basic_make_class(Class, tmp, LIST1(Top), NIL);

  VCELL(tmp) = Object;

  /* <top> <object> and <class> were partly uninitialized. Correct them here */
  THE_SLOT_OF(Class, S_direct_supers)   = LIST1(Object);
  THE_SLOT_OF(Class, S_cpl)		= LIST3(Class, Object, Top);

/* 
   THE_SLOT_OF(Object, S_cpl) 		= CDR(THE_SLOT_OF(Class, S_cpl));
   THE_SLOT_OF(Top, S_cpl)		= CDR(CDR(THE_SLOT_OF(Class, S_cpl)));
*/
  /* protect Top, Object and Class  against garbage collection */
  gc_protect(Top);
  gc_protect(Object);
  gc_protect(Class);
}


static void make_prim_type(SCM *var, char *name, SCM meta, SCM supers)
{
   SCM tmp = intern(name);
   gc_protect(*var = basic_make_class(meta, tmp, supers, NIL));
   VCELL(tmp) = *var;
}
	   
static void make_primitive_classes(void)
{
  SCM tmp = VCELL(intern("<procedure-class>"));

  make_prim_type(&Boolean, 	"<boolean>",	Class, LIST1(Top));
  make_prim_type(&Char,		"<char>",	Class, LIST1(Top));
  make_prim_type(&Pair,		"<pair>",	Class, LIST1(Top));
  make_prim_type(&Null,		"<null>", 	Class, LIST1(Top));
  make_prim_type(&String,	"<string>",	Class, LIST1(Top));
  make_prim_type(&Symbol,	"<symbol>",	Class, LIST1(Top));
  make_prim_type(&Vector,	"<vector>",	Class, LIST1(Top));
  make_prim_type(&Number,	"<number>",	Class, LIST1(Top));
  make_prim_type(&Real,		"<real>",	Class, LIST1(Number));
  make_prim_type(&Integer,	"<integer>",	Class, LIST1(Real));
  make_prim_type(&Keyword,	"<keyword>",	Class, LIST1(Top));
  make_prim_type(&Unknown,	"<unknown>",	Class, LIST1(Top));
  make_prim_type(&Procedure,	"<procedure>",	tmp,   LIST1(Top));
#ifdef USE_TK
  make_prim_type(&Widget,	"<widget>",	Class, LIST1(Top));
#endif
}  


/******************************************************************************/

static PRIMITIVE instancep(SCM obj)
{
  return INSTANCEP(obj)? truth: ntruth;
}

PRIMITIVE class_of(SCM obj)
{
  if (INSTANCEP(obj)) return CLASS_OF(obj);

  switch (TYPE(obj)) {
    case tc_boolean:	return Boolean;
    case tc_char:	return Char;
    case tc_cons:	return Pair;
    case tc_string:	return String;
    case tc_symbol:	return Symbol;
    case tc_vector:	return Vector;
    case tc_flonum:	return Real;
    case tc_integer:
    case tc_bignum:	return Integer;
    case tc_keyword:	return Keyword;
#ifdef USE_TK
    case tc_tkcommand:	return Widget;
#endif
    default: 		return (procedurep(obj) == truth)? Procedure: Unknown;
  }
}
static PRIMITIVE class_name(SCM obj)
{
  if (NINSTANCEP(obj)) err("class-name: bad class", obj);
  return slot_ref(obj, intern("name"));
}
static PRIMITIVE class_direct_supers(SCM obj)
{
  if (NINSTANCEP(obj)) err("class-direct-supers: bad class", obj);
  return slot_ref(obj, intern("direct-supers"));
}
static PRIMITIVE class_direct_slots(SCM obj)
{
  if (NINSTANCEP(obj)) err("class-direct-slots: bad class", obj);
  return slot_ref(obj, intern("direct-slots"));
}
static PRIMITIVE class_cpl(SCM obj)
{
  if (NINSTANCEP(obj)) err("class-recedence-list: bad class", obj);
  return slot_ref(obj, intern("cpl"));
}
static PRIMITIVE class_slots(SCM obj)
{
  if (NINSTANCEP(obj)) err("class-slots: bad class", obj);
  return slot_ref(obj, intern("slots"));
}

PRIMITIVE slot_existsp(SCM obj, SCM slot_name)
{
  if (NSYMBOLP(slot_name)) err("slot-exists?: bad slot name", slot_name);
  if (NINSTANCEP(obj))     err("slot-exists?: bad object", obj);
  return assq(slot_name, ACCESSORS_OF(obj)) == ntruth ? ntruth : truth;
}


/******************************************************************************
 *
 * slot-ref, slot-set! and slot-bound?
 *
 ******************************************************************************/

PRIMITIVE slot_ref(SCM obj, SCM slot_name)
{
  register SCM entry;
  SCM res;

  if (NINSTANCEP(obj)) err("slot-ref: bad instance", obj);
  
  entry = assq(slot_name, ACCESSORS_OF(obj));
  if (entry == ntruth) 
    err("slot-ref: no slot with name", slot_name);

  /* Two cases here:
   *	- if (cdr entry) is an integer (the offset of this slot in the slots vector
   *	- otherwise (cadr entry) is the reader function to apply
   */
  res = INTEGERP(CDR(entry)) ? THE_SLOT_OF(obj, INTEGER(CDR(entry)))
    			     : apply(cadr(entry), LIST1(obj));
  if (res == UNBOUND) err("slot-ref: slot unbound", slot_name);

  return res;
}

PRIMITIVE slot_set(SCM obj, SCM slot_name, SCM value)
{
  register SCM entry;

  if (NINSTANCEP(obj)) err("slot-set!: bad instance", obj);
  
  entry = assq(slot_name, ACCESSORS_OF(obj));
  if (entry == ntruth) 
    err("slot-set!: no slot with name", slot_name);

  /* Two cases here:
   *	- if (cdr entry) is an integer (the offset of this slot in the slots vector)
   *	- otherwise (caddr entry) is the writer function to apply
   */
  if (INTEGERP(CDR(entry)))
    THE_SLOT_OF(obj, INTEGER(CDR(entry))) = value;
  else 
    apply(caddr(entry), LIST2(obj, value));

  return UNDEFINED;
}

PRIMITIVE slot_boundp(SCM obj, SCM slot_name)
{
  register SCM entry;
  SCM res;

  if (NINSTANCEP(obj)) err("slot-bound?: bad instance", obj);
  
  entry = assq(slot_name, ACCESSORS_OF(obj));
  if (entry == ntruth) 
    err("slot-bound?: no slot with name", slot_name);

  res = INTEGERP(CDR(entry)) ? THE_SLOT_OF(obj, INTEGER(CDR(entry)))
    			     : apply(cadr(entry), LIST1(obj));

  return (res == UNBOUND) ? ntruth : truth;
}

/******************************************************************************
 *
 * %allocate-instance (the low level instance allocation primitive)
 *
 ******************************************************************************/
 
PRIMITIVE allocate_instance(SCM class)
{
  int kind;

  if (NCLASSP(class)) err("%allocate-instance: bad class", class);
  
  kind = SUBCLASSP(class, VCELL(intern("<generic>"))) ? TYPE_GENERIC: TYPE_INSTANCE;

  return make_instance(class, 
		       integer_value(THE_SLOT_OF(class, S_nfields)),
		       kind);
}

/******************************************************************************
 *
 * %modify-instance (used by change-class to modify in place)
 * 
 ******************************************************************************/
PRIMITIVE modify_instance(SCM old, SCM new)
{
  SCM z;

  if (NINSTANCEP(old) || NINSTANCEP(new)) 
    err("%modify-instance: both parameters must be instances", NIL);

  /* First make a clone to avoid gc-problems */
  NEWCELL(z, tc_instance);

  EXTDATA(z)	  = EXTDATA(old);
  CLASS_OF(z)	  = CLASS_OF(old);
  SLOTS_OF(z) 	  = SLOTS_OF(old);
  INST_TYPE(z)	  = INST_TYPE(old);
  ACCESSORS_OF(z) = ACCESSORS_OF(old);

  /* Now copy new content in old */
  EXTDATA(old)	    = EXTDATA(new);
  CLASS_OF(old)	    = CLASS_OF(new);
  SLOTS_OF(old)     = SLOTS_OF(new);
  INST_TYPE(old)    = INST_TYPE(new);
  ACCESSORS_OF(old) = ACCESSORS_OF(new);
  
  /* Now z will probaly be garbaged later since nobody points it */
  return old;
}

static PRIMITIVE stklos_version(void)
{
  return makestrg(strlen(STKLOS_VERSION), STKLOS_VERSION);
}

/******************************************************************************/

PRIMITIVE init_stklos()
{
  long flag;
  int old_error_context = error_context;
  
  /* Define new types */
  tc_instance = add_new_type(&instance_type);

  /* Bootstrap system. Bootstrap phase is non interruptible */
  error_context = ERR_FATAL;  flag = no_interrupt(1);

  create_Top_Object_Class();
  init_gf();
  make_primitive_classes();

  error_context = old_error_context;  no_interrupt(flag);
  
  
  /* Define new primitives */
  add_new_primitive("stklos-version",	       tc_subr_0, stklos_version);
  add_new_primitive("instance?", 	       tc_subr_1, instancep);
  add_new_primitive("slot-ref",		       tc_subr_2, slot_ref);
  add_new_primitive("slot-set!",	       tc_subr_3, slot_set);
  add_new_primitive("slot-bound?",	       tc_subr_2, slot_boundp);

  add_new_primitive("class-of",		       tc_subr_1, class_of); 
  add_new_primitive("class-name",	       tc_subr_1, class_name);
  add_new_primitive("class-direct-supers",     tc_subr_1, class_direct_supers);
  add_new_primitive("class-direct-slots",      tc_subr_1, class_direct_slots);
  add_new_primitive("class-precedence-list",   tc_subr_1, class_cpl);
  add_new_primitive("class-slots",	       tc_subr_1, class_slots);
  add_new_primitive("slot-exists?",	       tc_subr_2, slot_existsp);

  add_new_primitive("%allocate-instance",      tc_subr_1, allocate_instance);
  add_new_primitive("%initialize-object",      tc_subr_2, initialize_object);
  add_new_primitive("%compute-slots",	       tc_subr_1, compute_slots);
  add_new_primitive("%compute-initializers",   tc_subr_1, build_initializers);
  add_new_primitive("%modify-instance",	       tc_subr_2, modify_instance);
  return UNDEFINED;
}

#endif /* defined USE_STKLOS */
