(require "bugs")

(define (bg-compile-loop code agenda)
  (if (null? agenda)
      code
      (let ((fn (caar agenda))
	    (env (cadar agenda))
	    (stack-image (cddar agenda)))
	(bg-compile-fn
	 fn env stack-image (cdr agenda)
	 (lambda (c agenda)
	   (bg-compile-loop (append! (cons (bg-clambda-name fn) c) code)
			    agenda))))))
  
  
; similar to rabbit's TRIVIALIZE

(define bg-label (bg-make-identifier-family "L"))

(define (bg-var-location var env stack-image)
  (if (bg-var-properties-global? var)
      (list 'global (bg-var-properties-name var))
      (let ((stack-pos (position var stack-image)))
	(if stack-pos
	    (list 'sp (- stack-pos)
		  (list 'comment (bg-var-properties-alpha-name var)))
	    (list 'fp
		  (- (bg-frame-depth env)
		     (bg-frame-depth (bg-var-properties-frame var)))
		  (bg-var-properties-offset var)
		  (list 'comment (bg-var-properties-alpha-name var)))))))

(define (bg-gen-triv node env stack-image)
  (cond
   ((bg-constant? node)
    `((push-const ,(bg-constant-value node))))
   ((bg-var-ref? node)
    `((push ,@(bg-var-location (bg-var-ref-properties node) env stack-image))))
   ((bg-set!-? node)
    `(,@(bg-gen-triv (bg-set!-body node) env stack-image)
      (set ,@(bg-var-location (bg-set!-var node) env stack-image))))
   ((bg-if? node)
    (let ((els (bg-label))
	  (endif (bg-label)))
      `(,@(bg-gen-triv (bg-if-pred node) env stack-image)
	(branch-false ,els)
	,@(bg-gen-triv (bg-if-con node) env stack-image)
	(branch ,endif)
	,els
	,@(bg-gen-triv (bg-if-alt node) env stack-image)
	,endif)))
   ((bg-combination? node)
    (let* ((args (bg-combination-args node))
	   (fn (car args))
	   (params (cdr args)))
      (if (bg-var-ref? fn)		;must be a built-in function
	  `(,@(apply append
		     (map (lambda (p) (bg-gen-triv p env stack-image)) params))
	    (call-prim ,(bg-var-ref-name fn)))
	  (begin
	    (if (not (bg-lambda? fn))	;must be a lambda w/ trivial body
		(bg-internal-error "bg-gen-triv: confusing triv combination"
				   fn))
	    (if (not (eq? (length params) (length (bg-lambda-vars-bound fn))))
		(bg-internal-error "bg-gen-triv: wrong # args to lambda"
				   node))
	    (let ((args
		   (map (lambda (p a)
			  (cons (list 'comment 'computing
				      (bg-var-properties-alpha-name p))
				(bg-gen-triv a env stack-image)))
			(bg-lambda-vars-bound fn) params)))
	      `(,@(apply append args)
		,@(bg-gen-triv (bg-lambda-body fn) env
			       (append (reverse (bg-lambda-vars-bound fn))
				       stack-image))
		(roll-pop ,(1+ (length (bg-lambda-vars-bound fn))))))))))))


;
;
; the code returned will be one big reversed list of instructions implementing
; the nodes in the agenda in reverse order.  The top level should reverse this
; list.
;
; continue takes (continue code fns)

(define (bg-compile-fn clambda env stack-image agenda continue)
  (if (not (bg-clambda? clambda))
      (bg-internal-error
       "bg-compile: only functions (not code fragments) can be compiled"
       clambda)
      (let* ((body-env (or (bg-created-frame clambda) env))
	     (initial-stack-image
	      (append (bg-clambda-vars clambda) stack-image))
	     (body-stack-image
	      (if (eq? body-env env)
		  initial-stack-image
		  (map (lambda (v) (if (bg-closed-early? v) #f v))
		       initial-stack-image))))
	(bg-compile-body (bg-clambda-body clambda)
			 body-env
			 body-stack-image
			 agenda
			 (lambda (code fns)
			   (continue
			    (bg-prepend-prologue clambda
						 code
						 initial-stack-image)

			    fns))))))

						      

;; bg-closed-early? is #t iff v is moved to a consed frame on entry to the
;; scope of its declaration.	   
(define (bg-closed-early? v)
  (not (null? (bg-var-properties-write-refs v))))

; with some additional complication we could avoid creating frames which are 
; not frame-lowest?.
(define (bg-created-frame cl)
  (bg-clambda-frame cl))

(define (bg-compile-body cnode env stack-image agenda continue)
  (cond
   ((bg-cset!-? cnode)
    (bg-val-code (bg-cset!-body cnode) env stack-image agenda
		 (lambda (code fns)
		   (bg-finish-call
		    (append! code
			     `((set ,@(bg-var-location (bg-cset!-var cnode)
						       env stack-image))))
		    (bg-cset!-cont cnode)
		    1 env stack-image agenda continue))))
   ((bg-ccombination? cnode)
    (bg-comp-ccombination cnode env stack-image agenda continue))
   ((bg-cif? cnode)
    (bg-val-code
     (bg-cif-pred cnode) env stack-image agenda
     (lambda (pred agenda)
       (bg-compile-body
	(bg-cif-con cnode) env stack-image agenda
	(lambda (con agenda)
	  (bg-compile-body
	   (bg-cif-alt cnode) env stack-image agenda
	   (lambda (alt agenda)
	     (continue
	      (let ((el (bg-label)))
		`(,@pred
		  (branch-false ,el)
		  ,@con
		  ,el
		  ,@alt))
	      agenda))))))))
   (else
    (bg-internal-error "bg-compile-body: don't understand cnode." cnode))))

(define (bg-comp-ccombination cnode env stack-image agenda continue)
  (let ((fn (car (bg-ccombination-args cnode))))
    (cond
     ((and (bg-trivial? fn) (bg-trivial-fn? (bg-trivial-node fn)))
      (bg-map-val-code
       (cddr (bg-ccombination-args cnode)) ; compile args except continuation
       env stack-image agenda
       (lambda (args agenda)
	 (bg-finish-call
	  (append args `((call-prim ,(bg-var-ref-name (bg-trivial-node fn)))))
	  (cadr (bg-ccombination-args cnode))
	  1 env stack-image agenda continue))))

     (else (bg-map-val-code
	    (cdr (bg-ccombination-args cnode))
	    env stack-image agenda
	    (lambda (code agenda)
	      (bg-finish-call code
			      (car (bg-ccombination-args cnode))
			      (1- (length (bg-ccombination-args cnode)))
			      env stack-image agenda continue)))))))



(define (bg-known-fn-ref cnode)
  (and (bg-trivial? cnode)
       (bg-var-ref? (bg-trivial-node cnode))
       (bg-var-properties-known-function
	(bg-var-ref-properties (bg-trivial-node cnode)))))

(define (bg-finish-call code cnode args-stacked env stack-image agenda cont)
  (cond

   ((and (bg-known-fn-ref cnode) (bg-clambda-closure (bg-known-fn-ref cnode)))
    (let ((fn (bg-known-fn-ref cnode)))
      (case (bg-clambda-closure fn)
	((ezclose)
	 (bg-val-code
	  cnode env stack-image agenda
	  (lambda (fncode agenda)
	    (cont
	     (append code fncode
		     `((pop-fp-call-abs
			,(bg-clambda-name (bg-known-fn-ref cnode)))))
	     agenda))))
	((noclose)
	 (cont
	  (append code
		  (if (bg-clambda-comp-env fn)
		      `((fp-cdr ,(- (bg-frame-depth (bg-clambda-comp-env fn)
						    (bg-frame-depth env)))))
		      `((fp-nillify)))
		  `((call-abs ,(bg-clambda-name (bg-known-fn-ref cnode)))))
	  agenda)))))
   ; the case of a known function of closure type #f is caught in the else
   ; clause of this cond

   ((bg-clambda? cnode)
    ; This case could be slightly more intellegent if it cooperated with
    ; bg-comp-ccombination.  For example, continuations passed to clambda's are
    ; almost never referenced (and so don't really need to be passed).  Other
    ; unreferenced paramters, evalled only for side effect don't have to be
    ; passed either.  Neither opt. would make much difference for the byte-code
    ; machine though (which always has `enough' registers).
    (let* ((init-stack (append (bg-clambda-vars cnode) stack-image))
	   (body-stack (map (lambda (v) (and v (or (bg-closed-early? v) v)))
			    init-stack)))
      (bg-compile-body
       (bg-clambda-body cnode)
       (or (bg-created-frame cnode) env)
       body-stack
       agenda
       (lambda (body agenda)
	 (cont
	  (append code
		  `((comment inline-call))
		  (bg-prepend-prologue cnode body init-stack))
	  agenda)))))
      
   (else (bg-val-code cnode env stack-image agenda
		      (lambda (fncode agenda)
			(cont
			 (append code fncode `((call ,args-stacked)))
			 agenda))))))

(define (bg-map-val-code cnodes env stack-image agenda continue)
  (let ((answers
	 (map (lambda (c)
		(bg-val-code c env stack-image '() (lambda (c a) (cons c a))))
	      cnodes)))
    (continue
     (apply append! (nreverse (map car answers)))
     (append (apply append (map cdr answers)) agenda))))
		       

(define (bg-val-code cnode env stack-image agenda continue)
  (cond
   ((bg-trivial? cnode)
    (continue (bg-gen-triv (bg-trivial-node cnode) env stack-image)
	      agenda))
   ((bg-cvar-ref? cnode)
    (continue
     `((push ,@(bg-var-location
		(bg-cvar-ref-properties cnode) env stack-image)))
     agenda))
   ((bg-clambda? cnode)
    (case (bg-clambda-closure cnode)
      ((noclose)
       (continue
	`((push noclose-fn (comment nonclosure ,(bg-clambda-name cnode)))) 
	(cons (cons cnode (cons env stack-image)) agenda)))
      ((ezclose)
       (continue `(,@(bg-close-code cnode env stack-image)
		   (push-fp))
		 (cons (cons cnode (cons env '())) agenda)))
      ((#f)
       (continue `(,@(bg-close-code cnode env stack-image)
		   (close ,(bg-clambda-name cnode)))
		 (cons (cons cnode (cons env '())) agenda)))))
   (else (bg-internal-error "bg-val-code: don't understand cnode." cnode))))

(define (bg-close-code cnode env stack-image)
  `((comment closing ,(bg-clambda-name cnode))
    ,@(map
       (lambda (v)
	 `(move ,(bg-var-location v #f stack-image)
		,(bg-var-location v env '())))
       (intersect (bg-clambda-closerefs cnode) stack-image))))

; close over all vars ever framed which are set!'ed below this clambda

(define (bg-prepend-prologue clambda code init-stack)
  (append!
   (let ((f (bg-created-frame clambda)))
     (if f
	 `((comment prolog of ,(bg-clambda-name clambda))
	   (mkframe ,(length (bg-frame-vars f)))
	   ,@(map
	      (lambda (v)
		`(move ,(bg-var-location v #f init-stack)
		       ,(bg-var-location v f '())))
	      (intersect (bg-frame-vars f)
			 (bg-clambda-set!-vars clambda))))
	 '()))
   code))
