; -*- Mode: Scheme; Syntax: Scheme; Package: Scheme; -*-
; File emit.scm / Copyright (c) 1989 Jonathan Rees / See file COPYING

;;;; Common Lisp code emission utilities

; This is intimately tied up with the GENERATE module, but is
; separated for the purpose of producing alternate implementations of
; GENERATE with different internal calling conventions.  Thus GENERATE
; may know a lot about this module, but not vice versa.


; If @TARGET-PACKAGE is #f, leave unqualified globals in the SCHEME
; package.  Otherwise, intern them in the target package.

(define @target-package (make-fluid #f))


; @TRANSLATING-TO-FILE? This controls a number of inconsequential code
; generation decisions, e.g. whether the (IF #F X) should return
; unspecified and whether local variables should be turned into
; symbols in the target package.

(define @translating-to-file? (make-fluid #f))


; Global variable management:

(define (get-global-reference global)
  (if (symbol? (variable-substitution global))
      (variable-substitution global)
      (let ((path (variable-path global)))
	(change-package (variable-name global)
			(if path
			    ;; Collapse FOO:BAR:BAZ to simply BAR:BAZ
			    ;; INTERN will barf if package doens't exist
			    (variable-name path)
			    (fluid @target-package))))))

(define (emit-global-set! var CL-sym rhs-code)
  (cond ((mutable-global? var)
	 `(lisp:setq ,CL-sym ,rhs-code))
	(else
	 ;; Do an ENVIRONMENT-SET!
	 `(schi:set!-aux
	   ,(environment-marker (fluid @target-package))
	   (lisp:quote ,(variable-name var))
	   ,rhs-code
	   (lisp:quote ,CL-sym)))))

; SUBSTITUTE-AND-PEEP
; LISP:SUBLIS would suffice here, but this additionally does some
; peephole optimizations.  Careful -- this is semantically blind!
; In particular, never put lambda-bindings in SUBST-type definitions.

(define (substitute-and-peep alist cl-form)
  (cond ((symbol? cl-form)
	 (let ((probe (assq cl-form alist)))
	   (if probe (cdr probe) cl-form)))
	((pair? cl-form)
	 (let ((yow (map (lambda (z) (substitute-and-peep alist z)) cl-form)))
	   (case (car yow)
	     ((lisp:funcall) (funcallify (cadr yow) (cddr yow)))
	     ((lisp:and lisp:or)
	      `(,(car yow)
		,@(apply append (map (lambda (yiz)
				       (if (car-is? yiz (car yow))
					   (cdr yiz)
					   (list yiz)))
				     (cdr yow)))))
	     (else yow))))))

; Dinky utilities

(define (insert-&rest l)
  (if (null? (cdr l))
      `(lisp:&rest ,@l)
      (cons (car l) (insert-&rest (cdr l)))))

(define (cl-externalize-locals vars env)
  (map (lambda (var)
	 (cl-externalize-local (variable-name var) env))
       vars))

(define (cl-externalize-local name env)
  (if (qualified-symbol? name)
      ;; Don't touch local variables that aren't named by ordinary
      ;; Scheme symbols.
      name
      (if (name-in-use? name env)
	  (in-target-package (make-name-from-uid name (generate-uid)))
	  (in-target-package name))))

(define (name-in-use? name env)
  (let loop ((env env))
    (cond ((procedure? env)
	   (maybe-binding name env))
	  ((eq? name (variable-name (car env))) #t)
	  (else (loop (cdr env))))))

; Kludge -- use it heuristically only!

(define (mutable-global? var)
  (let ((name (variable-name var)))
    (and (not (qualified-symbol? name))
	 (let* ((s (symbol->string name))
		(n (string-length s)))
	   (and (>= n 3)
		(char=? (string-ref s 0) #\*)
		(char=? (string-ref s (- n 1)) #\*))))))


; Package crud

(define (in-target-package sym)		;For pretty output
  (if (fluid @translating-to-file?)
      (change-package sym (fluid @target-package))
      sym))

(define (change-package sym package)
  (if (and package (not (qualified-symbol? sym)))
      (intern-renaming-perhaps (symbol->string sym) package)
      sym))

; Code emission utilities; peephole optimizers

(define (prognify form-list)
  (if (null? (cdr form-list))
      (car form-list)
      `(lisp:progn ,@form-list)))

(define (deprognify cl-form)
  (if (car-is? cl-form 'lisp:progn)
      (cdr cl-form)
      (list cl-form)))

(define (funcallify fun args)
  (cond ((car-is? fun 'lisp:function)
	 ;; Peephole optimization
	 (let ((fun (cadr fun)))
	   (cond ((and (car-is? fun 'lisp:lambda)
		       (not (memq 'lisp:&rest (cadr fun)))
		       (= (length (cadr fun))
			  (length args)))
		  (letify (map list (cadr fun) args)
			  (prognify (cddr fun))))
		 (else
		  `(,fun ,@args)))))
	(else
	 `(lisp:funcall ,fun ,@args))))

;+++ To do: turn nested singleton LET's into LET*

(define (letify specs body)
  (if (null? specs)
      body
      `(lisp:let ,specs ,@(deprognify body))))

(define (sharp-quote-lambda? exp)
  (and (car-is? exp 'lisp:function)
       (car-is? (cadr exp) 'lisp:lambda)))

; The following hack has the express purpose of suppressing obnoxious
; warnings from losing Common Lisp compilers.  The problem would be
; mitigated if Common Lisp had some way to proclaim a variable to be
; lexical (or "not misspelled", as Moon calls it), AND if compilers treated
; variables like they did functions, permitting forward references.

(define @CL-variable-references (make-fluid 'dont-accumulate))

(define (noting-references-to-globals thunk)
  (let-fluid @CL-variable-references '() thunk))

(define (locally-specialize form-list)
  (let ((vars (fluid @CL-variable-references)))
    (if (or (null? vars)
	    (and (pair? form-list)
		 (pair? (car form-list))
		 (memq (caar form-list)
		       '(lisp:defun lisp:defstruct lisp:deftype))))
	form-list
	`((lisp:locally (lisp:declare
			  (lisp:special ,@(map get-global-reference vars)))
	    ,@form-list)))))

(define (emit-function-proclamation CL-sym);sym -> form*
  (if (memq ':dec lisp:*features*)
      (if (fluid @translating-to-file?)
	  `(,(make-photon
	      (lambda (port)
		(display "#+:DEC " port)
		(lisp:prin1 `(lisp:proclaim
			      (lisp:quote
			       (lisp:function ,CL-sym)))
			    port))))
	  `((lisp:proclaim (lisp:quote (lisp:function ,CL-sym)))))
      `()))

(define (emit-sharp-plus feature code)
  (cond ((fluid @translating-to-file?)
	 `(,(make-photon
	     (lambda (port)
	       (display "#+" port)
	       (lisp:prin1 feature port)))
	   ,code))
	((memq feature lisp:*features*)
	 `(,code))
	(else
	 `())))

(define *top-level-forms-lose?*
  (memv :lispm lisp:*features*))  ;or maybe just :symbolics

(define (emit-top-level code)		;form* -> form
  (if (and *top-level-forms-lose?*
	   (fluid @lambda-encountered?))
      (let ((g (lisp:gentemp "TOP\\")));;!?!?
	`(lisp:progn (lisp:defun ,g () ,@code)
		     (,g)))
      (prognify code)))
