
;;; A quick and dirty decimal world dump analyzer
;;; (C) Kevin Lang, Fall '86, CMU Oaklisp project.


(herald analyze (env t))

(set (load-noisily?) nil)


(define int-tag 0)
(define imm-tag 1)
(define loc-tag 2)
(define ptr-tag 3)


(define-local-syntax (when test . body)
  `(if ,test (block . ,body)))

(define-local-syntax (dolist . restuff) 
  (destructure ((((var l) . body) restuff))
    (let* ((itt-var (generate-symbol 'g))
	   (v-x (generate-symbol 'g)))
      `(labels (( (,v-x ,itt-var)
		 (if (null? ,itt-var) nil
		     (let ((,var (car ,itt-var)))
		       ,@body
		       (,v-x (cdr ,itt-var)))) ))
	 (,v-x ,l)))))


(define (split ref)
  (cond ((not (number? ref))
	 (error "~s is not a ref" ref))
	((> (abs ref) (* 1024 1024))
	 (cons 'w 'int))
	(else
	 (cons (ash ref -2)
	       (nth '(int imm loc ptr) (logand ref 3))))))


(lset world nil)
(lset varstart nil)
(lset symstart nil)

(define (init-aworld inlist)
  (let ((array-size (length inlist)))
    (set world (make-vector array-size))
    (iterate Step ((i 0)(l inlist))
      (if (< i array-size)
	  (block (set (vref world i) (car l))
		 (Step (+ i 1) (cdr l)))))))


(define (split-ww i)
  (split (vref world i)))

(lset ref-table (make-table))

(define (intern-ref ref name)
  (let* ((s (split ref))
	 (val (car s))
	 (tag (cdr s)))
    (cond ((eq? 'int tag)
	   val)
	  ((eq? 'imm tag)
	   s)
	  (else
	   (let* ((try (table-entry ref-table val))
		  (interned-name
		   (or try
		       (let ((name (or name (addr-label val))))
			 (set (table-entry ref-table val) name)))))
	     (if (eq? 'loc tag)
		 (list 'loc interned-name)
		 interned-name))))))


(define (lookup-ref ref addr)
  (let* ((lk (intern-ref ref nil))
	 (lr (cond ((and (pair? lk)
			 (eq? 'loc (first lk))
			 (pair? (second lk))
			 (eq? 'var (first (second lk))))
		    (list 'glo (second (second lk))))
		   (else
		    lk))))
    (if (and (>= addr varstart)
	     (< addr symstart)
	     (or (symbol? lr)(pair? lr)))
	(list (split ref) lr)
	lr)))


(define (analyze-file file)
  (let ((dumpin  (read-entire-file-ext file 'du))
	(symin (read-file-ext file 'sym)))
    (complain-maybe = (length dumpin) (+ 4 (fourth dumpin)) 'dumplength)
    (init-aworld (nthcdr dumpin 4))
    (format t "words: ~s~%" (fourth dumpin))
    (let* ((worldend (fourth dumpin))
	   (var-list (second (first symin)))
	   (sym-list (second (second symin)))
	   (varad (car (split-ww (cdr-assq '%%varloc var-list))))
	   (symad (car (split-ww (cdr-assq '%%symloc var-list)))))
      (set varstart varad)
      (set symstart symad)
      (set ref-table (make-table 'ref-table))
      (dolist (p sym-list)
	(let ((name (car p))
	      (ref (tagize-ptr (cdr p))))
	  (intern-ref ref (list 'sym name))))
      (dolist (p var-list)
	(let ((name (car p))
	      (addr (cdr p))
	      (ref (tagize-ptr (cdr p))))
	  (intern-ref ref (list 'var name))
	  (intern-ref (vref world addr) name)))
      (iterate step ((i 0))
	(when (< i worldend)
	  (intern-ref (vref world i) nil)
	  (step (1+ i))))
      (with-open-streams ((outfile (open (->filename-ext file 'an) '(out))))
	(print-init outfile)
	(iterate step ((i 0))
	  (when (< i worldend)
	    (let ((here (table-entry ref-table i)))
	      (if here (print-labo here outfile)))
	    (print-dato (lookup-ref (vref world i) i) outfile)
	    (step (1+ i))))
	(format outfile "~%")))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;  random stuff
; 


(define (read-entire-file-ext file-symbol ext)
  (with-open-streams ((infile (open (->filename-ext file-symbol ext) '(in))))
    (iterate step ((otl '())(try (read infile)))
      (if (eof? try)
	  (reverse! otl)
	  (step (cons try otl)(read infile))))))

(define (read-file-ext file-symbol ext)
  (with-open-streams ((infile (open (->filename-ext file-symbol ext) '(in))))
    (read infile)))



(define first car)
(define second cadr)
(define third caddr)
(define fourth cadddr)


(define (tagize-int x)  (+ int-tag (* x 4)))
(define (tagize-ptr x)  (+ ptr-tag (* x 4)))
(define (tagize-loc x)  (+ loc-tag (* x 4)))


(define (addr-label n)
  (list (if (< n varstart)
	    'cod
	    'm)
	n))

(define (cdr-assq x l)
  (let ((a (assq x l)))
    (if (not a)
      (error "cdr-assq: ~s not found in ~s" x l)
      (cdr a))))

(define (complain-maybe pred thing1 thing2 check)
  (if (not (pred thing1 thing2))
      (error "~s: ~s and ~s not ~s ~&" check thing1 thing2 pred)))

(define (->filename-ext name ext)
    (->filename
	  (string->symbol
	   (string-append (symbol->string name) "." (symbol->string ext)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;  special output stuff
; 


(define (stringize-ref ref)
  (string-downcase
   (cond ((pair? ref)
	  (if (symbol? (cdr ref))
	      (format nil "~a.~a" (car ref)(cdr ref))
	      (format nil "~a.~a" (first ref)(stringize-ref (second ref)))))
	 ((symbol? ref)
	  (symbol->string (abbreviate-symbol ref)))
	 (else
	  (format nil "~a" ref)))))


(lset maxlen nil)
(define maxpagewid 77)
(define labextra 0)

(define (print-init fil)
      (format fil "~%")
      (set (hpos fil) (+ maxlen 3)))

(define (print-labo ref fil)
  (format fil "~%~a:" (truncate-string (stringize-ref ref) (+ labextra maxlen)))
  (set (hpos fil) (+ maxlen (+ 2 labextra))))

(define (print-dato ref fil)
  (let ((the-string (stringize-ref ref)))
    (when (> (+ (display-width the-string) (hpos fil)) maxpagewid)
      (print-init fil))
    (format fil "~a " the-string)))


(define (truncate-string s1 trlen)
  (let ((len (string-length s1)))
    (if (<= len trlen)
	s1
	(substring s1 0 trlen))))


(define (init-maxlen n)
  (set maxlen n)
  (set abbrev-table (make-table 'abbrev-table)))

(init-maxlen 15)


(define (abbreviate-symbol sym)
  (let ((try (table-entry abbrev-table sym)))
    (or try
	(set (table-entry abbrev-table sym)
	     (crude-abbreviate-symbol sym)))))

(define (crude-abbreviate-symbol sym)
    (string->symbol (truncate-string  (symbol->string sym) maxlen)))


(define (abbsym sym)
  (let ((nkilled 0)
	(must-kill 2)
	(trylist nil))
    (labels (((finish l)
	      (string->symbol (truncate-string (list->string (map char-upcase l)) maxlen)))
	     ((ntokill l1 l2)
	      (- (+ (length l1)(length l2)) maxlen))
	     ((finish-maybe l r)
	      (if (or (< (ntokill l r) 1)
		      (null? trylist))
		  (finish (append (reverse l) r))
		  (if (< nkilled must-kill)
		      ((car trylist) l r)
		      (block (set trylist (append (cdr trylist)(list (car trylist))))
			     (set nkilled 0)
			     (let ((guy-to-try (car trylist)))
			       (if (eq? guy-to-try killvow)
				   (guy-to-try (append (reverse r) l) nil)
				   (guy-to-try nil (append (reverse l) r))))))))
	     ((killphy l r)
	      (cond ((null? l)
		     (killphy (cons (car r) l)(cdr r)))
		    ((null? r)
		     (if (= nkilled 0) (set trylist (cdr trylist)))
		     (set nkilled must-kill)
		     (finish-maybe l r))
		    ((and (lowercase? (car l))(eq? #\- (car r)))
		     (increment nkilled)
		     (finish-maybe (cons (car r) (cdr l)) (cdr r)))
		    (else
		     (killphy (cons (car r) l)(cdr r)))))
	     ((killvow l r)
	      (cond ((null? l)
		     (if (= nkilled 0) (set trylist (cdr trylist)))
		     (set nkilled must-kill)
		     (finish-maybe l r))
		    ((memq? (car l) '(#\a #\e #\i #\o #\u #\y))
		     (increment nkilled)
		     (finish-maybe (cdr l) r))
		    (else
		     (killvow (cdr l) (cons (car l) r)))))
	     ((initup l r)
	      (if (null? r)
		  (block (set nkilled 0) (set trylist (cdr trylist))(killvow l r))
		  (if (or (null? l) (memq? (car l) '(#\- #\%)))
		      (initup (cons (char-upcase (car r)) l) (cdr r))
		      (initup (cons (car r) l) (cdr r)))))

	     )
	    (set trylist (list initup killvow killphy))
	    (finish-maybe nil (map char-downcase (string->list (symbol->string sym)))))))
		   
    
    

