;;; A quick and dirty Oaklisp file linker.
;;; (C) Kevin Lang, Fall '86, CMU Oaklisp project.


(herald tool (env t))


; the input to this program is a bunch of files, each of which
; contains a big list whose format is
; 
; ((
;   ((constant 14 ((foo bar) baz))
;    (variable 2 append)  ; these offsets are 
;    (code 28 6))         ; in terms of 16 shortwords
;   (100 343 232 ... )    ; 16 bit opcodes
;   )
;  ( 
;     another code block
;   )
; )



(define cell-size 1)
(define pair-size 3)
(define symbol-size 2)
(define null-size 1)
(define type-size 9)
(define coercable-type-size 10)

(define big-endian? t)

(define int-tag 0)
(define imm-tag 1)
(define loc-tag 2)
(define ptr-tag 3)

(define return-opcode (* 24 256))
(define noop-opcode 0)

(define value-stack-size   (* 16 1024))
(define context-stack-size (* 16 1024))
(define total-world-size   (* 768 1024))
(lset world-array-size 0)	

(define cache-pairs? t)

(set div quotient)



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;macros for speed
;

; this code depends on bignums

(define (tagize-int x)
  (+ int-tag (* 4
		(if (negative? x)
		    (+ x (expt 2 30))
		    x))))

(define-local-syntax (tagize-imm x)  `(fx+ imm-tag (ash ,x 2)))
(define-local-syntax (tagize-ptr x)  `(fx+ ptr-tag (ash ,x 2)))
(define-local-syntax (tagize-loc x)  `(fx+ loc-tag (ash ,x 2)))

(define-local-syntax (store-world-int x word-addr) `(store-world-word (tagize-int ,x) ,word-addr))
(define-local-syntax (store-world-ptr x word-addr) `(store-world-word (tagize-ptr ,x) ,word-addr))
(define-local-syntax (store-world-loc x word-addr) `(store-world-word (tagize-loc ,x) ,word-addr))

(define-local-syntax (zero-enough? x) `(or (eq? ,x 0) (alikeq? ,x '(0 . 0))))

(define-local-syntax (store-world-word word word-addr)
  `(let ((oldword (vref world ,word-addr)))
     (if (zero-enough? oldword)
	 (set (vref world ,word-addr) ,word)
	 (error "attempted overwrite <~s ~s> ~&" ,word ,word-addr))))

(define-local-syntax (store-world-opcodes o1 o2 word-addr) 
  `(store-world-word (cons ,o1 ,o2) ,word-addr))

(define-local-syntax (toble-probe key toble)
  `(or (table-entry ,toble ,key)
       (block
	(set (table-entry ,toble ,key) (make-cell nil))
	nil)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;top level stuff
;


(define-local-syntax (when test . body)
  `(if ,test (block . ,body)))


(lset stashed-inlist nil)

(lset var-table nil)
;impose an ordering on variables
(lset var-list nil) 
(lset blk-table nil)
(lset sym-table nil)
(lset pair-table nil)
;(lset string-table nil)


(define (init-tables)
  (set var-list '())
  (set var-table (make-toble))
  (set blk-table (make-toble))
  (set sym-table (make-toble))
; (set string-table (make-toble))
  (set pair-table (make-toble)))


(define (tool-files in-files out-file)
  (format t "reading ...~%")
  (let ((in-names (map (lambda (na) (file-name-ext na 'oa)) in-files)))
    (tool (map (lambda (name)
		 (format t "~a~%" name)
		 (read-oa-file name))
	       in-names))
    (dump-tables (file-name-ext out-file 'sym))
    (format t "symbol-table~%")
    (dump-world (file-name-ext out-file 'cold))
    (cons 'world out-file)))


(define (tool inlist)
  (set stashed-inlist inlist)
  (init-tables)
  (count-things inlist)
  (format t "counts~%")
  (compute-base-addresses)
  (format t "base-addrs~%")
  (init-world)
  (format t "world-init~%")
  (layout-symbols-and-variables)
  (format t "syms-and-vars~%")
  (layout-handbuilt-data)
  (format t "handbuilt~%")
  (patch-symbols)
  (format t "symbol-patches~%")
  (build-blk-table inlist)
  (format t "blk-table~%")
  (spew-opcodes inlist)
  (format t "opcodes~%")
  (set stashed-inlist nil)
)







;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;first pass
;  count everything
;  initialize symbol and variable tables


(define reg-code-delta 4)   ;extra opcode units per block
(define top-code-delta -2)  ;in actual world

(lset blk-count 0)
(lset opc-count 0)
(lset var-count 0)
(lset sym-count 0)
(lset dat-count 0) ;words
(lset max-blks 0)  ;most blocks in one file

;

(define (count-things inlist)
  (set dat-count 0)
  (set blk-count 0)
  (set opc-count (- reg-code-delta top-code-delta)) ; header for top code blk
  (walk count-variable vars-to-preload)
  (walk (lambda (fil)
	  (let ((nblks (length fil)))
	    (set opc-count (fx+ opc-count
				(fx+ top-code-delta
				     (fx* reg-code-delta (fx- nblks 1)))))
	    (set max-blks (max nblks max-blks))
	    (set blk-count (fx+ nblks blk-count))
	    (walk (lambda (blk)
		    (print-dot)
		    (count-opcodes blk)
		    (walk (lambda (x)
			    (let ((keyword (first x)))
			      (cond ((eq? 2 keyword);constant
				     (count-data (third x))
				     )
				    ((eq? 0 keyword);variable
				     (count-variable (third x)))
				    ((eq? 1 keyword);code
				     nil)
				    ((eq? 'constant keyword)
				     (count-data (third x))
				     )
				    ((eq? 'variable keyword)
				     (count-variable (third x)))
				    ((eq? 'code keyword)
				     nil)
				    (else
				     (error "bad inline patch keyword ~s ~&" keyword))
				    )))
			  (first blk)))
		  fil)))
	inlist)
  (set var-list (reverse! var-list))
  (set var-count (toble-nkeys var-table))
  (set sym-count (toble-nkeys sym-table))
  (set dat-count (fx+ dat-count (handbuilt-data-size)))
  (format t "~&ops:~s  vars:~s  syms:~s cells:~s~%"
	  opc-count var-count sym-count dat-count))
  


(define (count-opcodes blk)
  (let ((op-co (length (second blk))))
    (if (odd? op-co)
	(error "<~s> odd # of opcodes is ~n ~&" op-co blk))
    (set opc-count (fx+ opc-count op-co))))


(define (count-variable v)
  (if (not (toble-probe v var-table))
      (push var-list v))
  (count-symbol v))


(define (count-symbol v)
  (set dat-count (fx+ (string-size (symbol->string v)) dat-count))
  (toble-probe v sym-table))


(define (count-data d)
  (cond ((symbol? d)
	 (count-symbol d))
	((number? d)
	 nil)
	((char? d)
	 nil)
 	((null? d)
	 nil)
	((pair? d)
	 (set dat-count (fx+ pair-size dat-count))
	 (count-data (car d))
	 (count-data (cdr d)))
	((string? d)
	 (set dat-count (fx+ (string-size d) dat-count)))
	(else (error "count: bad inline constant <~s> ~&" d))))




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;second pass
;  layout symbols and variables
;  build block table


(lset start-of-opc-space 0)
(lset start-of-var-space 0)
(lset start-of-sym-space 0)
(lset start-of-dat-space 0)

(lset next-free-dat 0)

(define  (compute-base-addresses)
  (set start-of-opc-space 0)
  (set start-of-var-space (+ start-of-opc-space (quotient opc-count 2)))
  (set start-of-sym-space (+ start-of-var-space (* var-count cell-size)))
  (set start-of-dat-space (+ start-of-sym-space (* sym-count symbol-size)))
  (set world-array-size (+ start-of-dat-space dat-count))
  (set next-free-dat start-of-dat-space))


(define (alloc-dat n)
  (let ((old-addr next-free-dat))
    (set next-free-dat (fx+ next-free-dat n))
    (if (fx> next-free-dat world-array-size)
	(space-error 'data)
	old-addr)))


(define (layout-symbols-and-variables)
  (let ((nextvar start-of-var-space)
	(nextsym start-of-sym-space))
    (walk (lambda (name)
	    (toble-set name nextsym sym-table)
	    (set nextsym (fx+ nextsym symbol-size))
	    (toble-set name nextvar var-table)
	    (set nextvar (fx+ nextvar cell-size)))
	  var-list)
    (toble-walk (lambda (name addr)
		  (if (not addr)
		      (block (toble-set name nextsym sym-table)
			     (set nextsym (fx+ nextsym symbol-size)))))
		sym-table)))


(define (patch-symbols)
  (toble-walk (lambda (name addr)
		(store-world-ptr where-symbol-lives addr)
		(store-world-word (string-alloc (symbol->string name)) (fx+ 1 addr)))
	      sym-table))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; The blk table is a little strange.
; Its keys are special code numbers computed by uniq-blkno.
; The entries are conses (addr . kind), where kind says
; what sort of code munching needs to be done to append the
; top level blocks.


(lset first-regular-blk-addr 0)

(define (uniq-blkno filno blkno)
  (fx+ blkno (fx* max-blks filno)))

(define (build-blk-table inlist)
  (let* ((nfils (length inlist))
	 (next-blk-addr (+ start-of-opc-space
			   (quotient (- reg-code-delta top-code-delta) 2)))
	 (allocate-blk (lambda (blk toplevelp)
			 (let ((old-addr next-blk-addr)
			       (nwords (quotient (fx+ (length (second blk))
						      (if toplevelp
							  top-code-delta
							  reg-code-delta))
						 2)))
			   (set next-blk-addr (fx+ next-blk-addr nwords))
			   (if (fx> next-blk-addr start-of-var-space)
			       (space-error 'code)
			       old-addr)))))

    (iterate filStep ((fils inlist)(filno 0))
      (when fils
	(let ((blk (first (first fils))))
	  (set (contents (toble-install (uniq-blkno filno 0)
					blk-table))
	       (cons (allocate-blk blk t)
		     (if (fx= filno (fx- nfils 1)) 'lastoplevel 'toplevel))))
	(filStep (rest fils)(fx+ 1 filno))))

    (set first-regular-blk-addr next-blk-addr)

    (iterate filStep ((fils inlist)(filno 0))
      (when fils
	(iterate blkStep ((blks (rest (first fils)))(blkno 1))
	  (when blks
	    (let ((blk (first blks)))
	      (set (contents (toble-install (uniq-blkno filno blkno)
					    blk-table))
		   (cons (allocate-blk blk nil)
			 'regular)))
	    (blkStep (rest blks)(fx+ 1 blkno))))
	(filStep (rest fils)(fx+ 1 filno))))))




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;third pass
;  throw code out into the world


(define (spew-opcodes inlist)
  (store-world-ptr where-%code-vector-lives start-of-opc-space)
  (store-world-int first-regular-blk-addr (+ 1 start-of-opc-space)) 
  (store-world-ptr where-nil-lives (+ 2 start-of-opc-space))
  (iterate filStep ((fils inlist)(filno 0))
    (when fils
      (iterate blkStep ((blks (first fils))(blkno 0))
	(when blks
	  (print-dot)
	  (let* ((blk (first blks))
		 (patches (first blk))
		 (opcodes (second blk))
		 (info (toble-get (uniq-blkno filno blkno) blk-table))
		 (base-addr (car info))
		 (blk-kind (cdr info))
		 (regp (eq? blk-kind 'regular))
		 (delta (quotient (if regp reg-code-delta top-code-delta) 2))
		 (delbase-addr (fx+ delta base-addr)))
	    (when regp
	      (store-world-ptr where-%code-vector-lives base-addr)
	      (store-world-int (fx+ 2 (quotient (length opcodes) 2)) (fx+ 1 base-addr)))
	    (iterate opStep ((ops opcodes)(addr delbase-addr))
	      (cond (ops
		     (if (fx>= addr base-addr)
			 (store-world-opcodes (first ops) (second ops) addr))
		     (opStep (cddr ops) (fx+ 1 addr)))
		    (else (if (eq? 'toplevel blk-kind)
			      (changereturntonoop (fx- addr 1))))))
	    (iterate patStep ((pats patches))
	      (when pats
		(let* ((pat (first pats))
		       (patkind (first pat))
		       (pataddr (fx+ delbase-addr (quotient (second pat) 2)))
		       (patval (third pat))
		       (patref
			(xcond ((eq? 2 patkind) (constant-refgen patval))
			       ((eq? 0 patkind) (tagize-loc (toble-get patval var-table)))
			       ((eq? 1 patkind) (tagize-ptr (car (toble-get (uniq-blkno filno patval)
									    blk-table))))
			       ((eq? 'constant patkind) (constant-refgen patval))
			       ((eq? 'variable patkind) (tagize-loc (toble-get patval var-table)))
			       ((eq? 'code patkind) (tagize-ptr (car (toble-get
								      (uniq-blkno filno patval) blk-table))))
			       )))
		  (if (fx>= pataddr base-addr)
		      (store-world-word patref pataddr)))
		(patStep (rest pats))))
	    (blkStep (rest blks)(fx+ 1 blkno)))))
      (filStep (rest fils)(fx+ 1 filno)))))



;this is a hack to string the top-level blocks together
(define (changereturntonoop addr)   
  (let* ((them (get-world-opcodes addr))
	 (op1 (car them))
	 (op2 (cdr them)))
    (cond ((fx= op2 return-opcode)
	   (overwrite-world-opcodes op1 noop-opcode addr))
	  ((and (fx= op1 return-opcode)
		(fx= op2 noop-opcode))
	   (overwrite-world-opcodes noop-opcode noop-opcode addr))
	  (else
	   (error "bad ops in toplvl blk end <~s ~s> ~&" op1 op2)))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;cons up inline constants
;

(define (constant-refgen c)
  (cond ((symbol? c)
	 (tagize-ptr (toble-get c sym-table)))
 	((null? c)
	 (tagize-ptr where-nil-lives))
	((number? c)
	 (tagize-int c))
	((char? c)
	 (tagize-imm (ash (char->ascii c) 6)))
	((and (pair? c) cache-pairs?)
	 (caching-pair-alloc c))
	((pair? c)
         (pair-alloc c))
	((string? c)
         (string-alloc c))
	(else (error "refgen: bad constant data <~s> ~&" c))))


(define (pair-alloc c)
	 (let ((newpair (alloc-dat pair-size)))
	   (store-world-ptr where-cons-pair-lives newpair)
	   (store-world-word (constant-refgen (car c)) (fx+ 1 newpair))
	   (store-world-word (constant-refgen (cdr c)) (fx+ 2 newpair))
	   (tagize-ptr newpair)))


(define (caching-pair-alloc c)
  (let* ((thc (tree-hash c))
	 (probe (toble-probe thc pair-table)))
    (if (and probe (alikeq? c (car (contents probe))))
	(cdr (contents probe))
	(let ((newp (pair-alloc c)))
	  (toble-set thc (cons c newp) pair-table)
	  newp))))


(define (string-size c)
  (let* ((strlen (string-length c))
	 (strwordlen (fx+ 3 (div (fx+ strlen (fx- 3 1)) 3))))
    strwordlen))


(define (string-alloc c)
  (let* ((strlen (string-length c))
	 (strwordlen (fx+ 3 (quotient (fx+ strlen (fx- 3 1)) 3)))
	 (newstring (alloc-dat  strwordlen))
	 (strlist (map char->ascii (string->list c))))
;   (toble-probe c string-table)
;   (set (contents (toble-probe c string-table)) newstring);
    (store-world-ptr where-string-lives newstring)
    (store-world-int strwordlen  (fx+ 1 newstring))
    (store-world-int strlen  (fx+ 2 newstring))
    (string-alloc-aux (fx+ 3 newstring) strlist strlen)
    (tagize-ptr newstring)))


(define (string-alloc-aux i l to-do)
  (iterate aux ((i i)(l l)(to-do to-do))
    (cond ((fx= to-do 0) nil)
	  ((fx= to-do 1)
	   (store-world-int (car l) i))
	  ((fx= to-do 2)
	   (store-world-int (logior (car l) (ash (cadr l) 8)) i))
	  (t
	   (let* ((c0 (car l))
		  (l1 (cdr l))
		  (c1 (car l1))
		  (l2 (cdr l1))
		  (c2 (car l2)))
	     (store-world-int (logior c0 (logior (ash c1 8) (ash c2 16))) i)
	     (aux (fx+ i 1) (cdr l2) (fx- to-do 3)))))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;hand-built things
;

(define vars-to-preload '(nil cons-pair %code-vector string symbol
			      %%symloc %%nsyms %%symsize
			      %%varloc %%nvars
			      %%toploc
			      %%big-endian? %%booted?))

(lset where-nil-lives nil)
(lset where-string-lives nil)
(lset where-symbol-lives nil)
(lset where-cons-pair-lives nil)
(lset where-%code-vector-lives nil)

(define (handbuilt-data-size)
  (+ null-size
     (+ (* 2 type-size)
	(* 2 coercable-type-size))))

(define (layout-handbuilt-data)
  (set where-nil-lives (alloc-dat null-size))
  (set where-string-lives (alloc-dat coercable-type-size))
  (set where-symbol-lives (alloc-dat coercable-type-size))
  (set where-cons-pair-lives (alloc-dat type-size))
  (set where-%code-vector-lives (alloc-dat type-size))
  (store-world-ptr where-nil-lives (toble-get 'nil var-table))
  (store-world-ptr where-string-lives (toble-get 'string var-table))
  (store-world-ptr where-symbol-lives (toble-get 'symbol var-table))
  (store-world-ptr where-cons-pair-lives (toble-get 'cons-pair var-table))
  (store-world-ptr where-%code-vector-lives (toble-get '%code-vector var-table))
  (store-world-loc 0                  (toble-get '%%toploc var-table))
  (store-world-loc start-of-sym-space (toble-get '%%symloc var-table))
  (store-world-loc start-of-var-space (toble-get '%%varloc var-table))
  (store-world-int sym-count (toble-get '%%nsyms var-table))
  (store-world-int var-count (toble-get '%%nvars var-table))
  (store-world-int symbol-size (toble-get '%%symsize var-table))
  (store-world-ptr where-nil-lives (toble-get '%%booted? var-table))
  (store-endian))

(define (store-endian)
  (overwrite-world-word 0 (toble-get '%%big-endian? var-table))
  (if big-endian?
      (store-world-int 1 (toble-get '%%big-endian? var-table))
      (store-world-ptr where-nil-lives (toble-get '%%big-endian? var-table))))


      



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;random junk
;


(define first car)
(define second cadr)
(define third caddr)
(define rest cdr)

(define	 (do-something) nil)
(define	 (do-nothing) nil)

(define (make-cell x)
  (locative x))

(define (space-error which)
  (error "out of ~s space ~&" which))

(define (print-dot)
    (write-char (standard-output) #\.)
    (force-output (standard-output)))

(define (print-vector x)
    (walk (lambda (y)
	    (format t "~s " y))
	  (vector->list x)))


(define (pw)
  (print-vector (return-world)))

(let ((*oaklisp-read-table*
       (make-read-table *standard-read-table* '*oaklisp-read-table*)))
  (set (read-table-entry *oaklisp-read-table* #[char 22]);^V
       (lambda (stream ch read-table)
	 (ignore ch)
	 (list 'fluid (read-refusing-eof stream))))
  (set (read-table-entry *oaklisp-read-table* #[char 25]);^Y
       (lambda (stream ch read-table)
	 (ignore ch)
	 (list 'coercer (read-refusing-eof stream))))

  (define (read-file file-symbol)
    (with-open-streams ((infile (open (->filename file-symbol) '(in))))
       (set (port-read-table infile) *oaklisp-read-table*)
       (read infile))))

(define (file-name-ext name ext)
  (string->symbol
   (string-append (symbol->string name) "." (symbol->string ext))))



(define (read-oa-file file)		;file already has an extension
  (let ((red (read-file file)))
    (if (pair? (caar red))
	red
	(make-oa-list red))))


(define (make-oa-list oaf-list)
  (let ((sym-vec (list->vector (first oaf-list))))
    (labels (((rewrite-syms clause)
	      (let ((car-clause (car clause)))
		(if (> car-clause (- 5 1))
		    (list (- car-clause 5)
			  (second clause)
			  (vref sym-vec (third clause)))
		    clause))))
      (map! (lambda (blk)
	      (list (map! rewrite-syms (triplify! (first blk)))
		    (second blk)))
	    (second oaf-list)))))


; this function reverses the order of the triples.
; also, it is extremely side-effecting, so watch out.

(define (triplify! inlist)
  (iterate step ((in inlist) (out '()))
    (if in
	(let* ((cddrin (cdr (cdr in)))
	       (nxtrip (cdr cddrin)))
	  (step nxtrip
		(block
		 (set (cdr cddrin) nil)
		 (cons in out))))
	 out)))



(lset expt-16-3 (expt 16 3))
(lset expt-16-2 (expt 16 2))
(lset expt-16-1 (expt 16 1))
(lset expt-16-0 (expt 16 0))


(define (print-hex num ndigits outfile)
  (cond ((null? ndigits)
	 (format outfile "~x" num))
	((fx= 4 ndigits)
	 (write-char outfile (digit->char (mod (quotient num expt-16-3) 16) 16))
	 (write-char outfile (digit->char (mod (quotient num expt-16-2) 16) 16))
	 (write-char outfile (digit->char (mod (quotient num expt-16-1) 16) 16))
	 (write-char outfile (digit->char (mod (quotient num expt-16-0) 16) 16)))
	(else
	 (iterate step ((diglist '())(ndig ndigits)(no num))
	   (cond ((fx> ndig 0)
		  (step (cons (mod no 16) diglist)(fx- ndig 1)(quotient no 16)))
		 (diglist
		  (write-char outfile (digit->char (car diglist) 16))
		  (step (cdr diglist) ndig no))
		 ((not (zero? no))
		  (error "Can't print ~s with ~s 16 ~s digits~%" num ndigits 16)))))))

  

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;test stuff
;

(define testlist 
  '(	
    (					; start of file 1
     (					; start of block

      ((constant 6 ((foo bar) 1))
       (code     10 1)
       (constant 0 ())
       (variable 2 append))
      (0 0 0 0 100 343 0 0 232 1 0 0 2 3 4 5 5 6 7 6144) ;18
      )					; end of block
     (					; start of block
      ((constant 4 12)

       (constant 0 (ivar1 ivar2))
       (constant 6 dog)
       (variable 10 dog))
      (0 0 2 3 0 0 0 0 100 343 0 0 54 23 6144 0)          ;14
      )					; end of block
     )					; end of file
    (					; start of file 2
     (					; start of block
      ((constant 0 ())
       (variable 2 nil)
       (variable 4 foo))
      (0 0 0 0 0 0 9 8 7 6 5 6144)                   ;12
      )					; end of block
     )					;end of file
    )
  )





;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;world accessor functions
;

(define two16 (expt 2 16))

(lset world nil)			;this holds an array of words.
					;opcodes are stored as pairs.

(define (init-world)
  (let ((array-size  world-array-size))
    (set world (make-vector array-size))
    (iterate Step ((i 0))
      (if (fx< i array-size)
	  (block (set (vref world i) 0)
		 (Step (fx+ i 1)))))))

(comment
 (define (store-world-word word word-addr)
   (let ((oldword (vref world word-addr)))
     (if (zero-enough? oldword)
	 (set (vref world word-addr) word)
	 (error "attempted overwrite <~s ~s> ~&" word word-addr))))

 (define (store-world-opcodes o1 o2 word-addr) 
   (store-world-word (cons o1 o2) word-addr))
 )

(define (overwrite-world-opcodes o1 o2 word-addr) 
  (set (vref world word-addr) (cons o1 o2)))

(define (overwrite-world-word word word-addr) 
  (set (vref world word-addr) word))

(define (get-world-word word-addr)
  (vref world word-addr))

(define (get-world-opcodes word-addr)
  (vref world word-addr))

(define (return-world)
  world)


(comment
 (define (store-world-int x word-addr) (store-world-word (tagize-int x) word-addr))
 (define (store-world-ptr x word-addr) (store-world-word (tagize-ptr x) word-addr))
 (define (store-world-loc x word-addr) (store-world-word (tagize-loc x) word-addr))

 (define (tagize-int x)  (fx+ int-tag (fx* x 4)))
 (define (tagize-ptr x)  (fx+ ptr-tag (fx* x 4)))
 (define (tagize-loc x)  (fx+ loc-tag (fx* x 4)))


 (define (zero-enough? x)
   (or (eq? x 0) (alikeq? x '(0 . 0))))
)




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;write output files
;

(define (dump-world file-symbol)
  (let* ((world (return-world))
	 (actual-size next-free-dat))
    (with-open-streams ((outfile (open (->filename file-symbol) '(out))))
      (write-world-header outfile)
      (iterate Step ((i 0)(j 0))
	(when (fx< i actual-size)
	  (if (zero? j) (format outfile "~%"))
	  (let* ((guy (vref world i)))
	    (xcond ((number? guy)
		    (print-hex guy nil outfile)
		    (format outfile " "))
		   ((pair? guy)
		    (when big-endian?
		      (print-hex (car guy) nil outfile)
		      (print-hex (cdr guy) 4 outfile))
		    (when (not big-endian?)
		      (print-hex (cdr guy) nil outfile)
		      (print-hex (car guy) 4 outfile))
		    (format outfile " "))))
	  (Step (fx+ i 1) (mod (fx+ j 1) 8))))
      (format outfile "~%"))
    (format t "~&total words:~s~%" actual-size)))


(define	(write-world-header outfile)
  (print-hex value-stack-size nil outfile) (format outfile " ")
  (print-hex context-stack-size nil outfile) (format outfile " ")
  (print-hex total-world-size nil outfile) (format outfile " ")
  (print-hex next-free-dat nil outfile)    (format outfile " ")
  (print-hex start-of-opc-space nil outfile) (format outfile "~%"))


(define (dump-tables file-symbol)
  (with-open-streams ((outfile (open (->filename file-symbol) '(out))))
    (pretty-print
     (list `(variables ,(ordered-toble->alist var-table var-list))
	   `(symbols   ,(reverse (toble->alist sym-table))))
     outfile)))



	;	     `(strings   ,(reverse (toble->alist string-table))))





;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; toble module
;

(comment
 (define (toble-probe key toble)
   (or (table-entry toble key)
       (block
	(set (table-entry toble key) (make-cell nil))
	nil))))

(define (toble-install key toble)
  (if (toble-probe key toble)
      (error "key ~s already installed in table ~s ~&" key toble)
      (toble-probe key toble)))


(define (toble-get key toble)
  (let ((slot (toble-probe key toble)))
    (if (not slot)
	(error "key ~s not found in table ~s ~&" key toble)
	(contents slot))))

(define (toble-set key value toble)
  (let ((slot (toble-probe key toble)))
    (if (not slot)
	(error "key ~s not found in table ~s ~&" key toble)
	(set (contents slot) value))))



(define (toble-walk lamder toble)
  (walk-table (lambda (x y)
		(lamder x (contents y)))
	      toble))


(define (toble->alist toble)
  (let ((out '()))
    (walk-table (lambda (x y)
		  (push out (cons x (contents y))))
		toble)
    out))


(define (ordered-toble->alist toble keylist)
  (let ((out '()))
    (walk (lambda (x)
	    (let ((y (table-entry toble x)))
	      (push out (cons x (contents y)))))
	  keylist)
    (reverse! out)))


(define (toble-nkeys toble)
  (let ((count 0))
    (walk-table (lambda (x y) (increment count)) toble)
    count))



(define tabsize 5000)

(define (make-toble)
    (make-table-with-size tabsize))

(define (toble-clear x)
    (clean x))





