/* Copyright (C) 1987 Barak Pearlmutter and Kevin Lang, CMU Oaklisp project. */

/* An emulator for the CMU Oaklisp bytecode.  */


#include <stdio.h>
#include <ctype.h>
#include "emulator.h"
#include "stacks.h"


#ifdef BIG_ENDIAN
#define NEW_IFETCH
#endif

#define CASE_FOUR 1



#ifdef FAST

#define trace_insts	0
#define trace_stcon	0
#define trace_cxcon	0
#define trace_meth	0
#define trace_segs	0
#define trace_mcache	0

#else

bool trace_insts = FALSE;	/* trace instruction execution */
bool trace_stcon = FALSE;	/* trace stack contents */
bool trace_cxcon = FALSE;	/* trace contents stack contents */
bool trace_meth	 = FALSE;	/* trace method lookup */
bool trace_segs	 = FALSE;	/* trace stack segment manipulation */
bool trace_mcache= FALSE;	/* trace method cache hits and misses */

#endif

bool trace_traps = FALSE;		/* trace tag traps */
bool trace_files = FALSE;		/* trace file opening */
bool trace_gc	 = FALSE;		/* trace gc carefully */

bool dump_after  = FALSE;		/* dump world after running */
bool gc_before_dump = FALSE;		/* do a GC before dumping the world */



/*
 * Processor registers
 */

stack val_stk, cxt_stk;

unsigned int e_nargs;

ref *e_bp, *e_env, e_t, e_nil,
  e_fixnum_type, e_loc_type, e_cons_type, e_env_type, *e_subtype_table,
  e_object_type, e_segment_type, e_boot_code, e_code_segment,
  *e_arged_tag_trap_table, *e_argless_tag_trap_table, e_current_method;

unsigned short *e_pc;



#define maybe_dump_world(dumpstackp)	\
{					\
  UNOPTV(val_stk.ptr = val_stk_ptr);	\
  UNOPTC(cxt_stk.ptr = cxt_stk_ptr);	\
  maybe_dump_world_proc((dumpstackp));	\
}


void maybe_dump_world_proc(dumpstackp)
     int dumpstackp;
{
  if (dumpstackp)
    {
      printf("value ");
      dump_stack_proc(&val_stk);
      printf("context ");
      dump_stack_proc(&cxt_stk);
    }

  if (dump_after)
    {
      if (gc_before_dump && !dumpstackp)
	gc(1, "impending world dump");
      dump_world(gc_before_dump && !dumpstackp);
    }
}




void printref(refin)
     ref refin;
{
  long i;
  char suffex = '?';

  if (refin&2)
    {
      ref *p = (refin&1) ? REF_TO_PTR(refin) : LOC_TO_PTR(refin);
      
      if (SPATIC_PTR(p))
	{
	  i = p - spatic.start;
	  suffex = 's';
	}
      else if (NEW_PTR(p))
	{
	  i = p - new.start + spatic.size;
	  suffex = 'n';
	}
      else i = (long)p >> 2;

      (void)printf("%ld~%ld%c", i, refin&TAG_MASK, suffex);
    }
  else
    (void)printf("%ld~%ld", refin>>2, refin&TAG_MASK);
}      



#define TRACEMETHOD(zz) {if (trace_meth) {printf("meth-trace%ld  ",zz);          \
					 printref("obj_type:%ld~%ld  ",obj_type); \
   				         printref("alist:%ld~%ld  ",alist);       \
				         printref("mptr:%ld~%ld\n",*method_ptr);  }}

#define TRACEASSQ(zz) {if (trace_meth) {printf("aq-trace%ld  ",zz);          \
   				         printref("elem:%ld~%ld  ",elem);       \
				         printref("list:%ld~%ld\n",list);  }}

#define TRACEPASSQ(zz) {printf("aq-trace%ld  ",zz);    \
			  printref("elem:%ld~%ld  ",elem);       \
			    printref("l:%ld~%ld ",l);  \
			    printref("cdr(l):%ld~%ld ",cdr(l));  \
			    if (locl) printref("*locl:%ld~%ld\n",*locl); }


/* these are inline coded now

ref assq(elem, list)
     ref elem, list;
{
  while (list != e_nil && car(car(list)) != elem) {
    list = cdr(list);
  }
  return ((list == e_nil)? e_nil : car(list));
}

ref old_pseudo_assq(elem, list)
     ref elem, list;
{
  while (list != e_nil && car(car(list)) != elem) {
    list = cdr(list);
  }
  return list;
}
*/


/* The following code uses the bring-to-front heuristic,
   and eventually needs a register to inhibit this behavior.
   This code is now inserted inline in the one place it is used.
ref pseudo_assq(elem, loclist)
ref elem, *loclist;
{
  ref thelist = *loclist;

  register  ref l = thelist;
  register  ref *locl = NULL;

 iterate:
  if (l == e_nil)
    return e_nil;
  if (car(car(l)) == elem) {
    if (locl) {
      *locl = cdr(l);
      *loclist = l;
      cdr(l) = thelist;
    }
    return l;
  }
  locl = &(cdr(l));
  l = cdr(l);
  goto iterate;
}
*/




#define get_type(x)							   \
((x)&1 ?								   \
 ((x)&2 ? REF_SLOT(x, 0) : *(e_subtype_table + ((x&SUBTAG_MASK) >> 2))) : \
 ((x)&2 ? e_loc_type : e_fixnum_type))

/* ((unsigned short *) (REF_TO_PTR(seg) + CODE_CODE_START_OFF)) */
#define CODE_SEG_FIRST_INSTR(seg) \
  ((unsigned short *)&REF_SLOT(seg,CODE_CODE_START_OFF))


void find_method_type_pair(op, obj_type, method_ptr, type_ptr)
     ref op, obj_type, *method_ptr, *type_ptr;
{
  register ref alist;
  register ref *locl = NULL;
  ref thelist;
  ref *loclist;

 tail_recurse:
  /* First look for it here: */
  /*alist = pseudo_assq(op, (REF_TO_PTR(obj_type) + TYPE_OP_METHOD_ALIST_OFF));*/

  /******************************************/
  loclist = &REF_SLOT(obj_type, TYPE_OP_METHOD_ALIST_OFF);
  thelist = *loclist;
  alist = thelist;
  locl = NULL;
 iterate:
  if (alist == e_nil)
    goto iterate_end2;
  if (car(car(alist)) == op) {
    if (locl) {
      *locl = cdr(alist);
      *loclist = alist;
      cdr(alist) = thelist;
    }
    goto iterate_end;
  }
  locl = &(cdr(alist));
  alist = cdr(alist);
  goto iterate;
 iterate_end:
  /******************************************/

  if (alist != e_nil)
    {
      *method_ptr = cdr(car(alist));
      *type_ptr = obj_type;
      return;
    }

 iterate_end2:
  /* Loop looking for it on supertypes: */
  alist = REF_SLOT(obj_type, TYPE_SUPER_LIST_OFF);
  if (alist == e_nil) {
    return;
  }
  while (1)
    {
      if (cdr(alist) == e_nil)
	{
	  obj_type = car(alist);
          goto tail_recurse;
	}

      find_method_type_pair(op, car(alist), method_ptr, type_ptr);
      if (*method_ptr != e_nil) {
	/* Found on a supertype, that's it: */
	return;
      }
      alist = cdr(alist);
    }
}




main(argc,argv)
     int argc;
     char **argv;
{
  new.size = DEFAULT_NEW_SPACE_SIZE;

  {
    char **aargv = argv + 1;
    int aargc = argc - 1;

    while (aargc > 1 && (*aargv)[0] == '-')
      {
	switch ((*aargv)[1])
	  {
#ifndef FAST
	  case 'i':
	    trace_insts = 1;
	    break;
	  case 'c':
	    trace_stcon = 1;
	    break;
	  case 'x':
	    trace_cxcon = 1;
	    break;
	  case 'm':
	    trace_meth = 1;
	    break;
	  case 'S':
	    trace_segs = 1;
	    break;
	  case 'M':
	    trace_mcache = 1;
	    break;
#endif
	  case 'T':
	    trace_traps = 1;
	    break;
	  case 'F':
	    trace_files = 1;
	    break;
	  case 'd':
	    dump_after = 1;
	    break;
	  case 'h':
	    aargc -= 1;
	    aargv += 1;
	    new.size = string_to_int(*aargv) / sizeof(ref);
	    break;
	  case '9':
	    dump_decimal = 1;
	    break;
	  case 'b':
	    dump_binary = 1;
	    break;
	  case 'G':
	    gc_before_dump = 1;
	    break;
	  case 'g':
	    trace_gc = 1;
	    break;
	  default:
	    (void)printf("Unknown option %s.\n", aargv[0]);
	    break;
	  }
	aargc -= 1;
	aargv += 1;
      }

    if (aargc != 1)
      {
#ifndef FAST
	(void)printf("Usage: %s [-icxmSMTFd9bGg] [-h bytes] oaklisp-image\n",
		     argv[0]);
#else
	(void)printf("Usage: %s [-TFd9bGg] [-h bytes] oaklisp-image\n",
		     argv[0]);
#endif
	exit(1);
      }

    read_world(aargv[0]);

    alloc_space(&new);
    free_point = new.start;

    init_stk(&val_stk);
    init_stk(&cxt_stk);
  }

  /* Get the registers set to the boot code. */

  e_current_method = e_boot_code;
  e_env = REF_TO_PTR(REF_SLOT(e_current_method, METHOD_ENV_OFF));
  e_code_segment = REF_SLOT(e_current_method, METHOD_CODE_OFF);
  e_pc = CODE_SEG_FIRST_INSTR(e_code_segment);

  /* Put a reasonable thing in e_bp so GC doesn't get pissed. */
  e_bp = e_env;

  /* Tell the boot function the truth: */
  e_nargs = 0;

  /* Okay, gentlemen, lets go: */

  {

    /* This is used for instructions to communicate with the trap code
       when a fault is encountered. */
    unsigned int trap_nargs;

    register unsigned short instr;
    register ref x;
    ref y;
    register ref *val_stk_ptr = val_stk.ptr;
    ref *cxt_stk_ptr = cxt_stk.ptr;

#define TRAP0(N) {trap_nargs=(N); goto arg0_tt;}
#define TRAP1(N) {trap_nargs=(N); goto arg1_tt;}

#define TRAP0_IF(C,N) {if (C) TRAP0(N);}
#define TRAP1_IF(C,N) {if (C) TRAP1(N);}

#define CHECKTAG0(X,TAG,N) TRAP0_IF(!TAG_IS(X,TAG),N)
#define CHECKTAG1(X,TAG,N) TRAP1_IF(!TAG_IS(X,TAG),N)

#define CHECKCHAR0(X,N) \
    TRAP0_IF(/*!TAG_IS(X,INT_TAG)&&*/!SUBTAG_IS(X,CHAR_SUBTAG),N)
#define CHECKCHAR1(X,N) \
    TRAP1_IF(/*!TAG_IS(X,INT_TAG)&&*/!SUBTAG_IS(X,CHAR_SUBTAG),N)

#define CHECKWP0(X,N) TRAP0_IF(!TAG_IS(X,INT_TAG) && !SUBTAG_IS(X,WP_SUBTAG),N)
#define CHECKWP1(X,N) TRAP1_IF(!TAG_IS(X,INT_TAG) && !SUBTAG_IS(X,WP_SUBTAG),N)

#define CHECKTAGS1(X0,T0,X1,T1,N) \
    TRAP1_IF( !TAG_IS(X0,T0) || !TAG_IS(X1,T1), N)


    /* This is the big instruction fetch/execute loop. */
    while (1)
      {

#ifndef FAST
	if (trace_stcon)
	  dump_val_stk();
	if (trace_cxcon)
	  dump_cxt_stk();
#endif

	instr = *e_pc;

#define arg_field (instr>>8)
/* #define signed_arg_field SIGN_8BIT_ARG(arg_field) */
#define signed_arg_field ((short)((short)instr >> 8))
#define op_field  ((instr & 0xFF) >> 2)

#ifndef FAST
	if (trace_insts)
	  {
	    (void)printf("PC %ld (%d, %d)\n",
			 (SPATIC_PTR((ref *)e_pc)
			  ? e_pc - (unsigned short *)spatic.start
			  : e_pc - (unsigned short *)new.start
			    + 2*spatic.size),
			 op_field, arg_field);
	    (void)fflush(stdout);
	  }
#endif
	e_pc += 1;

	switch (op_field)
	  {
	  case (CASE_FOUR*0):	/* ARGLESS-INSTRUCTION xxxx */
	    switch (arg_field)
	      {

	      case 0:		/* NOOP */
		break;

	      case 1:		/* PLUS */
		POPVAL(x);
		y = PEEKVAL();
		CHECKTAGS1(x,INT_TAG,y,INT_TAG,2);
		{
		  register ref z;
		  /*
		    Luckily, the tags are 0 so instead of this:
		    z = INT_TO_REF(REF_TO_INT(x)+REF_TO_INT(y));
		    we can write this:
		    */
		  z = x+y;

		  OVERFLOW(ovl_plus,
			   (long)x<0 && (long)y<0 && (long)z>0 ||
			   (long)x>0 && (long)y>0 && (long)z<0,
			   TRAP1(2));

		  PEEKVAL() = z;
		}
		break;

	      case 2:		/* NEGATE */
		x = PEEKVAL();
		CHECKTAG0(x,INT_TAG,1);

		/* On a twos complement machine, the most negative integer
		   can not be negated. */
		if (x == MIN_REF)
		  TRAP0(1);

		/* Tag is 0, so this:
		   PEEKVAL() = INT_TO_REF(-REF_TO_INT(x));
		   goes to this:
		   */
		PEEKVAL() = -x;
		break;

	      case 3:		/* EQ? */
		POPVAL(x);
		PEEKVAL() = x==PEEKVAL() ? e_t : e_nil;
		break;

	      case 4:		/* NOT */
		PEEKVAL() = PEEKVAL() == e_nil ? e_t : e_nil;
		break;

	      case 5:		/* TIMES */
		POPVAL(x);
		y = PEEKVAL();
		CHECKTAGS1(x,INT_TAG,y,INT_TAG,2);
		{
		  register ref z;
		  
		  /* Tag winage changes this:
		     z = INT_TO_REF( REF_TO_INT(x) * REF_TO_INT(y) );
		     to this:
		     */
		  z = REF_TO_INT(x) * (long)y;

		  /* NOTE: this needs the C overflow checking code. */
		  OVERFLOW(ovl_times, FALSE, TRAP1(2));

		  PEEKVAL() = z;
		}
		break;

	      case 6:		/* LOAD-IMM ; INLINE-REF */
		/* allign pc to next word boundary: */
		if ((unsigned long)e_pc & 2)
		  e_pc += 1;
		PUSHVAL(*(ref *)e_pc);
		e_pc += sizeof(ref) / sizeof(*e_pc);
		break;

	      case 7:		/* DIVIDE */
		POPVAL(x);
		y = PEEKVAL();
		CHECKTAGS1(x,INT_TAG,y,INT_TAG,2);
		/* Can't divide by 0, or the most negative number by -1. */
		if (y == INT_TO_REF(0) ||
		    y == INT_TO_REF(-1) && x == MIN_REF)
		  TRAP1(2);
		/* Tag winage changes this:
		   PEEKVAL() = INT_TO_REF( REF_TO_INT(x) / REF_TO_INT(y) );
		   to this:
		   */
		PEEKVAL() = INT_TO_REF((long)x/(long)y);
		break;

	      case 8:		/* =0? */
		x = PEEKVAL();
		CHECKTAG0(x,INT_TAG,1);
		PEEKVAL() = x == INT_TO_REF(0) ? e_t : e_nil;
		break;

	      case 9:		/* GET-TAG */
		PEEKVAL() = INT_TO_REF(PEEKVAL() & TAG_MASK);
		break;

	      case 10:		/* GET-DATA */
		/* With the moving gc, this should *NEVER* be used.

		   For ease of debugging with the multiple spaces, this
		   makes it seem like spatic and new spaces are contiguous,
		   is compatible with print_ref, and also with CRUNCH. */
		x = PEEKVAL();
		if (x&2)
		  {
		    ref *p = (x&1) ? REF_TO_PTR(x) : LOC_TO_PTR(x);

		    PEEKVAL() =
		      INT_TO_REF(
				 SPATIC_PTR(p) ?
				 p - spatic.start :
				 NEW_PTR(p) ?
				 (p - new.start) + spatic.size :
				 ( /* This is one weird reference: */
				  printf("GET-DATA of "),
				  printref(x),
				  printf("\n"),
				  -(long)p - 1 )
				 );

		    /*
		    if (SPATIC_PTR(p))
		      PEEKVAL() = INT_TO_REF(p - spatic.start);
		    else if (NEW_PTR(p))
		      PEEKVAL() = INT_TO_REF((p - new.start) + spatic.size);
		    else
		      {
			PEEKVAL() = INT_TO_REF(-(long)p - 1);
			printf("GET-DATA of ");
			printref(x);
			printf("\n");
		      }
		      */
		  }
		else
		  PEEKVAL() = x&~TAG_MASK | INT_TAG;

		break;

	      case 11:		/* CRUNCH */
		POPVAL(x); /* data */
		y = PEEKVAL(); /* tag */
		CHECKTAGS1(x,INT_TAG,y,INT_TAG,2);
		{
		  int tag = REF_TO_INT(y)&3;
		  ref z;

		  if ((tag&2) != 0)
		    {
		      long i = REF_TO_INT(x);
		    
		      /* For now, preclude creation of very odd references. */
		      if (i < 0)
			{
			  trap_nargs = 2;
			  goto arg1_tt;
			}
		      else if (i < spatic.size)
			z = PTR_TO_LOC(spatic.start + i);
		      else if (i < (spatic.size + new.size))
			z = PTR_TO_LOC(new.start + (i - spatic.size));
		      else
			{
			  trap_nargs = 2;
			  goto arg1_tt;
			}
		    }
		  else
		    z = x;

		  PEEKVAL() = z | tag;
		}
		break;

	      case 12:		/* GETC */
		/***************************** OBSOLETE? *********************/
		/* Used in emergency cold load standard-input stream. */
		PUSHVAL_IMM(CHAR_TO_REF(getc(stdin)));
		break;

	      case 13:		/* PUTC */
		/* Used in emergency cold load standard-output stream, and
		   for the warm boot message. */
		x = PEEKVAL();
		CHECKCHAR0(x,1);
		(void)putc(REF_TO_CHAR(x), stdout);
		(void)fflush(stdout);
		if (trace_insts || trace_stcon || trace_cxcon)
		  (void)printf("\n");
		break;

	      case 14:		/* CONTENTS */
		x = PEEKVAL();
		CHECKTAG0(x,LOC_TAG,1);
		PEEKVAL() = *LOC_TO_PTR(x);
		break;

	      case 15:		/* SET-CONTENTS */
		POPVAL(x);
		CHECKTAG1(x,LOC_TAG,2);
		*LOC_TO_PTR(x) = PEEKVAL();
		break;

	      case 16:		/* LOAD-TYPE */
		PEEKVAL() = get_type(PEEKVAL());
		break;

	      case 17:		/* CONS */
		{
		  ref *p;

		  ALLOCATE_SS(p, 3, "space crunch in CONS instruction");

		  *p = e_cons_type;
		  POPVAL(x);
		  *(p+CONS_PAIR_CAR_OFF) = x;
		  *(p+CONS_PAIR_CDR_OFF) = PEEKVAL();
		  PEEKVAL() = PTR_TO_REF(p);
		}
		break;

	      case 18:		/* <0? */
		x = PEEKVAL();
		CHECKTAG0(x,INT_TAG,1);
		PEEKVAL() = (long)x < 0 ? e_t : e_nil;
		break;

	      case 19:		/* MOD */
		POPVAL(x);
		y = PEEKVAL();
		CHECKTAGS1(x,INT_TAG,y,INT_TAG,2);
		if (y == INT_TO_REF(0))
		  TRAP1(2);
		{
		  long b = REF_TO_INT(y);
		  long c = REF_TO_INT(x) % b;

		  PEEKVAL() = INT_TO_REF(c<0 ? c+b : c);
		}
		break;

	      case 20:		/* ASH */
		POPVAL(x);
		y = PEEKVAL();
		CHECKTAGS1(x,INT_TAG,y,INT_TAG,2);
		/* NOTE: this has no overflow check and needs one. */
		{
		  long b = REF_TO_INT(y);

		  PEEKVAL() = ( b<0 ? x>>-b : x<<b ) & ~3;
		}
		break;

	      case 21:		/* ROT */
		POPVAL(x);
		y = PEEKVAL();
		CHECKTAGS1(x,INT_TAG,y,INT_TAG,2);
		{
		  unsigned long a = (unsigned)x;
		  long b = REF_TO_INT(y);

		  PEEKVAL()
		    = ( b<0 ? (a>>-b | a<<30+b) : (a<<b | a>>30-b) ) & ~3;
		}
		break;

	      case 22:		/* STORE-BP-I */
		POPVAL(x);
		CHECKTAG1(x,INT_TAG,2);
		*(e_bp + REF_TO_INT(x)) = PEEKVAL();
		break;

	      case 23:		/* LOAD-BP-I */
		x = PEEKVAL();
		CHECKTAG0(x,INT_TAG,1);
		PEEKVAL() = *(e_bp + REF_TO_INT(x));
		break;

	      case 24:		/* RETURN */
		POP_CONTEXT();
		break;

	      case 25:		/* ALLOCATE */
		POPVAL(x);
		y = PEEKVAL();
		CHECKTAG1(y,INT_TAG,2);
		{
		  ref *p;
		  
		  ALLOCATE1(p, REF_TO_INT(y),
			    "space crunch in ALLOCATE instruction", x);

		  *p = x;

		  PEEKVAL() = PTR_TO_REF(p++);

		  while (p < free_point)
		    *p++ = REF_TO_INT(0);
		}
		break;

	      case 26:		/* ASSQ */
		{
		  register ref z;
		  
		  POPVAL(z);
		  x = PEEKVAL();
		  /* y = assq(z,x); */
		  while (x != e_nil && car(car(x)) != z)
		    x = cdr(x);
		}
		PEEKVAL() = ((x == e_nil) ? e_nil : car(x));
		break;

	      case 27:		/* LOAD-LENGTH */
		PEEKVAL() = INT_TO_REF(get_length(PEEKVAL()));
		break;

	      case 28:		/* PEEK */
		PEEKVAL() = INT_TO_REF( *(short *)PEEKVAL() );
		break;

	      case 29:		/* POKE */
		POPVAL(x);
		*(short *)x = REF_TO_INT(PEEKVAL());
		break;

	      case 30:		/* MAKE-CELL */
		{
		  ref *p;
		  
		  ALLOCATE_SS(p,1,"space crunch in MAKE-CELL instruction");

		  *p = PEEKVAL();
		  PEEKVAL() = PTR_TO_LOC(p);
		}
		break;

	      case 31:		/* SUBTRACT */
		POPVAL(x);
		y = PEEKVAL();
		CHECKTAGS1(x,INT_TAG,y,INT_TAG,2);
		{
		  register ref z;
		  /* Tag winage changes this:
		     z = INT_TO_REF(REF_TO_INT(x)-REF_TO_INT(y));
		     to this:
		     */
		  z = x-y;
		  OVERFLOW(ovl_subtr,
			   (long)x<0 && (long)y>0 && (long)z>0 ||
			   (long)x>0 && (long)y<0 && (long)z<0,
			   TRAP1(2));
		  PEEKVAL() = z;
		}
		break;

	      case 32:		/* = */
		POPVAL(x);
		y = PEEKVAL();
		CHECKTAGS1(x,INT_TAG,y,INT_TAG,2);
		PEEKVAL() = x == y ? e_t : e_nil;
		break;

	      case 33:		/* < */
		POPVAL(x);
		y = PEEKVAL();
		CHECKTAGS1(x,INT_TAG,y,INT_TAG,2);
		/* Tag trickery turns REF_TO_INT(x) < REF_TO_INT(y) into: */ 
		PEEKVAL() = (long)x < (long)y ? e_t : e_nil;
		break;

	      case 34:		/* LOG-NOT */
		x = PEEKVAL();
		CHECKTAG0(x,INT_TAG,1);
		/* Tag winage changes this:
		   PEEKVAL() = INT_TO_REF(~REF_TO_INT(x));
		   to this:
		   */
		PEEKVAL() = (ref)(~3L & ~(unsigned long)x);
		break;

	      case 35:		/* LONG-BRANCH distance (signed) */
		e_pc += ASHR2(SIGN_16BIT_ARG(*e_pc))+1;
		break;

	      case 36:		/* LONG-BRANCH-NIL distance (signed) */
		POPVAL(x);
		if (x == e_nil)
		  e_pc += ASHR2(SIGN_16BIT_ARG(*e_pc))+1;
		else
		  e_pc += 1;
		break;

	      case 37:		/* LONG-BRANCH-T distance (signed) */
		POPVAL(x);
		if (x != e_nil)
		  e_pc += ASHR2(SIGN_16BIT_ARG(*e_pc))+1;
		else
		  e_pc += 1;
		break;

	      case 38:		/* LOCATE-BP-I */
		x = PEEKVAL();
		CHECKTAG0(x,INT_TAG,1);
		PEEKVAL() = PTR_TO_LOC(e_bp + REF_TO_INT(x));
		break;

	      case 39:		/* LOAD-IMM-CON ; INLINE-REF */
		/* This is like a LOAD-IMM followed by a CONTENTS. */
		/* allign pc to next word boundary: */
		if ((unsigned long)e_pc & 2)
		  e_pc += 1;

		x = *(ref *)e_pc;
		e_pc += 2;

		CHECKTAG1(x,LOC_TAG,1);
		PUSHVAL(*LOC_TO_PTR(x));
		break;

		/* Cons instructions. */

#define CONSINSTR(a,ins)				\
		{					\
		  x = PEEKVAL();			\
		  CHECKTAG0(x,PTR_TAG, a);		\
		  if (REF_SLOT(x,0) != e_cons_type)	\
		    {					\
		      if (trace_traps)			\
			(void)printf("Type trap in ins.\n");	\
		      trap_nargs = a;			\
		      goto arg0_tt;			\
		    }					\
		}

	      case 40:		/* CAR */
		CONSINSTR(1,CAR);
		PEEKVAL() = car(x);
		break;

	      case 41:		/* CDR */
		CONSINSTR(1,CAR);
		PEEKVAL() = cdr(x);
		break;

	      case 42:		/* SET-CAR */
		CONSINSTR(2,SET-CAR);
		POPVALS(1);
		car(x) = PEEKVAL();
		break;

	      case 43:		/* SET-CDR */
		CONSINSTR(2,SET-CDR);
		POPVALS(1);
		cdr(x) = PEEKVAL();
		break;

	      case 44:		/* LOCATE-CAR */
		CONSINSTR(1,LOCATE-CAR);
		PEEKVAL() = PTR_TO_LOC(&car(x));
		break;

	      case 45:		/* LOCATE-CDR */
		CONSINSTR(1,LOCATE-CDR);
		PEEKVAL() = PTR_TO_LOC(&cdr(x));
		break;

		/* Done with cons instructions. */

	      case 46:		/* PUSH-CXT-LONG rel */
		PUSH_CONTEXT(ASHR2(SIGN_16BIT_ARG(*e_pc)) + 1);
		e_pc += 1;
		break;

	      case 47:		/* OLD-FILLTAG : reuse this opcode */
		printf("Obsolete call to OLD-FILLTAG.\n");
                instr &= 0xFF;
		goto new_filltag;
		break;

	      case 48:		/* THROW */
		POPVAL(x);
		CHECKTAG1(x,PTR_TAG,2);
		y = PEEKVAL();
		bash_val_height(REF_TO_INT(REF_SLOT(x,ESCAPE_OBJECT_VAL_OFF)));
		bash_cxt_height(REF_TO_INT(REF_SLOT(x,ESCAPE_OBJECT_CXT_OFF)));
		PUSHVAL(y);
		POP_CONTEXT();
		break;

	      case 49:		/* GET-WP */
		PEEKVAL() = ref_to_wp(PEEKVAL());
		break;

	      case 50:		/* WP-CONTENTS */
		x = PEEKVAL();
		CHECKWP0(x,1);
		PEEKVAL() = wp_to_ref(x);
		break;

	      case 51:		/* GC */
		gc(0, "explicit call");
		PUSHVAL(e_nil);
		break;

	      case 52:		/* BIG-ENDIAN? */
#ifdef BIG_ENDIAN
		PUSHVAL(e_t);
#else
		PUSHVAL(e_nil);
#endif
		break;

	      case 53:		/* VLEN-ALLOCATE */
		POPVAL(x);
		y = PEEKVAL();
		CHECKTAG1(y,INT_TAG,2);
		{
		  ref *p;
		  
		  ALLOCATE1(p, REF_TO_INT(y),
			    "space crunch in VARLEN-ALLOCATE instruction", x);

		  PEEKVAL() = PTR_TO_REF(p);

		  *p++ = x;
		  *p++ = y;

		  while (p < free_point)
		    *p++ = REF_TO_INT(0);
		}
		break;

	      case 54:		/* INC-LOC */
		/* Increment a locative by an amount.  This is an instruction
		   rather than (%crunch (+ (%pointer loc) index) %locative-tag)
		   to avoid a window of gc vulnerability.  All such windows
		   must be fully closed before engines come up. */
		POPVAL(x);
		y = PEEKVAL();
		CHECKTAGS1(x,LOC_TAG,y,INT_TAG,2);
		PEEKVAL() = PTR_TO_LOC(LOC_TO_PTR(x)+REF_TO_INT(y));
		break;

	      case 55:		/* FILL-CONTINUATION */
		/* This instruction fills a continuation object with
		   the appropriate values. */
		CHECKVAL_POP(1);
		FLUSHVAL(2);
		FLUSHCXT(0);
#ifndef FAST
		/* debugging check: */
		if (val_stk_ptr != &val_stk.data[1])
		  printf("Value stack flushing error.\n");
		if (cxt_stk_ptr != &cxt_stk.data[-1])
		  printf("Context stack flushing error.\n");
#endif
		x = PEEKVAL();
		/* CHECKTAG0(x,PTR_TAG,1); */
		REF_SLOT(x,CONTINUATION_VAL_SEGS) = val_stk.segment;
		REF_SLOT(x,CONTINUATION_VAL_OFF)
		  = INT_TO_REF(val_stk.pushed_count);
		REF_SLOT(x,CONTINUATION_CXT_SEGS) = cxt_stk.segment;
		REF_SLOT(x,CONTINUATION_CXT_OFF)
		  = INT_TO_REF(cxt_stk.pushed_count);
		/* Maybe it's a good idea to reload the buffer, but I'm
		   not bothering and things seem to work. */
		/* CHECKCXT_POP(0); */
		break;

	      case 56:		/* CONTINUE */
		/* Continue a continuation. */
		/* Grab the continuation. */
		POPVAL(x);
		/* CHECKTAG1(x,PTR_TAG,1); */
		y = PEEKVAL();
		/* Pull the crap out of it. */

		val_stk.segment = REF_SLOT(x,CONTINUATION_VAL_SEGS);
		val_stk.pushed_count
		  = REF_TO_INT(REF_SLOT(x,CONTINUATION_VAL_OFF));
		val_stk_ptr = &val_stk.data[-1];
		PUSHVAL_NOCHECK(y);

		cxt_stk.segment = REF_SLOT(x,CONTINUATION_CXT_SEGS);
		cxt_stk.pushed_count
		  = REF_TO_INT(REF_SLOT(x,CONTINUATION_CXT_OFF));
		cxt_stk_ptr = &cxt_stk.data[-1];
		POP_CONTEXT();
		break;

	      case 57:		/* REVERSE-CONS */
		/* This is just like CONS except that it takes its args
		   in the other order.  Makes open coded LIST better. */
		{
		  ref *p;

		  ALLOCATE_SS(p, 3, "space crunch in CONS instruction");

		  *p = e_cons_type;
		  POPVAL(x);
		  *(p+CONS_PAIR_CDR_OFF) = x;
		  *(p+CONS_PAIR_CAR_OFF) = PEEKVAL();
		  PEEKVAL() = PTR_TO_REF(p);
		}
		break;
#ifndef FAST
	      default:
		(void)printf("\nIllegal ARGLESS instruction %d.\n", arg_field);
		maybe_dump_world(1);
		exit(1);
#endif
	      }
	    break;

	  case (CASE_FOUR*1):	/* HALT n */
	    {
	      int halt_code = arg_field;
	      
	      if (halt_code != 0)
		(void)printf("\nHalt code %d.\n", halt_code);
	      (void)printf("\nOaklisp stopped itself...\n");
	      maybe_dump_world(halt_code);
	      exit(halt_code);
	    }

	  case (CASE_FOUR*2):	/* LOG-OP log-spec */
	    POPVAL(x);
	    y = PEEKVAL();
	    CHECKTAGS1(x,INT_TAG,y,INT_TAG,2);
	    /* Tag winage changes this:
	    {
	      long x1 = REF_TO_INT(x), y1 = REF_TO_INT(y);
	      PEEKVAL() = INT_TO_REF(  (arg_field&1 ? x1&y1 : 0)
					 | (arg_field&2 ? ~x1&y1 : 0)
					 | (arg_field&4 ? x1&~y1 : 0)
					 | (arg_field&8 ? ~x1&~y1 : 0) );
	    }
	    to this:
	    */

	    PEEKVAL()
	      = (~0x3) &
		(  (instr&(1<<8) ? x&y : 0)
		 | (instr&(2<<8) ? ~x&y : 0)
		 | (instr&(4<<8) ? x&~y : 0)
		 | (instr&(8<<8) ? ~x&~y : 0) );
	    break;

	  case (CASE_FOUR*3):	/* BLT-STACK stuff,trash */
#ifdef COMMENTOUT
	    /**************************************************************
	     *
	     * This is a hairy instruction, doubly so because it is quite
	     * common and hence must be efficient; a procedure call in the
	     * event that a segment pop is not necessary is not acceptable.
	     *
	     * We break the problem up into cases depending on where the
	     * nearest segment break is.
	     *
	     *
             *   TOS \ --------
	     *         |s- - -| \
	     *         |t  .  |	|
	     *         |u  .  |  > Case D
	     *         |f  .  |	|
	     *         |f- - -| /
	     *         -------- \
	     *         |  gap |  > Case C
	     *newTOS \ -t------ /
             *         |r- - -| \
	     *         |a  .  |	|
	     *         |s  .  |	 > Case B
	     *         |h  .  |	|
	     *         | - - -| /
	     *         -------- \
	     *         |      | |
	     *         |      |	 > Case A
	     *         |      | |
	     *         |stack | |
             *         V      V V
	     *
	     * In case A, everything happens in the current segment.
	     *
	     * In case B, we don't need to pop the current segment, but
	     * we do need to transfer part of the stuff in the previous
	     * segment and part of it in the current segment.
	     *
	     * In case C, we're going to have to pop the current segment,
	     * but all the stuff comes from one segment and goes to another.
	     *
	     * In case D, some of the stuff comes from the current segment
	     * and some from the previous segment, and it all goes to the
	     * previous segment, and we must pop the current segment.
	     *
	     * This all assumes that trash >= stuff.  In the event that
	     * trash < stuff, we relabel case C so that it means overlap
	     * rather than gap, and in that event both the source and the
	     * destination contain segment breaks, and no segment pop is
	     * needed.  Cases A, B and D remain the same.
	     *
	     *		Split	Split	Pop		- = NO
	     *	Case	Source	Dest	Segment		X = YES
	     *						left = trash >= stuff
	     *	A	- -	- -	- -		right = trash<stuff
	     *	B	- -	X X	- -
	     *	C	- X	- X	X -
	     *	D	X X	- -	X X
	     ****************************************************************/

	    /* Attempt to reimplement the code for slight efficiency gain
	       in path to case A and for additional code sharing. */
	    {
	      int stuff = arg_field&0xF ,trash = (arg_field>>4) + 1;
	      ref *src = val_stk.ptr - stuff
		, *dest = src - trash;

	      if (dest >= val_stk.bottom)
		{
		  /* case A */
		  do {
		    *++dest = *++src;
		  } while (src != val_stk.ptr);

		  val_stk.ptr = dest;
		}
	      else
		{
		  /* case B, C1, C2 or D */
		  ref *prev_seg_top
		    = & ((segment *)
			 REF_TO_PTR(val_stk.segment->previous_segment))
		      ->data[DATA_PER_SEGMENT - HYSTERESIS - 1];
		  int dist = val_stk.ptr - val_stk.bottom;

		  dest = prev_seg_top - (stuff+trash-dist);

		  if (src >= val_stk.bottom)
		    {
		      /* case B or C1 */
		      if (trash < stuff || dist > trash)
			{
			  /* case B */
#ifndef FAST
			  if (trace_segs) printf("b");
#endif
			  do {
			    *++dest = *++src;
			  } while (dest != prev_seg_top);

			  dest = val_stk.bottom;

			  do {
			    *++dest = *++src;
			  } while (src != val_stk.ptr);

			  val_stk.ptr = dest;
			}
		      else
			{
			  /* case C1 */
#ifndef FAST
			  if (trace_segs) printf("c1");
#endif
			  do {
			    *++dest = *++src;
			  } while (src != val_stk.ptr);

			  val_stk.ptr = val_stk.bottom;
			  pop_segment(&val_stk);
			  val_stk.ptr = dest;
			}
		    }
		  else
		    {
		      /* case D or C2 */
		      src = dest + trash;

		      if (trash >= stuff || dist <= trash)
			{
			  /* case D */
#ifndef FAST
			  if (trace_segs) printf("d");
#endif
			  do {
			    *++dest = *++src;
			  } while (src != prev_seg_top);

			  src = val_stk.bottom;

			  do {
			    *++dest = *++src;
			  } while (src != val_stk.ptr);

			  val_stk.ptr = val_stk.bottom;
			  pop_segment(&val_stk);
			  val_stk.ptr = dest;
			}
		      else
			{
			  /* case C2 */
#ifndef FAST
			  if (trace_segs) {
			    printf("c2(%d,%d,%d)", stuff, trash, dist);
			    (void)fflush(stdout);
			  }
#endif
			  do {
			    *++dest = *++src;
			  } while (src != prev_seg_top);

			  src = val_stk.bottom;

			  do {
			    *++dest = *++src;
			  } while (dest != prev_seg_top);

			  dest = val_stk.bottom;

			  do {
			    *++dest = *++src;
			  } while (src != val_stk.ptr);

			  val_stk.ptr = dest;
			}
		    }
		}
	    }
#endif /* COMMENTOUT */
	    /* New fast clear working code: */
	    {
	      register int stuff = arg_field&0xF
		, trash_m1 = (instr>>(8+4));

	      CHECKVAL_POP(stuff+trash_m1);
	      
	      {
		register ref *src = val_stk_ptr - stuff
		  , *dest = src - (trash_m1+1);

		while (src < val_stk_ptr)
		  *++dest = *++src;

		val_stk_ptr = dest;
	      }
	    }
	    break;

	  case (CASE_FOUR*4):	/* BRANCH-NIL distance (signed) */
	    POPVAL(x);
	    if (x == e_nil)
	      e_pc += signed_arg_field;
	    break;

	  case (CASE_FOUR*5):	/* BRANCH-T distance (signed) */
	    POPVAL(x);
	    if (x != e_nil)
	      e_pc += signed_arg_field;
	    break;

	  case (CASE_FOUR*6):	/* BRANCH distance (signed) */
	    e_pc += signed_arg_field;
	    break;

	  case (CASE_FOUR*7):	/* POP n */
	    POPVALS(arg_field);
	    break;

	  case (CASE_FOUR*8):	/* SWAP n */
	    {
	      ref *other;

	      MAKE_BACK_VAL_PTR(other,arg_field);

	      x = PEEKVAL();
	      PEEKVAL() = *other;
	      *other = x;
	    }
	    break;

	  case (CASE_FOUR*9):	/* BLAST n */
	    CHECKVAL_POP((int)arg_field);
	    {
	      ref *other = val_stk_ptr - arg_field;
	      *other = POPVAL_NOCHECK();
	    }
	    break;

	  case (CASE_FOUR*10):	/* LOAD-IMM-FIX signed-arg */
	    /* Tag trickery and opcode knowledge changes this
	       PUSHVAL_IMM(INT_TO_REF(signed_arg_field));
	       to this: */
	    PUSHVAL_IMM((ref) (((short)instr)>>6));
	    break;

	  case (CASE_FOUR*11):	/* STORE-STK n */
	    {
	      ref *other;

	      MAKE_BACK_VAL_PTR(other,arg_field);
	      *other = PEEKVAL();
	    }
	    break;

	  case (CASE_FOUR*12):	/* LOAD-BP n */
	    PUSHVAL(*(e_bp + arg_field));
	    break;

	  case (CASE_FOUR*13):	/* STORE-BP n */
	    *(e_bp + arg_field) = PEEKVAL();
	    break;

	  case (CASE_FOUR*14):	/* LOAD-ENV n */
	    PUSHVAL(*(e_env + arg_field));
	    break;

	  case (CASE_FOUR*15):	/* STORE-ENV n */
	    *(e_env + arg_field) = PEEKVAL();
	    break;

	  case (CASE_FOUR*16):	/* LOAD-STK n */
	    /* All attempts to start this with if (arg_field == 0) for speed
	       have failed, so benchmark carefully before trying it. */
	    {
	      ref *other;

	      MAKE_BACK_VAL_PTR(other,arg_field);
	      PUSHVAL(*other);
	    }
	    break;

	  case (CASE_FOUR*17):	/* MAKE-BP-LOC n */
	    PUSHVAL(PTR_TO_LOC(e_bp + arg_field));
	    break;

	  case (CASE_FOUR*18):	/* MAKE-ENV-LOC n */
	    PUSHVAL(PTR_TO_LOC(e_env + arg_field));
	    break;

	  case (CASE_FOUR*19):	/* STORE-REG reg */
	    x = PEEKVAL();
	    switch (arg_field)
	      {
	      case 0:
		e_t = x;
		break;
	      case 1:
		e_nil = x;
		break;
	      case 2:
		e_fixnum_type = x;
		break;
	      case 3:
		e_loc_type = x;
		break;
	      case 4:
		e_cons_type = x;
		break;
	      case 5:
		CHECKTAG1(x,PTR_TAG,1);
		e_subtype_table = REF_TO_PTR(x) + 2;
		break;
	      case 6:
		CHECKTAG1(x,LOC_TAG,1);
		e_bp = LOC_TO_PTR(x);
		break;
	      case 7:
		CHECKTAG1(x,PTR_TAG,1);
		e_env = REF_TO_PTR(x);
		break;
	      case 8:
		CHECKTAG1(x,INT_TAG,1);
		e_nargs = REF_TO_INT(x);
		break;
	      case 9:
		e_env_type = x;
		break;
	      case 10:
		CHECKTAG1(x,PTR_TAG,1);
		e_argless_tag_trap_table = REF_TO_PTR(x) + 2;
		break;
	      case 11:
		CHECKTAG1(x,PTR_TAG,1);
		e_arged_tag_trap_table = REF_TO_PTR(x) + 2;
		break;
	      case 12:
		e_object_type = x;
		break;
	      case 13:
		e_boot_code = x;
		break;
	      case 14:
		CHECKTAG1(x,LOC_TAG,1);
		free_point = LOC_TO_PTR(x);
		break;
	      case 15:
		CHECKTAG1(x,LOC_TAG,1);
		new.end = LOC_TO_PTR(x);
		break;
	      case 16:
		e_segment_type = x;
		BASH_SEGMENT_TYPE(x);
		break;
	      default:
		(void)printf("STORE-REG %d, unknown register.\n", arg_field);
		break;
	      }
	    break;

	  case (CASE_FOUR*20):	/* LOAD-REG reg */
	    {
	      ref z;
	      
	      switch (arg_field)
		{
		case 0:
		  z = e_t;
		  break;
		case 1:
		  z = e_nil;
		  break;
		case 2:
		  z = e_fixnum_type;
		  break;
		case 3:
		  z = e_loc_type;
		  break;
		case 4:
		  z = e_cons_type;
		  break;
		case 5:
		  z = PTR_TO_REF(e_subtype_table - 2);
		  break;
		case 6:
		  z = PTR_TO_LOC(e_bp);
		  break;
		case 7:
		  z = PTR_TO_REF(e_env);
		  break;
		case 8:
		  z = INT_TO_REF((long)e_nargs);
		  break;
		case 9:
		  z = e_env_type;
		  break;
		case 10:
		  z = PTR_TO_REF(e_argless_tag_trap_table - 2);
		  break;
		case 11:
		  z = PTR_TO_REF(e_arged_tag_trap_table - 2);
		  break;
		case 12:
		  z = e_object_type;
		  break;
		case 13:
		  z = e_boot_code;
		  break;
		case 14:
		  z = PTR_TO_LOC(free_point);
		  break;
		case 15:
		  z = PTR_TO_LOC(new.end);
		  break;
		case 16:
		  z = e_segment_type;
		  break;
		default:
		  (void)printf("LOAD-REG %d, unknown register.\n", arg_field);
		  z = e_nil;
		  break;
		}
	      PUSHVAL(z);
	    }
	    break;

	  case (CASE_FOUR*21):	/* FUNCALL-CXT, FUNCALL-CXT-BR distance (signed) */
	    /* NOTE: (FUNCALL-CXT) == (FUNCALL-CXT-BR 0) */
	    PUSH_CONTEXT(signed_arg_field);

	    /* Fall through to tail recursive case: */

	  case (CASE_FOUR*22):	/* FUNCALL-TAIL */
	    /* This label allows us to branch here from the tag trap code. */
	  funcall_tail:
	    POPVAL(x);
	    y = PEEKVAL();
	    CHECKTAG1(x,PTR_TAG,e_nargs+1);
	    {
	      ref *oper = REF_TO_PTR(x);

	      e_current_method = SLOT(oper,OPERATION_LAMBDA_OFF);

	      if (e_current_method == e_nil)
		{
		  /* SEARCH */
		  ref y_type = (e_nargs == 0) ? e_object_type : get_type(y);

#ifdef METH_CACHE
		  /* Check for cache hit: */
		  if (y_type == SLOT(oper,OPERATION_CACHE_TYPE_OFF))
		    {
#ifndef FAST
		      if (trace_mcache)
			{
			  printf("H");
			  (void)fflush(stdout);
			}
#endif /* FAST */
		      e_current_method = *(oper + OPERATION_CACHE_METH_OFF);
		      e_bp =
			REF_TO_PTR(y) +
			  REF_TO_INT(*(oper + OPERATION_CACHE_TYPE_OFF_OFF));
		    }
		  else
#endif /* METH_CACHE */
		    {
		      /* Search the type heirarchy. */
		      ref meth_type, offset;
#ifdef METH_CACHE
#ifndef FAST
		      if (trace_mcache)
			{
			  printf("M");
			  (void)fflush(stdout);
			}
#endif /* FAST */
#endif /* METH_CACHE */
		      find_method_type_pair(x,y_type,&e_current_method,&meth_type);

		      if (e_current_method == e_nil)
			{
			  if (trace_traps)
			    (void)printf("No handler for operation!\n");
			  trap_nargs = e_nargs+1;
			  goto arg1_tt;
			}

		      /* This could be dispensed with this if meth_type has no
			 ivars and isn't variable-length-mixin. */
			
		      x = REF_SLOT(y_type, TYPE_TYPE_BP_ALIST_OFF);
		      while (x != e_nil && car(car(x)) != meth_type)
			x = cdr(x);

		      offset = (x != e_nil) ? cdr(car(x)) : INT_TO_REF(0);
		      e_bp = REF_TO_PTR(y) + REF_TO_INT(offset);
#ifdef METH_CACHE
		      /* Cache the results of this search. */
		      SLOT(oper,OPERATION_CACHE_TYPE_OFF) = y_type;
		      SLOT(oper,OPERATION_CACHE_METH_OFF) = e_current_method;
		      SLOT(oper,OPERATION_CACHE_TYPE_OFF_OFF) = offset;
#endif
		    }
		}
	      else if (e_current_method == INT_TO_REF(0))
		{
		  /* TAG TRAP */
		  if (trace_traps)
		    (void)printf("No handler ever defined on anything for operation!\n");
		  trap_nargs = e_nargs+1;
		  goto arg1_tt;
		}
	      else
		{
		  /* LAMBDA */
		}
	    }

	    {
	      ref *p = REF_TO_PTR(e_current_method);
	      
	      e_env = REF_TO_PTR(SLOT(p, METHOD_ENV_OFF));
	      e_code_segment = SLOT(p, METHOD_CODE_OFF);
	    }
	    e_pc = CODE_SEG_FIRST_INSTR(e_code_segment);

	    break;

	  case (CASE_FOUR*23):	/* STORE-NARGS n */
	    e_nargs = arg_field;
	    break;

	  case (CASE_FOUR*24):	/* CHECK-NARGS n */
	    if (e_nargs != arg_field)
	      {
		if (trace_traps)
		  (void)printf("\n%d args passed; %d expected.\n", e_nargs, arg_field);
		trap_nargs = e_nargs;
		goto arg0_tt;
	      }
	    break;

	  case (CASE_FOUR*25):	/* CHECK-NARGS-GTE n */
	    if (e_nargs < arg_field)
	      {
		if (trace_traps)
		  (void)printf("\n%d args passed; %d or more expected.\n", e_nargs, arg_field);
		trap_nargs = e_nargs;
		goto arg0_tt;
	      }
	    break;

	  case (CASE_FOUR*26):	/* STORE-SLOT n */
	    POPVAL(x);
	    CHECKTAG1(x,PTR_TAG,2);
	    REF_SLOT(x, arg_field) = PEEKVAL();
	    break;

          case (CASE_FOUR*27):	/* LOAD-SLOT n */
	    CHECKTAG0(PEEKVAL(),PTR_TAG,1);
	    PEEKVAL() = REF_SLOT(PEEKVAL(), arg_field);
	    break;

	  case (CASE_FOUR*28):	/* MAKE-CLOSED-ENVIRONMENT n */
	    /* NOTE: if we knew that arg_field would never be 0, the two
	       if statements below could be eliminated, since the test would
	       always be true. */
	    {
	      ref *p;
	      ref z;

              register int zarg_field = arg_field;

	      ALLOCATE_SS(p,zarg_field+2,
			  "space crunch in MAKE-CLOSED-ENVIRONMENT");

	      z = PTR_TO_REF(p);

	      *p++ = e_env_type;
	      *p++ = INT_TO_REF(zarg_field+2);

	      /* This loop is replaced by the loop and if below for speed.
	      while (zarg_field--)
		POPVAL(*p++);
		*/

	      while (zarg_field > 1)
		{
		  POPVAL(*p++);
		  zarg_field -= 1;
		}
	      
	      if (zarg_field == 1)
		{
		  *p = PEEKVAL();
		  PEEKVAL() = z;
		}
	      else
		PUSHVAL(z);
	    }
	    break;

	  case (CASE_FOUR*29):	/* PUSH-CXT rel */
	    PUSH_CONTEXT(signed_arg_field);
	    break;

	  case (CASE_FOUR*30):	/* LOCATE-SLOT n */
	    PEEKVAL()
	      = PTR_TO_LOC( REF_TO_PTR( PEEKVAL() ) + arg_field );
	    break;

	  case (CASE_FOUR*31):	/* STREAM-PRIMITIVE n */
	    switch (arg_field)
	      {
	      case 0:		/* n=0: get standard input stream. */
		PUSHVAL((ref)stdin);
		break;
	      case 1:		/* n=1: get standard output stream. */
		PUSHVAL((ref)stdout);
		break;
	      case 2:		/* n=2: get standard error output stream. */
		PUSHVAL((ref)stderr);
		break;
	      case 3:		/* n=3: fopen, mode READ */
	      case 4:		/* n=4: fopen, mode WRITE */
	      case 5:		/* n=5: fopen, mode APPEND */
		POPVAL(x);
		/* How about a CHECKTAG(x,LOC_TAG,) here, eh? */
		y = PEEKVAL();
		{
		  long len = REF_TO_INT(y);
		  char *stuff = my_malloc(len+1);
		  unsigned long *p = (unsigned long *)LOC_TO_PTR(x);
		  int i = 0, j = 0;

		  while (i+2<len)
		    {
		      unsigned long pp = *p++;
		      stuff[j++] = (pp >> 2) & 0xFF;
		      stuff[j++] = (pp >> 10) & 0xFF;
		      stuff[j++] = (pp >> 18) & 0xFF;
		      i += 3;
		    }
		  if (i+1<len)
		    {
		      unsigned long pp = *p;
		      stuff[j++] = (pp >> 2) & 0xFF;
		      stuff[j++] = (pp >> 10) & 0xFF;
		      i += 2;
		    }
		  else if (i<len)
		    {
		      stuff[j++] = (*p >> 2) & 0xFF;
		      i += 1;
		    }
		  stuff[j] = 0;

		  if (trace_files)
		    (void)printf("About to open '%s'.\n", stuff);
		  PEEKVAL()
		    = (ref)fopen(stuff,
				 arg_field == 3 ? READ_MODE :
				 arg_field == 4 ? WRITE_MODE : APPEND_MODE) ;
		  free(stuff);
		}
		break;
	      case 6:		/* n=6: fclose */
		PEEKVAL()
		  = fclose((FILE *)PEEKVAL()) == EOF ? e_nil : e_t;
		break;
	      case 7:		/* n=7: fflush */
		PEEKVAL()
		  = fflush((FILE *)PEEKVAL()) == EOF ? e_nil : e_t;
		break;
	      case 8:		/* n=8: putc */
		POPVAL(x);
		y = PEEKVAL();
		CHECKCHAR1(y,2);
		PEEKVAL()
		  = putc(REF_TO_CHAR(y), (FILE *)x) == EOF ? e_nil : e_t;
		break;
	      case 9:		/* n=9: getc */
		{
		  int c=getc((FILE *)PEEKVAL());

		  PEEKVAL() = (c==EOF) ? e_nil : CHAR_TO_REF(c);
		}
		break;
	      default:
		(void)printf("\nIllegal arg %d to STREAM-PRIMITIVE bytecode.\n",
			     arg_field);
		maybe_dump_world(1);
		exit(1);
	      }
	    break;

	  case (CASE_FOUR*32):	/* FILLTAG n */
	  new_filltag:		/* This label goes with OLD-FILLTAG. */
	    x =  PEEKVAL();
	    CHECKTAG0(x,PTR_TAG,1);
	    REF_SLOT(x,ESCAPE_OBJECT_VAL_OFF) = INT_TO_REF( val_height()
							   - arg_field );
	    REF_SLOT(x, ESCAPE_OBJECT_CXT_OFF) = INT_TO_REF( cxt_height() );
	    break;
#ifndef FAST
	  default:
	    (void)printf("\nIllegal Bytecode %d.\n", op_field);
	    maybe_dump_world(1);
	    exit(1);
#endif
	  }
      }

    /* The above loop is infinite; we branch down to here when instructions
       fail, normally from tag traps, and then branch back. */

  arg1_tt:
    CHECKVAL_PUSH(3);
    PUSHVAL_NOCHECK(x);
  arg0_tt:
    if (trace_traps)
      {
	(void)printf("\nTag trap: opcode %d, argfield %d.\n",
		     op_field, arg_field);
	(void)printf("Top of stack: ");
	printref(x);
	(void)printf(", pc =  %ld\n",
		     (SPATIC_PTR((ref *)e_pc)
		      ? e_pc - (unsigned short *)spatic.start
		      : e_pc - (unsigned short *)new.start
		        + 2*spatic.size));
      }

    /* Trick: to preserve tail recursiveness, push context only if next
       instruction isn't a RETURN and current instruction wasn't a FUNCALL. */

    /* NOTE: It might be worth making sure op_field isn't recomputed
       many times here if your compiler is stupid. */

    if (*e_pc != 24*256 + 0 && op_field != 21 && op_field != 22)
      PUSH_CONTEXT(0);

    /* Trapping instructions stash their argument counts here: */
    e_nargs = trap_nargs;

    if (op_field == 0)
      {
	/* argless instruction. */
	PUSHVAL_NOCHECK(*(e_argless_tag_trap_table + arg_field));
      }
    else
      {
	/* arg'ed instruction, so push arg field as extra argument */

	PUSHVAL_NOCHECK(INT_TO_REF(arg_field));
	e_nargs += 1;

	PUSHVAL_NOCHECK(*(e_arged_tag_trap_table + op_field));
      }

    if (trace_traps)
      {
	(void)printf("Dispatching to ");
	printref(PEEKVAL());
	(void)printf(" with NARGS = %d.\n", e_nargs);
      }

    /* Set the instruction dispatch register in case the FUNCALL fails. */

    instr = (22<<2);

    goto funcall_tail;
  }
}



