;;; -*-Scheme-*-
;;;
;;;	$Header: vmspth.scm,v 1.7 87/08/20 04:00:17 GMT cph Rel $
;;;
;;;	Copyright (c) 1987 Massachusetts Institute of Technology
;;;
;;;	This material was developed by the Scheme project at the
;;;	Massachusetts Institute of Technology, Department of
;;;	Electrical Engineering and Computer Science.  Permission to
;;;	copy this software, to redistribute it, and to use it for any
;;;	purpose is granted, subject to the following restrictions and
;;;	understandings.
;;;
;;;	1. Any copy made of this software must include this copyright
;;;	notice in full.
;;;
;;;	2. Users of this software agree to make their best efforts (a)
;;;	to return to the MIT Scheme project any improvements or
;;;	extensions that they make, so that these may be included in
;;;	future releases; and (b) to inform MIT of noteworthy uses of
;;;	this software.
;;;
;;;	3. All materials developed as a consequence of the use of this
;;;	software shall duly acknowledge such use, in accordance with
;;;	the usual standards of acknowledging credit in academic
;;;	research.
;;;
;;;	4. MIT has made no warrantee or representation that the
;;;	operation of this software will be error-free, and MIT is
;;;	under no obligation to provide any services, by way of
;;;	maintenance, update, or otherwise.
;;;
;;;	5. In conjunction with products arising from the use of this
;;;	material, there shall be no use of the name of the
;;;	Massachusetts Institute of Technology nor of any adaptation
;;;	thereof in any advertising, promotional, or sales literature
;;;	without prior written consent from MIT in each case.
;;;

;;;; VMS pathname parsing and unparsing.

(declare (usual-integrations))

;;;; Parse

(define (symbol->pathname symbol)
  (string->pathname (symbol->string symbol)))

(define parse-pathname)
(define pathname-as-directory)
(let ()

(set! parse-pathname
  (named-lambda (parse-pathname string receiver)
    (let ((start 0)
	  (end (string-length string))
	  (index)
	  (device false)
	  (directory '())
	  (name false)
	  (type false)
	  (version false))

      (define (find-char char)
	(set! index (substring-find-next-char string start end char))
	index)

      (if (find-char #\:)
	  (begin
	    (set! device (wildify string start index))
	    (set! start (1+ index))))
      (if (and (< start end)
	       (char=? #\[ (string-ref string start))
	       (find-char #\]))
	  (begin
	    (set! directory (parse-directory string (1+ start) index))
	    (set! start (1+ index))))
      (if (< start end)
	  (cond ((find-char #\.)
		 (set! name (wildify string start index))
		 (set! start (1+ index))
		 (if (not (find-char #\;))
		     (set! type (wildify string start end))
		     (begin
		       (set! version (parse-version string (1+ index) end))
		       (set! type
			     (wildify string start (if version index end))))))
		((find-char #\;)
		 (set! version (parse-version string (1+ index) end))
		 (set! name (wildify string start (if version index end))))
		(else
		 (set! name (wildify string start end))
		 (set! start end))))
    (receiver device directory name type version))))

(set! pathname-as-directory
  (named-lambda (pathname-as-directory pathname)
    (make-pathname
     (pathname-device pathname)
     (let ((directory (pathname-directory pathname)))
       (let ((file (pathname-unparse-name (pathname-name pathname)
					  (pathname-type pathname)
					  (pathname-version pathname))))
	 (if (string-null? file)
	     directory
	     (let ((file-components
		    (list (parse-directory-component file
						     0
						     (string-length file)))))
	       (cond ((or (null? directory) (eq? directory 'UNSPECIFIC))
		      file-components)
		     ((pair? directory) (append directory file-components))
		     (else (error "Illegal pathname directory" directory)))))))
     false false false)))

(define (parse-directory string start end)
  (define (loop start)
    (let ((index (substring-find-next-char string start end #\.)))
      (if index
	  (cons (parse-directory-component string start index)
		(loop (1+ index)))
	  (list (parse-directory-component string start end)))))
  ;; This assumes that "dev:[]foo.scm" is absolute, while
  ;; "dev:foo.scm" is relative.  I'm not sure this is right.
  (cond ((= start end) (list 'ROOT))
	((char=? #\. (string-ref string start)) (loop (1+ start)))
	(else
	 (let ((components (loop start)))
	   (if (eq? 'UP (car components))
	       components
	       (cons 'ROOT components))))))

(define (parse-directory-component string start end)
  (cond ((substring=? string start end "*" 0 1) 'WILD)
	((substring=? string start end "-" 0 1) 'UP)
	(else (substring string start end))))

(define (parse-version string start end)
  (cond ((= start end) false)
	((substring=? string start end "*" 0 1) 'WILD)
	((substring=? string start end "-1" 0 2) 'NEWEST)
	((substring=? string start end "0" 0 1) false)
	(else
	 (digits->number (reverse! (substring->list string start end)) 1 0))))

(define (digits->number digits weight accumulator)
  (if (null? digits)
      accumulator
      (let ((value (char->digit (car digits) 10)))
	(and value
	     (digits->number (cdr digits)
			     (* weight 10)
			     (+ (* weight value) accumulator))))))

(define (wildify string start end)
  (if (substring=? string start end "*" 0 1)
      'WILD
      (substring string start end)))

;;; end LET.
)

;;;; Unparse

(define pathname-unparse)
(define pathname-unparse-name)
(let ()

(set! pathname-unparse
  (named-lambda (pathname-unparse device directory name type version)
    (string-append (let ((device-string (unparse-component device)))
		     (if device-string
			 (string-append device-string ":")
			 ""))
		   (unparse-directory directory)
		   (pathname-unparse-name name type version))))

(define (unparse-directory directory)
  (cond ((or (null? directory) (eq? directory 'UNSPECIFIC)) "")
	((pair? directory)
	 (string-append
	  "["
	  (cond ((not (memq (car directory) '(ROOT UP)))
		 (string-append "." (unparse-directory-components directory)))
		((null? (cdr directory)) "")
		(else (unparse-directory-components (cdr directory))))
	  "]"))
	(else
	 (error "Unrecognizable directory" directory))))

(define (unparse-directory-components directory)
  (if (null? (cdr directory))
      (unparse-directory-component (car directory))
      (string-append (unparse-directory-component (car directory))
		     "."
		     (unparse-directory-components (cdr directory)))))

(define (unparse-directory-component component)
  (cond ((eq? component 'WILD) "*")
	((eq? component 'UP) "-")
	((string? component) component)
	(else (error "Illegal directory component" component))))

(set! pathname-unparse-name
  (named-lambda (pathname-unparse-name name type version)
    (let ((name (unparse-component name))
	  (type (unparse-component type))
	  (version (unparse-version version)))
      (cond ((not name) "")
	    ((not type) name)
	    ((not version) (string-append name "." type))
	    (else (string-append name "." type ";" version))))))

(define (unparse-component component)
  (cond ((or (not component) (string? component)) component)
	((eq? component 'UNSPECIFIC) false)
	((eq? component 'WILD) "*")
	(else (error "Illegal pathname component" component))))

(define (unparse-version version)
  (cond ((or (not version) (string? version)) version)
	((eq? version 'UNSPECIFIC) false)
	((eq? version 'WILD) "*")
	((eq? version 'NEWEST) "-1")
	((and (integer? version) (> version 0))
	 (list->string (number->digits version '())))
	(else (error "Illegal pathname version" version))))

(define (number->digits number accumulator)
  (if (zero? number)
      accumulator
      (let ((qr (integer-divide number 10)))
	(number->digits (integer-divide-quotient qr)
			(cons (digit->char (integer-divide-remainder qr))
			      accumulator)))))

)

;;;; Working Directory

(define working-directory-pathname)
(define set-working-directory-pathname!)

(define working-directory-package
  (make-environment

(define pathname)

(define (reset!)
  ;; Right now there is no primitive to supply the initial working
  ;; directory.  Someone who understands VMS should fix this.
  (set! pathname (make-pathname false '(ROOT) false false false)))

(set! working-directory-pathname
  (named-lambda (working-directory-pathname)
    pathname))

(set! set-working-directory-pathname!
  (named-lambda (set-working-directory-pathname! name)
    (set! pathname
	  (pathname-as-directory
	   (pathname->absolute-pathname (->pathname name))))
    pathname))

))

(define (home-directory-pathname)
  (string->pathname ""))

(define init-file-pathname
  (string->pathname "scheme.init"))

(define pathname-newest
  false)