/* -*-C-*-

Copyright (c) 1987 Massachusetts Institute of Technology

This material was developed by the Scheme project at the Massachusetts
Institute of Technology, Department of Electrical Engineering and
Computer Science.  Permission to copy this software, to redistribute
it, and to use it for any purpose is granted, subject to the following
restrictions and understandings.

1. Any copy made of this software must include this copyright notice
in full.

2. Users of this software agree to make their best efforts (a) to
return to the MIT Scheme project any improvements or extensions that
they make, so that these may be included in future releases; and (b)
to inform MIT of noteworthy uses of this software.

3. All materials developed as a consequence of the use of this
software shall duly acknowledge such use, in accordance with the usual
standards of acknowledging credit in academic research.

4. MIT has made no warrantee or representation that the operation of
this software will be error-free, and MIT is under no obligation to
provide any services, by way of maintenance, update, or otherwise.

5. In conjunction with products arising from the use of this material,
there shall be no use of the name of the Massachusetts Institute of
Technology nor of any adaptation thereof in any advertising,
promotional, or sales literature without prior written consent from
MIT in each case. */

/* $Header: fileio.c,v 9.24 87/07/07 20:59:07 GMT cph Rel $ */

/* Primitives to perform I/O to and from files. */

#include "scheme.h"
#include "primitive.h"
#include "character.h"
#include "stringprim.h"

extern FILE   *OS_file_open();
extern Boolean OS_file_close();
extern Boolean OS_file_eof_p();
extern long    OS_file_length();
extern long    OS_file_read_chars();
extern Boolean OS_file_write_chars();
extern Boolean OS_file_copy();
extern Boolean OS_file_rename();
extern Boolean OS_file_remove();
extern Boolean OS_file_link_physical();
extern Boolean OS_file_link_symbolic();
extern Pointer OS_working_dir_pathname();
extern Boolean OS_set_working_dir_pathname();
extern Boolean OS_directory_make();
extern Pointer OS_directory_open();
extern Pointer OS_directory_read();

#define channel_number_to_long(location, argument, wrong_type, bad_range) \
{ if ((pointer_type (argument)) != TC_FIXNUM)				\
    Primitive_Error (wrong_type);					\
  Range_Check (location, argument, 0, FILE_CHANNELS, bad_range);	\
  if ((Channels[location]) == NULL)					\
    Primitive_Error (bad_range);					\
}

long
channel_to_long (argument, wrong_type, bad_range)
     Pointer argument;
     int wrong_type, bad_range;
{
  long i;
  Pointer channel;

  if ((pointer_type (argument)) != TC_HUNK3)
    Primitive_Error (wrong_type);
  channel = (Vector_Ref (argument, 0));
  Touch_In_Primitive (channel, channel);
  channel_number_to_long (i, channel, bad_range, bad_range);
  return (i);
}

FILE *
channel_to_file (argument, wrong_type, bad_range)
     Pointer argument;
     int wrong_type, bad_range;
{
  return (Channels[(channel_to_long (argument, wrong_type, bad_range))]);
}

#define arg1_to_file()						\
  (channel_to_file (Arg1, ERR_ARG_1_WRONG_TYPE, ERR_ARG_1_BAD_RANGE))

#define arg2_to_file()						\
  (channel_to_file (Arg2, ERR_ARG_2_WRONG_TYPE, ERR_ARG_2_BAD_RANGE))

/* (FILE-OPEN-CHANNEL filename output?)
   Open a file called FILENAME, returning a channel number.
   The argument OUTPUT?, if false, means open an input file.
   Otherwise, open an output file. */

Built_In_Primitive (Prim_File_Open_Channel, 2, "FILE-OPEN-CHANNEL", 0x54)
{
  long i;
  Primitive_2_Args ();

  CHECK_ARG (1, STRING_P);
  for (i = 1; (i <= FILE_CHANNELS); i++)
    if ((Channels[i]) == NULL)
      {
	Channels[i] =
	  OS_file_open ((Scheme_String_To_C_String (Arg1)), (Arg2 != NIL));
	if ((Channels[i]) == NULL)
	  error_bad_range_arg (1);
	Open_File_Hook (i);
	return (Make_Unsigned_Fixnum (i));
      }
  Primitive_Error (ERR_OUT_OF_FILE_HANDLES);
}

/* (FILE-CLOSE-CHANNEL channel_number)
   Closes the file represented by channel_number. */

Built_In_Primitive (Prim_File_Close_Channel, 1, "FILE-CLOSE-CHANNEL", 0x55)
{
  long channel;
  Boolean result;
  Primitive_1_Arg ();

  channel_number_to_long (channel, Arg1,
			  ERR_ARG_1_WRONG_TYPE,
			  ERR_ARG_1_BAD_RANGE);
  result = (OS_file_close (Channels[channel]));
  Channels[channel] = NULL;
  Close_File_Hook ();
  if (! result)
    error_external_return ();
  return (TRUTH);
}

Built_In_Primitive (Prim_File_Eof_P, 1, "FILE-EOF?", 0xC1)
{
  Primitive_1_Arg ();

  return ((OS_file_eof_p (arg1_to_file ())) ? TRUTH : NIL);
}

Built_In_Primitive (Prim_File_Length, 1, "FILE-LENGTH", 0xC4)
{
  Primitive_1_Arg ();
    
  return (Make_Unsigned_Fixnum (OS_file_length (arg1_to_file ())));
}

Built_In_Primitive (Prim_File_Read_Char, 1, "FILE-READ-CHAR", 0xC2)
{
  char chr;
  Primitive_1_Arg ();

  if ((OS_file_read_chars ((arg1_to_file ()), &chr, 1)) != 1)
    error_external_return ();
  return (c_char_to_scheme_char (chr));
}

Built_In_Primitive (Prim_File_Fill_Input_Buffer, 2, "FILE-FILL-INPUT-BUFFER", 0xC3)
{
  char *buffer;
  Primitive_2_Args ();

  CHECK_ARG (2, STRING_P);
  buffer = (Scheme_String_To_C_String (Arg2));
  return (Make_Unsigned_Fixnum (OS_file_read_chars ((arg1_to_file ()), buffer,
						    (string_length (Arg2)))));
}

Built_In_Primitive (Prim_File_Write_Char, 2, "FILE-WRITE-CHAR", 0xC5)
{
  char chr;
  Primitive_2_Args ();

  chr = (arg_ascii_char (1));
  if (! (OS_file_write_chars ((arg2_to_file ()), &chr, 1)))
    error_external_return ();
  return (NIL);
}

Built_In_Primitive (Prim_File_Write_String, 2, "FILE-WRITE-STRING", 0xC6)
{
  char *buffer;
  Primitive_2_Args ();

  CHECK_ARG (1, STRING_P);
  buffer = (Scheme_String_To_C_String (Arg1));
  if (! (OS_file_write_chars ((arg2_to_file ()), buffer,
			      (string_length (Arg1)))))
    error_external_return ();
  return (NIL);
}

Built_In_Primitive (Prim_File_Exists, 1, "FILE-EXISTS?", 0x10A)
{
  Primitive_1_Arg ();

  CHECK_ARG (1, STRING_P);
  return
    (((OS_file_existence_test (Scheme_String_To_C_String (Arg1))) > 0)
     ? TRUTH
     : NIL);
}

/* File System Operations */

/* (COPY-FILE old-name new-name)
   Make a new copy of the file OLD-NAME, called NEW-NAME. */

Built_In_Primitive (Prim_Copy_File, 2, "COPY-FILE", 0x111)
{
  Primitive_2_Args ();

  CHECK_ARG (1, STRING_P);
  CHECK_ARG (2, STRING_P);
  if (! (OS_file_copy ((Scheme_String_To_C_String (Arg1)),
		       (Scheme_String_To_C_String (Arg2)))))
    error_external_return ();
  return (NIL);
}

/* (RENAME-FILE old-name new-name)
   Moves the file from OLD-NAME to NEW-NAME. */

Built_In_Primitive (Prim_Rename_File, 2, "RENAME-FILE", 0x112)
{
  Primitive_2_Args ();

  CHECK_ARG (1, STRING_P);
  CHECK_ARG (2, STRING_P);
  if (! (OS_file_rename ((Scheme_String_To_C_String (Arg1)),
			 (Scheme_String_To_C_String (Arg2)))))
    error_external_return ();
  return (NIL);
}

/* (REMOVE-FILE filename)
   Delete the given file from the file system.  Signals an error if
   unable to delete the file. */

Built_In_Primitive (Prim_Remove_File, 1, "REMOVE-FILE", 0x113)
{
  Primitive_1_Arg ();

  CHECK_ARG (1, STRING_P);
  if (! (OS_file_remove (Scheme_String_To_C_String (Arg1))))
    error_external_return ();
  return (NIL);
}

/* (LINK-FILE old-file new-file physical?)
   Make NEW-FILE be a link pointing at OLD-FILE.  If PHYSICAL? is
   true, a physical link is created, otherwise a symbolic link is
   created. */

Built_In_Primitive (Prim_Link_File, 3, "LINK-FILE", 0x114)
{
  char *old_file, *new_file;
  Primitive_3_Args ();

  CHECK_ARG (1, STRING_P);
  old_file = (Scheme_String_To_C_String (Arg1));
  new_file = (Scheme_String_To_C_String (Arg2));
  if (! ((Arg3 != NIL)
	 ? (OS_file_link_physical (old_file, new_file))
	 : (OS_file_link_symbolic (old_file, new_file))))
    error_external_return ();
  return (NIL);
}

/* (WORKING-DIRECTORY-PATHNAME)
   Returns the current working directory as a string.*/

Built_In_Primitive (Prim_working_dir_pathname, 0,
		    "WORKING-DIRECTORY-PATHNAME", 0x19E)
{
  Pointer result;
  Primitive_0_Args ();

  result = (OS_working_dir_pathname ());
  if (result == NIL)
    error_external_return ();
  return (result);
}

/* (SET-WORKING-DIRECTORY-PATHNAME! string)
   Changes the current working directory to the given string.
   Returns the old working directory as its value. */

Built_In_Primitive (Prim_set_working_dir_pathname, 1,
		    "SET-WORKING-DIRECTORY-PATHNAME!", 0x117)
{
  Pointer result;
  Primitive_1_Arg ();

  CHECK_ARG (1, STRING_P);
  result = (OS_working_dir_pathname ());
  if (result == NIL)
    error_external_return ();
  if (! (OS_set_working_dir_pathname (Scheme_String_To_C_String (Arg1))))
    error_bad_range_arg (1);
  return (result);
}

/* (MAKE-DIRECTORY pathname)
   Create a new directory of the given name. */

Built_In_Primitive (Prim_Make_Directory, 1, "MAKE-DIRECTORY", 0x115)
{
  Primitive_1_Arg ();

  CHECK_ARG (1, STRING_P);
  if (! (OS_directory_make (Scheme_String_To_C_String (Arg1))))
    error_bad_range_arg (1);
  return (NIL);
}

/* (OPEN-DIRECTORY pathname)
   Attempts to open the directory specified by the string `pathname'.
   If successful, it returns the first file in the directory, as a string.
   If there is no such file, or the directory cannot be opened, returns #F. */

Built_In_Primitive (Prim_open_directory, 1, "OPEN-DIRECTORY", 0x19F)
{
  Primitive_1_Arg ();

  CHECK_ARG (1, STRING_P);
  return (OS_directory_open (Scheme_String_To_C_String (Arg1)));
}

/* (DIRECTORY-READ)
   Returns the next file in the directory opened by `open-directory',
   or #F if there are no more files in the directory. */

Built_In_Primitive (Prim_directory_read, 0, "DIRECTORY-READ", 0x1A0)
{
  Primitive_0_Args ();

  return (OS_directory_read ());
}
