;;; -*- Mode: Scheme; Syntax: Scheme; Package: (SCHEME :USE (PSEUDOSCHEME)) -*-

;;;
;;;	$Header$
;;;
;;;	Copyright (c) 1986, 1987 Massachusetts Institute of Technology
;;;     Initial implementation due to Ken Haase (KWH@AI.AI.MIT.EDU)
;;;
;;;	This material was developed by the Scheme project at the
;;;	Massachusetts Institute of Technology, Department of
;;;	Electrical Engineering and Computer Science.  Permission to
;;;	copy this software, to redistribute it, and to use it for any
;;;	purpose is granted, subject to the following restrictions and
;;;	understandings.
;;;
;;;	1. Any copy made of this software must include this copyright
;;;	notice in full.
;;;
;;;	2. Users of this software agree to make their best efforts (a)
;;;	to return to the MIT Scheme project any improvements or
;;;	extensions that they make, so that these may be included in
;;;	future releases; and (b) to inform MIT of noteworthy uses of
;;;	this software.
;;;
;;;	3. All materials developed as a consequence of the use of this
;;;	software shall duly acknowledge such use, in accordance with
;;;	the usual standards of acknowledging credit in academic
;;;	research.
;;;
;;;	4. MIT has made no warrantee or representation that the
;;;	operation of this software will be error-free, and MIT is
;;;	under no obligation to provide any services, by way of
;;;	maintenance, update, or otherwise.
;;;
;;;	5. In conjunction with products arising from the use of this
;;;	material, there shall be no use of the name of the
;;;	Massachusetts Institute of Technology nor of any adaptation
;;;	thereof in any advertising, promotional, or sales literature
;;;	without prior written consent from MIT in each case.
;;;

(declare (usual-integrations))
(declare (integrate-external "/u/kwh/programs/utility/plus"))

;;; This defines a variety of useful mapping functions for SCHEME.

;;;; Mapping and iteration functions

;;; SOME? and EVERY? takes arguments in the opposite order of Common 
;;; LISP, and also only allow unary tests; but I think it's more 
;;; natural in this way.  (Who knows?)  They also are true predicates,
;;; returning only true or false.

(define (some? of-list test)
  ;; Returns #T if some element of OF-LIST satisfies TEST.
  (define (check-list list)
    (if (null? list) #F
	(if (test (car list)) #T
	    (check-list (cdr list)))))
  (check-list of-list))

(define (every? of-list test)
  ;; Returns #T if every element of OF-LIST satisfies TEST.
  (define (check-list list)
    (if (null? list) #T
	(if (test (car list)) (check-list (cdr list))
	    #F)))
  (check-list of-list))

(define (maprange fcn from to)
  ;; Applies FCN to every integer between FROM and TO (including FROM)
  ;; and returns a list of the results in order.
  (define (iter-range base l tail)
    (if (>= from to) l
	(let ((elt (cons (fcn base) ())))
	  (set-cdr! tail elt)
	  (iter-range (1+ base) l elt))))
  (let ((start (list (fcn (ceiling from)))))
    (iter-range (ceiling from) start start)))

(define (for-range fcn from to)
  ;; Applies FCN to every integer between FROM and TO (including FROM).
  (define (count-up i)
    (fcn i)
    (let ((next (1+ i))) (if (>= next to) i (count-up next))))
  (define (count-down i)
    (fcn i)
    (let ((next (-1+ i))) (if (<= next to) i (count-down next))))
  (if (> to from) (count-up (ceiling from)) (count-down (floor from))))

(define (extract predicate list)
  ;; Returns a sublist of LIST made from elements satisfying PREDICATE.
  (define (extracting predicate from-list into-list)
    (if (null? from-list) into-list
	(extracting predicate (cdr from-list)
		    (if (predicate (car from-list))
			(cons (car from-list) into-list)
			into-list))))
  (extracting predicate list ()))

(define (collect fcn over-list)
  ;; Collects the non-false results of applying FCN to elements of OVER-LIST.
  (define (merge-into-list x list)
    (if (false? x) list (cons x list)))
  (define (collecting fcn over-list collection)
    (if (null? over-list) collection
	(collecting fcn (cdr over-list)
		    (merge-into-list (fcn (car over-list)) collection))))
  (collecting fcn over-list ()))

(define (mapunion fcn over-list)
  ;; UNIONS the lists resulting from applying FCN to the elements of OVER-LIST.
  (apply union (map fcn over-list)))
(define merge mapunion)

(define (mapappend fcn over-list)
  ;; APPENDS the lists resulting from applying FCN to the elements of OVER-LIST.
  (apply append (map fcn over-list)))


;;;; Iterating over trees

;;; These procedures all map other procedures over trees defined by a
;;; set of roots and a `tree-function' for extracting the children of
;;; the tree.

(define (maptree map-function roots tree-function)
  ;; Applies MAP-FUNCTION to each node in the tree, 
  ;; expanding only those nodes for which MAP-FUNCTION
  ;; returned TRUE (non-FALSE).
  (let ((nodes-visited ()))
    (define (walker nodes) 
      (define (not-visited? node)
	(not (or (memq node nodes-visited) (memq node nodes))))
      (if (null? nodes) #T
	  (let ((first-node (car nodes)))
	    (set! nodes-visited (cons first-node nodes-visited))
	    (if (map-function first-node)
		(walker (append! (cdr nodes)
				 (extract not-visited? (tree-function first-node))))
		(walker (cdr nodes))))))
    (walker (apply list roots))))

(define (search-tree test roots tree-function)
  ;; Maps over the tree, returning whatever node first passes TEST.
  (let ((nodes-visited ()))
    (define (walker nodes)
      (define (not-visited? node)
	(not (or (memq node nodes-visited) (memq node nodes))))
      (if (null? nodes) #T
	  (let ((first-node (car nodes)))
	    (set! nodes-visited (cons first-node nodes-visited))
	    (if (test first-node) first-node
		(walker (append! (extract not-visited? (tree-function first-node))
				 (cdr nodes)))))))
    (walker (apply list roots))))

(define (collect-fringe property roots tree-function)
  ;; Collects the fringe of the tree which satisfies PROPERTY.
  (define collection ())
  (define (collect x) (set! collection (cons x collection)))
  (define (collecting node) 
    (cond ((property node) (collect node) #F)
	  (else #T)))
  (maptree collecting roots tree-function)
  collection)

(define (collect-tree fcn roots tree-function)
  ;; Collects the non-false results of applying FCN to the nodes of
  ;; the tree.
  (define collection ())
  (define (collector x)
    (let ((fruits (fcn x)))
      (if (null? fruits) #F
	  (set! collection (cons fruits collection)))
      #T))
  (maptree collector roots tree-function)
  collection)


;;;; Random functions.

(define (return-true ignore) #T)
(define (return-false ignore) #F)

