;;; -*- Mode: Scheme; Syntax: Scheme; Package: (SCHEME :USE (PSEUDOSCHEME)) -*-

;;;
;;;	$Header$
;;;
;;;	Copyright (c) 1986, 1987 Massachusetts Institute of Technology
;;;     Initial implementation due to Ken Haase (KWH@AI.AI.MIT.EDU)
;;;
;;;	This material was developed by the Scheme project at the
;;;	Massachusetts Institute of Technology, Department of
;;;	Electrical Engineering and Computer Science.  Permission to
;;;	copy this software, to redistribute it, and to use it for any
;;;	purpose is granted, subject to the following restrictions and
;;;	understandings.
;;;
;;;	1. Any copy made of this software must include this copyright
;;;	notice in full.
;;;
;;;	2. Users of this software agree to make their best efforts (a)
;;;	to return to the MIT Scheme project any improvements or
;;;	extensions that they make, so that these may be included in
;;;	future releases; and (b) to inform MIT of noteworthy uses of
;;;	this software.
;;;
;;;	3. All materials developed as a consequence of the use of this
;;;	software shall duly acknowledge such use, in accordance with
;;;	the usual standards of acknowledging credit in academic
;;;	research.
;;;
;;;	4. MIT has made no warrantee or representation that the
;;;	operation of this software will be error-free, and MIT is
;;;	under no obligation to provide any services, by way of
;;;	maintenance, update, or otherwise.
;;;
;;;	5. In conjunction with products arising from the use of this
;;;	material, there shall be no use of the name of the
;;;	Massachusetts Institute of Technology nor of any adaptation
;;;	thereof in any advertising, promotional, or sales literature
;;;	without prior written consent from MIT in each case.
;;;

(declare (integrate-external "/u/kwh/programs/utility/plus")
	 (integrate-external "/u/kwh/programs/utility/mutable")
	 (integrate-external "/u/kwh/programs/typical/kernel"))


;;;; Tracing

;;; This determines if each call to INDEX is traced.
(define trace-object-indexing #F)

;;; This determines if daemon executions are traced.
(define trace-daemon-execution #F)

;;; Special forms of MESSAGE which check the flags above.
(definline (indexing-message . printout-args)
  (if trace-object-indexing (apply message printout-args)))
(definline (daemon-message . printout-args)
  (if trace-daemon-execution (apply message printout-args)))


;;;; Daemons

;;; This is a mapping for storing daemons.
;;;  Daemons are stored in both directions.
(define daemons-for-type (td-property 'daemons))
(define type-daemons     daemons-for-type)
(define types-for-daemon (make-mutable))

;;; The DAEMON type.
(define (daemon? thing) (defined? (types-for-daemon thing)))
(define daemons (type "Daemons" simple-type daemon? procedures))

;;; Formatting a daemon prints its description as well:
(define ($daemon d)
  ($nested ($procedure d) " [" ($procedure-description d) "]")) 

;;; Adding a daemon is straightforward:
(define (add-daemon! daemon type)
 ((mutator types-for-daemon) daemon (pusher type))
 ((mutator daemons-for-type) type (pusher daemon))
 daemon)

;;; As is removing a daemon:
(define (remove-daemon! daemon type)
  ((mutator daemons-for-type) type (deletor daemon))
  ((mutator types-for-daemon) daemon (deletor type))
  type)


;;;; Habituations & Inhibitions
;;; This keeps daemons from being run repeatedly on an object.

;;; This is a modifiable store (just a CONS, really) for inhibited or
;;; already-run daemons.
(define (make-daemon-store for-x) (list for-x))

;;; These are daemons which have already been run on an object.
(define already-run-cache (make-mutable))
(define already-run (generate-memoizer already-run-cache make-daemon-store))
(define (daemons-already-run x) (apply-if-defined cdr (already-run-cache x)))

;;; These are daemons which have been actively inhibited for an object.
(define inhibitions-cache (make-mutable))
(define inhibitions (generate-memoizer inhibitions-cache make-daemon-store))
(define (daemons-inhibited x) (apply-if-defined cdr (inhibitions-cache x)))

;;; Procedures for inhibiting daemons.
(define (inhibit-daemon! daemon for-x)
  (let ((inhibitions (inhibitions for-x)))
    (if (memq daemon (cdr inhibitions)) #F
	(set-cdr! inhibitions (cons daemon (cdr inhibitions))))))
(define (uninhibit-daemon! daemon for-x)
  (let ((inhibitions (inhibitions for-x)))
    (set-cdr! inhibitions (delete daemon (cdr inhibitions)))))


;;; A higher order procedure which makes an inhibitor.
(define (make-censor daemon)
  (define (inhibit-daemon x) (inhibit-daemon! daemon x))
  (procedure! inhibit-daemon 'INHIBITION "Inhibit the daemon " ($daemon daemon))
  inhibit-daemon)
(define censor (simple-cache make-censor))
(define inhibitor censor)


;;;; The INDEX procedure.

(define (index-under x type)
  (indexing-message $NL "Indexing the object " x " under " type "..... ")
  (let ((start-time (systime))
	(inhibitions (inhibitions x))
	(already-run (already-run x)))
    (define (execute-daemon daemon)
      (cond ((memq daemon (cdr already-run)) #F)
	    ((memq daemon (cdr inhibitions))
	     (daemon-message $NL "&&& Not running inhibited daemon " ($daemon daemon))
	     #F)
	    (ELSE (daemon-message $Nl "&&& Running daemon " ($daemon daemon))
		  (daemon x)
		  (set-cdr! already-run (cons daemon (cdr already-run)))
		  #T)))
    (define (run-type-daemons type)
      (let ((daemons (daemons-for-type type)))
	(if (defined? daemons) (for-each execute-daemon daemons))))
    (maptypes-under run-type-daemons x type)
    (indexing-message $NL ".... Finished indexing the object " x " in "
		      ($count (/ (- (systime) start-time) 100) "second"))
    x))
(define (index x) (index-under x lattice-top))

(define (assert! object property)
  (cond ((in? object property) object)
	((in? object (complement property))
	 (take-from-collection! object (complement property))
	 (put-in-collection! object property)
	 (reset-cache! (type-cache object))
	 (index object))
	(else (put-in-collection! object property)
	      (index-under object property))))
