;;; -*- Mode: Scheme; Syntax: Scheme; Package: (SCHEME :USE (PSEUDOSCHEME)) -*-

;;;
;;;	$Header$
;;;
;;;	Copyright (c) 1986, 1987 Massachusetts Institute of Technology
;;;     Initial implementation due to Ken Haase (KWH@AI.AI.MIT.EDU)
;;;
;;;	This material was developed by the Scheme project at the
;;;	Massachusetts Institute of Technology, Department of
;;;	Electrical Engineering and Computer Science.  Permission to
;;;	copy this software, to redistribute it, and to use it for any
;;;	purpose is granted, subject to the following restrictions and
;;;	understandings.
;;;
;;;	1. Any copy made of this software must include this copyright
;;;	notice in full.
;;;
;;;	2. Users of this software agree to make their best efforts (a)
;;;	to return to the MIT Scheme project any improvements or
;;;	extensions that they make, so that these may be included in
;;;	future releases; and (b) to inform MIT of noteworthy uses of
;;;	this software.
;;;
;;;	3. All materials developed as a consequence of the use of this
;;;	software shall duly acknowledge such use, in accordance with
;;;	the usual standards of acknowledging credit in academic
;;;	research.
;;;
;;;	4. MIT has made no warrantee or representation that the
;;;	operation of this software will be error-free, and MIT is
;;;	under no obligation to provide any services, by way of
;;;	maintenance, update, or otherwise.
;;;
;;;	5. In conjunction with products arising from the use of this
;;;	material, there shall be no use of the name of the
;;;	Massachusetts Institute of Technology nor of any adaptation
;;;	thereof in any advertising, promotional, or sales literature
;;;	without prior written consent from MIT in each case.
;;;

(declare (usual-integrations))
(declare (integrate-external "/u/kwh/programs/utility/plus")
	 (integrate-external "/u/kwh/programs/utility/mutable")
	 (integrate-external "/u/kwh/programs/utility/tuple")
	 (integrate-external "/u/kwh/programs/typical/kernel"))


;;;; Menus
;;; A menu is a procedure which takes another procedure and calls it
;;; with an "item display function" which will create a menu-item.

(define menu-count 0)

(define (make-menu proc)
  (let ((index 0) (menu ()) (menu-id menu-count))
    (define (new-item x) (set! index (1+ index)) (set! menu (cons x menu)))
    (set! menu-count (1+ menu-count))
    (define (setup-menu)
      (printout $NL ";;; ------- MENU #" menu-id " -------")
      (set! index 0) (set! menu ())
      (with-left-margin-procedure
       (lambda (p) (display "<" p) (display index p) (display ">  " p))
       (lambda () (proc new-item)))
      (set! menu (reverse menu)))
    (define (select)
      (printout $NL "Choose an item from menu #" menu-id ": ")
      (let ((index (read)))
	(if (and (integer? index) (positive? index) (<= index (length menu)))
	    (list-ref menu (-1+ index))
	    (if (zero? index) (sequence (setup-menu) (select)) (select-again)))))
    (define (select-again)
      (printout $NL "Select an entry from menu #" menu-id
		" between 1 and " (length menu)
		$NL "(Zero redisplays the menu)" $NL)
      (let ((index (read)))
	(if (and (integer? index) (positive? index) (<= index (length menu)))
	    (list-ref menu (-1+ index))
	    (if (zero? index) (sequence (setup-menu) (select-again))
		(select-again)))))
    (setup-menu)
    select))

(define (choose-one from-list printer)
  (define (print-list menu)
    (for-each (lambda (x) (menu x) (printer x)) from-list))
  (if (null? (rest from-list)) (first from-list)
      ((make-menu print-list))))


;;;; The editor, EF

;;; The editor determines properties to display and commands to offer
;;; based on the position of the edited object in the lattice.

(define ef-commands (td-property 'ef-commands))
(define index-on-edit? #T)
(define list-is-a-on-edit? #T)
(define *EF*)

(define (ef object)
  (if index-on-edit? (index object))
  (let ((template (editor-template object)))
    (define (generate-menu m) (property-menu object (second template) m))
    (printout $NL ":::> Editing the object " object)
    (if list-is-a-on-edit?
	(sequence
	  (printout $NL ":::> which is in these classes (and their generalizations):")
	  (for-each (lambda (c) (if (or show-complements-in-df (not (in? c complements)))
				    (printout $NL ">>>>        " c)))
		    (first template))))
    (printout $NL ":::> with the following properties:")
    (editor-loop object (make-menu generate-menu) (third template))))
(define (et spec) (ef (->td spec)))


;;;; EF Support functions.

(define (editor-template object)
  (let ((fringe ()) (properties ()) (commands ()))
    (define (check-type type)
      (let ((relevant-properties (type-properties type))
	    (relevant-commands (ef-commands type)))
	(if (defined? relevant-properties)
	    (set! properties (append relevant-properties properties)))
	(if (defined? relevant-commands)
	    (set! commands (append relevant-commands commands)))
	(if (every? fringe (lambda (f) (not (<<? f type))))
	    (set! fringe (cons type fringe)))))
    (maptypes check-type object)
    (let ((specials (unique-properties object)))
      (if (defined? specials)
	  (list fringe (append specials properties) (reverse commands))
	  (list fringe properties (reverse commands))))))

(define (property-menu object properties menu)
  (define (property-item p)
    (let ((value (p object)))
      (if (defined? value)
	  (let ((printer (property-printer p)))
	    (menu (list p value))
	    (if (defined? printer) (printer value)
		(printout $NL ($procedure p) ":  " (p value)))))))
  (for-each property-item properties))

(define (editor-loop object properties commands)
  (printout $NL ":::> Editing " object " >>> ")
  (set! *EF* object)
  (let ((input (read)))
    (if (and (symbol? input) (assq input commands))
	((cadr (assq input commands)) object properties commands)
	(sequence (printout $NL "???>  I don't know the command `" input "' ?!?!")
		  (editor-loop object properties commands)))))

(define (defcommand-for-type! command type description proc)
  (define (update-commands commands)
    (if (undefined? commands)
	(list (list command proc description))
	(let ((pair (assq command commands)))
	  (cond (pair (set-cdr! pair (list proc description)) commands)
		(ELSE (cons (list command proc description) commands))))))
  ((mutator ef-commands) type update-commands))


;;;; Standard editor commands.

(define (ef-reset object ignore ignore) (ef object))
(define (ef-quit  object ignore ignore) object)

(define (ef-help object properties commands)
  (define (describe-command-entry command-entry)
    (printout $NL ":::>        " (first command-entry) " -- " (third command-entry)))
  (printout $NL ":::>  To call a command, type either the command name or (<command-name> . args)")
  (printout $NL ":::>  Currently active commands are: ")
  (for-each describe-command-entry commands)
  (editor-loop object properties commands))

(define (ef-edit object properties commands)
  (ef (second (properties)))
  (editor-loop object properties commands))
(define (ef-edit-property object properties commands)
  (ef (first (properties)))
  (editor-loop object properties commands))

(defcommand-for-type! 'RESET lattice-top "Resets the editor, redisplaying the property menu." ef-reset)
(defcommand-for-type! 'REDISPLAY lattice-top "Resets the editor, redisplaying the property menu." ef-reset)
(defcommand-for-type! 'QUIT lattice-top "Exits the editor." ef-quit)
(defcommand-for-type! 'EXIT lattice-top "Exits the editor." ef-quit)
(defcommand-for-type! '? lattice-top "Gives rudimentary help on using the editor." ef-help)
(defcommand-for-type! 'HELP lattice-top "Gives rudimentary help on using the editor." ef-help)
(defcommand-for-type! 'EF lattice-top "Calls EF recursively on a property." ef-edit)
(defcommand-for-type! 'EDIT lattice-top "Calls EF recursively on a property." ef-edit)
(defcommand-for-type! 'EDIT-PROPERTY lattice-top "Calls EF recursively on a property's description."
  ef-edit-property)


;;;; PROCEDURE->COMMAND

;;; Many commands just take regular procedures and apply them to
;;; editor-relative references.
	
(define type-reader (make-mutable))

(define (make-command-from-procedure procedure . args)
  (define (command-from-procedure object properties commands)
    (define (get-argument x)
      (cond ((eq? x 'VALUE) (second (properties)))
	    ((eq? x 'PROPERTY) (first (properties)))
	    ((eq? x 'OBJECT) object)
	    ((string? x) (newline) (display x) (read))
	    ((defined? (type-reader x)) ((type-reader x)))
	    (ELSE x)))
    (apply procedure (map get-argument args))
    (editor-loop object properties commands))
  command-from-procedure)
(define procedure->command (canonical-cache make-command-from-procedure))

