;;; -*- Mode: Lisp; Syntax: Common-LISP; Package: PSEUDOSCHEME; Patch-File: Yes -*-

(require "pseudoscheme")

(in-package 'pseudoscheme)
(or (find-package "SCHEME")
    (make-package "SCHEME" :use '(PSEUDOSCHEME)
		  :shadow '(MERGE)))

(setq cl::*print-radix* NIL)
#+SYMBOLICS
(global:setq-standard-value CL::*PRINT-RADIX* NIL)

(define-pseudo (make-empty-bit-string ignore) 0)
(define empty-bit-string 0)
(define-pseudo (bit-string-or b1 b2) (logior b1 b2))
(define-pseudo (bit-string-and b1 b2) (logand b1 b2))
(define-pseudo (check-bit b i) (logbitp i b))
(define-pseudo (bit-string-modify b i v)
  (dpb (if v 1. 0.) (byte 1 i) b))
(define-pseudo (bit-string-modify! b i v)
  (dpb (if v 1. 0.) (byte 1 i) b))

(define (scheme::make-empty-type-cache ignore) (cons 0 0))

(defun hours-minutes-seconds ()
  (multiple-value-bind (seconds minutes hours) (get-decoded-time)
    (list hours minutes seconds)))
(setq hours-minutes-seconds #'hours-minutes-seconds)

(define-pseudo (fixnum? x) (typep x 'fixnum))

#+SYMBOLICS
(setf (symbol-function 'structure?) #'cli::structurep)
(setq structure? #'cli::structurep)

#+SYMBOLICS
(define-pseudo[subst] (assq elt list) (zl:assq elt list))

(export '(make-empty-bit-string empty-bit-string
				bit-string-or bit-string-and check-bit
				bit-string-modify bit-string-modify!
				hours-minutes-seconds fixnum? structure?))


;;; List operations.

(define-pseudo[subst] (delete! elt list) (lisp:delete elt list :test #'equal))
(define-pseudo[subst] (delq! elt list)   (lisp:delete elt list :test #'eq))
(define-pseudo[subst] (delq elt list)    (lisp:remove elt list :test #'equal)) 
(define-pseudo (append! l1 l2) (nconc l1 l2))

(defun delete-equal (elt list) (remove elt list :test #'equal))
(define-pseudo (scheme::delete elt list) (delete-equal elt list)) 

(define-pseudo[subst] (scheme::first l) (car l))
(define-pseudo[subst] (scheme::rest l) (cdr l))
(define-pseudo[subst] (scheme::second l) (cadr l))
(define-pseudo[subst] (scheme::third l) (caddr l))
(define-pseudo[subst] (scheme::fourth l) (cadddr l))

(define-pseudo[subst] (list? x) (listp x))
(define-pseudo[subst] (false? x) (null x))

(defun union* (&rest list-of-lists)
  (lisp:do ((lists list-of-lists (rest lists))
	    (result () (union (first lists) result :test #'eq)))
	   ((null lists) result)))
(defparameter union* #'union*)
(define (scheme::union . args) (apply union* args))

(export '(delete! delq! delq append! list? false?))


;;;; Weak conses

(define-pseudo[subst] (scheme::weak-car x) (car x))
(define-pseudo[subst] (scheme::set-weak-car! x v) (setf (car x) v))
(define-pseudo[subst] (scheme::weak-cdr x) (cdr x))
(define-pseudo[subst] (scheme::set-weak-cdr! x v) (setf (cdr x) v))
(define-pseudo[subst] (scheme::weak-cons x y) (cons x y))
(define-pseudo[subst] (scheme::weak-assq key alist) (assq key alist))



;;;; Random procedures 

(defun dynamic-wind (wind do unwind)
  (unwind-protect (block wind-body (funcall wind) (funcall do))
    (funcall unwind)))
(setq dynamic-wind #'dynamic-wind)

(defun procedure-name (thing)
  "Returns the name of a procedure."
  (lisp:let ((raw-name 
	       (cond ((sys:function-name thing))
		     ((zl:typep thing ':lexical-closure)
		      (SYS:FUNCTION-NAME (SI:UNDIGEST (SI:%P-CONTENTS-OFFSET thing 1))))
		     ((sys:validate-function-spec thing) thing)
		     (T NIL))))
    (flet ((mklist (x) (if (listp x) x (list x))))
      (cond ((null raw-name) NIL)
	    ((and (listp raw-name) (eq (first raw-name) ':INTERNAL))
	     (append (mklist (procedure-name (fourth raw-name)))
		     (mklist (procedure-name (second raw-name)))))
	    (T raw-name)))))
(setq procedure-name #'procedure-name)

(defvar *time-ratio* (/ 100 internal-time-units-per-second))
(defun systime () (floor (* (get-internal-real-time) *time-ratio*)))
(setq systime #'systime)

(defun unparse-with-brackets (thunk)
  (unwind-protect (progn (format *standard-output* "#[") (funcall thunk))
    (format *standard-output* "]")))
(setq unparse-with-brackets #'unparse-with-brackets)

(setq identity #'identity)

(export '(procedure-name dynamic-wind systime unparse-with-brackets identity))


;;;; Random macros.
(define-macro (scheme::unwind-protect expression . cleanup)
  `(dynamic-wind (lambda ()) (lambda () ,expression)
		 (lambda () ,@cleanup)))

(define-macro (fluid-let var-list . body)
  (let* ((vars (mapcar car var-list))
	 (temp-vars
	   (mapcar (lambda (v) (make-symbol (string v))) vars))
	 (outer-bindings (mapcar list temp-vars vars))
	 (sets (mapcar (lambda (x) (cons 'scheme::set! x)) var-list))
	 (resets (mapcar (lambda (v temp) `(set! ,v ,temp)) vars temp-vars)))
    `(let ,outer-bindings
       (scheme::unwind-protect (begin ,@sets ,@body)
	 ,@resets))))
(export '(fluid-let))

(define-macro (definline spec . body)
  `(define ,spec ,@body))
(export '(definline))

(define-macro (named-lambda define-spec . body)
  `(let () (define ,define-spec ,@body) ,(first define-spec)))
(export '(named-lambda))

(defun usual-integrations ())
(export '(usual-integrations))

(defmacro scheme::declare (&rest decls)
  (declare (ignore decls))
  ''DECLARATIONS)


;;;; Defstruct

(define-macro (scheme::define-structure name slots print-form)
  (define (symbol-append . args)
    (string->symbol (apply string-append (map symbol->string args))))
  (define (generate-accessors slots index accessors)
    (if (null? slots) accessors
	(generate-accessors
	 (rest slots) (1+ index)
	 `((definline (,(symbol-append name '- (first slots)) ,name)
		      (vector-ref ,name ,index))
	   ,@accessors))))
  (define (generate-modifiers slots index accessors)
    (if (null? slots) accessors
	(generate-modifiers
	 (rest slots) (1+ index)
	 `((definline (,(symbol-append 'set- name '- (first slots) '!) ,name ,(first slots))
	     (vector-set! ,name ,index ,(first slots)))
	   ,@accessors))))
  `(begin (define (,(symbol-append 'CONS- name) ,@slots)
	    #-SYMBOLICS(vector ',name ,@slots)
	    #+SYMBOLICS(zl:fillarray (make-array ,(length slots)
						 :leader-length 1
						 :named-structure-symbol ',name)
				     (list ,@slots)))
	  (definline (,(symbol-append name '?) x)
		     #-SYMBOLICS(and (vector? x) (eq? (vector-ref x 0) ',name))
		     #+SYMBOLICS(and (typep x 'structure)
				     (eq? (zl:named-structure-symbol x) ',name)))
	  ,@(generate-accessors slots 0 ())
	  ,@(generate-modifiers slots 0 ())
	  (define (,(symbol-append 'PRINT- name) ,name) ,print-form)
	  (SCHEME::SET-STRUCTURE-PRINTER! ',name ,(symbol-append 'PRINT- name))))

#+SYMBOLICS
(DEFUN SCHEME::SET-STRUCTURE-PRINTER! (NAME PROC)
  (LISP:FLET
    ((handler (message &rest args)
       (lisp:case message
	 (:which-operations '(:which-operations :print-self))
	 (:print-self
	   (lisp:let ((*standard-output* (second args)))
	     (funcall proc (first args)))))))
    (setf (get name 'scl:named-structure-invoke) #'handler)))


;;;; Margin hacking procedures.

#+SYMBOLICS
(defun with-left-margin-procedure (margin proc &rest args)
  (LISP:let ((outer-output *standard-output*))
    (LISP:labels
      ((margin-terpri () (lisp:terpri outer-output) (funcall margin outer-output))
       (margin-string-out (string &optional (from 0) to)
	 (lisp:do ((index from (1+ index))
		   (to (or to (string-length string)))
		   (base from))
		  ((>= index to) (scl::send outer-output :string-out string base to))
	   (when (char= (char string index) #\cr)
	     (cond ((= index base) (margin-terpri))
		   (T (margin-output :line-out string base (1- index))
		      (lisp:setq base (1+ index)))))))
       (margin-output (message &rest args)
	 (cond ((eq message :newline) (margin-terpri))
	       ((and (eq message :tyo) (char= (first args) #\cr))
		(margin-terpri))
	       ((eq message :line-out)
		(lisp:apply #'scl:send outer-output :string-out args)
		(margin-terpri))
	       ((eq message :string-out) (apply #'margin-string-out args))
	       (T (apply outer-output message args)))))
      (lisp:let ((*standard-output* #'margin-output))
	(apply proc args)))))
(setq with-left-margin-procedure #'with-left-margin-procedure)

(defun with-left-margin (margin proc &rest args)
  (apply #'with-left-margin-procedure
	 #'(lisp:lambda () (princ margin))
	 proc args))
(setq with-left-margin #'with-left-margin)

(export '(with-left-margin-procedure with-left-margin))


;;;; Lookup operations.

(setq scheme::make-empty-lookup #'lisp:make-hash-table)
(setf (symbol-function 'scheme::make-empty-lookup) #'lisp:make-hash-table)

(setq scheme::do-lookup #'lisp:gethash)
(setf (symbol-function 'scheme::do-lookup) #'lisp:gethash)

(defun scheme::mutate-lookup! (key table fcn default)
  (setf (gethash key table) (funcall fcn (gethash key table default))))
(setq scheme::mutate-lookup! #'scheme::mutate-lookup!) 

(defun scheme::modify-lookup! (key table value)
  (setf (gethash key table) value))
(setq scheme::modify-lookup! #'scheme::modify-lookup!)

(setf (symbol-function 'scheme::over-lookup) #'lisp:maphash)
(setf scheme::over-lookup #'lisp:maphash)
