;;; -*- Mode: Scheme; Syntax: Scheme; Package: (SCHEME :USE (PSEUDOSCHEME)) -*-

(declare (usual-integrations))
(declare (integrate-external "/u/kwh/programs/utility/plus")
	 (integrate-external "/u/kwh/programs/utility/mutable")
	 (integrate-external "/u/kwh/programs/typical/kernel"))


;;;; Tracing focus events.

(define trace-focus-events #T)


;;;; Foci and potential foci.
;;; For control purposes, concepts/types are divided into two
;;; empirical classes of foci and potential foci.  There is also a
;;; class of faded foci which are used to limit some generation
;;; processes.  These classes divide the control structure into two
;;; processes: a concept generation process producing potential foci
;;; from known-to-be-interesting foci and an analysis process
;;; producing foci from potential foci.

(define potential-foci (type "Potential Foci" divided-collection lisp-objects))
(define foci (type "Foci" divided-collection potential-foci))
(define faded-foci (type "Faded Foci" empirical-collection potential-foci))

(define (potential-focus! x)
  (if trace-focus-events
      (if (not (in? x potential-foci))
	  (message $NL "*** Potential Focus: " x " might be interesting.")))
  (assert! x potential-foci))
(procedure! potential-focus! 'POTENTIAL-FOCUS! "Indexed as potentially interesting.")
(define (focus! x)
  (if trace-focus-events
      (if (not (in? x foci))
	  (message $NL "*** Focus: " x " is interesting!")))
  (assert! x potential-foci)
  (assert! x foci))
(procedure! focus! 'FOCUS! "Indexed as interesting!")


;;;; Foci transformations.

;;; This sets up a criterion for migration from potential foci into
;;; foci.  Essentially, any definition in FROM-SPACE which
;;; accidentally satisfies PROPERTY is declared a foci; furthermore,
;;; any potential foci in FROM-SPACE is hypothesized as satisfying
;;; PROPERTY, setting up confirmation/disconfirmation proceedings.
(define (focus-property! property from-space)
  (add-daemon! focus!
	       (<AND> property from-space potential-foci
		      (complement (given-instances property))))
  (add-daemon! (hypothesize property)
	       (<AND> potential-foci from-space
		      (empirical-specialization-of property))))


;;;; `Processes' (daemons and tasks combined)

(define (process! procedure domain-type)
  (add-daemon! procedure domain-type)
  (slow-map procedure (known-elements domain-type)))


;;;; Concept Generators

(define (foci-class class) (<AND> class foci))

;;; This creates a concept generator daemon (used the cached version CONCEPT-GENERATOR).
(define (make-concept-generator generator)
  (define (generate-from instance)
    (if (in? instance faded-foci) (%undefined)
	(potential-focus! (generator instance))))
  (procedure! generate-from ($procedure generator)
	      "Generate new potential foci by applying "
	      ($procedure generator) " to foci. ["
	      ($procedure-description generator) "]"))
(define concept-generator (simple-cache make-concept-generator))

;;; This creates a concept generator generator daemon (used the cached
;;; version GENERATOR-GENERATOR).  The generated daemon must have a
;;; domain and range declared so as to place it appropriately in the
;;; lattice. 
(define (make-generator-generator generator)
  (define (make-new-concept-generator from)
    (let ((new-generator (generator from)))
      (process! (concept-generator new-generator)
		(foci-class (function-domain new-generator)))))
  make-new-concept-generator)
(define generator-generator (simple-cache make-generator-generator))

;;; This generates new, ALWAYS interesting, concepts from foci.
;;;  Warning: This is potentially explosive.
(define (foci-generator generator)
  (define (generate-focus from-x) (focus! (generator from-x)))
  (procedure! generate-focus (list ($function generator) '-FOCUS!)
	      "Generate foci by applying " ($procedure-description generator)))






