;;; -*-Scheme-*-
;;;
;;;	$Header: binlyz.scm,v 1.1 87/10/16 16:23:37 GMT cph Exp $
;;;
;;;	Copyright (c) 1987 Massachusetts Institute of Technology
;;;
;;;	This material was developed by the Scheme project at the
;;;	Massachusetts Institute of Technology, Department of
;;;	Electrical Engineering and Computer Science.  Permission to
;;;	copy this software, to redistribute it, and to use it for any
;;;	purpose is granted, subject to the following restrictions and
;;;	understandings.
;;;
;;;	1. Any copy made of this software must include this copyright
;;;	notice in full.
;;;
;;;	2. Users of this software agree to make their best efforts (a)
;;;	to return to the MIT Scheme project any improvements or
;;;	extensions that they make, so that these may be included in
;;;	future releases; and (b) to inform MIT of noteworthy uses of
;;;	this software.
;;;
;;;	3. All materials developed as a consequence of the use of this
;;;	software shall duly acknowledge such use, in accordance with
;;;	the usual standards of acknowledging credit in academic
;;;	research.
;;;
;;;	4. MIT has made no warrantee or representation that the
;;;	operation of this software will be error-free, and MIT is
;;;	under no obligation to provide any services, by way of
;;;	maintenance, update, or otherwise.
;;;
;;;	5. In conjunction with products arising from the use of this
;;;	material, there shall be no use of the name of the
;;;	Massachusetts Institute of Technology nor of any adaptation
;;;	thereof in any advertising, promotional, or sales literature
;;;	without prior written consent from MIT in each case.
;;;

;;;; Binary File Analyzer

(declare (usual-integrations))

(set! generate-cref			;defined in some outer environment.
(named-lambda (generate-cref output-file input-files #!optional long-names?)
  (if (unassigned? long-names?) (set! long-names? false))
  (let ((usage-cache (make-usage-cache)))
    (let ((file-entries (analyze&intern usage-cache input-files long-names?)))
      (sort-entries (btree-fringe usage-cache)
	(lambda (primitive bound unbound defined multiple)
	  (newline)
	  (write-string "Write output file")
	  (fluid-let ((*transcript-port* (current-output-port)))
	    (with-output-to-file output-file
	      (lambda ()
		(write-header&entries "Unbound names:" unbound)
		(write-header&entries "Global names:" bound)
		(write-header&entries "Primitives:" primitive)
		(if (not (null? multiple))
		    (begin (write-string "Multiple definitions:")
			   (newline)
			   (format-definition-entries multiple)
			   (write-char #\Page)
			   (newline)))
		(format-file-entries file-entries)
		(with-output-to-port *transcript-port*
		  (lambda ()
		    (newline)
		    (write-string "Write definitions index")))
		(write-string "Index to definitions:")
		(newline)
		(format-definition-entries defined))))))))
  *the-non-printing-object*))

(define *transcript-port*)

(define (write-header&entries header entries)
  (if (not (null? entries))
      (begin (write-string header) (newline)
	     (format-reference-entries entries)
	     (write-char #\Page) (newline))))

(define (analyze&intern usage-cache filenames long-names?)
  (let ((pathnames (map ->pathname filenames)))
    (map (lambda (pathname name)
	   (intern-file! usage-cache (analyze-file pathname name)))
	 pathnames
	 (pathnames->names pathnames long-names?))))

(define (build-usage-cache filenames long-names?)
  (let ((usage-cache (make-usage-cache))
	(pathnames (map ->pathname filenames)))
    (for-each (lambda (pathname name)
		(intern-file! usage-cache (analyze-file pathname name)))
	      pathnames
	      (pathnames->names pathnames long-names?))
    usage-cache))

(define (pathnames->names pathnames long-names?)
  (if long-names?
      (map pathname->string pathnames)
      (let ((prefix (length (greatest-common-prefix pathnames))))
	(map (lambda (pathname)
	       (pathname-extract-string
		(pathname-new-directory
		 pathname
		 (list-tail (pathname-directory pathname) prefix))
		'DIRECTORY 'NAME))
	     pathnames))))

(define (greatest-common-prefix pathnames)
  (if (null? pathnames)
      '()
      (let ((prefix))
	(for-each (lambda (pathname)
		    (let ((directory (pathname-directory pathname)))
		      (set! prefix
			    (if (unassigned? prefix)
				directory
				(let common-prefix ((x prefix) (y directory))
				  (if (or (null? x)
					  (null? y)
					  (not (equal? (car x) (car y))))
				      '()
				      (cons (car x)
					    (common-prefix (cdr x)
							   (cdr y)))))))))
		  pathnames)
	prefix)))

(define analyze-file)
(define intern-file!)
(define file-entry-name)
(define file-entry-definitions)
(define file-entry-references)
(define sort-entries)
(define format-file-entries)
(define format-file-entry)
(define format-reference-entries)
(define format-definition-entries)
(define make-usage-cache)
(let ()

;;;; Analysis

(set! sort-entries
(named-lambda (sort-entries entries receiver)
  (define (loop entries primitives bound unbound defined multiple)
    (if (null? entries)
	(receiver (reverse! primitives)
		  (reverse! bound)
		  (reverse! unbound)
		  (reverse! defined)
		  (reverse! multiple))
	(let ((entry (car entries))
	      (entries (cdr entries)))
	  (cond ((null? (cache-definitions entry))
		 (let ((name (cache-name entry)))
		   (cond ((symbol? name)
			  (if (lexical-unbound? system-global-environment name)
			      (loop entries
				    primitives bound (cons entry unbound)
				    defined multiple)
			      (loop entries
				    primitives (cons entry bound) unbound
				    defined multiple)))
			 ((primitive-procedure? name)
			  (loop entries
				(cons entry primitives) bound unbound
				defined multiple))
			 (else (error "Bad name" name)))))
		((null? (cdr (cache-definitions entry)))
		 (loop entries
		       primitives bound unbound
		       (cons entry defined) multiple))
		(else
		 (loop entries
		       primitives bound unbound
		       defined (cons entry multiple)))))))
  (loop entries '() '() '() '() '())))

(set! analyze-file
(named-lambda (analyze-file pathname name)
  (let ((expression (fasload pathname)))
    (scan-defines (if (comment? expression)
		      (comment-expression expression)
		      expression)
      (lambda (defined declarations expression)
	(vector name
		(eq?-difference (free-expression expression) defined)
		defined))))))

(define file-tag vector-first)
(define file-free vector-second)
(define file-defined vector-third)

(set! intern-file!
(named-lambda (intern-file! usage-cache analyzed-file)
  (let ((file-name (file-tag analyzed-file)))
    (newline)
    (write-string "Process definitions: ")
    (write-string file-name)
    (let ((definitions
	   (map (lambda (name)
		  (let ((cache-entry (insert-name! usage-cache name)))
		    (add-cache-definition! cache-entry file-name)
		    cache-entry))
		(file-defined analyzed-file))))
      (newline)
      (write-string "Process references: ")
      (write-string file-name)
      (let ((references
	     (map (lambda (name)
		    (let ((cache-entry (insert-name! usage-cache name)))
		      (add-cache-reference! cache-entry file-name)
		      cache-entry))
		  (file-free analyzed-file))))
	(vector file-name definitions references))))))

(set! file-entry-name vector-first)
(set! file-entry-definitions vector-second)
(set! file-entry-references vector-third)

;;;; Formatting

(define maximum-offset 80)

(set! format-file-entries
(named-lambda (format-file-entries entries)
  (if (not (null? entries))
      (let loop ((entries entries))
	(with-output-to-port *transcript-port*
	 (lambda ()
	   (newline)
	   (write-string "Write references: ")
	   (write-string (file-entry-name (car entries)))))
	(format-file-entry (car entries))
	(write-char #\Page)
	(newline)
	(if (not (null? (cdr entries)))
	    (loop (cdr entries)))))))

(set! format-file-entry
(named-lambda (format-file-entry file-entry)
  (write-string "Symbols defined in " )
  (write-string (file-entry-name file-entry))
  (write-string ":")
  (newline)
  (format-reference-entries (file-entry-definitions file-entry))))

(define (format-cache-entries cache-references m)
  (define (format-cache-entry cache-entry)
    (let ((name (cache-name cache-entry))
	  (references (cache-references cache-entry)))
      (let ((tag (string-append (write-to-string name) " ")))
	(write-string tag)
	(if (null? references)
	    (write-string m)
	    (let loop ((offset (+ (string-length tag) 1))
		       (references references))
	      (if (not (null? references))
		  (loop (let ((s (string-length (car references))))
			  (let ((new-offset (+ s offset 3)))
			    (if (< new-offset maximum-offset)
				(begin (write-string " ")
				       (write (car references))
				       new-offset)
				(begin (newline)
				       (write-string "    ")
				       (write (car references))
				       (+ s 6)))))
			(cdr references))))))
      (newline)))
  (lambda (cache-entries)
    (newline)
    (for-each format-cache-entry
	      (sort cache-entries cache-name-less?))))

(define (cache-name-less? x y)
  (name-less? (cache-name x) (cache-name y)))

;;;; Usage Cache

(set! make-usage-cache
(named-lambda (make-usage-cache)
  (make-btree name-less?
	      (lambda (name)
		(cons name
		      (cons (make-btree string<?
					identity-procedure
					identity-procedure)
			    (make-btree string<?
					identity-procedure
					identity-procedure))))
	      car)))

(define (name-less? x y)
  (cond ((symbol? x)
	 (cond ((symbol? y) (symbol-less? x y))
	       (else false)))
	((primitive-procedure? x)
	 (cond ((primitive-procedure? y)
		(symbol-less? (primitive-procedure-name x)
			      (primitive-procedure-name y)))
	       (else true)))
	(else true)))

(define (symbol-less? x y)
  (string<? (symbol->string x)
	    (symbol->string y)))

(define (btree-insert*! btree key)
  (btree-insert! btree key identity-procedure identity-procedure))

(define (insert-name! usage-cache name)
  (btree-insert*! usage-cache name))

(define cache-name car)

(define (add-cache-definition! cache-entry definition)
  (btree-insert*! (cadr cache-entry) definition))

(define (cache-definitions cache-entry)
  (btree-fringe (cadr cache-entry)))

(define (add-cache-reference! cache-entry reference)
  (btree-insert*! (cddr cache-entry) reference))

(define (cache-references cache-entry)
  (btree-fringe (cddr cache-entry)))

(set! format-reference-entries
  (format-cache-entries cache-references " no references"))

(set! format-definition-entries
  (format-cache-entries cache-definitions " no definitions"))

;;; end LET
)