/* %W% %G% */

#ifndef _LIST_H
#define _LIST_H

#include "stuff.h"
#include "memory.h"

/* lists are represented by a pointer to a pointer of list elements */
typedef struct listnode_s {
    Ptr item; /* item to hold in list */
    struct listnode_s *next; /* pointer to next element in list */
} listnode_t, *list_t;

typedef list_t *List;

/* list memory allocator struct */
extern Mem listAlloc;

/* list iterator */

typedef list_t listIter;
/* initialize listIter */
#define listIterInit(list, iter) ((iter) = *(list))
/* increment listIter */
#define listIterNext(iter) ((iter) = (iter)->next)
/* reference to current item in iter */
#define listIterRef(iter) ((iter)->item)

/* list walking macros */

/* EXPR is evaluated for each item in LIST (known locally as ITEM) and may
 * change the value of ITEM which is written back into the list. EXPR may
 * also break or continue the for loop but the actual item in the node, in
 * this case remains unmodifed. */
#define LIST_TO_EACH(LIST, ITEM, EXPR) \
{ \
    list_t _cur_; \
    for (_cur_ = *(LIST); _cur_; _cur_ = _cur_->next) { \
	Ptr ITEM = _cur_->item; \
	EXPR; _cur_->item = ITEM; \
    } \
}

/* EXPR is evaluated for each item in LIST (known locally as ITEM).
 * EXPR may also break or continue the for loop. */
#define LIST_FOR_EACH(LIST, ITEM, EXPR) \
{ \
    list_t _cur_; \
    for (_cur_ = *(LIST); _cur_; _cur_ = _cur_->next) { \
	Ptr ITEM = _cur_->item; \
	EXPR; \
    } \
}

/* TRUE if list is empty */
#define listEmpty(list) (*(list) == (list_t)NULL)

#ifdef __STDC__

/* Initialize the memory allocator (to have nblocks list nodes) used by
 * the list routines.  The allocator, listAlloc, is global. You can
 * share the allocator with other data types or use another type's
 * allocator by setting listAlloc appropriately (in this case, do not
 * call listInit). */
void listInit(int nblocks);

/* Return an empty list */
extern List listNew(void);

/* Add item to the front of l */
extern void listPush(Ptr item, List l);

/* Add item to the front of l and return a pointer to item */
extern Ptr *listRef(Ptr item, List l);

/* Return the first item in l or NULL if l is empty. */
extern Ptr listPeek(List l);

/* Remove the first item in the list and return it or NULL if l is empty. */
extern Ptr listPop(List l);

/* return the list starting with the next item in l or NULL if l has one or
 * less items. */
extern List listNext(List l);

/* Apply f with (item, arg2) to each item in l */
extern void listApply(List l, F_VOID f, Ptr arg2);

/* Apply f with (item) to each item in l */
extern void listApply1(List l, F_VOID f);
					   
/* Empty l and if f is not NULL free all items with it. */
extern void listClear(List l, F_VOID f);

/* Free all memory used by l and if f is not NULL free all items with it. */
extern void listFree(List l, F_VOID f);

/* Find the sublist in l starting with the first item that satisfies
 * (*test)(item, arg2).  Return the sublist or NULL if test fails on all
 * items. If test is NULL, == is used. */
extern List listSublist(List l, F_BOOLEAN test, Ptr arg2);

/* Find the item in l that satisfies (*test)(item, arg2) or NULL if test
 * fails on all items. If test is NULL, == is used. */
extern Ptr listFind(List l, F_BOOLEAN test, Ptr arg2);

/* Remove the first item in l that satisfies (*test)(item, arg2).  If
 * the item is not found, nothing happens. If test is NULL, == is used. */
extern void listDelete(List l, F_BOOLEAN test, Ptr arg2);

/* Add obj before the first item in l satisfying (*sort)(obj, item) or
 * at the end if no such items exist. */
extern void listAddSorted(Ptr obj, List l, F_BOOLEAN sort);

/* Return the number of items in l */
extern int listLength(List l);

/* Destructively reverse the order of items in l */
extern void listReverse(List l);

/* Sort the items in l so that (*comp)(item, next item) is always TRUE. */
extern void listSort(List l, F_BOOLEAN comp);

#else

extern void listInit();
extern List listNew();
extern void listPush();
extern Ptr *listRef();
extern Ptr listPop();
extern Ptr listPeek();
extern List listNext();
extern void listApply1();
extern void listApply();
extern void listClear();
extern void listFree();
extern List listSublist();
extern Ptr listFind();
extern void listDelete();
extern void listAddSorted();
extern int listLength();
extern void listReverse();
extern void listSort();

#endif /* defined(__STDC__) */

#endif /* defined(_LIST_H) */
