;;; -*- Scheme -*-

(declare (usual-integrations))

#|
This file contains procedures for drawing towns, streets, and block
groups.  The top level procedures reference the global variables
*GRAPHICS-WINDOW*
*CENSUS-DATA*
*GEOGRAPHIC-DATA*
*NAMED-STREETS*
*BOUNDARY-SEGMENTS*

which are defined in the setup file.

Major procedures are

DRAW-TOWN
SHADE-BLKGRP <block> <color-function>

MOUSE-SELECT-BLOCK
MOUSE-LABEL-STREET

FLASH-STREET <street-name>

|#

;;; restore the town from the saved region

(define (draw-town)
  (graphics-clear *graphics-window*)
  (restore-screen-to-graphics-device *graphics-window* *graphics-save-region*))

;;; Draw a town, scaled to fit in a given window.

(define (initial-draw-town)
  (let ((town *geographic-data*)
	(window *graphics-window*))
    (let ((min-lat (town/min-lat town))
	  (max-lat (town/max-lat town))
	  (min-long (town/min-long town))
	  (max-long (town/max-long town))
	  (pairs-list (town/streets town)))
      (set-default-coordinate-function! window
					min-long min-lat max-long max-lat)
      (draw-streets window pairs-list default-color-function)
      (copy-screen-to-saved-region *graphics-window* *graphics-save-region*)
    'done)))

(define (draw-streets window street-list color-function)
  (for-each
   (lambda (x)
     (let ((color (color-function x)))
       (if color
	   (begin
	     (graphics-operation window 'set-foreground-color color)
	     (graphics-draw-line window
				 (street/from-long x) (street/from-lat x)
				 (street/to-long x) (street/to-lat x))))))
   street-list))


(define x-graphics-copy-area
  (make-primitive-procedure 'x-graphics-copy-area))

(define x-graphics-device/xw
  (access x-graphics-device/xw (->environment '(runtime x-graphics))))

; save a bit rectangle from X-DEVICE into SAVE-DEVICE 
(define (copy-screen-to-saved-region x-device save-device)
  (with-values
      (lambda () (graphics-coordinate-limits x-device))
    (lambda (x-left y-bottom x-right y-top)
					; setup same vitual coordinate limits
      (graphics-set-coordinate-limits save-device x-left y-bottom x-right y-top)
					; copy the screen
      (x-graphics-copy-area
       (x-graphics-device/xw x-device)
       (x-graphics-device/xw save-device)
       (exact->inexact x-left)
       (exact->inexact y-top)
       (exact->inexact (- x-left x-right))
       (exact->inexact (- y-top y-bottom))
       (exact->inexact x-left)
       (exact->inexact y-top))
      (graphics-flush save-device))))


; restore a bit rectangle from SAVE-DEVICE to X-DEVICE:
(define (restore-screen-to-graphics-device x-device save-device)
  (copy-screen-to-saved-region save-device x-device))


;;;; Color instructions

;;; SHOW-BLOCKGROUPS specifies that blockgroup boundaries will be in
;;; black, other streets in grey. 

(define (show-blockgroups street-segment) ; street->color
  (cond ((blkgrp-boundary? street-segment) "black")
	((road? street-segment) "gray75")	
	(else #f)))

;;; Alternative to SHOW-BLOCKGROUPS

(define (show-blockgroups-and-other street-segment) ; street->color
  (cond ((blkgrp-boundary? street-segment) "black")
	((really-major-road? street-segment) "gray25")
	((major-road? street-segment) "gray50")
	((neighborhood-road? street-segment) "gray75")
	((railroad? street-segment) "gray25")
	((water? street-segment) "gray50")
	(else #f)))

;;; Another alternative to SHOW-BLOCKGROUPS

(define show-streets
  (let ((color #f)) ; kludge, code should figure this out for itself
    (if color
	(lambda (street-segment)
	  (cond ((really-major-road? street-segment) "green")
		((major-road? street-segment) "green")
		((neighborhood-road? street-segment) "grey")
		((water? street-segment) "blue")
		((railroad? street-segment) "red")
		(else "white")))
	(lambda (street-segment)
	  (cond ((really-major-road? street-segment) "gray25")
		((major-road? street-segment) "gray50")
		((neighborhood-road? street-segment) "gray75")
		((water? street-segment) "gray50")
		((railroad? street-segment) "gray25")
		(else #f))))))

(define default-color-function show-blockgroups-and-other)


(define make-point list)
(define point-xcor car)
(define point-ycor cadr)

;;;; COORDINATE FUNCTIONS

;;; standard coordinate function scales things to fit the window,
;;; based upon the max and min extent of the streets

(define (set-standard-coordinate-function! window min-long min-lat max-long max-lat)
  (let ((lat-diff (- max-lat min-lat))
	(long-diff (- max-long min-long)))
    (if (> lat-diff long-diff)
	(graphics-set-coordinate-limits 
	 window (- min-long (/ (- lat-diff long-diff) 2.0))
	 min-lat (+ max-long (/ (- lat-diff long-diff) 2.0)) max-lat)
	(graphics-set-coordinate-limits 
	 window
	 min-long (- min-lat (/ (- long-diff lat-diff) 2.0))
	 max-long (+ max-lat (/ (- long-diff lat-diff) 2.0)))
	)))

;;; another choice for a coordinate function is to create a "section"
;;; that expands a sub-part of the town to fill an entire window

(define (make-section-coord-function llx lly urx ury)
  (lambda (window min-long min-lat max-long max-lat)
    (let ((lat-diff (- max-lat min-lat))
	  (long-diff (- max-long min-long))
	  (llxr (* 1. min-long))
	  (llyr (* 1. min-lat))
	  (urxr (* 1. max-long))
	  (uryr (* 1. max-lat)))
      (if (> lat-diff long-diff)
	  (begin
	    (set! llxr (- min-long (/ (- lat-diff long-diff) 2.0)))
	    (set! urxr (+ max-long (/ (- lat-diff long-diff) 2.0))))
	  (begin
	    (set! llyr (- min-lat (/ (- long-diff lat-diff) 2.0)))
	    (set! uryr (+ max-lat (/ (- long-diff lat-diff) 2.0)))))
      (let ((f1 (round->exact (+ llxr (* llx (- urxr llxr)))))
	    (f2 (round->exact (+ llyr (* lly (- uryr llyr)))))
	    (f3 (round->exact (+ llxr (* urx (- urxr llxr)))))
	    (f4 (round->exact (+ llyr (* ury (- uryr llyr))))))
	(graphics-set-coordinate-limits window f1 f2 f3 f4)))))


(define set-default-coordinate-function! set-standard-coordinate-function!)


;;;; Flash a street with a given name

(define flash-delay 2000)
(define flash-iterations 10)

(define (flash-street street-name)
  (flash *graphics-window*
	 street-name
	 default-color-function))

(define (flash window name color-function)
  (let ((the-streets (find-street name (town/streets *geographic-data*))))
    (define (iter num-of-times)
      (if (= num-of-times 0)
	  'done
	  (begin 
	    (draw-streets window the-streets (lambda (x) "black")) 
	    (delay-loop flash-delay)
	    (draw-streets window the-streets (lambda (x) "gray75")) 
	    (delay-loop flash-delay)
	    (iter (-1+ num-of-times)))))
    (if (null? the-streets)
	(string-append "Street not found: " name)
	(begin (iter flash-iterations)
	       (draw-streets window the-streets color-function)))))

(define (find-street name list-of-streets)
  (list-transform-positive 
      list-of-streets
    (lambda (record) 
      (string-prefix? name (street/name record)))))

(define (delay-loop count)
  (if (< count 0)
      'done
      (delay-loop (- count 1))))

;;;; Find the street closest to the mouse position, and label it.

;;;Find closest street to mouse position.  Use named streets only.

(define (mouse-label-street)
  ;;memoize computing the named streets
  (if (not *named-streets*)
      (set! *named-streets*
	    (remove-unnamed-streets
	     (town/streets *geographic-data*))))
  (let ((position (map floor->exact (mouse-x-y *graphics-window*))))
    (label-street (find-nearest-street position *named-streets*))))

(define (label-street the-street)
  (let ((name (street/name the-street))
	(point (midpoint the-street)))
    (graphics-operation *graphics-window*
			'set-foreground-color "black")
    (graphics-draw-text *graphics-window*
			(point-xcor point) (point-ycor point) name)))

;;;

(define standard-mouse-shape 34)
(define requesting-input-mouse-shape 29)

(define (mouse-x-y window)
  (x-graphics/discard-events window)
  (graphics-operation window 'set-mouse-shape requesting-input-mouse-shape)
  (let ((thepoint (map floor->exact ((x-graphics/read-button window) list))))
    (graphics-operation window 'set-mouse-shape standard-mouse-shape)
    (make-point (car thepoint) (cadr thepoint))))

(define (remove-unnamed-streets list-of-streets)
  (list-transform-negative list-of-streets
    (lambda (record)
      (equal? "" (street/name record)))))


(define (midpoint the-street)
  (make-point
   (/ (+ (street/from-long the-street) (street/to-long the-street)) 2.0)
   (/ (+ (street/from-lat the-street) (street/to-lat the-street)) 2.0)))

;;; find the street whose midpoint is closest to the indicated
;;; position.

(define (find-nearest-street position list-of-streets)
  (define (dist^2 p1 p2)
    (let ((xd (- (point-xcor p2) (point-xcor p1)))
	  (yd (- (point-ycor p2) (point-ycor p1))))
      (+ (* xd xd) (* yd yd))))
  (let loop ((the-list (cdr list-of-streets))
	     (min-street (car list-of-streets))
	     (min-dist (dist^2 position (midpoint (car list-of-streets)))))
    (if (null? the-list)
	min-street
	(let ((current (car the-list))
	      (rest (cdr the-list)))
	  (if (< (dist^2 position (midpoint current)) min-dist)
	      (loop rest current (dist^2 position (midpoint current)))
	      (loop rest min-street min-dist))))))


;;;; Shade a census block (indicated by a pop-data structure)
;;;; Does it by XOR so underlying map is preserved.

(define (shade-blkgrp pop-data shading-color)
  (let ((polygon (pop-data/blkgrp-boundary pop-data)))
    (let ((poly-vec #f))
      (let loop ((s-list polygon)
		 (p-list '()))
	(if (null? s-list)
	    (set! poly-vec (list->vector (reverse p-list)))
	    (let ((s1 (car s-list))
		  (s2 (cdr s-list)))
	      (loop s2 (cons (cdr s1) (cons (car s1) p-list))))))
      (graphics-operation *graphics-window*
			  'set-foreground-color shading-color)
      (graphics-bind-drawing-mode
       *graphics-window*
       11
       (lambda ()
	 (graphics-operation *graphics-window* 'fill-polygon poly-vec))))))

;;; Locate a block group based on the mouse position
;;; If no block found then returns (arbitrarily) the first census block.

(define (mouse-select-block)
  (let* ((point (mouse-x-y *graphics-window*))
	 (bs (locate-block-boundary-street (point-xcor point) (point-ycor point))))
    (if (null? bs)
	(car *census-data*)
	(let ((pd (street->pop-data bs)))
	  (if (null? pd)
	      (car *census-data*)
	      pd)))))

(define (street->pop-data street)
  (let* ((baz (real-right street))
	 (block (car baz))
	 (tract (cdr baz)))
    (list-search-positive *census-data*
      (lambda (pop-data)
	(and (= block (pop-data/blkgrp pop-data))
	     (= tract (pop-data/tract pop-data)))))))

;;; midpoint approximation made here
;;; should throw out horizontal lines but doesn't
;;; does not seem to matter much

;;;memoizes boundary-segments.  compute this when initializing data

(define (locate-block-boundary-street long lat)
  (if (not *boundary-segments*)
      (set! *boundary-segments*
	    (list-transform-positive (town/streets *geographic-data*)
	      blkgrp-boundary?)))
  (let* ((horiz (list-transform-positive
		    *boundary-segments*
		  (lambda (x) 
		    (between lat (street/from-lat x) (street/to-lat x)))))
	 (sorted (sort horiz
		       (lambda (x y)
			 (> (point-xcor (midpoint x))
			    (point-xcor (midpoint y)))))))
    (list-search-positive sorted
      (lambda (y) (> long (point-xcor (midpoint y)))))))


(define (between value point1 point2)
  (or (> point2 value point1)
      (> point1 value point2)))

;;; pick the right hand blkgrp and track even when the vector points down
(define (real-right street)
  (if (> (street/from-lat street)
	 (street/to-lat street))
      (cons (street/blkgrp-l street)
	    (street/tract-l street))
      (cons (street/blkgrp-r street)
	    (street/tract-r street))))




