
% OL(P): Object Layer for Prolog -- system
% Version 1.1 for SICStus Prolog and QUINTUS Prolog
% (C) 1993 Markus P.J. Fromherz.  All Rights Reserved.
% (C) 1993 Xerox Corporation.     All Rights Reserved.


% ol_compile_resource(+File, +Options) -- compile resource File with Options,
%    as in ol_compile_resource(resource, [verbose])

ol_compile_resource(File, Options) :-
   % resource.rc -> resource.pl
   name(File, FileString),
   ol_append(FileString, ".rc", InFileString),
   ol_append(FileString, ".pl", OutFileString),
   name(InFile, InFileString),
   name(OutFile, OutFileString),
   ( ol_member(verbose, Options) ->
        format("compiling ~w -> ~w~n", [InFile,OutFile])
      ;
        true
   ),
   open(InFile, read, InStream),
   open(OutFile, write, OutStream),
   ol_version(V),
   format(OutStream, "% OL(P) ~w~n", [V]),
   format(OutStream, "% Source: ~w~2n", [InFile]),
   repeat,
     read(InStream, InStat),
     ol_compile_rc_statement(InStat, OutStream),
   !,
   ( ol_prolog(Prolog),
     ol_meta_predicate(DProlog, BL, MetaPredicate),
     DProlog = Prolog,
     ol_expand_meta_predicate(MetaPredicate, BL, CompileClause),
     portray_clause(OutStream, CompileClause),
     fail
    ;
     true
   ),
   close(InStream),
   close(OutStream).


% see resource file (resource.rc) for these predicates
:- dynamic ol_prolog/1, ol_meta_predicate/3.


% ol_compile_rc_statement(+InStat, +OutStream) -- compile statement InStat
%    (ol_meta_predicate/3 is expanded, while the rest is mostly just moved
%    from InStream to OutStream)
ol_compile_rc_statement(end_of_file, _).
ol_compile_rc_statement((:-op(P,T,N)), OutStream) :-
   op(P, T, N),
   portray_clause(OutStream, (:-op(P,T,N))),
   !, fail.
ol_compile_rc_statement(ol_prolog(Prolog), OutStream) :-
   retractall(ol_prolog(_)), assert(ol_prolog(Prolog)),
   retractall(ol_meta_predicate(_,_,_)),
   portray_clause(OutStream, (:- abolish(ol_prolog/1))),
   portray_clause(OutStream, ol_prolog(Prolog)),
   !, fail.
ol_compile_rc_statement(ol_meta_predicate(Prolog,BL,MetaPredicate), _) :-
   assert(ol_meta_predicate(Prolog,BL,MetaPredicate)),
   !, fail.
ol_compile_rc_statement((ol_meta_predicate(Prolog,BL,MetaPredicate):-B), _) :-
   assert((ol_meta_predicate(Prolog,BL,MetaPredicate):-B)),
   !, fail.
ol_compile_rc_statement(Statement, OutStream) :-
   portray_clause(OutStream, Statement),
   fail.


% ol_expand_meta_predicate(+Term, +BaseLanguage, -CompileClause) --
%    construct a compiling clause ol_compile_goals/6 for specification Term,
%    for each for Term, ::Term, and O::Term
ol_expand_meta_predicate(T, BL,
                         (ol_compile_goals(OG,BL,This,Self,Inst,CG):-!,CompileBody)) :-
   T =.. [N|As],
   ( OG=G, O=none ; OG= ::G, O=self ; OG=O::G ),
   ol_expand_meta_predicate_list(As, O, Ps, BL, This, Self, Inst, CPs, CompileGoals),
   G =.. [N|Ps],
   CG =.. [N|CPs],
   ol_list_conj(CompileGoals, CompileBody).

% ol_expand_meta_predicate_list(+As, -Ps, +BL, ?This, ?Self, ?Inst, -CPs, -CompileGoals) --
%    arg.s As of Term, Ps of G, and CPs of CG, where G is the goal to be
%    compiled to CG according to the specification of Term (cf. above);
%    CompileGoals is the list of goals that do the compilation, This and
%    Self are parameters for the pseudo-variables
ol_expand_meta_predicate_list([], _, [], _, _, _, _, [], []).
ol_expand_meta_predicate_list([::|As], O, [G|Gs], BL, T, S, I, [CG|CGs],
                              [ol_compile_body(G,BL,T,S,I,CG)|CompileGoals]) :-
   O == none,
   !,
   ol_expand_meta_predicate_list(As, O, Gs, BL, T, S, I, CGs, CompileGoals).
ol_expand_meta_predicate_list([::|As], O, [G|Gs], BL, T, S, I, [CG|CGs],
                              [ol_compile_body(::G,BL,T,S,I,CG)|CompileGoals]) :-
   O == self,
   !,
   ol_expand_meta_predicate_list(As, O, Gs, BL, T, S, I, CGs, CompileGoals).
ol_expand_meta_predicate_list([::|As], O, [G|Gs], BL, T, S, I, [CG|CGs],
                              [ol_compile_body(O::G,BL,T,S,I,CG)|CompileGoals]) :-
   !,
   ol_expand_meta_predicate_list(As, O, Gs, BL, T, S, I, CGs, CompileGoals).
ol_expand_meta_predicate_list([-|As], O, [P|Gs], BL, T, S, I, [P|CGs], CompileGoals) :-
   !,
   ol_expand_meta_predicate_list(As, O, Gs, BL, T, S, I, CGs, CompileGoals).
ol_expand_meta_predicate_list([A|As], O, [G|Gs], BL, T, S, I, [CG|CGs], CompileGoals) :-
   A =.. [N|As1],
   ol_expand_meta_predicate_list(As1, O, Ps, BL, T, S, I, CPs, CompileGoals1),
   G =.. [N|Ps],
   CG =.. [N|CPs],
   ol_expand_meta_predicate_list(As, O, Gs, BL, T, S, I, CGs, CompileGoals2),
   ol_append(CompileGoals1, CompileGoals2, CompileGoals).


% ol_project(Project) -- the main Project
% ol_project_files(Project, Files) -- a Project (without path) and its Files
% ol_project_path(Project, PFString, FileString) -- PFString is the full file
%    path name if FileString is instantiated (cf. ol_path_project)
:- multifile ol_project_files/2, ol_project_path/3.
:- dynamic ol_project/1, ol_project_files/2, ol_project_path/3.

ol_objects([(object),ol,project]).

% ol_new_project(+Project, +Files) -- initialize new Project with Files
ol_new_project(Project, Files) :-
   atom(Project),
   ol_list(Files),
   \+ ( ol_member(File, Files), \+ atom(File) ),
   ol_path_project(Project, ProjectName, PFString, FileString),
   retractall(ol_project(_)),
   assert(ol_project(ProjectName)),
   retractall(ol_project_files(_,_)),
   assert(ol_project_files(ProjectName, Files)),
   retractall(ol_project_path(_,_,_)),
   assert(ol_project_path(ProjectName, PFString, FileString)),
   % retract all object data not pertaining to the system objects
   ( ProjectName == ol_project ->
        ol_clear_object_info(_)
      ;
        ol_clear_user_object_info(_)
   ),
   ol_save_project_info(ProjectName, [verbose]).

% ol_rename_project(+NewProject) -- rename current project to NewProject
ol_rename_project(NewProject) :-
   atom(NewProject),
   retract(ol_project(Project)),
   assert(ol_project(NewProject)),
   retract(ol_project_files(Project, Files)),
   assert(ol_project_files(NewProject, Files)),
   retract(ol_project_path(Project, PFString, FileString)),
   assert(ol_project_path(NewProject, PFString, FileString)),
   \+ ( retract(ol_object(Object, Project, File)),
        \+ assert(ol_object(Object, NewProject, File)) ),
   ol_save_project_info(NewProject, [verbose]),
   name(Project, FileString),   % note: variable used above
   ol_append(PFString, ".pl", ProjectFileString),
   ol_append("rm ", ProjectFileString, RemoveOldProjectString),
   name(RemoveOldProject, RemoveOldProjectString),
   unix(shell(RemoveOldProject)).

% ol_add_project_files(+NewFiles) -- add NewFiles to project
ol_add_project_files(NewFiles) :-
   ol_project(Project),
   ol_list(NewFiles),
   \+ ( ol_member(File, NewFiles), \+ atom(File) ),
   retract(ol_project_files(Project,Files)),
   !,
   ol_append(Files, NewFiles, AllFiles),
   assert(ol_project_files(Project, AllFiles)),
   ol_save_project_info(Project, [verbose]).

% ol_remove_project_files(OldFiles) --
ol_remove_project_files(OldFiles) :-
   ol_project(Project),
   retract(ol_project_files(Project,Files)),
   !,
   ol_remove_all(OldFiles, Files, AllFiles),
   assert(ol_project_files(Project, AllFiles)),
   \+ ( ol_member(File, OldFiles),
        ol_object(O, Project, File),
        \+ ol_clear_object_info(O) ),
   ol_save_project_info(Project, [verbose]).


% ol_add_library(+LibProject) -- add LibProject as library
ol_add_library(LibProject) :-
   ol_project(Project),
   ol_append_suffix(LibProject, ".pl", LibProjectFile),
   ( ( predicate_property(absolute_file_name(_,_,_), built_in) ->
          absolute_file_name(LibProjectFile,[access(read)],_)
        ;
          unix(access(LibProjectFile, 0)) 
     ) ->
        true
      ;
        format(" * library project file ~w doesn't exist", [LibProjectFile]),
        fail
   ),
   ol_path_project(LibProject, LibProjectName, PFString, FileString),
   consult(LibProjectFile),
   retractall(ol_project_path(LibProjectName,_,_)),
   assert(ol_project_path(LibProjectName, PFString, FileString)),
   ol_save_project_info(Project, [verbose]).

% ol_remove_library(+LibProject) -- remove library LibProject
ol_remove_library(LibProject) :-
   ol_project(Project),
   retractall(ol_project_files(LibProject,_)),
   retractall(ol_project_path(LibProject,_,_)),
   ol_clear_user_object_info(LibProject),
   ol_save_project_info(Project, [verbose]).


% ol_switch_project(+Project) -- switch to Project (i.e. get its data);
%    Project is assumed to be the project name with path
ol_switch_project(Project) :-
   ol_append_suffix(Project, ".pl", ProjectFile),
   ( ( predicate_property(absolute_file_name(_,_,_), built_in) ->
          absolute_file_name(ProjectFile,[access(read)],_)
        ;
          unix(access(ProjectFile, 0))
     ) ->
        true
      ;
        format(" * project file ~w doesn't exist", [ProjectFile]),
        fail
   ),
   ol_clear_user_object_info(_),
   ol_path_project(Project, ProjectName, PFString, FileString),
   retractall(ol_project(_)),
   assert(ol_project(ProjectName)),
   retractall(ol_project_files(_,_)),
   retractall(ol_project_path(_,_,_)),
   consult(ProjectFile),
   ol_syntax(ol_project_files(_,_),
             true,
             (format("** ~w is not a project file - abort", [Project]), abort)),
   retractall(ol_project_path(ProjectName,_,_)),
   assert(ol_project_path(ProjectName, PFString, FileString)).

ol_clear_user_object_info(Project) :-
   ol_objects(SystemObjects),
   \+ ( ol_object(O, Project, _), \+ ol_member(O, SystemObjects),
        \+ ( retractall(ol_object(O,_,_)),
             retractall(ol_super(O,_)),
             retractall(ol_state(O,_)),
             retractall(ol_unfold(O,_,_)),
             retractall(ol_instance(O,_)),
             retractall(ol_info(O,_,_)),
             ol_interface_head(O, _, _, _, _, IH),
             retractall(IH) ) ).

ol_clear_object_info(O) :-
   retractall(ol_object(O,_,_)),
   retractall(ol_super(O,_)),
   retractall(ol_state(O,_)),
   retractall(ol_unfold(O,_,_)),
   retractall(ol_instance(O,_)),
   retractall(ol_info(O,_,_)),
   ol_interface_head(O, _, _, _, _, IH),
   retractall(IH).


% ol_do_project(+Action, +Project, +WhichFiles, ?BaseLanguage, ?Extension) --
%    switch to Project (if necessary) and do Action to all its files (if
%    WhichFiles is 'all') or all changed files (if Whichfiles is 'new');
%    Action might be consult, fcompile, load, translate etc. (arity 1);
%    Project is assumed to be the project name with path or a variable; in
%    the latter case, the current project and its libraries are affected;
%    only the files in BaseLanguage are taken (can be a variable to take all);
%    if Extension is a variable or [BaseLanguage], the actual file listed
%    in ol_project_files/2 is taken; otherwise, Extension must be a list
%    of atoms which will be concatenated and replace the file's extension
%    BaseLanguage
ol_do_project(Action, Project, WhichFiles, BL, Ext) :-
   ol_member(WhichFiles, [all,new]),
   Ext = [_|_],
   ( \+ \+ ol_project_files(Project, _) ->
        ProjectName = Project     % project or library currently loaded
      ;
        format("Switch to project ~w first? (y/n) ", [Project]),
        read(y),
        ol_switch_project(Project),
        ol_project(ProjectName)
   ),
   ol_project(CurrentProject),
   % load libraries first, then main project, in case of variable
   ol_do_project_cond(Action, ProjectName, WhichFiles, BL, Ext, ProjectName\==CurrentProject),
   ol_do_project_cond(Action, ProjectName, WhichFiles, BL, Ext, ProjectName==CurrentProject).

ol_do_project_cond(Action, ProjectName, WhichFiles, BL, Ext, Condition) :-
   FileAction =.. [Action,PathFile],
   \+ ( ol_project_files(ProjectName, Files),
        Condition,
        ol_project_path(ProjectName, PFString, FileString),
        ol_member(File, Files),
          name(File, FString), ol_base_language(FString, FRString, BL),
          ( ol_changed_file(File) -> true ; WhichFiles == all ),
        ( (Action==consult;Action==load) ->
             retractall(ol_changed_file(File))
            ;
             true
        ),
        ( Ext = [BL] -> FileName = File ; ol_append_extension(FRString, Ext, FileName) ),
        ol_path_file(FileName, PFString, FileString, PathFile),
        \+ FileAction ).
   

% ol_path_project(+Project, -ProjectName, -PFString, ?FileString) --
%    extract path to Project (PFString is the path string, with the tail
%    pointed to by FileString)
ol_path_project(Project, ProjectName, PFString, FileString) :-
   name(Project, ProjectString),
   ( (ol_append(PathString, [47|ProjectNameString], ProjectString),
      \+ ol_member(47, ProjectNameString)) ->             % /
        name(ProjectName, ProjectNameString),
        ol_append(PathString, [47|FileString], PFString)
      ;
        ProjectName = Project,
        PFString = FileString
   ).

% ol_path_file(File, PFString, FileString, PathFile) -- PathFile is File with
%    the project's path added (PFString is the path string, with the tail
%    pointed to by FileString; cf. ol_path_project/4)
ol_path_file(File, PFString, FileString, PathFile) :-
   name(File, FileString),
   name(PathFile, PFString).


% compiling files

ol_compile_project(Project, Files, Options) :-
   ol_project(Project),
   ol_project_files(Project, AllFiles),
   ol_project_path(Project, PFString, FileString),
   \+ ( ol_member(File, Files),
        \+ ( ol_member(File, AllFiles),
             ol_path_file(File, PFString, FileString, PathFile),
             ol_compile_file(Project, PathFile, File, Options)
           ),
        format("** error while compiling ~w", [File])
      ),
   ( ol_compiled(_) ->
        ol_reevaluate_compiled(Project, Options),
        ol_save_project_info(Project, Options)
      ;
        true
   ).


% ol_compile_file(+Project, +PathFile, +File, +Options) -- compile PathFile with Options,
%    where PathFile is assumed to be a file name with at least one extension,
%    the one of the base language, to which '.ol' (e.g., example.pl.ol) append;
%    it also has the path; File is the file name only
ol_compile_file(Project, OutFile, File, Options) :-
   name(OutFile, OutFileString),
   ol_append(OutFileString, ".ol", InFileString),
   name(InFile, InFileString),
   ol_base_language(OutFileString, _, BL),
   ( ol_member(verbose, Options) ->
        format("compiling ~w -> ~w~n", [InFile,OutFile])
      ;
        true
   ),
   open(InFile, read, InStream),
   open(OutFile, write, OutStream),
   ol_version(V),
   format(OutStream, "% OL(P) ~w~n", [V]),
   format(OutStream, "% Source: ~w~n~n", [InFile]),
   ol_read(BL, InStream, InStat, Init, ReadTerm),
   ol_write(BL, OutStream, OutStat, WriteTerm),
   Init,
   repeat,
     ReadTerm,
     ( InStat == end_of_file
      ;
       ol_compile(InStat, OutStat, Project, OutStream, File, Options, BL),
       WriteTerm,
       fail
     ),
   !,
   close(InStream),
   close(OutStream),
   assert(ol_changed_file(File)).


% compiling objects and predicates

% ol_object(O, P, F) -- O is an object in file F of project P
% ol_super(O, SO) -- SO is a super-object of O
% ol_info(O, K, I) -- O has info I of kind K
% ol_state(O, S) -- O is in state S (compiled, , autonomous)
% ol_unfold(O, M1, M2) -- in O, M2 is to be unfolded in M1
% ol_instance(O, I) -- I is the default instance for O;
%                      with attribute-value list: [Ai=Vi,...|_]
%                      with position term:        O(V1,...,Vn,[(A1,1),...,(An,n)])
% ol_compiled(O) -- (the interface of) O has been recompiled
% ol_current_object(O) -- O is the current object during the compilation
% ol_changed_file(F) -- F is a file that has been compiled and was not
%                       consulted or loaded since

:- multifile ol_object/3, ol_state/2, ol_super/2, ol_info/3, ol_unfold/3,
             ol_instance/2, ol_send/2.
:- dynamic ol_object/3, ol_state/2, ol_super/2, ol_info/3, ol_unfold/3,
           ol_instance/2, ol_send/2,
           ol_current_object/1, ol_compiled/1, ol_changed_file/1.

% ol_compile(+Statement, -CompiledStatement, +Project, +OutStream, +File, +Options, +OutStream) --
%    CompiledStatement is the compiled version of the OL Statement,
%    Options is a list of those, OutStream is the file currently being
%    written to, File the file name

% "object Object is_a SuperObjects."
ol_compile((object O is_a SOs), _, Project, OutStream, File, Options, _) :-
   !,
   ol_syntax(( atom(O), ON=O, As=[] ; O=..[ON,As] ),
             true,
             (format("** invalid object name: ~q - abort~n", [O]), abort)),
   ( ol_member(verbose, Options) -> format("   object: ~w~n", [ON]) ; true ),
   format(OutStream, "~n% * object ~q~2n", [ON]),
   ol_objects(SystemObjects),
   ol_syntax(\+ (ol_member(ON, SystemObjects), Project\==ol_project),
             true,
             (format("** object ~w uses name of system object - abort", [ON]), abort)),
 % ol_syntax(ol_list_of_el(As, A, functor(A,=,2)),
 %           true,
 %           format(" * invalid attribute-value list ~q~n", [As])),
   findall(A=V, ol_member(A=V,As), DAs),   % no overriding if name only
   ( ol_object(ON, _, OF) ; true ),
   ol_syntax(OF=File,
             true,
             format(" * object ~w in file ~w overridden in file ~w~n", [ON,OF,File])),
   ol_syntax(ol_conj_of_el(SOs, SO, (atom(SO),ol_object(SO,_,_))),
             true,
             format(" * invalid super objects: ~q~n", [SOs])),
   retractall(ol_current_object(_)),  assert(ol_current_object(ON)),
   retractall(ol_object(ON,_,_)),     assert(ol_object(ON,Project,File)),
   retractall(ol_state(ON,_)),        assert(ol_state(ON,compiled)),
   retractall(ol_super(ON, _)),       ( ol_member_conj(SO,SOs),
                                        assert(ol_super(ON,SO)), fail ; true ),
   retractall(ol_info(ON,_,_)),       assert(ol_info(ON,att,DAs)),
   retractall(ol_unfold(ON, _, _)),
   retractall(ol_instance(ON,_)),
   assert(ol_compiled(ON)),
   !, fail.    % write nothing to file
ol_compile((object O), _, Project, OutStream, File, Options, _) :-
   O == (object),
   !,
   ( ol_member(verbose, Options) -> format("   object: ~w~n", [O]) ; true ),
   format(OutStream, "~n% * object ~q~2n", [O]),
   ( Project\==ol_project ->
        format("** object ~w uses name of system object - abort~n", [O]),
        abort
      ;
        true
   ),
   retractall(ol_current_object(_)),  assert(ol_current_object(O)),
   retractall(ol_object(O,_,_)),      assert(ol_object(O,Project,File)),
   retractall(ol_state(O,_)),         assert(ol_state(O,compiled)),
   retractall(ol_super(O, _)),
   retractall(ol_info(O,_,_)),        assert(ol_info(O,att,[])),
   assert(ol_compiled(O)),
   !, fail.    % write nothing to file
ol_compile((object O), CS, Project, OutStream, File, Options, BL) :-
   !,
   ol_compile((object O is_a (object)), CS, Project, OutStream, File, Options, BL).

% "end_object Object."
ol_compile((end_object O), _, _, _, _, _, _) :-
   ol_current_object(CO),
   !,
   retract(ol_current_object(CO)),
   ol_syntax(O == CO,
             true,
             format(" * names not matching: object ~w - end_object ~w~n", [CO,O])),
   !, fail.    % write nothing to file

% ":- Declare References."
ol_compile((:- publish Refs), _, _, _, _, _, _) :-
   ol_current_object(O),
   !,
   ol_member_conj(Ref, Refs),
   ol_syntax((Ref = N/A, atom(N), integer(A)),
             assert(ol_info(O,pub,Ref)),
             format(" * invalid reference in publish declaration: ~q~n", [Ref])),
   fail.    % write nothing to file
ol_compile((:- override Refs), _, _, _, _, _, _) :-
   ol_current_object(O),
   !,
   ol_member_conj(Ref, Refs),
   ol_syntax((Ref = N/A, atom(N), integer(A)),
             assert(ol_info(O,ovr,Ref)),
             format(" * invalid reference in override declaration: ~q~n", [Ref])),
   fail.    % write nothing to file
ol_compile((:- unfold Unfoldings1), _, _, _, _, _, _) :-
   ol_current_object(O),
   !,
   ol_replace(Unfoldings1, this, O, Unfoldings),
   ol_member_conj(Unfolding, Unfoldings),
   ol_syntax((Unfolding = (N1/A1 in N2/A2), atom(N1), integer(A1), atom(N2), integer(A2)),
             assert(ol_unfold(O,N2/A2,N1/A1)),
             format(" * invalid reference in unfold declaration: ~q~n", [Unfolding])),
   fail.    % write nothing to file
ol_compile((:- dynamic), (:- dynamic O/3), _, _, _, _, _) :-
 % ol_method_head(_, O, _, _, MethodHead), functor(MethodHead, O, A),  % A=3
   ol_current_object(O),
   !.

% "Head :- Body."
ol_compile(Clause, CClause, _, _, _, _, BL) :-
   ol_current_object(O),
   !,
   ol_replace_pv(Clause, O, Self, RClause),
   ( RClause = (H:-B) -> true ; Clause = H, B = true ),
   ol_compile_clause(H, B, BL, O, Self, CClause),
   functor(H, N, A),
   ( ol_info(O,def,N/A) -> true ; assert(ol_info(O,def,N/A)) ).

ol_compile((:-op(P,T,N)), (:-op(P,T,N)), _, _, _, _, _) :-
   !,
   op(P, T, N).
ol_compile(Statement, Statement, _, _, _, _, _) :-
   \+ ol_current_object(_).


% ol_compile_clause(Head, Body, BL, This, Self, CompiledClause) --
%    CompiledClause is the compiled version of (Head:-Body) in object This
%    and context variable Self for base language BL
ol_compile_clause(Head, Body, BL, This, Self, (CHead:-CBody)) :-
   ol_replace_pv(Head, This, Self, Head1),
   ol_replace(Head1, true, true, Head2),
   ol_method_head(Head2, This, Self, Inst, CHead),
   ol_compile_body(Body, BL, This, Self, Inst, CBody).

% ol_method_head(MethodGoal, This, Self, Instance, MethodHead)
ol_method_head(MethodGoal, This, Self, Inst, MethodHead) :-
   MethodHead =.. [This,MethodGoal,Self,Inst].
% ol_method_head(MethodName, This, Self, Args, MethodHead) :-
%    MethodHead =.. [MethodName,ts(This,Self,Inst)|Args].

% to be traced in debugging and to be unfolded in optimization
:- dynamic ol_call/3.
ol_call(O, M, This) :-
   atom(O) -> ol_send(M, rsd(O,This,loc,_))
            ; O=..[ON,Inst], ol_send(M, rsd(ON,This,loc,Inst)).
 % ol_interface_head(O, This, loc, M, Inst, InterfaceHead).
:- dynamic ol_new/2, ol_new/3, ol_assign/3.
ol_new(O, I) :-
   ol_instance(O, I).
ol_new(O, As, I) :-
   ol_instance(O, J), ol_append(As, J, I).
ol_assign(J, As, I) :-
   ol_append(As, J, I).
/* with position instead of list:
ol_new(O, As, I) :-
   ol_instance(O, J),
   functor(J, N, A),      % ol_assign(J,As,I) from here
   functor(I, N, A),
   arg(A, J, Ps),
   arg(A, I, Ps),
   ol_replace_attributes(Ps, As, J, I).
ol_assign(J, As, I) :-
   functor(J, N, A),
   functor(I, N, A),
   arg(A, J, Ps),
   arg(A, I, Ps),
   ol_replace_attributes(Ps, As, J, I).
*/
:- dynamic ol_val_e/3, ol_val_eq/3, ol_val_neq/3.
ol_val_e(Inst, A, V) :-
   ol_val(Inst, A, V).
ol_val_eq(Inst, A, V) :-
   ol_val(Inst, A, W), V==W.
ol_val_neq(Inst, A, V) :-
   ol_val(Inst, A, W), V\==W.

:- dynamic ol_no_portray/0.

% portray(Term) -- for OL(P) encoded terms
portray(_) :-
   ol_no_portray, !, fail.
portray(ol_send(MethodGoal, rsd(Receiver,Sender,Def,Inst))) :-
 % ol_interface_head(R, S, D, M, Inst, InterfaceHead),
   format("from ~q: ~q(~q)/~q::~q", [Sender,Receiver,Inst,Def,MethodGoal]).
portray(MethodHead) :-
   ol_method_head(MethodGoal, This, Self, Inst, MethodHead),
   ol_object(This, _, _),
   functor(MethodGoal, N, A),
   \+ \+ ol_info(This, def, N/A),
   format("  in ~q: ~q(~q)::~q", [This,Self,Inst,MethodGoal]).
portray(ol_new(O,I)) :-
   writeq(O::new(I)).
portray(ol_new(O,As,I)) :-
   writeq(O::new(As,I)).
portray(ol_call(O,M,_)) :-
   writeq(O::M).
portray(ol_call(G,_,_,_,_)) :-
   writeq(G).
portray(ol_assign(J,As,I)) :-
   format("~q := ~q.~q", [I,J,As]).
portray(ol_val_e(I,A,V)) :-
   format("~q.~q = ~q", [I,A,V]).
portray(ol_val_eq(I,A,V)) :-
   format("~q.~q == ~q", [I,A,V]).
portray(ol_val_neq(I,A,V)) :-
   format("~q.~q \== ~q", [I,A,V]).

ol_reevaluate_compiled(Project, Options) :-
   ( ol_member(verbose,Options) ->
        format("recompiling interface for:~n", [])
      ;
        true
   ),
   ol_find_descendants([(object)], [], [], Ds),
   ol_member(O, [(object)|Ds]), \+ \+ ol_compiled(O),
     ol_object(O, Project, _),
     ol_interface_head(O, _, _, _, _, IH),
     retractall(IH),
     ( ol_member(verbose,Options) -> format("   object: ~w~n", [O]) ; true ),
     ol_add_interface(O),
     retractall(ol_compiled(O)),
     \+ ( (ol_super(Child, O), \+ ol_compiled(Child)),   % for all ...
          \+ assert(ol_compiled(Child)) ),               % do ...
   fail.
ol_reevaluate_compiled(_, _).


ol_find_descendants([O|Os], SOs1, Ls1, SOs) :-
   findall(SO, (ol_super(SO, O), \+ \+ ol_super(_, SO)), SOs2),
   findall(SO, (ol_super(SO, O), \+ ol_super(_, SO)), Ls2),
   ol_remove_all(SOs2, SOs1, SOs1R),
   ol_remove_all(SOs2, Os, Os1),
   ol_remove_all(Ls2, Ls1, Ls3),
   ol_append(Os1, SOs2, Os2),
   ol_append(SOs1R, SOs2, SOs3),
   ol_append(Ls3, Ls2, Ls),
   ol_find_descendants(Os2, SOs3, Ls, SOs).
ol_find_descendants([], SOs1, Ls, SOs) :-
   ol_append(SOs1, Ls, SOs).

ol_find_ancestors([O|Os], SOs1, Ls1, SOs) :-
   findall(SO, (ol_super(O, SO), \+ \+ ol_super(SO, _)), SOs2),
   findall(SO, (ol_super(O, SO), \+ ol_super(SO, _)), Ls2),
   ol_remove_all(SOs2, SOs1, SOs1R),
   ol_remove_all(SOs2, Os, Os1),
   ol_remove_all(Ls2, Ls1, Ls3),
   ol_append(Os1, SOs2, Os2),
   ol_append(SOs1R, SOs2, SOs3),
   ol_append(Ls3, Ls2, Ls),
   ol_find_ancestors(Os2, SOs3, Ls, SOs).
ol_find_ancestors([], SOs1, Ls, SOs) :-
   ol_append(SOs1, Ls, SOs).

ol_descendant(O, SO) :-
   ol_super(SO, O).
ol_descendant(O, SO) :-
   ol_super(SO1, O),
   ol_descendant(SO1, SO).

ol_ancestor(O, SO) :-
   ol_super(O, SO).
ol_ancestor(O, SO) :-
   ol_super(O, SO1),
   ol_ancestor(SO1, SO).

ol_add_interface(O) :-
   retractall(ol_info(O,inh,_)),
   ol_handled(O, MAs),
     ( ol_member(m(HO,N/A), MAs),                              % only in HO
         functor(M, N, A),
         ol_method_head(M, HO, O, I, TM),
         ol_interface_head(O, _, loc, I, M, IH),
         asserta((IH:-TM)),
         ( HO\==O -> asserta(ol_info(O,inh,(HO,N/A))) ; true )
      ;
       ol_member(s(HO,N/A), MAs),                              % in HO, overridden in O
         functor(M, N, A),
         ol_method_head(M, O,  O, I, TM),
         ol_method_head(M, HO, O, I, HTM),
         ol_interface_head(O, _, loc, I, M, IH),
         ol_interface_head(O, _, inh, I, M, HIH),
         assert((IH:-TM)),
         assert((HIH:-HTM))
     ),
   fail.
ol_add_interface(O) :-
   ol_info(O,def,N/A), \+ ol_info(O,pub,N/A), \+ ol_info(O,ovr,N/A),
     functor(M, N, A),
     ol_method_head(M, O, O, I, TM),
     ol_interface_head(O, O, loc, I, M, IH),
     assert((IH:-TM)),
   fail.
ol_add_interface(O) :-
   ol_default_attributes(O, AllAs),
   retractall(ol_info(O,allatt,_)),     assert(ol_info(O,allatt,AllAs)),
   ol_extract_attribute_list(AllAs, DAs),
   Inst = DAs,
   /* with position instead of list:
   ol_create_instance(O, DAs, Inst),
   */
   retractall(ol_instance(O,_)),        assert(ol_instance(O,Inst)),
   !.


% ol_handled(+CurrentObject, -Methods)
%    The algorithm basically goes through the inheritance graph depth-first,
%    noting visible (public or overridden) methods as m(O,N/A), those that
%    are overridden as s(O,N/A). It is assumed that the interfaces of all
%    superobjects have already been determined, and that the `inh' ol_info's
%    have been determined, so that it is sufficient to look at the current
%    object and its immediate superobjects.
ol_handled((object), Ms) :-
   findall(m((object),NA), (ol_info((object),def,NA), ol_info((object),pub,NA)), Ms),
   !.   
ol_handled(CO, Ms) :-
   findall(m(CO,NA), ol_info(CO,pub,NA), CMs),
   findall(SO, ol_super(CO, SO), SOs),
   findall((HO,NA),
           ( ol_member(SO, SOs),
             ((ol_info(SO,pub,NA);ol_info(SO,ovr,NA)), HO=SO ; ol_info(SO,inh,(HO,NA)))
           ),
           HONAs),
   ol_handled(HONAs, CO, CMs, Ms),
   ( ol_info(CO,ovr,NA), \+ ol_member(s(_,NA), Ms),
       format(" * method ~q declared as overriding in ~q is not inherited~n", [NA,CO]),
       fail
    ;
     true
   ),
   !.

% ol_handled(+VisibleMethods, +CurrentObject, +MethodsAccu, -Methods) --
%    Methods are all of CurrentObject's visible methods,
ol_handled([], _, Ms, Ms).
ol_handled([(HO,NA)|HONAs], CO, Ms1, Ms) :-
   % add the method NA (which is visible in ancestor HO of CO) as
   %   s(HO,NA) if overridden in CO, or as
   %   m(HO,NA) if non-existing in CO;
   % in both cases, an already existing s(SO,NA) or m(SO,NA) takes precedence
   % if SO is not a superobject of O, and has to be removed from Ms otherwise
   % (all methods of HO also appearing in CO are treated the same, except that
   % a warning is given if the method is not declared as overriding)
   ( ( ol_info(CO,ovr,NA) ; ol_info(CO,pub,NA) ; ol_info(CO,def,NA) ) ->
        ( ol_remove(s(SO,NA), Ms1, Ms2) ->
             ( ol_ancestor(HO, SO) -> Ms3 = [s(HO,NA)|Ms2] ; Ms3 = Ms1 )
           ;
             Ms3 = [s(HO,NA)|Ms1]
        ),
        ol_syntax(ol_info(CO,ovr,NA),
                  true,
                  format(" * method ~q inherited from ~q overridden in ~q~n",
                         [NA,HO,CO]))
      ;
        ( ol_remove(m(SO,NA), Ms1, Ms2) ->
             ( ol_ancestor(HO, SO) -> Ms3 = [m(HO,NA)|Ms2] ; Ms3 = Ms1 )
           ;
             Ms3 = [m(HO,NA)|Ms1]
        )
   ),
   !,
   ol_handled(HONAs, CO, Ms3, Ms).

% ol_default_attributes(+CurrentObject, -Attributes)
ol_default_attributes(CO, As) :-
   findall((CO,AV), ( ol_info(CO, att, DAs), ol_member(AV, DAs) ), CAs),
   findall(SO, ol_super(CO, SO), SOs),
   findall((HO,AV),
           ( ol_member(SO, SOs),
             ol_info(SO, allatt, DAs),
             ol_member((HO,AV), DAs)
           ),
           HOAs),
   ol_default_attributes(HOAs, CO, CAs, As),
   !.

% ol_default_attributes(+DefaultAttributes, +CurrentObject, +AttsAccu, -Attributes) --
%    Attributes are all of CurrentObject's default attributes
%    (this is a simplified version of ol_handled/4 for attributes)
ol_default_attributes([], _, As, As).
ol_default_attributes([(HO,A=V)|HOAs], CO, As1, As) :-
   ( ol_remove((SO,A=_), As1, As2) ->
        ( ol_ancestor(HO, SO) -> As3 = [(HO,A=V)|As2] ; As3 = As1 )
      ;
        As3 = [(HO,A=V)|As1]
   ),
   !,
   ol_default_attributes(HOAs, CO, As3, As).

ol_extract_attribute_list([], _OpenTail).
ol_extract_attribute_list([(_,AV)|As], [AV|DAs]) :-
   ol_extract_attribute_list(As, DAs).

ol_create_instance(O, As, Inst) :-
   length(As, A),
   !,
   functor(Inst, O, A),
   A1 is A+1,
   arg(A1, Inst, Ps),
   ol_set_attributes(A, As, Ps, Inst).

ol_set_attributes(0, _, [], _) :-
   !.
ol_set_attributes(P, [A=V|As], [(A,P)|Ps], Inst) :-
   arg(P, Inst, V),
   P1 is P-1,
   ol_set_attributes(P1, As, Ps, Inst).


/*
   SOs = [FO|TOs],
   findall(NA,
           ( ol_info(FO,pub,NA) ; ol_info(FO,ovr,NA) )
           NAs),
   ol_handled(NAs, FO, TOs, CO, CMs, Ms),
   ( ol_info(CO,ovr,NA), \+ ol_member(s(_,NA), Ms),
       format(" * method ~q declared as overriding in ~q is not inherited~n", [NA,CO]),
       fail
    ;
     true
   ),
   !.

% ol_handled(+VisibleMethods, +SuperObject, +SuperObjects, +CurrentObject, +MethodsAccu, -Methods) --
%    Methods are all of CurrentObject's visible methods,
ol_handled([NA|NAs], O, Os, CO, Ms1, Ms) :-
   % add the method NA (which is visible in ancestor O of CO) as
   %   s(O,NA) if overridden in CO, or as
   %   m(O,NA) if non-existing in CO
   % in both cases, an already existing s(SO,NA) or m(SO,NA) takes precedence
   % if it is not a superobject of O, and has to be removed from Ms otherwise
   % (all methods of O also appearing in CO are treated the same, except that
   % a warning is given if the method is not declared as overriding)
   ( ( ol_info(CO,ovr,NA) ; ol_info(CO,pub,NA) ; ol_info(CO,def,NA) ) ->
        ( ol_remove(s(SO,NA), Ms1, Ms2) ->
             ( ol_ancestor(O, SO) -> Ms3 = [s(O,NA)|Ms2] ; Ms3 = Ms1 )
           ;
             Ms3 = [s(O,NA)|Ms1]
        ),
        ol_syntax(ol_info(CO,ovr,NA),
                  true,
                  format(" * method ~q inherited from ~q overridden in ~q~n",
                         [NA,O,CO]))
      ;
        ( ol_remove(m(SO,NA), Ms1, Ms2) ->
             ( ol_ancestor(O, SO) -> Ms3 = [m(O,NA)|Ms2] ; Ms3 = Ms1 )
           ;
             Ms3 = [m(O,NA)|Ms1]
        )
   ),
   !,
   ol_handled(NAs, O, Os, CO, Ms3, Ms).
ol_handled([], FO, Os, CO, Ms1, Ms) :-
   % no more visible methods for FO - replace by super-objects
   findall(SO, ol_super(FO, SO), SOs),
   ol_append(SOs, Os, SOs1),
   SOs1 = [FO1|TOs],
   findall(NA,
           ( ol_info(FO1,def,NA),
             ol_once(( ol_info(FO1,pub,NA); ol_info(FO1,ovr,NA) ))
           ),
           NAs),
   !,
   ol_handled(NAs, FO1, TOs, CO, Ms1, Ms).
ol_handled([], _, [], _, Ms, Ms).
*/


ol_save_project_info(Project, Options) :-
   name(Project, ProjectString),
   ol_project_path(Project, PFString, ProjectString),
   ol_append(PFString, ".pl", ProjectFileString),
   name(ProjectFile, ProjectFileString),
   ( ol_member(verbose, Options) ->
        format("saving object data -> ~w~n", [ProjectFile])
      ;
        true
   ),
   open(ProjectFile, write, OutStream),
   ol_version(V),
   format(OutStream, "% OL(P) ~w~n", [V]),
   format(OutStream, "% Source: object interfaces for project ~w", [Project]),
   ol_save_project_file_info(OutStream, Project),
   \+ ( ol_project_files(LibProject, _), LibProject \== Project,
        \+ ol_save_project_file_info(OutStream, LibProject) ),
   close(OutStream).

ol_save_project_file_info(OutStream, Project) :-
   format(OutStream, "~2n% *** project ~w~2n", [Project]),
   ol_portray_facts(OutStream, ol_project_files(Project,_)),
   ol_portray_facts(OutStream, ol_project_path(Project,_,_)),
   ol_project_files(Project, Files),
   \+ ( ol_member(File, Files),
        \+ ( format(OutStream, "~n% ** file ~w~n", [File]),
             \+ ( ol_object(O, Project, File),
                  \+ ( format(OutStream, "~n% * object ~q~2n", [O]),
                       ol_portray_facts(OutStream, ol_object(O,_,_)),
                       ol_portray_facts(OutStream, ol_super(O,_)),
                       ol_portray_facts(OutStream, ol_state(O,_)),
                       ol_portray_facts(OutStream, ol_unfold(O,_,_)),
                       ol_portray_facts(OutStream, ol_instance(O,_)),
                       ol_portray_facts(OutStream, ol_info(O,_,_)),
                       nl(OutStream),
                       ol_interface_head(O, _, _, _, _, IH),
                       \+ ( clause(IH, Body),
                            \+ portray_clause(OutStream, (IH:-Body)) )
                     )
                )
           )
      ).



% ol_make(+FileName, +CompilingFlag) -- save a state with OL(P) loaded
ol_make(FileName, Flag) :-
   ( predicate_property(save(_), built_in) ->  % regular save state
        save(FileName),
        ol_make_init(Flag)
      ;                                        % save state Quintus style
       save_program(FileName, ol_make_init(Flag))
   ).

% ol_make_init(+CompilingFlag) -- initialize for OL(P) loaded state
ol_make_init(Flag) :-
   ol_prolog(Prolog),
   ol_herald(Prolog, Herald),
   version(Herald),
   write(Herald), nl, 
   ( nonvar(Flag) ->       % for systems without compiling flag
        prolog_flag(compiling, _, Flag)
      ;
        true
   ).

% ol_version(?VersionNumber)
ol_version('1.1').

% ol_herald(+Prolog, -Herald) -- create the atom Herald, which includes name, version and (C)
ol_herald(sicstusProlog, '
OL(P): Object Layer for Prolog    Copyright (C) 1993 Xerox Corp.
Version 1.1 for SICStus Prolog    All rights reserved.    (MPJF)

Call project::help for a summary of project commands.
').
ol_herald(quintusProlog, '
OL(P): Object Layer for Prolog    Copyright (C) 1993 Xerox Corp.
Version 1.1 for QUINTUS Prolog    All rights reserved.    (MPJF)

Call project::help for a summary of project commands.
').
