/*
 * Copyright (C) 1990 by Ted Kim.
 *
 * This software work was developed at UCLA with support in part from
 * DARPA Contract F29601-87-C-0072.
 */

/*
 * pointer.c - pointer functions
 *
 * pxPointerState
 * pxWarpPointer
 *
 * pxGetPointerAttribute
 * pxSetPointerAttribute
 * pxQueryPointer
 * pxSetPointer
 *
 * pxGetPointerMapping
 * pxSetPointerMapping
 */

#include <X11/Xlib.h>

#include "dl.h"
#include "conn.h"
#include "ff.h"
#include "attrib.h"

/*
 * pxPointerState(+Connection, +Window, -Root, -Child, -RootX, -RootY, -WinX,
 *	-WinY, -State, 0)
 *
 *	integer: Connection, Window, Root, Child, RootX, RootY, WinX, WinY,
 *		State
 */
FFInteger
pxPointerState(pc, pw, pr, pch, prx, pry, pwx, pwy, pstate)
     FFInteger pc, pw, *pr, *pch, *prx, *pry, *pwx, *pwy, *pstate;
{
  PxConnection *pcp;

  PX_ErrorInit("xPointerState/9");
  *pr = *pch = *prx = *pry = *pwx = *pwy = *pstate = PX_Invalid;

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  if (XQueryPointer(pcp->xDisplay, pw, pr, pch, prx, pry, pwx, pwy, pstate) ==
      False)
    return FF_Fail;
  return FF_Ok;
}

/*
 * pxWarpPointer(+Connection, +Source, +Dest, +SrcX, +SrcY, +Width, +Height,
 *	+DestX, +DestY, 0)
 *
 *	integer: Connection, Source, Dest, SrcX, SrcY, Width, Height,
 *		DestX, DestY
 */
FFInteger
pxWarpPointer(pc, ps, pd, psx, psy, pw, ph, pdx, pdy)
     FFInteger pc, ps, pd, psx, psy, pw, ph, pdx, pdy;
{
  PxConnection *pcp;

  PX_ErrorInit("xWarpPointer/9");

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  XWarpPointer(pcp->xDisplay, ps, pd, psx, psy, pw, ph, pdx, pdy);
  return FF_Ok;
}

/*
 * pxGetPointerAttribute(+Attribute, -Value, 0)
 *	integer: Attribute, Value
 */

typedef struct {
  int numerator;
  int denominator;
  int threshold;
} PxPointerControl;

PxPointerControl ppc;

Bool numerator_changed = False,
  denominator_changed = False,
  threshold_changed = False;

#define PX_GetAttribute(NAME, MEMBER) \
  case NAME: *pv = ppc.MEMBER; \
             break

FFInteger
pxGetPointerAttribute(pa, pv)
     FFInteger pa, *pv;
{
  *pv = PX_Invalid;

  switch (pa) {
    PX_GetAttribute(0, numerator);
    PX_GetAttribute(1, denominator);
    PX_GetAttribute(2, threshold);
    default: PX_Error(unknown_attribute);
  }
  return FF_Ok;
}

/*
 * pxSetPointerAttribute(+Attribute, +Value, 0)
 *	integer: Attribute, Value
 */

#define PX_SetAttribute(A, MEM, FLAG) \
   case A: ppc.MEM = pv; \
           FLAG = True; \
           break

FFInteger
pxSetPointerAttribute(pa, pv)
     FFInteger pa, pv;
{
  PX_ErrorInit("xSetPointer/2");

  switch (pa) {
    PX_SetAttribute(0, numerator, numerator_changed);
    PX_SetAttribute(1, denominator, denominator_changed);
    PX_SetAttribute(2, threshold, threshold_changed);
    default: PX_Error(unknown_attribute);
  }
  return FF_Ok;
}

/*
 * pxQueryPointer(+Connection, 0)
 *	integer: Connection
 */
FFInteger
pxQueryPointer(pc)
     FFInteger pc;
{
  PxConnection *pcp;

  PX_ErrorInit("xQueryPointer/2");

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  XGetPointerControl(pcp->xDisplay, &(ppc.numerator), &(ppc.denominator),
		     &(ppc.threshold));
  return FF_Ok;
}

/*
 * pxSetPointer(+Connection, 0)
 *	integer: Connection
 */
FFInteger
pxSetPointer(pc)
     FFInteger pc;
{
  PxConnection *pcp;
  Bool acceleration_changed = False;
  int t1, t2;

  PX_ErrorInit("xSetPointer/2");

  if (PXLookupConnection(pc, &pcp)) {
    numerator_changed = denominator_changed = threshold_changed = False;
    return FF_Fail;
  }

  if (numerator_changed || denominator_changed || threshold_changed) {
    if (numerator_changed && !denominator_changed) 
      XGetPointerControl(pcp->xDisplay, &t1, &(ppc.denominator), &t2);
    if (!numerator_changed && denominator_changed)
      XGetPointerControl(pcp->xDisplay, &(ppc.numerator), &t1, &t2);
    if (numerator_changed || denominator_changed)
      acceleration_changed = True;

    XChangePointerControl(pcp->xDisplay,
			  acceleration_changed, threshold_changed,
			  ppc.numerator, ppc.denominator, ppc.threshold);
    numerator_changed = denominator_changed = threshold_changed = False;
  }
  return FF_Ok;
}

/*
 * pxGetPointerMapping(+Connection, +Length, 0)
 *	integer: Connection, Length
 */
FFInteger
pxGetPointerMapping(pc, pl)
     FFInteger pc, pl;
{
  PxConnection *pcp;

  PX_ErrorInit("xGetPointerMapping/2");

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  if (pl < 0)
    return FF_Fail;
  if (!pl) 
    return FF_Ok;

  XGetPointerMapping(pcp->xDisplay, pxMallocBlock, pl);
  return FF_Ok;
}

/*
 * pxSetPointerMapping(+Connection, 0)
 *	integer: Connection
 */
FFInteger
pxSetPointerMapping(pc)
     FFInteger pc;
{
  PxConnection *pcp;

  PX_ErrorInit("xSetPointerMapping/2");

  if (PXLookupConnection(pc, &pcp)) 
    return PXVectorFree();

  if (XSetPointerMapping(pcp->xDisplay, pxMallocBlock, pxElements) !=
      MappingSuccess)
    return PXVectorFree();
  return FF_Ok;
}

/*
 * eof
 */

 
