/*
 * Copyright (C) 1989 by The Regents of the University of California.
 *
 * This software work was developed at UCLA with support in part from
 * DARPA Contract F29601-87-C-0072.
 */

/*
 * text.c
 *
 * pxOpenFont
 * pxCloseFont
 *
 * pxLoadedFonts
 * pxLoadFont
 * pxUnloadFont
 *
 * pxGetFontPath
 * pxSetFontPath
 * pxListFonts
 *
 * pxGetFontAttribute
 * pxGetFontProperty
 * pxGetCharInfo
 *
 * pxTextExtents
 *
 * pxDrawText
 * pxImageText
 */

#include <stdio.h>
#include <X11/Xlib.h>

#include "dl.h"
#include "conn.h"
#include "ff.h"

/*
 * pxOpenFont(+Connection, +Name, -Font, 0)
 *	integer: Connection, Font
 *	string: Name
 */
FFInteger
pxOpenFont(pc, pn, pf)
     FFInteger pc, *pf;
     FFString pn;
{
  PxConnection *pcp;

  PX_ErrorInit("xOpenFont/3");
  *pf = PX_Invalid;

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  *pf = XLoadFont(pcp->xDisplay, pn);
  return FF_Ok;
}

/*
 * pxCloseFont(+Connection, +Font, 0)
 *	integer: Connection, Font
 */
FFInteger
pxCloseFont(pc, pf)
     FFInteger pc, pf;
{
  PxConnection *pcp;

  PX_ErrorInit("xCloseFont/2");

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  XUnloadFont(pcp->xDisplay, pf);
  return FF_Ok;
}

/*
 * pxLoadedFonts(-Next)
 *	integer: Next
 */

typedef struct _PxFont {
  struct _PxFont *next;
  int des;
  int conn;
  Font xfid;
  XFontStruct *xfsp;
} PxFont;

DL pxFontList = {NULL, sizeof(PxFont), 0};

void
pxLoadedFonts(pn)
     FFInteger *pn;
{
  if ((pxList = pxFontList.list) == NULL)
    *pn = PX_End;
  else {
    pxType = PX_List;
    *pn = PX_Cont;
  }
}

/*
 * pxLoadFont(+Connection, +Font, -LoadedFont, 0)
 *	integer: Connection, Font, LoadedFont
 */
FFInteger
pxLoadFont(pc, pf, plf)
     FFInteger pc, pf, *plf;
{
  PxFont *pfp;
  PxConnection *pcp;
  XFontStruct *fsp;

  PX_ErrorInit("xLoadFont/3");
  *plf = PX_Invalid;

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  if ((fsp = XQueryFont(pcp->xDisplay, pf)) == NULL)
    PX_Error("no such font");

  if (DLAdd(&pxFontList, plf, (DLHeader **)&pfp))
    PX_Error(no_memory);
  pfp->conn = pc;
  pfp->xfid = pf;
  pfp->xfsp = fsp;
  return FF_Ok;
}

/*
 * pxUnloadFont(+LoadedFont, 0)
 *	integer: LoadedFont
 */
FFInteger
pxUnloadFont(plf)
     FFInteger plf;
{
  PxFont *pfp;
  int des;

  PX_ErrorInit("xUnloadFont/1");

  if (DLLookup(pxFontList, (int)plf, (DLHeader **)&pfp))
    PX_Error(no_such_fil);
  des = pfp->des;

  _XFreeExtData(pfp->xfsp->ext_data); /* free the XFontStruct */
  if (pfp->xfsp->per_char)
    XFree((char *) pfp->xfsp->per_char);
  if (pfp->xfsp->properties)
    XFree((char *)pfp->xfsp->properties);
  XFree((char *)pfp->xfsp);

  DLRemove(&pxFontList, (int)des);
  return FF_Ok;
}

/*
 * pxGetFontPath(+Connection, -Next, 0)
 *	integer: Connection, Next
 */
FFInteger
pxGetFontPath(pc, pn)
     FFInteger pc, *pn;
{
  PxConnection *pcp;

  PX_ErrorInit("xGetFontPath/2");
  *pn = PX_Invalid;

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  pxStringPointers = XGetFontPath(pcp->xDisplay, &pxElements);
  if (pxElements) {
    pxMallocBlock = (char *)pxStringPointers;
    pxType = PX_VectorPath;
    *pn = PX_Cont;
  } else 
    *pn = PX_End;
  return FF_Ok;
}

/*
 * pxSetFontPath(+Connection, 0)
 *	integer: Connection
 */
FFInteger
pxSetFontPath(pc)
     FFInteger pc;
{
  PxConnection *pcp;

  PX_ErrorInit("xSetFontPath/2");

  if (PXLookupConnection(pc, &pcp))
    return PXVectorFree();

  XSetFontPath(pcp->xDisplay, pxMallocBlock, pxElements);
  (void)PXVectorFree();
  return FF_Ok;
}

/*
 * pxListFonts(+Connection, +Pattern, +MaxNames, -Next, 0)
 *	integer: Connection, MaxNames, Next
 *	string: Pattern
 */
FFInteger
pxListFonts(pc, pp, pmn, pn)
     FFInteger pc, pmn, *pn;
     FFString pp;
{
  PxConnection *pcp;

  PX_ErrorInit("xListFonts/4");
  *pn = PX_Invalid;

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  pxStringPointers = XListFonts(pcp->xDisplay, pp, pmn, &pxElements);
  if (pxElements) {
    pxMallocBlock = (char *)pxStringPointers;
    pxType = PX_VectorFont;
    *pn = PX_Cont;
  } else 
    *pn = PX_End;
  return FF_Ok;
}

/*
 * pxGetFontAttribute(+LoadedFont, +Attribute, -Value, 0)
 *	integer: LoadedFont, Attribute, Value
 */

#define PX_GetAttribute(A, V) \
   case A: *pv = (pfp->V); \
           break

FFInteger
pxGetFontAttribute(plf, pa, pv)
     FFInteger plf, pa, *pv;
{
  PxFont *pfp;

  PX_ErrorInit("xQueryFont/2");
  *pv = PX_Invalid;

  if (DLLookup(pxFontList, (int)plf, (DLHeader **)&pfp))
    PX_Error(no_such_fil);

  switch (pa) {
    PX_GetAttribute(0, xfsp->direction);
    PX_GetAttribute(1, xfsp->min_char_or_byte2);
    PX_GetAttribute(2, xfsp->max_char_or_byte2);
    PX_GetAttribute(3, xfsp->min_byte1);
    PX_GetAttribute(4, xfsp->max_byte1);
    PX_GetAttribute(5, xfsp->all_chars_exist);
    PX_GetAttribute(6, xfsp->default_char);
    PX_GetAttribute(7, xfsp->min_bounds.lbearing);
    PX_GetAttribute(8, xfsp->min_bounds.rbearing);;
    PX_GetAttribute(9, xfsp->min_bounds.width);
    PX_GetAttribute(10, xfsp->min_bounds.ascent);
    PX_GetAttribute(11, xfsp->min_bounds.descent);
    PX_GetAttribute(12, xfsp->min_bounds.attributes);
    PX_GetAttribute(13, xfsp->max_bounds.lbearing);
    PX_GetAttribute(14, xfsp->max_bounds.rbearing);
    PX_GetAttribute(15, xfsp->max_bounds.width);
    PX_GetAttribute(16, xfsp->max_bounds.ascent);
    PX_GetAttribute(17, xfsp->max_bounds.descent);
    PX_GetAttribute(18, xfsp->max_bounds.attributes);
    PX_GetAttribute(19, xfsp->ascent);
    PX_GetAttribute(20, xfsp->descent);
    PX_GetAttribute(21, conn);
    PX_GetAttribute(22, xfid);
    case 23:
      pxMallocBlock = NULL;		/* prevents a free from pxVectorGet */
      pxFontProperty = pfp->xfsp->properties;
      if (pxElements = pfp->xfsp->n_properties) {
        pxType = PX_FontProperty;    
        *pv = PX_Cont;
      } else 
        *pv = PX_End;
      break;
    default: PX_Error(unknown_attribute);
  }
  return FF_Ok;
}

/*
 * pxFontProperties(+LoadedFont, -Next, 0)
 *	integer: LoadedFont, Next
 */
FFInteger
pxFontProperties(plf, pn)
     FFInteger plf, *pn;
{
  PxFont *pfp;

  PX_ErrorInit("xFontProperties/2");
  *pn = PX_Invalid;

  if (DLLookup(pxFontList, (int)plf, (DLHeader **)&pfp))
    PX_Error(no_such_fil);


}

/*
 * pxGetFontProperty(+LoadedFont, +Property, -Value, 0)
 *	integer: LoadedFont, Property, Value
 */
FFInteger
pxGetFontProperty(plf, pp, pv)
     FFInteger plf, pp, *pv;
{
  PxFont *pfp;
  XFontProp *xfpp;
  int i;

  PX_ErrorInit("xQueryFontProperties/2");
  *pv = PX_Invalid;

  if (DLLookup(pxFontList, (int)plf, (DLHeader **)&pfp))
    PX_Error(no_such_fil);

  for (i = 0, xfpp = pfp->xfsp->properties;
       i < pfp->xfsp->n_properties; i++, xfpp++)
    if (pp == xfpp->name) {
      *pv = xfpp->card32;
      return FF_Ok;
    }
  PX_Error("no such font property");
}

/*
 * pxGetCharInfo(+LoadedFont, +Character, -LeftBearing, -RightBearing, -Width,
 *	-Ascent, -Descent, -Attributes, 0)
 *
 *	integer: LoadedFont, Character, LeftBearing, RightBearing, Width,
 *		Ascent, Descent, Attributes
 */
FFInteger
pxGetCharInfo(plf, pc, plb, prb, pw, pa, pd, pattrib)
     FFInteger plf, pc, *plb, *prb, *pw, *pa, *pd, *pattrib;
{
  PxFont *pfp;
  XFontStruct *fsp;
  XCharStruct *csp;
  int index, t, rw;

  PX_ErrorInit("xGetCharInfo/8");
  *plb = *prb = *pw = *pa = *pd = *pattrib = PX_Invalid;

  if (DLLookup(pxFontList, (int)plf, (DLHeader **)&pfp))
    PX_Error(no_such_fil);
  fsp = pfp->xfsp;
  
  if ((fsp->min_byte1 == 0) && (fsp->min_byte1 == fsp->max_byte1)) {
    if ((pc < fsp->min_char_or_byte2) || (pc > fsp->max_char_or_byte2))
      PX_Error(no_such_character);
    index = pc;
  } else {
    t = pc / (rw = fsp->max_char_or_byte2 - fsp->min_char_or_byte2 + 1);
    if ((t < fsp->min_byte1) || (t > fsp->max_byte1))
      PX_Error(no_such_character);
    index = pc - (rw * fsp->min_byte1);
  }  
  
  csp = (fsp->per_char == NULL) ? &(fsp->min_bounds) : &(fsp->per_char[index]);

  *plb = csp->lbearing;
  *prb = csp->rbearing;
  *pw = csp->width;
  *pa = csp->ascent;
  *pd = csp->descent;
  *pattrib = csp->attributes;
  return FF_Ok;  
}

/*
 * pxTextExtents(+LoadedFont, +Format, +XString, -LeftBearing, -RightBearing,
 *	-Width, -Ascent, -Descent, 0)
 *
 *	integer: LoadedFont, Format, LeftBearing, RightBearing, Width, Ascent,
 *		Descent
 *	string: XString
 */
FFInteger
pxTextExtents(plf, pformat, ps, plb, prb, pw, pa, pd)
     FFInteger plf, pformat, *plb, *prb, *pw, *pa, *pd;
     FFString ps;
{
  PxFont *pfp;
  int t;
  XCharStruct xcs;

  PX_ErrorInit("xTextExtents/7");
  *plb = *prb = *pw = *pa = *pd = PX_Invalid;

  if (DLLookup(pxFontList, (int)plf, (DLHeader **)&pfp)) {
    (void)PXVectorFree();
    PX_Error(no_such_fil);
  }
				/* error returns? */
  switch (pformat) {
  case PX_String:
    XTextExtents(pfp->xfsp, ps, strlen(ps), &t, pa, pd, &xcs);
    break;
  case PX_Vector8:
    XTextExtents(pfp->xfsp, pxMallocBlock, pxElements, &t, pa, pd, &xcs);
    (void)PXVectorFree();
    break;
  case PX_Vector16:
    XTextExtents16(pfp->xfsp, pxMallocBlock, pxElements, &t, pa, pd, &xcs);
    (void)PXVectorFree();
  }

  *plb = xcs.lbearing;
  *prb = xcs.rbearing;
  *pw = xcs.width;
  return FF_Ok;
}

/*
 * pxDrawText(+Connection, +Drawable, +GC, +X, +Y, 0)
 * 	integer: Connection, Drawable, GC, X, Y
 */
FFInteger
pxDrawText(pc, pd, pgc, px, py)
     FFInteger pc, pgc, px, py;
{
  PxConnection *pcp;
  PxGC *pgcp;

  PX_ErrorInit("xDrawText/6");

  switch (pxType) {
  case PX_TextItem8:
  case PX_TextItem16:
    break;
  default:
    PX_Error(not_text);
  }
  if (PXLookupConnection(pc, &pcp))
    return PXVectorFree();
  if (DLLookup(pxGCList, (int)pgc, (DLHeader **)&pgcp)) {
    (void)PXVectorFree();
    PX_Error(no_such_gc);
  }

  if (pxType == PX_TextItem8)
    XDrawText(pcp->xDisplay, pd, pgcp->xGC, px, py, pxMallocBlock, pxElements);
  else 
    XDrawText16(pcp->xDisplay, pd, pgcp->xGC, px, py, pxMallocBlock,
		pxElements);
  (void)PXVectorFree();
  return FF_Ok;
}

/*
 * pxImageText(+Connection, +Drawable, +GC, +X, +Y, +Type, +String, 0)
 *	integer: Connection, Drawable, GC, X, Y, Type
 *	string: String
 */
FFInteger
pxImageText(pc, pd, pgc, px, py, pt, ps)
     FFInteger pc, pd, pgc, px, py, pt;
     FFString ps;
{
  PxConnection *pcp;
  PxGC *pgcp;

  PX_ErrorInit("xImageText/6");

  switch (pt) {
  case PX_String:
  case PX_Vector8:
  case PX_Text16:
    break;
  default:
    PX_Error(not_text);
  }
   
  if (PXLookupConnection(pc, &pcp))
    return PXVectorFree();
  if (DLLookup(pxGCList, (int)pgc, (DLHeader **)&pgcp)) {
    (void)PXVectorFree();
    PX_Error(no_such_gc);
  }

  switch (pt) {
  case PX_String:
    XDrawImageString(pcp->xDisplay, pd, pgcp->xGC, px, py, ps, strlen(ps));
    break;
  case PX_Vector8:
    XDrawImageString(pcp->xDisplay, pd, pgcp->xGC, px, py, pxMallocBlock,
		     pxElements);
    (void)PXVectorFree();
    break;
  case PX_Text16:
    XDrawImageString16(pcp->xDisplay, pd, pgcp->xGC, px, py, pxMallocBlock,
		     pxElements);
    (void)PXVectorFree();
    break;
  }
  return FF_Ok;
}

/*
 * eof
 */
