;;  $Id: install-lispworks.lisp,v 1.4 1993/01/25 14:07:11 anjo Exp $
;;  
;;  File	install-lispworks.lisp
;;  Part of	PCE/Lisp interface
;;  Author	Anjo Anjewierden, anjo@swi.psy.uva.nl
;;  Purpose	Installation of PCE/Lisp for Harlequin LispWorks
;;  Works with	PCE 4.5,  LispWorks 3.1
;;  
;;  Notice	Copyright (c) 1992, 1993  University of Amsterdam
;;  
;;  History	26/11/92  (Created)
;;  		05/01/93  (Last Modified)


;;  ------------------------------------------------------
;;  Description
;;  ------------------------------------------------------
;;
;;  This file contains the installation and initialisation functions
;;  specific to LispWorks.
;;  The following functions are available for PCE/LispWorks installation.
;;
;;  PCE::COMPILE-PCE
;;  	Used to compile the Lisp source files.  This only needs to be
;;      done once.
;;  PCE::INSTALL-PCE
;;  	Generates a PCE/Lisp image.  The file "load.lisp" contains
;;      several Lisp variables that must be customised first.
;;  PCE::FAST-BOOT-PCE
;;      Loads PCE and then runs it.  Only used for PCE/Lisp development.
;;  PCE::DUMMY-BOOT-PCE
;;      Loads PCE only.  Only used for PCE/Lisp development.
;;
;;  A typical sequence would be:
;;
;;  -   Edit the file "load.lisp" and customise for the local environment.
;;      See under "Customisation".
;;  -	Start Lisp.
;;  -	Load the load file, i.e. (load "load").
;;  -	Compile the PCE/Lisp source code, i.e. (pce::compile-pce).
;;  -	Quit Lisp.
;;  -	Start Lisp.
;;  -	Load the load file, i.e. (load "load").
;;  -	Generate the PCE/Lisp image, i.e. (pce::install-pce).
;;  -	Quit Lisp.


;;  ------------------------------------------------------
;;  Directives
;;  ------------------------------------------------------

(in-package "PCE")


;;  ------------------------------------------------------
;;  Loading and Initialisation
;;  ------------------------------------------------------

;;! install-pce
;;
;;  Generate a PCE/Lisp image.  The image is saved in the file
;;  specified with *pce-saved-state* (see "load.lisp").

(defun install-pce ()
  (load-pce)                            ; load PCE image
  (saved-state-pce *pce-saved-state*))  ; save to disk


;;! fast-boot-pce
;;
;;  Loads the PCE object code and initialises PCE without creating a
;;  saved image.  Only used to test the PCE/Lisp interface.  This is
;;  the standard function to test new releases of PCE/Lisp.

(defun fast-boot-pce ()
  (load-pce)                            ; load PCE image
  (init-pce))                           ; initialise PCE


;;! dummy-boot-pce
;;
;;  Loads the PCE object code without initialising PCE itself.
;;  Used to debug the initialisation routines.

(defun dummy-boot-pce ()
  (load-pce))


;;! compile-pce
;;
;;  Compile the PCE/Lisp source code.

(defun compile-pce ()
  (dolist (file *pce-lisp-sources*)
	  (compile-file file :print 0)))


(defun load-pce ()
  (lw-load "sun4"))


(defun saved-state-pce (name)
  (in-package "USER")
  (use-package "PCE")
  (save-image name			; Name of the image
	      :restart-function #'pce::init-pce))


;;! init-pce
;;
;;  Initialisation of the PCE/Lisp interface.

(defun init-pce ()
  (initialise-pce)
  (pce-lispworks-initialise)
  (pce-lispworks-main-loop))


;;  ------------------------------------------------------
;;  Loading PCE object files
;;  ------------------------------------------------------

;;! find-pce-object-files files ==> object-files
;;
;;  Returns a list of all object file names part of PCE.  The names
;;  are found in a file called "files" in the machine specific PCE
;;  sub-directory.  A typical call is:
;;
;;=	(find-pce-object-files "sun4/files")

(defun find-pce-object-files (files)
  (let (all-files)
    (let* ((file-name (string-append *pce-directory* files)))
      (if (not (probe-file file-name))
	  (error "Cannot find ``~A''.~%Is variable *pce-directory* correct?~%"
		 file-name)
	(let ((file (open file-name)))
	  (loop
	   (let ((object-file (read-line file nil nil)))
	     (unless object-file
		     (close file)
		     (return nil))
	     (push object-file all-files))))))
    (reverse all-files)))


(defun lw-load (machine-type)
  (let ((old (change-directory "."))
	(objects (find-pce-object-files (string-append machine-type
						       "/files"))))
    (change-directory (string-append *pce-directory* machine-type))
    (apply #'read-foreign-modules
	   (append objects
		   (list (string-append (directory-namestring old)
					"pce-lispworks.o")
			 (string-append (directory-namestring old)
					"host-lispworks.o")
			 (string-append (directory-namestring old)
					"libgcc.a")
			 "-lXt"
			 "-lX11"
			 "-lpixrect"
			 "-lm")))
    (change-directory old)))
