
:- module(paper,[]).


:- ensure_loaded(library(ytoolkit)).
:- ensure_loaded(library('behaviors/resizeable')).
:- ensure_loaded(library(edipo)).

default(border,3).
default(border_pixmap,["# "," #"]).
default(cursor,crosshair).
%default(function,invert).
default(function,xor).

behavior(A,B) :- resizeable(A,B).

behavior(expose(0),Name) :- 
	clear,
	recorded(Name,paper_drawing(Drawing),_),
	Drawing.
behavior(buttonPress(1,X,Y),Name) :- 
	recorded(Name,paper_drawing(Drawing),Ref),
	in(Drawing,X,Y), !,
	erase(Ref),
	(Drawing=writestring(_,_,_) ->
		retype(Drawing,Name,NewDrawing)
	;
		resize(Drawing,PX,PY,Call,NewDrawing),
		window(Name,[cursor=sizing]), 
		while_moving(Drawing,Name,PX,PY,Call,NewDrawing),
		window(Name,[cursor=crosshair])
		
	),
	NewDrawing, recorda(Name,paper_drawing(NewDrawing),_),
	behavior(expose(0),Name).
behavior(buttonPress(2,X,Y),Name) :- 
	recorded(Name,paper_drawing(Drawing),Ref),
	in(Drawing,X,Y), !,
	erase(Ref),
	(Drawing=writestring(TX,TY,T) ->
		stringbox(T,W,H), TCY is TY-H, 
		Call=move_drawing(Drawing,X,Y,W,H,PX,PY,NX,_NY,RY),
		window(Name,[cursor=fleur]), 
		while_moving(crectangle(TX,TCY,W,H),Name,PX,PY,Call,crectangle(NX,RY,W,H)),
		window(Name,[cursor=crosshair]), 
		CY is RY+H, NewDrawing=writestring(NX,CY,T)
	;
		move(Drawing,X,Y,PX,PY,Call,NewDrawing),
		window(Name,[cursor=fleur]), 
		while_moving(Drawing,Name,PX,PY,Call,NewDrawing),
		window(Name,[cursor=crosshair])
	),
	NewDrawing, recorda(Name,paper_drawing(NewDrawing),_),
	behavior(expose(0),Name).
behavior(buttonPress(3,X,Y),Name) :- 
	recorded(Name,paper_drawing(Drawing),Ref),
	in(Drawing,X,Y), !,
	erase(Ref),
	window(Name,[cursor=pirate]),
	wait_for(Name,buttonRelease(3,_,_)),
	window(Name,[cursor=crosshair]),
	Drawing.

/***********************************************\
*						*
*	graphical object manipulation 		*
*						*
\***********************************************/

% in(Objects,X,Y)   point (X,Y) is in Object 

in(line(X,Y,X,Y),X,Y) :- !, line(X,Y,X,Y).
in(line(X1,Y1,X2,Y2),X,Y) :-
	d(X1,Y1,X2,Y2,D1), 
	d(X1,Y1,X,Y,D2), 
	d(X2,Y2,X,Y,D3),
	D2 + D3 - D1 < 0.1,
	(Y>=Y1, Y=<Y2; Y>=Y2, Y=<Y1).

in(poly([(X,Y),(X,Y)]),X,Y) :- !.
in(poly([(X,Y)|R]),SX,SY) :- in_poly(R,X,Y,SX,SY).

in_poly([(X2,Y2)|_],X1,Y1,SX,SY) :- in(line(X1,Y1,X2,Y2),SX,SY), !.
in_poly([(X,Y)|R],_,_,SX,SY) :- in_poly(R,X,Y,SX,SY).

in(rectangle(X,Y,0,0),X,Y) :- !.
in(rectangle(X,Y,_,H),X,SY) :- !, SY >= Y, SY =< Y+H.
in(rectangle(X,Y,W,_),SX,Y) :- !, SX >= X, SX =< X+W.
in(rectangle(X,Y,W,H),SX,SY) :- SX is X+W, !, SY >= Y, SY =< Y+H.
in(rectangle(X,Y,W,H),SX,SY) :- SY is Y+H, !, SX >= X, SX =< X+W.

in(frectangle(X,Y,0,0),X,Y) :- !.
in(frectangle(X,Y,W,H),SX,SY) :- 
	SX >= X, SX =< X+W,
	SY >= Y, SY =< Y+H.

in(circle(X,Y,0),X,Y) :- !.
in(circle(SX,SY,R),X,Y) :-
	d(SX,SY,X,Y,D), E is D-R, E > -2, E < 2.

in(fcircle(X,Y,0),X,Y) :- !.
in(fcircle(SX,SY,R),X,Y) :-
	d(SX,SY,X,Y,D), D =< R.


in(writestring(X,Y,[]),X,Y) :- !.
in(writestring(X,Y,C),SX,SY) :-
	stringbox(C,W,H),
	YC is Y-H,
	in(frectangle(X,YC,W,H),SX,SY).

%
%	 how to resize an object
%

resize(line(_,_,X2,Y2),PX,PY,line(PX,PY,X2,Y2),line(PX,PY,X2,Y2)).

resize(poly([])).

resize(rectangle(X,Y,_,_),PX,PY,
	(W is PX-X, H is PY-Y,
	change_rect(X,Y,W,H,NX,NY,NW,NH),
	rectangle(NX,NY,NW,NH)),
	rectangle(NX,NY,NW,NH)).


resize(frectangle(X,Y,_,_),PX,PY,
	(W is PX-X, H is PY-Y,
	change_rect(X,Y,W,H,NX,NY,NW,NH),
	frectangle(NX,NY,NW,NH)),
	frectangle(NX,NY,NW,NH)).



resize(circle(X,Y,_),SX,SY,(d(X,Y,SX,SY,NR),circle(X,Y,NR)),circle(X,Y,NR)).

resize(fcircle(X,Y,_),SX,SY,(d(X,Y,SX,SY,NR),fcircle(X,Y,NR)),fcircle(X,Y,NR)).


retype(writestring(X,Y,C),P,writestring(X,Y,NC)) :-
	writebox(0,CW,CH),
	new_widget(field,[
		parent=P,
		x=X-CW,
		y=Y-CH-2,
		chars=C,
		callback=ytoolkit:exit_inner_loop
	],Field),
	inner_loop,
	current_widget(_,[chars=NC],Field),
	kill_widget(_,[],Field).
	
		

%
% 	how to move an object
%

move(line(X1,Y1,X2,Y2),X,Y,PX,PY,
	move_line(X1,Y1,X2,Y2,X,Y,PX,PY,FX1,FY1,FX2,FY2),
	line(FX1,FY1,FX2,FY2)).

move_line(X1,Y1,X2,Y2,X,Y,PX,PY,FX1,FY1,FX2,FY2) :-
		DX is PX-X, DY is PY-Y, 
		FX1 is X1+DX, FY1 is Y1+DY,
		FX2 is X2+DX, FY2 is Y2+DY,
		line(FX1,FY1,FX2,FY2).

move(rectangle(X,Y,W,H),SX,SY,PX,PY,
	move_rect(X,Y,W,H,SX,SY,PX,PY,NX,NY),
	rectangle(NX,NY,W,H)).

move_rect(X,Y,W,H,SX,SY,PX,PY,NX,NY) :- 
	NX is X+PX-SX, NY is Y+PY-SY, 
	rectangle(NX,NY,W,H).


move(frectangle(X,Y,W,H),SX,SY,PX,PY,
	move_frect(X,Y,W,H,SX,SY,PX,PY,NX,NY),
	frectangle(NX,NY,W,H)).

move_frect(X,Y,W,H,SX,SY,PX,PY,NX,NY) :- 
	NX is X+PX-SX, NY is Y+PY-SY, 
	frectangle(NX,NY,W,H).


move(circle(X,Y,R),SX,SY,PX,PY,
	move_circle(X,Y,R,SX,SY,PX,PY,NX,NY),
	circle(NX,NY,R)).

move_circle(X,Y,R,SX,SY,PX,PY,NX,NY) :-
	NX is X+PX-SX, NY is Y+PY-SY, 
	circle(NX,NY,R).

move(fcircle(X,Y,R),SX,SY,PX,PY,
	move_fcircle(X,Y,R,SX,SY,PX,PY,NX,NY),
	fcircle(NX,NY,R)).

move_fcircle(X,Y,R,SX,SY,PX,PY,NX,NY) :-
	NX is X+PX-SX, NY is Y+PY-SY, 
	fcircle(NX,NY,R).


move_drawing(writestring(X,Y,T),SX,SY,_W,H,PX,PY,NX,NY,RY) :-
	NX is X+PX-SX, NY is Y+PY-SY, 
	writestring(NX,NY,T),
	RY is NY-H.

%---------------------------------------%
%					%
% 	private paper utils		%
%					%
%---------------------------------------%

% distance of two points	
%d(X1,Y1,X2,Y2,D) :- D is sqrt((X2-X1)^2+(Y2-Y1)^2). % ^ means xor in SICStus
% and sqrt is not an is/2 function.

d(X1,Y1,X2,Y2,D) :- W is (X2-X1)*(X2-X1) + (Y2-Y1)*(Y2-Y1), sqrt(W,D).

:- ytoolkit:engine(quintus) -> 
	ensure_loaded(library(math));
	asserta((sqrt(X,Y) :- Y is sqrt(X))).

% :- meta_predicate while_moving(?,?,?,:,:).

while_moving(Old,W,X,Y,Call,Draw) :-
	recorda('$recall',Old,_),
	repeat,
	event(W,E),
	recall(E,X,Y,Call,Draw), !.

recall(motionNotify(_,_),_,_,_,_) :- 
	recorded('$recall',Call,Ref), erase(Ref), Call, fail.
recall(motionNotify(X,Y),X,Y,Call,Draw) :- !,
	Call, recorda('$recall',Draw,_), fail.
recall(buttonRelease(_,X,Y),X,Y,_,Draw) :- !,
	recorded('$recall',Draw,Ref), !,
	Draw,
	erase(Ref).


/* drawing utils */

my_rectangle(X,Y,W,H) :- 
	change_rect(X,Y,W,H,NX,NY,NW,NH),
	rectangle(NX,NY,NW,NH).

change_rect(X,Y,W,H,NX,NY,NW,NH) :-
	(W > 0 ->
		NX=X, NW=W
	;	
		NX is X+W, NW is -W
	),
	(H > 0 ->
		NY=Y, NH=H
	;
		NY is Y+H, NH is -H
	).

my_circle(X,Y,XR,YR) :-
	W is (XR-X)*(XR-X)+(YR-Y)*(YR-Y),
	sqrt(W,R),
	circle(X,Y,R).

