/***********************************************************************\
*									*
*	C								*
*	O	This is just another implementation of a tetrix 	*
*	L	variant, made to have a different example of what	*
*	U	can be done in Prolog using Edipo, in particular	*
*	M	the two different interaction models (see bellow)	*
*	N								*
*	S	Ze' Paulo Leal, Easter 92				*
*									*
\***********************************************************************/

:- ensure_loaded(library(edipo)).

:- ensure_loaded(random).

speed(400).		% sun4
%speed(2000).		% mips 4680

/************************

Main 

************************/

columns :-
	initialize_interface,
	initialize_game,
	set_val(abort,false),
	repeat,
		do_game,
	get_val(abort,true).

do_game :-	wait_for_user,
	initialize_game,
	repeat,
		lower_one_colum,
	(complete, !, end_game; get_val(abort,true)).

lower_one_colum :-
		new_column,
		repeat,
			user_change,
		(
		    lower_column, !, 	% succeeds when hits the bottom
		    remove_symbols,
		    update
		; 
		    get_val(abort,true)
		).

/************************

Initializations and Finalizations

************************/


initialize_interface :-
	(get_val(inited,true) -> true 	% only have to do this once
    ;
	init_random,			% initialize random seed
	dont_flush,
	initialize_windows,

	speed(S), G is S*125,
	set_val(letters,5),		% number of different letters
	set_val(level_gap,G),		% time between level changes
	set_val(status,'Stop'),		% playing or not
	set_val(inited,true)
    ).

initialize_game :-
	speed(S),

	set_val(level,1),		% level counter  (reset each game)
	set_val(points,0),		% player points
	set_val(time,0),		% time counter
	set_val(speed,S),		% max user moves between indep. move

	unset(pos(_,_)),		% clear lane
	unset(remove(_,_)),
	unset(incol),

	current_window(lane,[height=HL]),
	current_window(col,[height=HC]),
	LIM is HL-HC,			% this is how far a columnn can fall

	set_cval(col(0),LIM),
	set_cval(col(1),LIM),
	set_cval(col(2),LIM),
	set_cval(col(3),LIM),
	set_cval(col(4),LIM),

	working_envir(lane), clear,
	redraw(lane),
	redraw(points).

initialize_windows :-
	choose_font(my_lane_font(LP),LP,LF),
	choose_font(my_button_font(BP),BP,BF),
	choose_font(my_points_font(PP),PP,PF),

	background(BKA,BKV),
	window(columns,[		% Top level window
    		width=340,height=440,
		BKA=BKV]),
	window(lane,[			% columns go down the lane ...
    		parent=columns,
		x=60,y=20,
		width=100,height=400,
		border=2]),
	gc(lane,[font=LF]),
	init_colors,
	window(col,[			% the falling column
		parent=lane,
		x=0,y=0,
		width=20,height=60]),
	gc(col,[font=LF]),
	init_colors,
	window(init,[			% init button - restarts game
    		parent=columns,
		x=220,y=200,
		width=80,height=30,
		border=1]),
	gc(wide,[line_width=4,font=BF]),% for the button's 3D look
	window(status,[			% status button - stop/cont. playing
		parent=columns,
		x=220,y=240,
		width=80,height=30,
		border=1]),
	gc(wide,[line_width=4,font=BF]),
	window(abort,[parent=columns,	% abort button - back to Prolog
		x=220,y=280,
		width=80,height=30,
		border=1]),
	gc(wide,[line_width=4,font=BF]),
	window(points,[			% where the points/level are shown
		parent=columns,
		x=200,y=60,
		width=120,height=60,
		border=1]),
	gc(points,[font=PF]).

complete  :- get_cval(col(_),V), V<0.

end_game :-
	bell(0), bell(0), bell(0),	
	set_val(status,'Stop'),
	redraw(status).

new_column :-
	random(V), X is (V mod 5)*20,
	window(col,[x=X,y=0]),
	get_val(letters,R),
	repeat,
/* when using letters instead of colors
		random(V0), S0 is "A" + V0 mod R,
		random(V1), S1 is "A" + V1 mod R,
		random(V2), S2 is "A" + V2 mod R,
*/
		random(V0), S0 is V0 mod R,
		random(V1), S1 is V1 mod R,
		random(V2), S2 is V2 mod R,
	\+ (S0==S1, S1==S2), !,
	set_val(incol,incol(S0,S1,S2)),
	redraw(col),
	inc_points(10).


/************************

Interaction

Two different inteaction modes are uses:
	user_change/0		reads events noblocking during game
	wait_for_user/0		reads event with blocking

************************/


user_change :- 	(events(W,E) ->	process(E,W); get_val(status,'Play')).

wait_for_user :- 
	repeat,
	event(W,E), process(E,W),
	(get_val(status,'Play'); get_val(abort,true)).



/************************

User interaction

************************/

/* game keys {'h'=left,'j'=rt.up,'k'=rt.down,'l'=left,' '=fall} */

process(keyPress("j",_,_,_),_) :-  get_val(status,'Play'), !,
	current_window(col,[x=X,y=Y,width=W]),
	C is X//W-1, 
	(get_cval(col(C),T) -> Y<T; true),
	NX is X-W, NX >= 0,
	window(col,[x=NX]).
process(keyPress("l",_,_,_),_) :-  get_val(status,'Play'), !,
	current_window(col,[x=X,y=Y,width=W]),
	C is X//W+1, 
	(get_cval(col(C),T) -> Y<T; true),
	NX is X+W, NX < 100,
	window(col,[x=NX]).
process(keyPress("k",_,_,_),_) :-  get_val(status,'Play'), !,
	get_val(incol,incol(A,B,C)),
	set_val(incol,incol(B,C,A)),
	redraw(col).
process(keyPress("i",_,_,_),_) :-  get_val(status,'Play'), !,
	get_val(incol,incol(B,C,A)),
	set_val(incol,incol(A,B,C)),
	redraw(col).
process(keyPress(" ",_,_,_),_) :-  get_val(status,'Play'), !,
	get_val(time,T),
	get_val(speed,S),
	NT is T-(T mod S)+S,
	set_val(time,NT),
	current_window(col,[x=X,y=OY,width=W]),
	C is X//W,
	get_cval(col(C),V),
	window(col,[y=V]),
	P is (V-OY)//W*5,
	inc_points(P).

/* button handling */

process(buttonPress(_,_,_),status) :- bell(0),
	get_val(status,S),
	toggle(S,NS),
	set_val(status,NS),
	redraw(status).
process(buttonPress(_,_,_),init) :-	columns.
process(buttonPress(_,_,_),abort) :-	do_flush, flush,
	set_val(abort,true).



/************************

(Re)Drawing

	this is the only place were (most of) actual drawing is done,
	in response to expose(0) events

************************/


process(expose(0),points) :-
	working_envir(points),
	get_val(points,P),
	get_val(level,L),
	writeterm(5,20,'points:                  '), writeterm(70,20,P),
	writeterm(5,40,'level:                   '), writeterm(70,40,L).


process(expose(0),col) :- 
	working_envir(col), clear,
	get_val(incol,incol(A,B,C)),
	working_envir(col,A), frectangle(0, 0,20,20),
	working_envir(col,B), frectangle(0,20,20,20),
	working_envir(col,C), frectangle(0,40,20,20).

/* if using letters instead of colors
	writestring(5,20,[A]),
	writestring(5,40,[B]),
	writestring(5,60,[C]).
*/

process(expose(0),lane) :-
	working_envir(lane),
%	(get_val(clear_lane,true) -> 
%	    crectangle(0,0,100,400), 
%	    set_val(clear_lane,false); true),
	(get_cval(pos(Line,Col),S),
/* when using letters instead of colors
		X is Col*20+5, Y is (Line+1)*20, 
		writestring(X,Y,[S]), 
*/
		X is Col*20, Y is Line*20, 
		working_envir(lane,S), frectangle(X,Y,20,20),
	fail; true),
	working_envir(col),
	(get_cval(col(C),V),
		XL is C*20+1,
		YL is V+57,
		frectangle(XL,YL,18,2),
	 fail).

/* drawing buttons */
process(expose(0),status) :- 
	get_val(status,S), 
	toggle(S,T),
	draw_button(status,T).
process(expose(0),init) :- draw_button(init,'Restart').
process(expose(0),abort) :- draw_button(abort,'Quit').

process(_,_).

redraw(X) :- process(expose(0),X). 	% sometimes one has to force redrawing

draw_button(B,S) :-
	working_envir(B),	clear,
	current_window(B,[width=W,height=H]),
	line(2,2,2,H),	line(2,2,W,2),
	working_gc(wide),
	line(2,H,W,H), line(W,2,W,H),
	writebox(S,SW,SH),
	SX is (W-SW)//2, SY is (H+SH)//2,
	writeterm(SX,SY,S).


toggle('Play','Stop').
toggle('Stop','Play').

/************************

Column handling

************************/


lower_column :- 
	(is_time ->
	    current_window(col,[x=X,y=Y,width=W]),
	    NY is Y+W,
	    C is X//W,
	    get_cval(col(C),V),
	    (NY > V -> true; window(col,[y=NY]), fail)
	),
	NV is V-60,
	set_cval(col(C),NV),
	L is Y//W,
	fix_col(C,L).

fix_col(C,L) :-
	get_val(incol,incol(S0,S1,S2)),
			set_cval(pos(L ,C),S0),	set_cval(new(L ,C),true),
	L1 is L+1,	set_cval(pos(L1,C),S1), set_cval(new(L1,C),true),
	L2 is L+2,	set_cval(pos(L2,C),S2), set_cval(new(L2,C),true),
	redraw(lane).	% this is really needed if you have backing store

remove_symbols :-
	(
	    get_cval(new(L,C),_), \+ get_val(remove(L,C),_),
	    	get_cval(pos(L,C),V),
		unset_cval(new(L,C)),
		remove_symbols(L,C,V),
	    fail
	).
remove_symbols :- 
	(get_cval(remove(_,_),_) -> % if there is at least one remotion
	    remove_above,
	    remove_symbols
	;
%	    set_val(clear_lane,true),
%	    redraw(lane)
	true
	).

remove_symbols(L,C,V) :-
	mark_remove(L,C),
	dir(LD,CD),
	try_remove(L,C,LD,CD,V,F),	% remove forward, on bcktr unremove
	try_remove(L,C,-LD,-CD,V,B),	% same  backwards
	F + B >= 2, !,			% total removed plus this one >= 3
	inc_points(100).

dir(0,1). 	% horizontal line
dir(1,0). 	% vertical line
dir(1,1). 	% diag /
dir(1,-1).	% diag \

try_remove(L,C,LD,LC,V,Rcount) :- 
	NL is L+LD, NC is C+LC, 
	(get_cval(pos(NL,NC),V) ->
	    mark_remove(NL,NC),
	    try_remove(NL,NC,LD,LC,V,NRcount),
	    Rcount is NRcount +1
	;
	    Rcount=0
	).

mark_remove(L,C) :- set_cval(remove(L,C),true).
mark_remove(L,C) :- unset_cval(remove(L,C)).		% on bcktr unremove

remove_above :- 
	(get_cval(remove(L,C),_) ->
		unset_cval(remove(L,C)),
		remove_above(L,C),
		remove_above   ;
		true
	).

remove_above(L,C) :-
	K is L-1,
	(get_cval(remove(K,C),_) -> 
	    unset_cval(remove(K,C)),
	    set_cval(remove(L,C),_)
	;
	    true
	),
	X is C*20, Y is L*20, 
	(get_cval(pos(K,C),V) ->
	    remove_above(K,C),
	    set_cval(pos(L,C),V),
	    working_envir(lane,V), frectangle(X,Y,20,20),
	    set_cval(new(L,C),true)
	;
	    unset_cval(pos(L,C)),
	    working_envir(lane),
	    crectangle(X,Y,20,20),
	    get_cval(col(C),T),
	    W is T+20, !,
	    set_cval(col(C),W)
	).


/************************

Points

************************/

inc_points(P) :-
	recorded(points,O,Ref), erase(Ref),
	Q is P+O,
	recorda(points,Q,_).

update :- 
	get_val(time,T),
	get_val(level,L),
	get_val(level_gap,G),
	(T > G ->
	    set_val(time,0),
	    NL is L+1, 
	    set_val(level,NL),
	    get_val(speed,S),
	    NS is S*7//10,
	    set_val(speed,NS)
	; true),
	redraw(points).

/************************

Fonts, colors amd bitmaps

************************/


choose_font(G,P,F) :- call(G), font_list(P,F), !.

my_lane_font('9x15bold').
my_lane_font('9x15').
my_lane_font(*).

my_button_font('-*-helvetica-bold-r-*-*-14-*-*-*-*-*-*-*').
my_button_font('9x15bold').
my_button_font(*).

my_points_font('-*-courier-medium-r-normal-*-14-*-*-*-*-*-*-*').
my_points_font('9x15').
my_points_font(*).


my_color :- color, \+  get_val(use_color,false).

background(BKA,BKV) :-	
	(color ->	
	    BKA=bkg_pixel,
	    BKV=pink
	;
	    BKA=bkg_pixmap,
	    BKV=["#   ","  # "," #  ","   #"]
	).

init_colors :- 
	    color_def(N,C,T),
	    (color -> I=C; I=black),
	    (gc(N,[foreground=I,tile=T,fill_style=tiled]) -> true;
		gc(N,[foreground=I,fill_style=tiled])
	    ),
	    fail.
init_colors.

color_def(0,rosyBrown		,["####","#   ","# ##","# ##"]).
color_def(1,slateBlue2		,["## #","  # ","## #","  # "]).
color_def(2,orangeRed			,["#   ","  # "," #  ","   #"]).
color_def(3,darkOliveGreen	,[" ###","## #","# ##","### "]).
color_def(4,lightYellow2	,["#  #","##  "," ## ","  ##"]).

/************************

Attributes and time

************************/



set_val(N,_) :- recorded(N,_,Ref),  erase(Ref),  fail.
set_val(N,V) :- recorda(N,V,_).

get_val(N,V) :- recorded(N,V,_).


set_cval(N,_)  :- recorded(N,cval(N,_),Ref),  erase(Ref),  fail.
set_cval(N,V) :- recordz(N,cval(N,V),_). % must be in the end for repeat-fail

get_cval(N,V) :- recorded(N,cval(N,V),_).

unset(C) :- (recorded(C,_,R), erase(R), fail; true).

unset_cval(C) :- (recorded(C,cval(C,_),R), erase(R), fail; true).

is_time :-
	recorded(speed,S,_),
	recorded(time,T,Ref), erase(Ref),
	NT is T+1,
	recorda(time,NT,_),
	0 is T mod S.



/************************

Debug

************************/

test(L) :- 
	set_lane(L),
%	repeat, event(_,keyPress("c",_,_,_)),
	get0(_),
	remove_symbols,
	working_window(lane), clear,
	redraw(lane).

set_lane(L) :- 
	initialize_game,
	set_lane(L,14),
	set_val(clear_lane,false),
	redraw(lane).

set_lane([],_).
set_lane([L|R],LN) :- 
	set_lane(L,LN,0), 
	LM is LN-1,
	set_lane(R,LM).

set_lane([],_,_).
set_lane([V|R],LN,CN) :- 
	(V==32 -> 
	    true; 
	    set_cval(pos(LN,CN),V),
	    T is (LN-3)*20,
	    set_cval(col(CN),T)
	),
	DN is CN+1, 
	set_lane(R,LN,DN).



mkst :- save('columns.ys'), columns.

:- nl, write('*** type this please *** 		| ?- columns.').

%EOF
