/*  COMMAND_HELP.PL  */


:- module command_help.


:- public command_help/3.


/*
SPECIFICATION
-------------

This module defines the short error messages and explanations for
commands.


PUBLIC command_help( CommandWord+, ErrorCode+, Short-, Explanation- ):

The error ErrorCode, detected in command CommandWord, has short error
message Short and explanation Explanation.
*/


/*
IMPLEMENTATION
--------------

Note that the module adds some extra clauses for 'output', to link
different mesages together via 'cex_', and to add some shorthands. See
the description of 'add_user_output' in OUTPUT.PL.
*/


:- needs
        for_in_list / 4,
        output / 1,
        token_contents / 2,
        add_user_output/1. 


/*
First, the messages for specific commands.
------------------------------------------
*/

command_help(
copy, bad_fact_number_2/_,
'There should be just one number after the word "to".'~,
cex_(copy)
).

command_help(
copy, bad_fact_number_1/_,
'There should be a number after the word "copy",  optionally followed by'~<>
'the word "to" and another number.'~,
cex_(copy)
).


command_help(
change, bad_fact_or_question_range/_,
'After the word "in", there should be something denoting a fact.'~,
cex_(change)
).

command_help(
change, missing_in,
'The replacement should be followed by the word "in" and a range.'~,
cex_(change)
).

command_help(
change, bad_replacement/_,
'There should be a replacement after the word "to".'~,
cex_(change)
).

command_help(
change, missing_to,
'There should be the word "to" after the thing to be replaced.'~,
cex_(change)
).

command_help(
change, bad_replacee/_,
'There should be something to be replaced after the word "change".'~,
cex_(change)
).

command_help(
edit, token_error(Message,Input),
'There is a syntax error in what you have edited'~,
seplist_(Message_,' ')~<>
'The error was detected near the point shown below:'~<>
puts_(Input)~
) :-
    upper_case_first_element( Message, Message_ ).


/*
The range errors will catch some others.
----------------------------------------
*/

command_help(
delete, E,
'There is a grammatical error in your command, but it\'s hard to be'~<>
'precise about where.'~,                    
cex_(delete)
).


/*
Now, for commands of which more than one has the same argument pattern.
(Actually, some don't, but they might in future).
*/


/*
Commands with no arguments.
---------------------------
*/

command_help(
C, excess_after,
'There must be nothing after the word'...q_(C)...'.',
'Just type:'~<>
'    '<>C<>'.'~
).


/*
Commands taking an optional printer name as argument.
-----------------------------------------------------
*/

command_help(
C, bad_printer_name/Rem,
qtokens_(Rem)...'is not the name of a printer.'~,
cex_(C)
).


/*
Commands taking a file name ("knowledge base") as argument.
-----------------------------------------------------------
*/

command_help(
C, bad_file_name/Rem,
qtokens_(Rem)...'is not acceptable as the name for a knowledge base.'~,
cex_(C)
).

command_help(
C, missing_file_name,
'The word'...q_(C)...'should be followed by the name of a knowledge base.'~,
cex_(C)
).


/*
Commands taking a script name as argument.
------------------------------------------
*/

command_help(
C, bad_script_name/Rem,
qtokens_(Rem)...'is not the name of a script.'~,
cex_(C)
).

command_help(
C, missing_script_name,
'The word'...q_(C)...'should be followed by the name of a script.',
cex_(C)
).

command_help(
script, file_not_found,
'Though the name of your script is correct, I can\'t find it.'~,
'Try another name, or ask your teacher to check the names of.'~<>
'all the scripts.'~
).


/*
Commands taking a section number as argument.
---------------------------------------------
*/

command_help(
C, bad_section_number/Rem,
qtokens_(Rem)...'is not a section number.'~,
cex_(C)
).

command_help(
C, no_section_found/Rem,
'That section is not in the script.'~,
cex_(C)
).

command_help(
C, section_zero,
'There is no section 0.'~,
none
).

command_help(
C, missing_section_number,
'The word'...q_(C)...'should be followed by a section number.'~,
cex_(C)
).


/*
Commands taking a fact or question range as argument.
-----------------------------------------------------
*/

command_help(
C, bad_fact_or_question_range/Rem,
qtokens_(Rem)...'does not denote a fact or question.'~,
cex_(C)
).


/*
Commands taking a fact or question range, or printer.
-----------------------------------------------------
*/

command_help(
C, bad_range_or_printer_name/Rem,
qtokens_(Rem)...'should be either a printer, or a fact or question range.'~,
cex_(C)
).


/*
File errors.
------------
*/

command_help(
save, no_space_for_file,
'There is not enough space left on the disc to hold your facts.',
'This probably means that your teacher will have to delete some'~<>
'old files. Please ask for help.'~
).

command_help(
save, directory_not_found,
'This shouldn\'t happen. Please tell your teacher that I can\'t'~<>
'find the directory for this file.'~,
'You will have to ask for help.'
).

command_help(
save, access_forbidden,
'This shouldn\'t happen. Please tell your teacher that I can\'t'~<>
'access this file.'~,
'You will have to ask for help.'
).

command_help(
save, invalid_filename,
'There is something wrong with the name of your file.',
-cex_(save)~<>
'You are not allowed funny symbols like $, or names starting with numbers,'~<>
'or names longer than 32 characters.'~
).

command_help(
Get, file_not_found,
'Though the name of your file is correct, I can\'t find it.'~,
'Try another name, or ask your teacher to check the names of.'~<>
'all the files.'~
) :- is_get( Get ).

command_help(
Get, directory_not_found,
'This shouldn\'t happen. Please tell your teacher that I can\'t'~<>
'find the directory for this file.'~,
'You will have to ask for help.'
) :- is_get( Get ).

command_help(
Get, access_forbidden,
'This shouldn\'t happen. Please tell your teacher that I can\'t'~<>
'access this file.'~,
'You will have to ask for help.'
) :- is_get( Get ).

command_help(
Get, invalid_filename,
'There is something wrong with the name of your file.',
-cex_(Get)~<>
'You are not allowed funny symbols like $, or names starting with numbers,'~<>
'or names longer than 32 characters.'~
) :- is_get( Get ).


/*
Ranges.
-------               
*/

command_help(
C, l_not_integer/L,
L...'is not an integer.'~,
cex_(C)
).

command_help(
C, u_not_integer/U,
U...'is not an integer.'~,
cex_(C)
).

command_help(
C, wrong_order/[L, U],
U...'is smaller than'...L<>'.'~,
'You can\'t say something like 60/40.'~<>
'The numbers are the wrong way round.'~
).


/*
Examples.
---------
*/


command_example( again,
'    again.             ---     to repeat the current section.'~<>
'The word "again" must always be followed immediately by a dot or'~<>
'question mark.'~
).


command_example(analyse,
'    analyse.           ---     to analyse everything.'~<>
'    analyse 1.         ---     to analyse fact 1.'~<>
'    analyse 19/45.     ---     to analyse facts 19 to 45.'~<>
'    analyse loves.     ---     to analyse all facts for "loves".'~<>
'    analyse q.         ---     to analyse the last question.'~<>
'And you can put a printer name just after the command to print the'~<>
'output:'~<>
'    analyse ctc 19/45  ---     to print the analysis of facts 19/45'~<>
'                               at CTC.'~<>
'    analyse ep loves   ---     to print the analysis of all facts for'~<>
'                               "loves" at Psychology.'~<>
'for example.'~
).


command_example(append,
'    append myfacts.'~<>
'    append tim1.'~<>
'to append what\'s in MYFACTS or TIM1 to your knowledge base.'~
).


command_example( bye,
'    bye.               ---     to leave the Tutor.'~<>
'The word "bye" must always be followed immediately by a dot or'~<>
'question mark.'~
).


command_example(change,
'    change x to X in all.'~<>
'    change john to mary in 10.'~<>
'    change john to mary in 19/45.'~<>
'    change \'john loves\' to \'jim lives\' in all.'~<>
'are examples of changing some facts.'~<>
'You can also change the last question and re-ask it:'~<>
'    change john to mary in q.'~
).


command_example(copy,
'    copy 10 to 33.     ---     copies fact 10 to 33, replacing 33'~<>
'                               if it was already there.'~<>
'    copy 10.           ---     copies fact 10 to make a new fact at'~<>
'                               the end of the knowledge base.'~
).


command_example(delete,
'    delete             ---     deletes the fact with the highest number.'~<>
'    delete 30          ---     deletes fact 30.'~<>
'    delete 19/45       ---     deletes facts 19 to 45.'~<>
'    delete loves       ---     deletes all facts for "loves".'~<>
'    delete i eat fish  ---     deletes that fact.'~
).


command_example(edit,
'    edit               ---     edits the fact with the highest number.'~<>
'    edit 30            ---     edits fact 30.'~<>
'    edit 19/45         ---     edits facts 19 to 45.'~<>
'    edit loves         ---     edits all facts for "loves".'~<>
'    edit i eat fish    ---     edits that fact.'~
).


command_example(load,
'    load myfacts.'~<>
'    load tim1.'~<>
'to load what\'s in MYFACTS or TIM1. These facts will get added to the'~<>
'database, but you will not see them if you do a "show", and you won\'t'~<>
'be able to change them.'~
).


command_example( logic,
'    logic.             ---     to go into Logic mode.'~<>
'The word "logic" must always be followed immediately by a dot or'~<>
'question mark.'~
).


command_example( next,
'    next.              ---     to move to the next section.'~<>
'The word "next" must always be followed immediately by a dot or'~<>
'question mark.'~
).


command_example( prolog,
'    prolog.            ---     to go into Prolog mode.'~<>
'The word "prolog" must always be followed immediately by a dot or'~<>
'question mark.'~
).


command_example(restore,
'    restore myfacts.'~<>
'    restore tim1.'~<>
'to restore what\'s in MYFACTS or TIM1 to your knowledge base. This'~<>
'command will delete what\'s already there.'~
).


command_example(save,
'    save myfacts.'~<>
'    save tim1.'~<>
'to save your knowledge base into MYFACTS or TIM1.'~
).


command_example(script,
'    script lesson1.'~<>
'    script lesson5.'~<>
'to change to Lesson 1 or Lesson 5.'~
).


command_example(section,
'    section 5.'~<>
'    section 23.'~<>
'to change to sections 5 or 23.'~
).


command_example(show,
'    show.              ---     to show everything.'~<>
'    show 1.            ---     to show fact 1.'~<>
'    show 19/45.        ---     to show facts 19 to 45.'~<>
'    show loves.        ---     to show all facts for "loves".'~<>
'    show q.            ---     to show the last question.'~<>
'And you can put a printer name just after the command to print the'~<>
'output:'~<>
'    show ctc 19/45     ---     to print facts 19/45 at CTC.'~<>
'    show ep loves      ---     to print all facts for'~<>
'                               "loves" at Psychology.'~<>
'for example.'~
).


command_example(trace,
'    trace.             ---     to trace the last question.'~<>
'And you can put a printer name just after the command to print the'~<>
'output:'~<>
'    trace ctc          ---     to print the trace at CTC.'~<>
'for example.'~
).


command_example(why,
'    why.               ---     to explain the answer to the last'~<>
'                               question.'~<>
'And you can put a printer name just after the command to print the'~<>
'output:'~<>
'    why ctc            ---     to print the output at CTC.'~<>
'for example.'~
).


:- add_user_output( command_help_output ).


command_help_output( cex_(Command) ) :-
    !,
    command_example( Command, Text ),
    output( Text ).

command_help_output( tokens_(Tokens) ) :-
    !,
    for_in_list( Token,
                 Tokens,
                 ( token_contents(Token,Raw), output(Raw) ),
                 output(' ')
               ).

command_help_output( qtoken_(X) ) :-
    !,
    token_contents( X, Raw ),
    output( '"'<>Raw<>'"' ).

command_help_output( q_(X) ) :-
    !,
    output( '"'<>X<>'"' ).

command_help_output( qtokens_(Tokens) ) :-
    !,
    output( '"' ),
    output( tokens_(Tokens) ),
    output( '"' ).


/*
Miscellaneous.
--------------
*/

is_get( append ).
is_get( load ).
is_get( restore ).


:- endmodule.
