/* XLISP-STAT 2.1 Copyright (c) 1990, by Luke Tierney                  */
/* Additions to Xlisp 2.1, Copyright (c) 1989 by David Michael Betz    */
/* You may give out copies of this software; for conditions see the    */
/* file COPYING included with this distribution.                       */

#include "xlisp.h"
#include "xlstat.h"

#define BRUSH_WIDTH 20
#define BRUSH_HEIGHT 40
#define AXIS_LABEL_GAP 4
#define AXIS_TICK_LENGTH 3
#define AXIS_LABEL_TEMPLATE "12345"
#define CLICK_WIDTH 4
#define CLICK_HEIGHT 4

typedef struct brush {
  int left, top, width, height, showing;
} Brush;

typedef struct clickrange {
  int width, height;
} ClickRange;

typedef struct content {
  int left, top, width, height, origin_x, origin_y, x_variable, y_variable;
} Content;

typedef struct {
  int left, top, right, bottom;
} Margin;

typedef struct {
  int showing, labeled, ticks, height, edge;
} Axis;

typedef struct iview {
  IViewData data;
  Content content;
  Margin margin;
  Axis x_axis, y_axis;
  Brush brush;
  ClickRange clickrange;
  MouseMode mouseMode;
  int showingLabels, fixed_aspect, dirty;
  LVAL links;
  double *scale, *shift;
} *IView;

typedef IView StGrInfo;

#define IViewGetIView(w) ((IView) StGWGetRefCon(IViewWindowWinInfo(w)))

/**************************************************************************/
/**                                                                      **/
/**                       IView Creation Functions                       **/
/**                                                                      **/
/**************************************************************************/

VOID IViewFreeMem(w)
     IVIEW_WINDOW w;
{
  IView iview = IViewGetIView(w);

  if (iview != nil) {
    if (IViewDataPtr(w) != nil) {
      IViewDataFree(IViewDataPtr(w));
      IViewSetData(w, nil);
    }
    StFree(iview->scale);
    StFree(iview->shift);
    StFree(iview);
    StGWSetRefCon(IViewWindowWinInfo(w), (long) nil);
  }
}

IVIEW_WINDOW IViewNew(object)
     LVAL object;
{
  IVIEW_WINDOW w = (IVIEW_WINDOW) IViewWindowNew(object, FALSE);
  IView iview;
  StGWWinInfo *gwinfo;
  int vars, i;

  gwinfo = StGWObWinInfo(object);
  get_iview_ivars(object, &vars);

  iview = (IView) StCalloc(sizeof(struct iview), 1);
  StGWSetRefCon(gwinfo, (long) iview);
  IViewSetData(w, IViewDataNew(vars));
  iview->scale = (double *) StCalloc(vars, sizeof(double));
  iview->shift = (double *) StCalloc(vars, sizeof(double));
  
  StGWSetFreeMem(gwinfo, IViewFreeMem);
  StGrSetContentVariables(gwinfo, 0, 1);
  IViewSetBrush(w, 0, 0, BRUSH_WIDTH, BRUSH_HEIGHT);
  StGrSetClickRange(gwinfo, CLICK_WIDTH, CLICK_HEIGHT);
  IViewSetShowingLabels(w, FALSE);
  IViewSetMouseMode(w, selecting);
  IViewSetLinks(w, NIL);
  StGrSetMargin(gwinfo, 0, 0, 0, 0);
  IViewSetFixedAspect(w, TRUE);
  iview->brush.showing = FALSE;
  
  for (i = 0; i < vars; i++) {
    IViewSetScaledRange(w, i, 0.0, 1.0);
    IViewSetScale(w, i, 1.0);
    IViewSetShift(w, i, 0.0);
  }
  return(w);
}

/**************************************************************************/
/**                                                                      **/
/**                 IView State Accessors and Mutators                   **/
/**                                                                      **/
/**************************************************************************/

int StGrDirty(gwinfo) 
	StGWWinInfo *gwinfo;
{
  StGrInfo gr = (StGrInfo) StGWGetRefCon(gwinfo);
  if (gr == nil) xlfail("no graph installed in this window");
  return(gr->dirty);
}

VOID StGrSetDirty(gwinfo, dirty) 
	StGWWinInfo *gwinfo;
	int dirty;
{
  StGrInfo gr = (StGrInfo) StGWGetRefCon(gwinfo);
  if (gr == nil) xlfail("no graph installed in this window");
  gr->dirty = dirty;
}

static  StGrInfo StGrGetGrInfo(gwinfo) 
	StGWWinInfo *gwinfo;
{
  StGrInfo gr = (StGrInfo) StGWGetRefCon(gwinfo);
  if (gr == nil) xlfail("no graph installed in this window");
  return(gr);
}

VOID StGrSetContentRect(gwinfo, left, top, width, height)
     StGWWinInfo *gwinfo;
     int left, top, width, height;
{
  StGrInfo gr = StGrGetGrInfo(gwinfo);

  gr->content.left = left; gr->content.top = top;
  gr->content.width = width; gr->content.height = height;
}

VOID StGrGetContentRect(gwinfo, left, top, width, height)
     StGWWinInfo *gwinfo;
     int *left, *top, *width, *height;
{
  StGrInfo gr = StGrGetGrInfo(gwinfo);

  if (left != nil) *left = gr->content.left;
  if (top != nil) *top = gr->content.top;
  if (width != nil) *width = gr->content.width;
  if (height != nil) *height = gr->content.height;
}

VOID StGrSetContentOrigin(gwinfo, x, y)
     StGWWinInfo *gwinfo;
     int x, y;
{
  StGrInfo gr = StGrGetGrInfo(gwinfo);

  gr->content.origin_x = x;
  gr->content.origin_y = y;
}

VOID StGrGetContentOrigin(gwinfo, x, y)
     StGWWinInfo *gwinfo;
     int *x, *y;
{
  StGrInfo gr = StGrGetGrInfo(gwinfo);

  if (x != nil) *x = gr->content.origin_x;
  if (y != nil) *y = gr->content.origin_y;
}

VOID StGrSetContentVariables(gwinfo, x, y)
     StGWWinInfo *gwinfo;
     int x, y;
{
  StGrInfo gr = StGrGetGrInfo(gwinfo);
  int vars = StGrNumVariables(gwinfo);
  
  gr->content.x_variable = (vars > x && x >= 0) ? x : 0;
  gr->content.y_variable = (vars > y && y >= 0) ? y : 1;
}

VOID StGrGetContentVariables(gwinfo, x, y)
     StGWWinInfo *gwinfo;
     int *x, *y;
{
  StGrInfo gr = StGrGetGrInfo(gwinfo);

  if (x != nil) *x = gr->content.x_variable;
  if (y != nil) *y = gr->content.y_variable;
}

VOID StGrSetClickRange(gwinfo, width, height)
     StGWWinInfo *gwinfo;
     int width, height;
{
  StGrInfo gr = StGrGetGrInfo(gwinfo);

  gr->clickrange.width = width;
  gr->clickrange.height = height;
}

VOID StGrGetClickRange(gwinfo, width, height)
     StGWWinInfo *gwinfo;
     int *width, *height;
{
  StGrInfo gr = StGrGetGrInfo(gwinfo);

  if (width != nil) *width = gr->clickrange.width;
  if (height != nil) *height = gr->clickrange.height;
}

VOID IViewSetMouseMode(w, mode)
	IVIEW_WINDOW w;
	MouseMode mode;
{
  StGWWinInfo *gwinfo = IViewWindowWinInfo(w);
  IView iview = IViewGetIView(w);
  if (iview == nil) return;

  if (iview->mouseMode == brushing) IViewEraseBrush(w);
  iview->mouseMode = mode;
  if (iview->mouseMode == brushing) IViewDrawBrush(w);
  switch (mode) {
  case brushing:  StGWSetCursor(gwinfo, BRUSH_CURSOR); break;
  case usermode:  StGWSetCursor(gwinfo, HAND_CURSOR);  break;
  case selecting:
  default:        StGWSetCursor(gwinfo, ARROW_CURSOR);
  }
}

MouseMode IViewMouseMode(w)
	IVIEW_WINDOW w;
{
  IView iview = IViewGetIView(w);
  if (iview == nil) return((MouseMode) 0);

  return(iview->mouseMode);
}

VOID IViewSetShowingLabels(w, show)
	IVIEW_WINDOW w;
	int show;
{
  IView iview = IViewGetIView(w);
  if (iview == nil) return;
  
  IViewUnselectAllPoints(w);
  iview->showingLabels = show;
}

int IViewShowingLabels(w)
     IVIEW_WINDOW w;
{
  IView iview = IViewGetIView(w);

  return(iview != nil && iview->showingLabels);
}

VOID IViewSetData(w, data)
	IVIEW_WINDOW w;
	IViewData data;
{
  IView iview = IViewGetIView(w);

  if (iview != nil) iview->data = data;
}

char *StGrData(gwinfo)
	StGWWinInfo *gwinfo;
{
  StGrInfo gr = StGrGetGrInfo(gwinfo);
  if (gr->data == nil) xlfail("No data in this IView");
  return((char *) gr->data);
}

IViewData IViewDataPtr(w)
     IVIEW_WINDOW w;
{
  IView iview = IViewGetIView(w);
  if (iview == nil) xlfail("No IView installed in this window");
  if (iview->data == nil) xlfail("No data in this IView");
  return(iview->data);
}

VOID StGrSetMargin(gwinfo, left, top, right, bottom)
     StGWWinInfo *gwinfo;
     int left, top, right, bottom;
{
  StGrInfo gr = StGrGetGrInfo(gwinfo);
  
  gr->margin.left = left;
  gr->margin.top = top;
  gr->margin.right = right;
  gr->margin.bottom = bottom;
}

VOID StGrGetMargin(gwinfo, left, top, right, bottom)
     StGWWinInfo *gwinfo;
     int *left, *top, *right, *bottom;
{
  StGrInfo gr = StGrGetGrInfo(gwinfo);
  
  if (left != nil) *left = gr->margin.left;
  if (top != nil) *top = gr->margin.top;
  if (right != nil) *right = gr->margin.right;
  if (bottom != nil) *bottom = gr->margin.bottom;
}

VOID IViewSetFixedAspect(w, fixed)
     IVIEW_WINDOW w;
     int fixed;
{
  IView iview = IViewGetIView(w);
  if (iview == nil) xlfail("No IView installed in this window");
  iview->fixed_aspect = fixed;
}

IViewFixedAspect(w)
     IVIEW_WINDOW w;
{
  IView iview = IViewGetIView(w);
  if (iview == nil) xlfail("No IView installed in this window");
  return(iview->fixed_aspect);
}

VOID IViewSetScale(w, var, scale)
     IVIEW_WINDOW w;
     unsigned var;
     double scale;
{
  IView iview = IViewGetIView(w);
  if (iview == nil) xlfail("No IView installed in this window");
  if (var >= IViewNumVariables(w) || scale <= 0.0) return;
  else iview->scale[var] = scale;
}

double IViewScale(w, var)
     IVIEW_WINDOW w;
     unsigned var;
{
  IView iview = IViewGetIView(w);
  if (iview == nil) xlfail("No IView installed in this window");
  if (var >= IViewNumVariables(w)) return(0.0);
  else return(iview->scale[var]);
}

VOID IViewSetShift(w, var, shift)
     IVIEW_WINDOW w;
     unsigned var;
     double shift;
{
  IView iview = IViewGetIView(w);
  if (iview == nil) xlfail("No IView installed in this window");
  if (var >= IViewNumVariables(w)) return;
  else iview->shift[var] = shift;
}

double IViewShift(w, var)
     IVIEW_WINDOW w;
     unsigned var;
{
  IView iview = IViewGetIView(w);
  if (iview == nil) xlfail("No IView installed in this window");
  if (var >= IViewNumVariables(w)) return(0.0);
  else return(iview->shift[var]);
}

/**************************************************************************/
/**                                                                      **/
/**                            Axis Functions                            **/
/**                                                                      **/
/**************************************************************************/

static set_axis(w, which, showing, labeled, ticks)
	IVIEW_WINDOW w;
	int which, showing, labeled, ticks;
{
  StGWWinInfo *gwinfo = IViewWindowWinInfo(w);
  IView iview = IViewGetIView(w);
  Axis *axis;
  
  if (iview == nil) xlfail("No IView installed in this window");
   
  switch (which) {
  case 'X': axis = &iview->x_axis; break;
  case 'Y': axis = &iview->y_axis; break;
  }

  axis->showing = showing;
  axis->labeled = labeled;
  axis->ticks = ticks;

  if (axis->showing) {
    axis->height = StGWTextAscent(gwinfo)
                 + StGWTextWidth(gwinfo, AXIS_LABEL_TEMPLATE) / 2
                 + AXIS_LABEL_GAP + AXIS_TICK_LENGTH;
    if (axis->labeled)
      axis->height += StGWTextAscent(gwinfo) + AXIS_LABEL_GAP;
    axis->edge = StGWTextWidth(gwinfo, AXIS_LABEL_TEMPLATE);
  }
  else {
    axis->height = 0;
    axis->edge = 0;
  }
}  
  
VOID IViewGetAxisMargin(w, left, top, right, bottom)
	IVIEW_WINDOW w;
	int *left, *top, *right, *bottom;
{
  IView iview = IViewGetIView(w);
  if (iview == nil) xlfail("No IView installed in this window");

  if (left != nil) 
    *left = (iview->x_axis.edge > iview->y_axis.height)
          ? iview->x_axis.edge : iview->y_axis.height;
  if (bottom != nil)
    *bottom = (iview->y_axis.edge > iview->x_axis.height)
            ? iview->y_axis.edge : iview->x_axis.height;
  if (top != nil) *top = iview->y_axis.edge;
  if (right != nil) *right = iview->x_axis.edge;
}

VOID IViewSetXaxis(w, showing, labeled, ticks)
	IVIEW_WINDOW w;
	int showing, labeled, ticks;
{
  set_axis(w, 'X', showing, labeled, ticks);
}

VOID IViewGetXaxis(w, showing, labeled, ticks)
	IVIEW_WINDOW w;
	int *showing, *labeled, *ticks;
{
  IView iview = IViewGetIView(w);
  if (iview == nil) xlfail("No IView installed in this window");
  if (showing != nil) *showing = iview->x_axis.showing;
  if (labeled != nil) *labeled = iview->x_axis.labeled;
  if (ticks != nil) *ticks = iview->x_axis.ticks;
}

VOID IViewSetYaxis(w, showing, labeled, ticks)
	IVIEW_WINDOW w;
	int showing, labeled, ticks;
{
  set_axis(w, 'Y', showing, labeled, ticks);
}

VOID IViewGetYaxis(w, showing, labeled, ticks)
	IVIEW_WINDOW w;
	int *showing, *labeled, *ticks;
{
  IView iview = IViewGetIView(w);
  if (iview == nil) xlfail("No IView installed in this window");
  if (showing != nil) *showing = iview->y_axis.showing;
  if (labeled != nil) *labeled = iview->y_axis.labeled;
  if (ticks != nil) *ticks = iview->y_axis.ticks;
}

static draw_tick(w, x, y, value, axis)
	IVIEW_WINDOW w;
	int x, y, axis;
	double value;
{
  char s[100];
  int offset;
  StGWWinInfo *gwinfo = IViewWindowWinInfo(w);
  
  offset = StGWTextWidth(gwinfo, AXIS_LABEL_TEMPLATE) / 3;
  switch (axis) {
  case 'X':
    offset += AXIS_TICK_LENGTH + StGWTextAscent(gwinfo);
    StGWDrawLine(gwinfo, x, y, x, y + AXIS_TICK_LENGTH);
    sprintf(s, "%.3g", value);
    StGWDrawText(gwinfo, s, x, y + offset, 1, 0);
    break;
  case 'Y':
    offset += AXIS_TICK_LENGTH + AXIS_LABEL_GAP;
    StGWDrawLine(gwinfo, x, y, x - AXIS_TICK_LENGTH, y);
    sprintf(s, "%.3g", value);
    StGWDrawTextUp(gwinfo, s, x - offset, y, 1, 0);
    break;
  }
}
  
VOID IViewDrawAxes(w)
	IVIEW_WINDOW w;
{
  IView iview = IViewGetIView(w);
  int left, top, width, height, right, bottom, x, y;
  double low, high, value;
  int offset, tick, i;
  StGWWinInfo *gwinfo = IViewWindowWinInfo(w);
  
  if (iview == nil) xlfail("No IView installed in this window");
  StGrGetContentVariables(gwinfo, &x, &y);
  StGrGetContentRect(gwinfo, &left, &top, &width, &height);
  right = left + width;
  bottom = top + height;
  
  offset = StGWTextWidth(gwinfo, AXIS_LABEL_TEMPLATE) / 3;
  if (iview->x_axis.showing) {
    StGWDrawLine(gwinfo, left, bottom + 1, right, bottom + 1);
    IViewGetRange(w, x, &low, &high);
    if (iview->x_axis.ticks >= 2) {
      draw_tick(w, left, bottom + 1, low, 'X'); 
      draw_tick(w, right, bottom + 1, high, 'X');
      for (i = 1; i < iview->x_axis.ticks - 1; i++) {
        tick = left + (((double) i) * width) / (iview->x_axis.ticks - 1);
        value = low + i * (high - low) / (iview->x_axis.ticks - 1);
        draw_tick(w, tick, bottom + 1, value, 'X');
      }
    }
    if (iview->x_axis.labeled) {
      offset += AXIS_TICK_LENGTH + AXIS_LABEL_GAP + 2 * StGWTextAscent(gwinfo);
      StGWDrawText(gwinfo, IViewVariableLabel(w, x),
                           (left + right) / 2, bottom + offset, 1, 0);
    }
  }
  offset = StGWTextWidth(gwinfo, AXIS_LABEL_TEMPLATE) / 3;
  if (iview->y_axis.showing) {
    StGWDrawLine(gwinfo, left - 1, bottom, left - 1, top);
    IViewGetRange(w, y, &low, &high);
    if (iview->y_axis.ticks >= 2) {
      draw_tick(w, left - 1, bottom, low, 'Y'); 
      draw_tick(w, left - 1, top, high, 'Y');
      for (i = 1; i < iview->y_axis.ticks - 1; i++) {
        tick = bottom - (((double) i) * height) / (iview->y_axis.ticks - 1);
        value = low + i * (high - low) / (iview->y_axis.ticks - 1);
        draw_tick(w, left - 1, tick, value, 'Y');
      }
    }
    if (iview->y_axis.labeled) {
      offset += AXIS_TICK_LENGTH + 2 * AXIS_LABEL_GAP + StGWTextAscent(gwinfo);
      StGWDrawTextUp(gwinfo, IViewVariableLabel(w, y),
                            left - offset, (top + bottom) / 2, 1, 0);
    }
  }
}

/**************************************************************************/
/**                                                                      **/
/**                           Brush Functions                            **/
/**                                                                      **/
/**************************************************************************/

VOID IViewSetBrush(w, x, y, width, height)
     IVIEW_WINDOW w;
     int x, y, width, height;
{
  IView iview = IViewGetIView(w);
  int showing = iview->brush.showing;
  if (iview == nil) return;

  if (showing) IViewEraseBrush(w);
  iview->brush.left = x - width;
  iview->brush.top = y - height;
  iview->brush.width = width;
  iview->brush.height = height;
  if (showing) IViewDrawBrush(w);
}

VOID IViewGetBrush(w, x, y, width, height)
     IVIEW_WINDOW w;
     int *x, *y, *width, *height;
{
  IView iview = IViewGetIView(w);
  if (iview == nil) return;

  if (x != nil) *x = iview->brush.left + iview->brush.width;
  if (y != nil) *y = iview->brush.top + iview->brush.height;
  if (width != nil) *width = iview->brush.width;
  if (height != nil) *height = iview->brush.height;
}

VOID IViewEraseBrush(w)
     IVIEW_WINDOW w;
{
  StGWWinInfo *gwinfo = IViewWindowWinInfo(w);
  IView iview = IViewGetIView(w);
  int mode, type;
  
  if (iview != nil && iview->brush.showing) {
    mode = StGWDrawMode(gwinfo); StGWSetDrawMode(gwinfo, 1);
    type = StGWLineType(gwinfo); StGWSetLineType(gwinfo, 1);
    StGWFrameRect(gwinfo, iview->brush.left, iview->brush.top,
		                  iview->brush.width, iview->brush.height);
    iview->brush.showing = FALSE;
    StGWSetDrawMode(gwinfo, mode);
    StGWSetLineType(gwinfo, type);
  }
}

VOID IViewDrawBrush(w)
     IVIEW_WINDOW w;
{
  StGWWinInfo *gwinfo = IViewWindowWinInfo(w);
  IView iview = IViewGetIView(w);
  int mode, type;

  if (iview != nil && ! iview->brush.showing) {
    mode = StGWDrawMode(gwinfo); StGWSetDrawMode(gwinfo, 1);
    type = StGWLineType(gwinfo); StGWSetLineType(gwinfo, 1);
    StGWFrameRect(gwinfo, iview->brush.left, iview->brush.top,
		                  iview->brush.width, iview->brush.height);
    iview->brush.showing = TRUE;
    StGWSetDrawMode(gwinfo, mode);
    StGWSetLineType(gwinfo, type);
  }
}

VOID IViewMoveBrush(w, x, y)
     IVIEW_WINDOW w;
{
  IView iview = IViewGetIView(w);
  if (iview == nil) return;

  IViewEraseBrush(w);
  iview->brush.left = x - iview->brush.width;
  iview->brush.top = y - iview->brush.height;
  IViewDrawBrush(w);
}

/**************************************************************************/
/**                                                                      **/
/**                      Mouse Action Functions                          **/
/**                                                                      **/
/**************************************************************************/

static struct {
  int x, y, left, top, width, height;
} dragRect;

static VOID drag(w, x, y)
     IVIEW_WINDOW w;
     int x, y;
{
  StGWWinInfo *gwinfo = IViewWindowWinInfo(w);
  
  if (dragRect.width != 0 && dragRect.height != 0) 
    StGWFrameRect(gwinfo, dragRect.left, dragRect.top, 
                          dragRect.width, dragRect.height);
  dragRect.width = abs(dragRect.x - x); 
  dragRect.height = abs(dragRect.y - y);
  dragRect.left = (x < dragRect.x) ? x : dragRect.x; 
  dragRect.top = (y < dragRect.y) ? y : dragRect.y; 
  if (dragRect.width != 0 && dragRect.height != 0) 
    StGWFrameRect(gwinfo, dragRect.left, dragRect.top, 
                          dragRect.width, dragRect.height);
}

VOID IViewStdSelectingMouseAction(w, x, y, type, mods)
     IVIEW_WINDOW w;
     int x, y;
     MouseEventType type;
     MouseClickModifier mods;
{
  int mode, line_type;
  int clickwidth, clickheight;
  StGWWinInfo *gwinfo = IViewWindowWinInfo(w);

  if (type == MouseClick) {
    if (mods != ExtendModifier) IViewUnselectAllPoints(w);
    StGrGetClickRange(gwinfo, &clickwidth, &clickheight);
    IViewAdjustPointsInRect(w, x - clickwidth / 2, y - clickheight / 2,
                               clickwidth, clickheight, pointSelected);
    
    mode = StGWDrawMode(gwinfo); StGWSetDrawMode(gwinfo, 1);
    line_type = StGWLineType(gwinfo); StGWSetLineType(gwinfo, 1);
    dragRect.x = x; dragRect.y = y;
    dragRect.left = x, dragRect.top = y;
    dragRect.width = 0; dragRect.height = 0;
    StGWWhileButtonDown(gwinfo, drag, TRUE);
    if (dragRect.width != 0 && dragRect.height != 0)
    StGWFrameRect(gwinfo, dragRect.left, dragRect.top, 
                          dragRect.width, dragRect.height);
    StGWSetDrawMode(gwinfo, mode);
    StGWSetLineType(gwinfo, line_type);

    IViewAdjustPointsInRect(w, dragRect.left, dragRect.top,
				               dragRect.width, dragRect.height, pointSelected);
  }
}

static VOID dragbrush(w, x, y)
     IVIEW_WINDOW w;
     int x, y;
{
  IView iview = IViewGetIView(w);
  
  IViewMoveBrush(w, x, y);
  IViewAdjustPointsInRect(w, iview->brush.left, iview->brush.top,
                             iview->brush.width, iview->brush.height,
                             pointSelected);
}

VOID IViewStdBrushingMouseAction(w, x, y, type, mods)
     IVIEW_WINDOW w;
     int x, y;
     MouseEventType type;
     MouseClickModifier mods;
{
  IView iview = IViewGetIView(w);
  StGWWinInfo *gwinfo = IViewWindowWinInfo(w);

  IViewMoveBrush(w, x, y);
  if (type == MouseClick) {
    if (mods != ExtendModifier) IViewUnselectAllPoints(w);
    StGWWhileButtonDown(gwinfo, dragbrush, TRUE);
  }
  else if (type == MouseMove) {
    IViewMoveBrush(w, x, y);
    IViewAdjustPointsInRect(w, iview->brush.left, iview->brush.top,
                               iview->brush.width, iview->brush.height, pointHilited);
  }
}

VOID IViewStdMouseAction(w, x, y, type, mods)
     IVIEW_WINDOW w;
     int x, y;
     MouseEventType type;
     MouseClickModifier mods;
{
  switch (IViewMouseMode(w)) {
  case selecting: IViewStdSelectingMouseAction(w, x, y, type, mods); break;
  case brushing:  IViewStdBrushingMouseAction(w, x, y, type, mods); break;
  }
}

VOID IViewStdUnselectAllPoints(w)
     IVIEW_WINDOW w;
{
  int i, n = IViewNumPoints(w);
  IViewCheckLinks(w);
  IViewClearPointMarks(w);
  for (i = 0; i < n; i++)
    if ((int) IViewPointState(w, i) > (int) pointNormal 
        && ! IViewPointMasked(w, i)) 
      IViewSetPointState(w, i, pointNormal);
  IViewAdjustScreens(w);
}

VOID IViewEraseSelection(w)
	IVIEW_WINDOW w;
{
  int n = IViewNumPoints(w), i;
  IViewCheckLinks(w);
  IViewClearPointMarks(w);
  for (i = 0; i < n; i++) 
    if (IViewPointState(w, i) == pointSelected)
      IViewSetPointState(w, i, pointInvisible);
  IViewAdjustScreens(w);
}

VOID IViewMaskSelection(w)
	IVIEW_WINDOW w;
{
  int n = IViewNumPoints(w), i;
  
  IViewClearPointMarks(w);
  for (i = 0; i < n; i++) 
    if (IViewPointState(w, i) == pointSelected)
      IViewSetPointMask(w, i, TRUE);
  IViewRedrawContent(w);
}

VOID IViewUnmaskAllPoints(w)
	IVIEW_WINDOW w;
{
  int n = IViewNumPoints(w), i;
  
  IViewClearPointMarks(w);
  for (i = 0; i < n; i++) IViewSetPointMask(w, i, FALSE);
  IViewRedrawContent(w);
}

VOID IViewShowAllPoints(w)
	IVIEW_WINDOW w;
{
  int n = IViewNumPoints(w), i;
  IViewCheckLinks(w);
  IViewClearPointMarks(w);
  for (i = 0; i < n; i++) IViewSetPointState(w, i, pointNormal);
  IViewAdjustScreens(w);
}

int IViewAllPointsShowing(w)
	IVIEW_WINDOW w;
{
  int result = TRUE, n = IViewNumPoints(w), i;
  
  for (i = 0; i < n && result; i++)
    if (IViewPointState(w, i) == pointInvisible) result = FALSE;
  return(result);
}

int IViewAllPointsUnmasked(w)
	IVIEW_WINDOW w;
{
  int result = TRUE, n = IViewNumPoints(w), i;
  
  for (i = 0; i < n && result; i++)
    if (IViewPointMasked(w, i)) result = FALSE;
  return(result);
}

int IViewAnyPointsSelected(w)
	IVIEW_WINDOW w;
{
  int result = FALSE, n = IViewNumPoints(w), i;
  
  for (i = 0; i < n && ! result; i++)
    if (IViewPointState(w, i) == pointSelected) result = TRUE;
  return(result);
}

/*************************************************************************/
/**                                                                     **/
/**                      IView Linking Functions                        **/
/**                                                                     **/
/*************************************************************************/

/**** storing links in internal structure is risky because of possible GC */

VOID IViewSetLinks(w, links)
     IVIEW_WINDOW w;
     LVAL links;
{
  IView iview = IViewGetIView(w);
  if (iview == nil) xlfail("No IView installed in this window");
  iview->links = links;
}

LVAL IViewGetLinks(w)
     IVIEW_WINDOW w;
{
  IView iview = IViewGetIView(w);
  if (iview == nil) xlfail("No IView installed in this window");
  return(iview->links);
}

int IViewIsLinked(w)
     IVIEW_WINDOW w;
{
  IView iview = IViewGetIView(w);
  if (iview == nil) return(FALSE);
  else return(iview->links != NIL);
}
