/* XLISP-STAT 2.1 Copyright (c) 1990, by Luke Tierney                  */
/* Additions to Xlisp 2.1, Copyright (c) 1989 by David Michael Betz    */
/* You may give out copies of this software; for conditions see the    */
/* file COPYING included with this distribution.                       */

#include "xlisp.h"
#include "xlstat.h"

#define LABEL_OFFSET 5

#ifdef USESTRINGS
extern char *IViewDataStringString();
extern double IViewDataStringValue(), IViewDataStringTransformedValue()
#endif /* USESTRINGS */

/**************************************************************************/
/**                                                                      **/
/**                    General IView Data Functions                      **/
/**                                                                      **/
/**************************************************************************/

int StGrNumVariables(gwinfo)
     StGWWinInfo *gwinfo;
{
  return(IViewDataNumVariables((IViewData) StGrData(gwinfo)));
}

int IViewNumVariables(w)
     IVIEW_WINDOW w;
{
  return(IViewDataNumVariables(IViewDataPtr(w)));
}

VOID IViewSetVariableLabel(w, var, s)
     IVIEW_WINDOW w;
     int var;
     char *s;
{
  IViewDataSetVariableLabel(IViewDataPtr(w), var, s);
}

char *IViewVariableLabel(w, var)
     IVIEW_WINDOW w;
     int var;
{
  return(IViewDataVariableLabel(IViewDataPtr(w), var));
}

VOID IViewSetRange(w, var, low, high)
     IVIEW_WINDOW w;
     int var;
     double low, high;
{
  double scale, shift;

  scale = IViewScale(w, var);
  shift = IViewShift(w, var);
  low = scale * low + shift;
  high = scale * high + shift;
  IViewSetScaledRange(w, var, low, high);
}

VOID IViewGetRange(w, var, low, high)
     IVIEW_WINDOW w;
     int var;
     double *low, *high;
{
  double scale, shift;

  scale = IViewScale(w, var);
  shift = IViewShift(w, var);
  IViewGetScaledRange(w, var, low, high);
  if (scale > 0.0) {
    if (low != nil) *low = (*low - shift) / scale;
    if (high != nil) *high = (*high - shift) / scale;
  }
}

VOID IViewSetScaledRange(w, var, low, high)
     IVIEW_WINDOW w;
     int var;
     double low, high;
{
  IViewDataSetRange(IViewDataPtr(w), var, low, high);
}

VOID IViewGetScaledRange(w, var, low, high)
     IVIEW_WINDOW w;
     int var;
     double *low, *high;
{
  IViewDataGetRange(IViewDataPtr(w), var, low, high);
}

VOID IViewSetScreenRange(w, var, low, high)
     IVIEW_WINDOW w;
     int var, low, high;
{
  IViewDataSetScreenRange(IViewDataPtr(w), var, low, high);
}

VOID IViewGetScreenRange(w, var, low, high)
     IVIEW_WINDOW w;
     int var, *low, *high;
{
  IViewDataGetScreenRange(IViewDataPtr(w), var, low, high);
}

VOID IViewSetIdentityTransformation(w)
	IVIEW_WINDOW w;
{
  IViewDataSetIdentityTransformation(IViewDataPtr(w));
}

VOID IViewSetTransformation(w, a)
	IVIEW_WINDOW w;
	double **a;
{
  IViewDataSetTransformation(IViewDataPtr(w), a);
}

double **IViewTransformation(w)
	IVIEW_WINDOW w;
{
  return(IViewDataTransformation(IViewDataPtr(w)));
}

int IViewIsTransformed(w)
	IVIEW_WINDOW w;
{
  return(IViewDataIsTransformed(IViewDataPtr(w)));
}

VOID IViewApplyTransformation(w, a, inbasis)
	IVIEW_WINDOW w;
	double **a;
	int *inbasis;
{
  IViewDataApplyTransformation(IViewDataPtr(w), a, inbasis);
}

double IViewEncodeValue(w, value, var)
	IVIEW_WINDOW w;
	double value;
	int var;
{
  double scale = IViewScale(w, var), shift = IViewShift(w, var);
  if (scale == 1.0 && shift == 0.0) return(value);
  else return(scale * value + shift);
}

double IViewDecodeValue(w, value, var)
	IVIEW_WINDOW w;
	double value;
	int var;
{
  double scale = IViewScale(w, var), shift = IViewShift(w, var);
  return((scale > 0.0) ? (value - shift) / scale : 0.0);
}

/**************************************************************************/
/**                                                                      **/
/**                      IView Point Data Functions                      **/
/**                                                                      **/
/**************************************************************************/

VOID IViewAddPoints(w, n)
     IVIEW_WINDOW w;
     int n;
{
  IViewDataAddPoints(IViewDataPtr(w), n);
}

VOID IViewClearPoints(w)
     IVIEW_WINDOW w;
{
  IViewDataClearPoints(IViewDataPtr(w));
}

int IViewNumPoints(w)
     IVIEW_WINDOW w;
{
  return(IViewDataNumPoints(IViewDataPtr(w)));
}

VOID IViewSetPointValue(w, var, point, value)
     IVIEW_WINDOW w;
     int var, point;
     double value;
{
  IViewDataSetPointValue(IViewDataPtr(w), var, point, IViewEncodeValue(w, value, var));
}

double IViewPointValue(w, var, point)
     IVIEW_WINDOW w;
     int var, point;
{
  return(IViewDecodeValue(w, IViewDataPointValue(IViewDataPtr(w), var, point), var));
}

VOID IViewSetPointScaledValue(w, var, point, value)
     IVIEW_WINDOW w;
     int var, point;
     double value;
{
  IViewDataSetPointValue(IViewDataPtr(w), var, point, value);
}

double IViewPointScaledValue(w, var, point)
     IVIEW_WINDOW w;
     int var, point;
{
  return(IViewDataPointValue(IViewDataPtr(w), var, point));
}

double IViewPointTransformedValue(w, var, point)
     IVIEW_WINDOW w;
     int var, point;
{
  return(IViewDataPointTransformedValue(IViewDataPtr(w), var, point));
}

int IViewPointScreenValue(w, var, point)
     IVIEW_WINDOW w;
     int var, point;
{
  return(IViewDataPointScreenValue(IViewDataPtr(w), var, point));
}

VOID IViewGetScreenPointValues(w, point, x)
	IVIEW_WINDOW w;
	int point, *x;
{
  IViewDataGetScreenPointValues(IViewDataPtr(w), point, x);
}

VOID IViewSetPointMask(w, point, masked)
     IVIEW_WINDOW w;
     int point;
     int masked;
{
  IViewDataSetPointMask(IViewDataPtr(w), point, masked);
}

int IViewPointMasked(w, point)
     IVIEW_WINDOW w;
     int point;
{
  return(IViewDataPointMasked(IViewDataPtr(w),  point));
}

VOID IViewSetPointColor(w, point, color)
     IVIEW_WINDOW w;
     int point;
     int color;
{
  IViewDataSetPointColor(IViewDataPtr(w), point, color);
}

int IViewPointColor(w, point)
     IVIEW_WINDOW w;
     int point;
{
  return(IViewDataPointColor(IViewDataPtr(w),  point));
}

VOID IViewSetPointState(w, point, state)
     IVIEW_WINDOW w;
     int point;
     PointState state;
{
  if (IViewPointState(w, point) != state) {
    IViewSetPointScreenState(w, point, IViewPointState(w, point));
    IViewDataSetPointState(IViewDataPtr(w), point, state);
    IViewAdjustOwnScreenPoint(w, point);
    IViewSetPointScreenState(w, point, IViewPointState(w, point));
    if (IViewIsLinked(w)) IViewMatchPointState(w, point);
  }
}

PointState IViewPointState(w, point)
     IVIEW_WINDOW w;
     int point;
{
  return(IViewDataPointState(IViewDataPtr(w),  point));
}

VOID IViewSetPointScreenState(w, point, state)
     IVIEW_WINDOW w;
     int point;
     PointState state;
{
  IViewDataSetPointScreenState(IViewDataPtr(w), point, state);
}

PointState IViewPointScreenState(w, point)
     IVIEW_WINDOW w;
     int point;
{
  return(IViewDataPointScreenState(IViewDataPtr(w),  point));
}

VOID IViewResetScreenStates(w)
     IVIEW_WINDOW w;
{
  IViewDataResetScreenStates(IViewDataPtr(w));
}

VOID IViewSetPointMark(w, point, marked)
     IVIEW_WINDOW w;
     int point;
     int marked;
{
  IViewDataSetPointMark(IViewDataPtr(w), point, marked);
}

int IViewPointMarked(w, point)
     IVIEW_WINDOW w;
     int point;
{
  return(IViewDataPointMarked(IViewDataPtr(w),  point));
}

VOID IViewClearPointMarks(w)
     IVIEW_WINDOW w;
{
  IViewDataClearPointMarks(IViewDataPtr(w));
}

VOID IViewSetPointLabel(w, point, s)
     IVIEW_WINDOW w;
     int point;
     char *s;
{
  IViewDataSetPointLabel(IViewDataPtr(w), point, s);
}

char *IViewPointLabel(w, point)
     IVIEW_WINDOW w;
     int point;
{
  return(IViewDataPointLabel(IViewDataPtr(w), point));
}

VOID IViewSetPointSymbol(w, point, sym, hsym)
     IVIEW_WINDOW w;
     int point, sym, hsym;
{
  IViewDataSetPointSymbol(IViewDataPtr(w),  point, sym, hsym);
}

VOID IViewGetPointSymbol(w, point, sym, hsym)
     IVIEW_WINDOW w;
     int point, *sym, *hsym;
{
  IViewDataGetPointSymbol(IViewDataPtr(w),  point, sym, hsym);
}

/**************************************************************************/
/**                                                                      **/
/**                      IView Line Data Functions                       **/
/**                                                                      **/
/**************************************************************************/

int IViewNumLines(w)
     IVIEW_WINDOW w;
{
  return(IViewDataNumLines(IViewDataPtr(w)));
}

VOID IViewAddLines(w, n)
     IVIEW_WINDOW w;
     int n;
{
  IViewDataAddLines(IViewDataPtr(w), n);
}

VOID IViewClearLines(w)
     IVIEW_WINDOW w;
{
  IViewDataClearLines(IViewDataPtr(w));
}

VOID IViewSetLineValue(w, var, line, value)
	IVIEW_WINDOW w;
	int var, line;
	double value;
{
  IViewDataSetLineValue(IViewDataPtr(w), var, line, IViewEncodeValue(w, value, var));
}

double IViewLineValue(w, var, line)
	IVIEW_WINDOW w;
	int var, line;
{
  return(IViewDecodeValue(w, IViewDataLineValue(IViewDataPtr(w), var, line), var));
}

VOID IViewSetLineScaledValue(w, var, line, value)
	IVIEW_WINDOW w;
	int var, line;
	double value;
{
  IViewDataSetLineValue(IViewDataPtr(w), var, line, value);
}

double IViewLineScaledValue(w, var, line)
	IVIEW_WINDOW w;
	int var, line;
{
  return(IViewDataLineValue(IViewDataPtr(w), var, line));
}

double IViewLineTransformedValue(w, var, line)
	IVIEW_WINDOW w;
	int var, line;
{
  return(IViewDataLineTransformedValue(IViewDataPtr(w), var, line));
}

int IViewLineScreenValue(w, var, line)
	IVIEW_WINDOW w;
	int var, line;
{
  return(IViewDataLineScreenValue(IViewDataPtr(w), var, line));
}

VOID IViewSetLineMask(w, line, masked)
     IVIEW_WINDOW w;
     int line;
     int masked;
{
  IViewDataSetLineMask(IViewDataPtr(w), line, masked);
}

int IViewLineMasked(w, line)
     IVIEW_WINDOW w;
     int line;
{
  return(IViewDataLineMasked(IViewDataPtr(w),  line));
}

VOID IViewSetLineColor(w, line, color)
     IVIEW_WINDOW w;
     int line;
     int color;
{
  IViewDataSetLineColor(IViewDataPtr(w), line, color);
}

int IViewLineColor(w, line)
     IVIEW_WINDOW w;
     int line;
{
  return(IViewDataLineColor(IViewDataPtr(w),  line));
}

VOID IViewSetNextLine(w, line, next)
     IVIEW_WINDOW w;
     int line;
     int next;
{
  IViewDataSetNextLine(IViewDataPtr(w), line, next);
}

int IViewNextLine(w, line)
	IVIEW_WINDOW w;
	int line;
{
  return(IViewDataNextLine(IViewDataPtr(w), line));
}

VOID IViewSetLineType(w, line, type)
     IVIEW_WINDOW w;
     int line;
     int type;
{
  IViewDataSetLineType(IViewDataPtr(w), line, type);
}

int IViewLineType(w, line)
	IVIEW_WINDOW w;
	int line;
{
  return(IViewDataLineType(IViewDataPtr(w), line));
}

VOID IViewSetLineWidth(w, line, width)
     IVIEW_WINDOW w;
     int line, width;
{
  IViewDataSetLineWidth(IViewDataPtr(w), line, width);
}

VOID IViewGetLineWidth(w, line, width)
	IVIEW_WINDOW w;
	int line, *width;
{
  IViewDataGetLineWidth(IViewDataPtr(w), line, (unsigned *) width);
}

#ifdef USESTRINGS
/**************************************************************************/
/**                                                                      **/
/**                     IView String Data Functions                      **/
/**                                                                      **/
/**************************************************************************/

IViewNumStrings(w)
     IVIEW_WINDOW w;
{
  return(IViewDataNumStrings(IViewDataPtr(w)));
}

IViewAddStrings(w, n)
     IVIEW_WINDOW w;
     int n;
{
  IViewDataAddStrings(IViewDataPtr(w), n);
}

IViewClearStrings(w)
     IVIEW_WINDOW w;
{
  IViewDataClearStrings(IViewDataPtr(w));
}

IViewSetStringValue(w, var, string, value)
	IVIEW_WINDOW w;
	int var, string;
	double value;
{
  IViewDataSetStringValue(IViewDataPtr(w), var, string, IViewEncodeValue(w, value, var));
}

double IViewStringValue(w, var, string)
	IVIEW_WINDOW w;
	int var, string;
{
  return(IViewDecodeValue(w, IViewDataStringValue(IViewDataPtr(w), var, string), var));
}

IViewSetStringScaledValue(w, var, string, value)
	IVIEW_WINDOW w;
	int var, string;
	double value;
{
  IViewDataSetStringValue(IViewDataPtr(w), var, string, value);
}

double IViewStringScaledValue(w, var, string)
	IVIEW_WINDOW w;
	int var, string;
{
  return(IViewDataStringValue(IViewDataPtr(w), var, string));
}

double IViewStringTransformedValue(w, var, string)
	IVIEW_WINDOW w;
	int var, string;
{
  return(IViewDataStringTransformedValue(IViewDataPtr(w), var, string));
}

IViewStringScreenValue(w, var, string)
	IVIEW_WINDOW w;
	int var, string;
{
  return(IViewDataStringScreenValue(IViewDataPtr(w), var, string));
}

IViewSetStringMask(w, string, masked)
     IVIEW_WINDOW w;
     int string;
     int masked;
{
  IViewDataSetStringMask(IViewDataPtr(w), string, masked);
}

IViewStringMasked(w, string)
     IVIEW_WINDOW w;
     int string;
{
  return(IViewDataStringMasked(IViewDataPtr(w),  string));
}

IViewSetStringColor(w, string, color)
     IVIEW_WINDOW w;
     int string;
     int color;
{
  IViewDataSetStringColor(IViewDataPtr(w), string, color);
}

IViewStringColor(w, string)
     IVIEW_WINDOW w;
     int string;
{
  return(IViewDataStringColor(IViewDataPtr(w),  string));
}

IViewSetStringString(w, string, str)
	IVIEW_WINDOW w;
	int string;
	char *str;
{
  IViewDataSetStringString(IViewDataPtr(w), string, str);
}

char *IViewStringString(w, string)
	IVIEW_WINDOW w;
	int string;
{
  return(IViewDataStringString(IViewDataPtr(w), string));
}

IViewSetStringModifiers(w, string, up, h, v)
	IVIEW_WINDOW w;
	int string, up, h, v;
{
  IViewDataSetStringModifiers(IViewDataPtr(w), string, up, h, v);
}

IViewGetStringModifiers(w, string, up, h, v)
	IVIEW_WINDOW w;
	int string, *up, *h, *v;
{
  IViewDataGetStringModifiers(IViewDataPtr(w), string, up, h, v);
}
#endif /* USESTRINGS */
	
/**************************************************************************/
/**                                                                      **/
/**                     IView Data Drawing Functions                     **/
/**                                                                      **/
/**************************************************************************/

VOID IViewDrawDataPoints(w, var1, var2, m, n)
     IVIEW_WINDOW w;
     unsigned var1, var2, m, n;
{
  IViewDataDrawPoints(IViewDataPtr(w), w, var1, var2, m, n, LABEL_OFFSET);
}

VOID IViewDrawDataLines(w, var1, var2, m, n)
     IVIEW_WINDOW w;
     unsigned var1, var2, m, n;
{
  IViewDataDrawLines(IViewDataPtr(w), w, var1, var2, m, n);
}


#ifdef DODO
static IViewDrawDataLine(w, var1, var2, line,
                         left, top, width, height, orig_x, orig_y,
                         use_color, draw_color)
     IVIEW_WINDOW w;
     unsigned var1, var2, line;
     int left, top, width, height, orig_x, orig_y, use_color, draw_color;
{
  int n = IViewNumLines(w);
  int x, y, nx, ny;
  int next;
/*  int right = left + width, bottom = top + height;*/
  int type, color, linewidth;
  char *gwinfo;

  gwinfo = IViewWindowWinInfo(w);
  
  if (line >= n || IViewLineMasked(w, line)) return;
  x = orig_x + IViewLineScreenValue(w, var1, line);
  y = orig_y - IViewLineScreenValue(w, var2, line);
/*  if (x < left || x > right) return;
  if (y < top || y > bottom) return;*/

  next = IViewNextLine(w, line);
  if (next >= n || next < 0 || IViewLineMasked(w, next)) return;
  nx = orig_x + IViewLineScreenValue(w, var1, next);
  ny = orig_y - IViewLineScreenValue(w, var2, next);
/*  if (nx < left || nx > right) return;
  if (ny < top || ny > bottom) return;*/
  
  IViewGetLineWidth(w, line, &linewidth);
  StGWSetLineWidth(gwinfo, linewidth);
  type = IViewLineType(w, line);
  if (use_color) {
    color = IViewLineColor(w, line);
    if (color != NOCOLOR) StGWSetDrawColor(gwinfo, color);
    else StGWSetDrawColor(gwinfo, draw_color);
  }
  StGWSetLineType(gwinfo, type);
  StGWDrawLine(gwinfo, x, y, nx, ny);
}

IViewDrawDataLines(w, var1, var2, m, n)
	IVIEW_WINDOW w;
	unsigned var1, var2, m, n;
{
  int vars = IViewNumVariables(w);
  int i, left, top, width, height, x, y, use_color;
  int line_type, line_width, draw_color;
  StGWWinInfo *gwinfo;

  gwinfo = IViewWindowWinInfo(w);
  
  if (var1 >= vars || var2 >= vars) return;
  if (n > IViewNumLines(w)) return;
  
  StGrGetContentRect(gwinfo, &left, &top, &width, &height);
  StGrGetContentOrigin(gwinfo, &x, &y);
  use_color = StGWUseColor(gwinfo);
  line_type = StGWLineType(gwinfo);
  StGWGetLineWidth(gwinfo, &line_width);
  if (use_color) draw_color = StGWDrawColor(gwinfo);

  for (i = m; i < n; i++)
    IViewDrawDataLine(w, var1, var2, i, left, top, width, height, x, y,
                      use_color, draw_color);

  StGWSetLineType(gwinfo, line_type);
  StGWSetLineWidth(gwinfo, line_width);
  if (use_color) StGWSetDrawColor(gwinfo, draw_color);
}
#endif /* DODO */

#ifdef USESTRINGS
static IViewDrawDataString(w, var1, var2,  string,
                         left, top, width, height, orig_x, orig_y,
                         use_color, draw_color)
     IVIEW_WINDOW w;
     unsigned var1, var2, string;
     int left, top, width, height, orig_x, orig_y;
     int use_color, draw_color;
{
  int n = IViewNumStrings(w);
  int x, y;
/*  int right = left + width, bottom = top + height; not needed JKL */
  int up, h, v;
  int color;
  char *s;
  char *gwinfo = IViewWindowWinInfo(w);

  if (string >= n || IViewStringMasked(w, string)) return;
  x = orig_x + IViewStringScreenValue(w, var1, string);
  y = orig_y - IViewStringScreenValue(w, var2, string);
/*  if (x < left || x > right) return;
  if (y < top || y > bottom) return;*/
  
  if (use_color) {
    color = IViewStringColor(w, string);
    if (color != NOCOLOR) StGWSetDrawColor(gwinfo, color);
    else StGWSetDrawColor(gwinfo, draw_color);
  }
  IViewGetStringModifiers(w, string, &up, &h, &v);
  s = IViewStringString(w, string);
  if (s != nil) {
    if (up) StGWDrawTextUp(gwinfo, s, x, y, h, v);
    else StGWDrawText(gwinfo, s, x, y, h, v);
  }
}

IViewDrawDataStrings(w, var1, var2, m, n)
	IVIEW_WINDOW w;
	unsigned var1, var2, m, n;
{
  int vars = IViewNumVariables(w);
  int i, left, top, width, height, x, y, use_color, draw_color;
  char *gwinfo;

  gwinfo = IViewWindowWinInfo(w);
  
  if (var1 >= vars || var2 >= vars) return;
  if (n > IViewNumStrings(w)) return;
  
  StGrGetContentRect(gwinfo, &left, &top, &width, &height);
  StGrGetContentOrigin(gwinfo, &x, &y);
  use_color = StGWUseColor(gwinfo);
  if (use_color) draw_color = StGWDrawColor(gwinfo);
  for (i = m; i < n; i++)
    IViewDrawDataString(w, var1, var2, i, left, top,
                        width, height, x, y, use_color, draw_color);
  if (use_color) StGWSetDrawColor(gwinfo, draw_color);
}
#endif /* USESTRINGS */

IViewDepthCuePoints(w, var, cut1, cut2, cut3, m, n)
     IVIEW_WINDOW w;
     unsigned var, cut1, cut2, cut3, m, n;
{
  IViewDataCuePoints(IViewDataPtr(w), var, cut1, cut2, cut3, m, n);
}

/**************************************************************************/
/**                                                                      **/
/**                     Standard Callback Functions                      **/
/**                                                                      **/
/**************************************************************************/

VOID IViewStdResize(w)
	IVIEW_WINDOW w;
{
  int left, top, width, height, x, y, size;
  int m_left, m_top, m_right, m_bottom;
  int i , vars = IViewNumVariables(w);
  StGWWinInfo *gwinfo = IViewWindowWinInfo(w);
  
  width = StGWCanvasWidth(gwinfo);
  height = StGWCanvasHeight(gwinfo);
  StGrGetMargin(gwinfo, &m_left, &m_top, &m_right, &m_bottom);
  left = m_left;
  top = m_top;
  width -= m_left + m_right;
  height -= m_top + m_bottom;
  IViewGetAxisMargin(w, &m_left, &m_top, &m_right, &m_bottom);
  left += m_left;
  top += m_top;
  width -= m_left + m_right;
  height -= m_top + m_bottom;
  if (IViewFixedAspect(w)) {
    size = (width > height) ? height : width;
    left += (width - size) / 2;
    top += (height - size) / 2;
    StGrSetContentRect(gwinfo, left, top, size, size);
    StGrSetContentOrigin(gwinfo, left, top + size);
    for (i = 0; i < vars; i++) IViewSetScreenRange(w, i, 0, size);
  }
  else {
    StGrSetContentRect(gwinfo, left, top, width, height);
    StGrSetContentOrigin(gwinfo, left, top + height);
    StGrGetContentVariables(gwinfo, &x, &y);
    IViewSetScreenRange(w, x, 0, width);
    IViewSetScreenRange(w, y, 0, height);
    for (i = 0; i < vars; i++) 
      if (i != x && i != y)
	IViewSetScreenRange(w, i, 0, width);
  }
  IViewResizeOverlays(w);
}

VOID IViewStdRedraw(w)
	IVIEW_WINDOW w;
{
  int left, top, height, width;
  StGWWinInfo *gwinfo = IViewWindowWinInfo(w);
  
  StGWStartBuffering(gwinfo);
  
  if (IViewMouseMode(w) == brushing) IViewEraseBrush(w);
  StGWGetViewRect(gwinfo, &left, &top, &width, &height);  
  StGWSetClipRect(gwinfo, TRUE, left, top, width, height);
  IViewRedrawBackground(w);
  IViewRedrawOverlays(w);
  IViewRedrawContent(w);
  StGWBufferToScreen(gwinfo, left, top, width, height);
  if (IViewMouseMode(w) == brushing) IViewDrawBrush(w);
  StGrSetDirty(gwinfo, FALSE);
}

VOID IViewStdRedrawBackground(w)
	IVIEW_WINDOW w;
{
  int left, top, height, width;
  StGWWinInfo *gwinfo = IViewWindowWinInfo(w);
  
  StGWStartBuffering(gwinfo);
  
  if (IViewMouseMode(w) == brushing) IViewEraseBrush(w);
  StGWGetViewRect(gwinfo, &left, &top, &width, &height);  
  StGWSetClipRect(gwinfo, TRUE, left, top, width, height);
  StGWEraseRect(gwinfo, left, top, width, height);
  IViewDrawAxes(w);
  StGWBufferToScreen(gwinfo, left, top, width, height);
  if (IViewMouseMode(w) == brushing) IViewDrawBrush(w);
}

VOID IViewGetContentMarginRect(w, left, top, width, height)
	IVIEW_WINDOW w;
	int *left, *top, *width, *height;
{
  int cleft, ctop, cwidth, cheight, mleft, mtop, mright, mbottom;
  int x_showing, y_showing;
  StGWWinInfo *gwinfo = IViewWindowWinInfo(w);
  
  StGrGetContentRect(gwinfo, &cleft, &ctop, &cwidth, &cheight);
#ifdef DODO
  StGrGetMargin(gwinfo, &mleft, &mtop, &mright, &mbottom);
#endif /* DODO */
  IViewGetAxisMargin(w, &mleft, &mtop, &mright, &mbottom);
  IViewGetXaxis(w, &x_showing, nil, nil);
  IViewGetYaxis(w, &y_showing, nil, nil);
  if (y_showing || x_showing) {
    cwidth += mright;
    ctop -= mtop; cheight += mtop;
    if (! y_showing) { cleft -= mleft; cwidth += mleft; }
    if (! x_showing) cheight += mbottom;
  }
  if (left != nil) *left = cleft;
  if (top != nil) *top = ctop;
  if (width != nil) *width = cwidth;
  if (height != nil) *height = cheight;
}

VOID IViewStdRedrawContent(w)
	IVIEW_WINDOW w;
{
  int left, top, width, height, vleft, vtop, vwidth, vheight;
  int x, y;
  StGWWinInfo *gwinfo;
  
  gwinfo = IViewWindowWinInfo(w);

  if (IViewMouseMode(w) == brushing) IViewEraseBrush(w);
  IViewGetContentMarginRect(w, &left, &top, &width, &height);
  StGrGetContentVariables(gwinfo, &x, &y);
  
  StGWStartBuffering(gwinfo);
  StGWSetClipRect(gwinfo, TRUE, left, top, width + 1, height + 1);
  StGWEraseRect(gwinfo, left, top, width + 1, height + 1);
  IViewDrawDataPoints(w, x, y, 0, IViewNumPoints(w));
  IViewDrawDataLines(w, x, y, 0, IViewNumLines(w));
#ifdef USESTRINGS
  IViewDrawDataStrings(w, x, y, 0, IViewNumStrings(w));
#endif /* USESTRINGS */
  StGWBufferToScreen(gwinfo, left, top, width + 1, height + 1);
  StGWGetViewRect(gwinfo, &vleft, &vtop, &vwidth, &vheight);
  StGWSetClipRect(gwinfo, TRUE, vleft, vtop, vwidth, vheight);
  if (IViewMouseMode(w) == brushing) IViewDrawBrush(w);
  IViewResetScreenStates(w);
}

VOID IViewStdMarkPointsInRect(w, left, top, width, height)
     IVIEW_WINDOW w;
     int left, top, width, height;
{
  int c_left, c_top, c_width, c_height, x, y, center_x, center_y;
  int i, n = IViewNumPoints(w), vars = IViewNumVariables(w);
  unsigned var1, var2;
  StGWWinInfo *gwinfo = IViewWindowWinInfo(w);
  
  StGrGetContentRect(gwinfo, &c_left, &c_top, &c_width, &c_height);
  StGrGetContentOrigin(gwinfo, &center_x, &center_y);
  StGrGetContentVariables(gwinfo, (int *) &var1, (int *) &var2);
  
  if (var1 >= vars || var2 >= vars) return;
  
  for (i = 0; i < n; i++) {
    x = center_x + IViewPointScreenValue(w, var1, i);
    y = center_y - IViewPointScreenValue(w, var2, i);
    if ((x >= left && x <= left + width && y >= top && y <= top + height)
        && (! IViewPointMasked(w, i) && IViewPointState(w, i) != pointInvisible))
      IViewSetPointMark(w, i, TRUE);
    else IViewSetPointMark(w, i, FALSE);
  }
}

static DrawPoint(w, i, state, var1, var2, left, top, width, height, 
                 center_x, center_y, use_color)
	IVIEW_WINDOW w;
	int i;
	PointState state;
	int var1, var2, left, top, width, height, center_x, center_y;
	int use_color;
{
  int x, y, sym, hsym;
  int color, oldcolor;
  StGWWinInfo *gwinfo;

  gwinfo = IViewWindowWinInfo(w);
  
  if (use_color) {
    oldcolor = StGWDrawColor(gwinfo);
    color = IViewPointColor(w, i);
    if (color != NOCOLOR) StGWSetDrawColor(gwinfo, color);
  }
  x = center_x + IViewPointScreenValue(w, var1, i);
  y = center_y - IViewPointScreenValue(w, var2, i);
  IViewGetPointSymbol(w, i, &sym, &hsym);
  if (state == pointNormal) StGWReplaceSymbol(gwinfo, hsym, sym, x, y);
  else StGWReplaceSymbol(gwinfo, sym, hsym, x, y);
  if (use_color && color != NOCOLOR) StGWSetDrawColor(gwinfo, oldcolor);
}

static DrawLabel(w, i, var1, var2, left, top, width, height, 
                 center_x, center_y)
	IVIEW_WINDOW w;
	int i;
	int var1, var2, left, top, width, height, center_x, center_y;
{
  StGWWinInfo *gwinfo = IViewWindowWinInfo(w);
  int x, y, mode = StGWDrawMode(gwinfo);

  StGWSetDrawMode(gwinfo, 1);
  x = center_x + IViewPointScreenValue(w, var1, i);
  y = center_y - IViewPointScreenValue(w, var2, i);
  StGWDrawString(gwinfo, IViewPointLabel(w, i), 
                           x + LABEL_OFFSET, y - LABEL_OFFSET);
  StGWSetDrawMode(gwinfo, mode);
}

VOID IViewStdAdjustScreen(w)
	IVIEW_WINDOW w;
{
  StGWWinInfo *gwinfo = IViewWindowWinInfo(w);
  if (StGrDirty(gwinfo)) IViewRedrawContent(w);
  StGrSetDirty(gwinfo, FALSE);
  IViewResetScreenStates(w);
}

VOID IViewStdAdjustPointsInRect(w, left, top, width, height, state)
	IVIEW_WINDOW w;
	int left, top, width, height;
	PointState state;
{
  int i, n = IViewNumPoints(w);
  PointState point_state;
  int masked, in_rect;
  
  IViewCheckLinks(w);
  
  if (IViewMouseMode(w) == brushing) IViewEraseBrush(w);
  
  IViewMarkPointsInRect(w, left, top, width, height);
  for (i = 0; i < n; i++) {
    masked = IViewPointMasked(w, i);
    point_state = IViewPointState(w, i);
    if (! masked && point_state != pointInvisible) {
      in_rect = IViewPointMarked(w, i);
      if (in_rect && (int) point_state < (int) state) {
        IViewSetPointState(w, i, state);
      }
      else if (! in_rect 
	       && state == pointHilited && point_state == pointHilited) {
        IViewSetPointState(w, i, pointNormal);
      }
    }
  }
  IViewAdjustScreens(w);
  if (IViewMouseMode(w) == brushing) IViewDrawBrush(w);
}


VOID IViewStdAdjustScreenPoint(w, i)
	IVIEW_WINDOW w;
	int i;
{
  unsigned var1, var2;
  int left, top, width, height, x, y;
  PointState point_state, screen_state;
  int masked, showingLabels = IViewShowingLabels(w);
  StGWWinInfo *gwinfo = IViewWindowWinInfo(w);
  int use_color = StGWUseColor(gwinfo);
  
  StGrGetContentRect(gwinfo, &left, &top, &width, &height);
  StGrGetContentOrigin(gwinfo, &x, &y);
  StGrGetContentVariables(gwinfo, (int *) &var1, (int *) &var2);
  
  masked = IViewPointMasked(w, i);
  point_state = IViewPointState(w, i);
  screen_state = IViewPointScreenState(w, i);
  if (! masked && point_state != screen_state) {
    IViewSetPointScreenState(w, i, point_state);
	if (point_state == pointInvisible || screen_state == pointInvisible) {
	  StGrSetDirty(gwinfo, TRUE);
	}
    else if ((int) point_state > (int) screen_state) { 
      if (IViewMouseMode(w) == brushing) IViewEraseBrush(w);
      DrawPoint(w, i, point_state, var1, var2, left, top, width, height, x, y, use_color);
      if (showingLabels)
        DrawLabel(w, i, var1, var2, left, top, width, height, x, y); /* to draw */
      if (IViewMouseMode(w) == brushing) IViewDrawBrush(w);
    }
    else {
      if (IViewMouseMode(w) == brushing) IViewEraseBrush(w);
      if (showingLabels)
        DrawLabel(w, i, var1, var2, left, top, width, height, x, y); /* to erase */
      DrawPoint(w, i, point_state, var1, var2, left, top, width, height, x, y, use_color);
      if (IViewMouseMode(w) == brushing) IViewDrawBrush(w);
    }
  }
}

/**************************************************************************/
/**                                                                      **/
/**                       IView Rotation Functions                       **/
/**                                                                      **/
/**************************************************************************/

VOID IViewRotate2(w, var1, var2, newalpha)
	IVIEW_WINDOW w;
	unsigned var1, var2;
	double newalpha;
{
  static int *inbasis;
  static double **a;
  static int maxvars = 0;
  static double alpha = 0.0, s = 0.0, c = 1.0;
  int i, j, vars = IViewNumVariables(w);
  
  if (var1 >= vars || var2 >= vars) return;
  
  if (vars > maxvars) {
    maxvars = vars;
    if (a != nil) StFree(a[0]);
    StFree(a);
    StFree(inbasis);
    a = (double **) StCalloc(sizeof(double *), maxvars);
    a[0] = (double *) StCalloc(sizeof(double), maxvars * maxvars);
    for (i = 1; i < vars; i++) a[i] = a[0] + i * maxvars;
    inbasis = (int *) StCalloc(sizeof(int), maxvars);
  }
  
  if (alpha != newalpha) {
    alpha = newalpha;
    s = sin(alpha);
    c = cos(alpha);
  }
  
  for (i = 0; i < vars; i++) {
    inbasis[i] = FALSE;
    for (j = 0; j < vars; j++) a[i][j] = (i == j) ? 1.0 : 0.0;
  }
  a[var1][var1] =  c; a[var1][var2] = -s;
  a[var2][var1] =  s; a[var2][var2] =  c;
  inbasis[var1] = TRUE; inbasis[var2] = TRUE;

  IViewApplyTransformation(w, a, inbasis);
}
  
/**************************************************************************/
/**                                                                      **/
/**                        Miscellaneous Functions                       **/
/**                                                                      **/
/**************************************************************************/

static double NiceValue(x)
     double x;
{
  long ilx;
  double lx, v1, v2, v3, v4;
  
  if (x <= 0.0) return (0.0);
  else {
    lx = log(x) / log(10.0);
    ilx = floor(lx);
    v1 = pow(10.0, (float) ilx);
    v2 = pow(10.0, (float) ilx) * 2.0;
    v3 = pow(10.0, (float) ilx) * 5.0;
    v4 = pow(10.0, (float) ilx + 1);
    
    if ((fabs(x - v1) < fabs(x - v2))
	&& (fabs(x - v1) < fabs(x - v3))
	&& (fabs(x - v1) < fabs(x - v4)))
      return(v1);
    else if ((fabs(x - v2) < fabs(x - v3))
	     && (fabs(x - v2) < fabs(x - v4)))
      return(v2);
    else if (fabs(x - v3) < fabs(x - v4))
      return(v3);
    else
      return(v4);
  }
}

VOID GetNiceRange(low, high, ticks)
     double *low, *high;
     int *ticks;
{
  double delta;
  
#ifdef IEEEFP
  if (! is_finite(*high) || ! is_finite(*low)) return;
#else
  if (fabs(*high) >= HUGE || fabs(*low) >= HUGE) return;
#endif /* IEEEFP */
  if ((*high <= *low) || (*ticks < 2)) return;
  
  delta = NiceValue((*high - *low) / (*ticks - 1));
  if (delta <= 0.0) return;
  
  *low = floor(*low / delta + .01) * delta;   /* adjust by .01 for rounding */
  *high = ceil(*high / delta - .01) * delta;
  
  *ticks = 1 + (.01 + (*high - *low) / delta); /* add .01 for rounding */
}

