/* mswdlg - Low Level Dialog Objects for Microsoft Windows             */
/* XLISP-STAT 2.1 Copyright (c) 1990, 1991, by Luke Tierney            */
/* Additions to Xlisp 2.1, Copyright (c) 1989 by David Michael Betz    */
/* You may give out copies of this software; for conditions see the    */
/* file COPYING included with this distribution.                       */
 
#include <string.h>
#include "dialogs.h"
#include "wxlisp.h"
#include "ledit.h"
#include "iview.h"

/* external variables */
extern HWND hWndFrame, hWndClient;
extern HANDLE hInst, hAccel;

/* static variables */
static char szXLSDlgClass[] = "XLSDialog";
static BOOL InModalDialog = FALSE;
static LVAL ModalItem = NIL;

/* function prototypes */
IVIEW_WINDOW GETDIALOGADDRESS(LVAL);
LVAL coerce_to_vector(LVAL);
LVAL cvfixnum(FIXTYPE);
LVAL make_string(char *);
LVAL send_message(LVAL, LVAL);
LVAL send_message_1L(LVAL, LVAL, LVAL);
LVAL slot_value(LVAL, LVAL);
LVAL xlenter(char *);
LVAL arraydata(LVAL);
LVAL displacedarraydim(LVAL);
LVAL integer_list_2(int, int);
LVAL xlbadtype(LVAL);
Point ListToPoint(LVAL);
void set_dialog_address(IVIEW_WINDOW, LVAL);
void set_slot_value(LVAL, LVAL, LVAL);
void standard_hardware_clobber(LVAL);
void xlerror(char *, LVAL);
void xlfail(char *);
int dialog_item_p(LVAL);
int button_item_p(LVAL);
int toggle_item_p(LVAL);
int choice_item_p(LVAL);
int text_item_p(LVAL);
int scroll_item_p(LVAL);
int list_item_p(LVAL);
int llength(LVAL);
int arrayp(LVAL);
int displacedarrayp(LVAL);
int matrixp(LVAL);
int simplevectorp(LVAL);
int valid_dialog_address(LVAL);
BOOL FAR PASCAL XLSDlgProc(HWND, unsigned, WORD, LONG);
void DialogRemove(LVAL);
void DialogSetDefaultButton(LVAL, LVAL);
static void InstallItem(IVIEW_WINDOW, LVAL);
static void InstallButtonItem(IVIEW_WINDOW, LVAL);
static void InstallToggleItem(IVIEW_WINDOW, LVAL);
static void InstallChoiceItem(IVIEW_WINDOW, LVAL);
static void InstallTextItem(IVIEW_WINDOW, LVAL);
static void InstallScrollItem(IVIEW_WINDOW, LVAL);
static void InstallListItem(IVIEW_WINDOW, LVAL);
static void SetClusterValue(IVIEW_WINDOW, int);

/***********************************************************************/
/**                                                                   **/
/**                       Defines and Data Types                      **/
/**                                                                   **/
/***********************************************************************/

typedef struct {
  LVAL object;
  int count, dflt;
  // item data;
} DialogData;

typedef struct {
  int type;
  int itemNumber, clusterLeader, clusterSize;
  HWND itemHandle;
  LVAL object;
} DialogItemData;

#define SETDIALOGOBJECT(w, d) SetWindowLong((HWND) (w), 0, (LONG) (d))
#define GETDIALOGOBJECT(w) ((LVAL) GetWindowLong((HWND) (w), 0))
#define SETDIALOGDATA(w, d) SetWindowWord((HWND) (w), 4, (WORD) (d))
#define GETDIALOGDATA(w) ((HANDLE) GetWindowWord((HWND) (w), 4))
#define GETITEMDATA(d) ((DialogItemData *) (((DialogData *) (d)) + 1))

#define MIN_DLG_WIDTH 50
#define MIN_DLG_HEIGHT 20
#define MIN_BUTTON_STRLEN 10

#define STATIC_TEXT_PAD 4
#define EDIT_TEXT_PAD 10

#define SCROLL_WIDTH 180
#define SCROLL_MIN 0
#define SCROLL_MAX 100
#define SCROLL_PAGE 5

#define LIST_ITEM_PAD 2
#define MAX_LIST_ROWS 12
#define LIST_COL_CHARS 15   // ### 20 ?
#define MAX_ENTRY_LENGTH 30

#define ITEM_INDEX_BASE 1024

/***********************************************************************/
/**                                                                   **/
/**                          Utility Functions                        **/
/**                                                                   **/
/***********************************************************************/

static void check_alloc(LONG p)
{
  if (! p) xlfail("allocation failed");
}

static void check_lock(void *p)
{
  if (! p) xlfail("lock failed");
}

static int FindItemType(LVAL item)
{
  if (consp(item)) return(ITEM_LIST);
  else if (button_item_p(item)) return(BUTTON_ITEM);
  else if (toggle_item_p(item)) return(TOGGLE_ITEM);
  else if (text_item_p(item)) return(TEXT_ITEM);
  else if (choice_item_p(item)) return(CHOICE_ITEM);
  else if (scroll_item_p(item)) return(SCROLL_ITEM);
  else if (list_item_p(item)) return(LIST_ITEM);
  else xlfail("item of unknown type");
  return(0);
}

static int count_hardware_items(LVAL items)
{ 
  LVAL temp;
  int n;

  if (! consp (items)) return(0);
  else {
    for (n = 0; consp(items); items = cdr(items)) {
      switch(FindItemType(car(items))) {
      case CHOICE_ITEM: 
	temp = slot_value(car(items), s_text);
	if (! consp(temp)) xlerror("not a list", temp);
	n += llength(temp);
        break;
      case ITEM_LIST: n += count_hardware_items(car(items)); break;
      default: n += 1;
      }
    }
  }
  return(n);
}

static Point AvCharSize(BOOL leaded)
{
  Point pt;
  HDC hDC;
  TEXTMETRIC tm;

  hDC = GetDC(hWndFrame);
  SelectObject(hDC, GetStockObject(SYSTEM_FIXED_FONT));
  GetTextMetrics(hDC, &tm);
  pt.h = tm.tmAveCharWidth;
  pt.v = tm.tmHeight;
  if (leaded) pt.v += tm.tmExternalLeading;
  ReleaseDC(hWndFrame, hDC);
  return(pt);
}

static Point text_size(char *s)
{
  char *bp;
  int w;
  Point sz;
  
  for (w = 0, sz.h = 0, sz.v = 0; *s != '\0'; s++) {
    for (bp = buf; *s != '\0' && *s != '\r' && *s != '\n'; s++, bp++)
      *bp = *s;
    *bp = '\0';
    w = strlen(buf);
    if (sz.h < w) sz.h = w;
    sz.v++;
    if (*s == '\0') break;
  }
  return(sz);
}

static void truncateListEntry(char *s)
{
  if (strlen(s) > MAX_ENTRY_LENGTH) {
    s = s + MAX_ENTRY_LENGTH - 3;
    s[0] = '.'; s[1] = '.'; s[2] = '.'; s[3] = '\0';
  }
}
 
static BOOL FindItemData(IVIEW_WINDOW theDialog,
			 WORD wParam,
			 DialogItemData *di)
{
  HANDLE hd;
  DialogData *dd;
  DialogItemData *did;
  int index, result;

  index = wParam - ITEM_INDEX_BASE;
  if (index < 0) return(FALSE);

  hd = GETDIALOGDATA(theDialog);
  dd = (DialogData *) GlobalLock(hd);
  check_lock(dd);
  did = GETITEMDATA(dd);
  if (index < dd->count) {
    *di = did[index];
    result = TRUE;
  }
  else result = FALSE;
  GlobalUnlock(hd);
  return(result);
}

static BOOL FindItemObjectData(LVAL item, DialogItemData *di)
{
  LVAL dialog;
  HANDLE hd;
  DialogData *dd;
  DialogItemData *did;
  int i, found;

  dialog = slot_value(item, s_dialog);
  if (dialog == NIL || ! check_dialog_address(dialog)) return(FALSE);

  hd = GETDIALOGDATA(GETDIALOGADDRESS(dialog));
  dd = (DialogData *) GlobalLock(hd);
  check_lock(dd);
  did = GETITEMDATA(dd);
  for (i = 0, found = FALSE; i < dd->count && ! found; i++) {
     if (did[i].object == item) {
       *di = did[i];
      found = TRUE;
    }
  }
  GlobalUnlock(hd);
  return(found);
}

/***********************************************************************/
/**                                                                   **/
/**                    Internal Dialog Data Functions                 **/
/**                                                                   **/
/***********************************************************************/

static HANDLE MakeDialogData(LVAL dialog)
{
  int numItems;
  HANDLE hItems;
  DialogData *dd;

  numItems = count_hardware_items(slot_value(dialog, s_items));
  hItems = GlobalAlloc(GMEM_MOVEABLE | GMEM_ZEROINIT,
		       sizeof(DialogData) + numItems * sizeof(DialogItemData));
  check_alloc(hItems);
  dd = (DialogData *) GlobalLock(hItems);
  check_lock(dd);
  dd->count = 0;
  dd->object = dialog;
  dd->dflt = -1;
  GlobalUnlock(hItems);
  return(hItems);
}

static void FreeDialogData(HANDLE hItems)
{
  GlobalFree(hItems);
}

static void InstallItemList(IVIEW_WINDOW theDialog, LVAL items)
{
  for (; consp(items); items = cdr(items))
    if (consp(car(items))) InstallItemList(theDialog, car(items));
    else InstallItem(theDialog, car(items));
}

static void InstallItem(IVIEW_WINDOW theDialog, LVAL item)
{
  if (! dialog_item_p(item)) xlerror("not a dialog item", item);
  switch (FindItemType(item)) {
  case BUTTON_ITEM: InstallButtonItem(theDialog, item); break;
  case TOGGLE_ITEM: InstallToggleItem(theDialog, item); break;
  case CHOICE_ITEM: InstallChoiceItem(theDialog, item); break;
  case TEXT_ITEM:   InstallTextItem(theDialog, item);   break;
  case SCROLL_ITEM: InstallScrollItem(theDialog, item); break;
  case LIST_ITEM:   InstallListItem(theDialog, item);   break;
  default: xlfail("unkown item type");
  }
}

/***********************************************************************/
/**                                                                   **/
/**                Initialization and Callback Functions              **/
/**                                                                   **/
/***********************************************************************/

BOOL InitApplDialogs(HANDLE hInstance)
{
  WNDCLASS wc;

  /* class structure for the dialog windows */
  wc.style = CS_HREDRAW | CS_VREDRAW;
  wc.lpfnWndProc = (FARPROC) XLSDlgProc;
  wc.cbClsExtra = 0;
  wc.cbWndExtra = sizeof(LONG) + sizeof(WORD);
  wc.hInstance = hInstance;
  wc.hIcon = NULL;
  wc.hCursor = LoadCursor((HWND) NULL, IDC_ARROW);
  wc.hbrBackground = COLOR_WINDOW + 1;
  wc.lpszMenuName = NULL;
  wc. lpszClassName = szXLSDlgClass;
  if (! RegisterClass(&wc)) return(FALSE);
  return(TRUE);
}

InitInstDialogs(HANDLE hInstance)
{
  return(TRUE);
}

BOOL FAR PASCAL XLSDlgProc(HWND hWnd,
			     unsigned message,
			     WORD wParam,
			     LONG lParam)
{
  LVAL dialog;
  DialogItemData di;
  int val, wid;

  switch(message) {
  case WM_CREATE:
    dialog = (LVAL) ((LPCREATESTRUCT) lParam)->lpCreateParams;
    SETDIALOGOBJECT(hWnd, dialog);
    SETDIALOGDATA(hWnd, MakeDialogData(dialog));
    InstallItemList((IVIEW_WINDOW) hWnd, slot_value(dialog, s_items));
    DialogSetDefaultButton(dialog, slot_value(dialog, s_default_button));
    break;
  case WM_COMMAND:
    switch (wParam) {
    case IDC_SHOWWINDOW:
    case IDC_HIDEWINDOW:
      break;
    default:
      if (FindItemData(hWnd, wParam, &di)) {
	if (di.object != NIL) {
	  switch (di.type) {
	  case TOGGLE_ITEM:
	    val = ! SendMessage(di.itemHandle, BM_GETCHECK, 0, 0);
	    SendMessage(di.itemHandle, BM_SETCHECK, val, 0);
	    set_slot_value(di.object, s_value, (val) ? s_true : NIL);
	    break;
	  case CHOICE_ITEM:
	    SetClusterValue(hWnd, di.itemNumber);
	    set_slot_value(di.object, s_value,
			   cvfixnum((FIXTYPE) di.itemNumber - di.clusterLeader));
	    break;
	  }
	  if (di.type == LIST_ITEM) {
	    switch (HIWORD(lParam)) {
	    case LBN_DBLCLK:
	      send_message_1L(di.object, sk_do_action, s_true);
	      break;
	    case LBN_SELCHANGE:
	    case LBN_SELCANCEL:
	      send_message(di.object, sk_do_action);
	      break;
	    }
	  }
	  else {
	    if (InModalDialog) ModalItem = di.object;
	    else send_message(di.object, sk_do_action);
	  }
	}
	return(0);
      }
    }
    break;
  case WM_VSCROLL:
  case WM_HSCROLL:
    wid = GetWindowWord(HIWORD(lParam), GWW_ID);
    switch (wParam) {
    case SB_PAGEDOWN:
    case SB_LINEDOWN:
    case SB_PAGEUP:
    case SB_LINEUP:
    case SB_TOP:
    case SB_BOTTOM:
      if (FindItemData(hWnd, wid, &di) && di.object != NIL) {
	int low, high, value, page;
	LVAL temp, item = di.object;

	temp = slot_value(item, s_min_value);
	low = fixp(temp) ? (int) getfixnum(temp) : SCROLL_MIN;
	temp = slot_value(item, s_max_value);
	high = fixp(temp) ? (int) getfixnum(temp) : SCROLL_MAX;
	temp = slot_value(item, s_value);
	value = (fixp(temp)) ? (int) getfixnum(temp) : low;
	temp = slot_value(item, s_page_increment);
	page = (fixp(temp)) ? (int) getfixnum(temp) : SCROLL_PAGE;
	switch (wParam) {
	case SB_PAGEDOWN: value += page; break;
	case SB_LINEDOWN: value++;       break;
	case SB_PAGEUP:   value -= page; break;
	case SB_LINEUP:   value--;       break;
	case SB_TOP:      value = low;   break;
	case SB_BOTTOM:   value = high;  break;
	}
	if (value < low) value = low;
	if (value > high) value = high;
	set_slot_value(item, s_value, cvfixnum((FIXTYPE) value));
	SetScrollPos(di.itemHandle, SB_CTL, value, TRUE);
	send_message(item, sk_scroll_action);
      }
      return(0);
    case SB_THUMBPOSITION:
      val = LOWORD(lParam);
      if (FindItemData(hWnd, wid, &di) && di.object != NIL) {
	set_slot_value(di.object, s_value, cvfixnum((FIXTYPE) val));
	SetScrollPos(di.itemHandle, SB_CTL, val, TRUE);
	send_message(di.object, sk_do_action);
      }
      return(0);
    }
    break;
  case WM_QUERYENDSESSION:
  case WM_CLOSE:
    if (! InModalDialog)
      send_message((LVAL) GetWindowLong(hWnd, 0), sk_close);
    return(0);
  case WM_DESTROY:
    FreeDialogData(GETDIALOGDATA(hWnd));
  }
  return((BOOL) DefWindowProc(hWnd, message, wParam, lParam));
}

void MSWResetDialogs(void)
{
  ModalItem = NIL;
  InModalDialog = FALSE;
}

/***********************************************************************/
/**                                                                   **/
/**                Allocation and Deallocation Functions              **/
/**                                                                   **/
/***********************************************************************/

void DialogAllocate(LVAL dialog)
{
  IVIEW_WINDOW theDialog;
  Point loc, size;
  Rect bounds;
  char *title;
  BOOL goAway, modeless;
  DWORD flags;
  HANDLE items, ref;

  if (check_dialog_address(dialog)) DialogRemove(dialog);
    
  if (! stringp(slot_value(dialog, s_title)))
    xlerror("not a string", slot_value(dialog, s_title));
  title = (char *) getstring(slot_value(dialog, s_title));

  goAway = (slot_value(dialog, s_go_away) != NIL) ? TRUE : FALSE;
  modeless = (slot_value(dialog, s_type) != s_modal) ? TRUE : FALSE;
  flags = WS_POPUP | WS_DLGFRAME | WS_VISIBLE;
  if (modeless) flags |= WS_CAPTION; // ### kill border so dlgframe works?
  if (goAway) flags |= WS_SYSMENU;

  loc = ListToPoint(slot_value(dialog, s_location));
  size = ListToPoint(slot_value(dialog, s_size));
  if (size.h < MIN_DLG_WIDTH) size.h = MIN_DLG_WIDTH;
  if (size.v < MIN_DLG_HEIGHT) size.v = MIN_DLG_HEIGHT;
  if (modeless) { // ### fix if manage to get proper frame
    size.h += 2 * GetSystemMetrics(SM_CXBORDER);
    size.v += GetSystemMetrics(SM_CYBORDER);
    size.v += GetSystemMetrics(SM_CYCAPTION);
  }
  else {
    size.h += 2 * GetSystemMetrics(SM_CXDLGFRAME);
    size.v += 2 * GetSystemMetrics(SM_CYDLGFRAME);
  }

  theDialog = CreateWindow(szXLSDlgClass,
			   title, flags,
			   loc.h, loc.v, size.h, size.v,
			   hWndFrame, 0, hInst, (LONG) dialog);
  check_alloc(theDialog);
  set_dialog_address(theDialog, dialog);
}

void DialogRemove(LVAL dialog)
{
  if (check_dialog_address(dialog))
    DestroyWindow((HWND) GETDIALOGADDRESS(dialog));
  if (objectp(dialog)) standard_hardware_clobber(dialog);
}

/***********************************************************************/
/**                                                                   **/
/**                       Dialog Item Functions                       **/
/**                                                                   **/
/***********************************************************************/

static void InstallBtnItem(IVIEW_WINDOW theDialog,
			   LVAL item,
			   DWORD flags,
			   int type,
			   DialogItemData *di)
{
  HANDLE hd;
  DialogItemData *data;
  DialogData *dialogData;
  HWND theItem;
  int itemIndex;
  char *text;
  Point loc, size;

  hd = GETDIALOGDATA(theDialog);
  dialogData = (DialogData *) GlobalLock(hd);
  check_lock(dialogData);

  itemIndex = (dialogData->count)++;

  if (! stringp(slot_value(item, s_text)))
    xlerror("not a string", slot_value(item, s_text));
  text = (char *) getstring(slot_value(item, s_text));

  loc = ListToPoint(slot_value(item, s_location));
  size = ListToPoint(slot_value(item, s_size));

  theItem = CreateWindow("button", text,
			 WS_CHILD | WS_VISIBLE | flags,
			 loc.h, loc.v, size.h, size.v,
			 (HWND) theDialog, itemIndex + ITEM_INDEX_BASE,
			 hInst, NULL);
  check_alloc(theItem);

  data = GETITEMDATA(dialogData);
  data[itemIndex].type = type;
  data[itemIndex].itemNumber = itemIndex;
  data[itemIndex].itemHandle = theItem;
  data[itemIndex].object = item;
  if (di) *di = data[itemIndex];

  GlobalUnlock(hd);
}

static void InstallButtonItem(IVIEW_WINDOW theDialog, LVAL item)
{
  InstallBtnItem(theDialog, item, BS_PUSHBUTTON, BUTTON_ITEM, NULL);
}

void DialogButtonGetDefaultSize(LVAL item, int *pwidth, int *pheight)
{
  LVAL text;
  Point csz;
  int n;

  text = slot_value(item, s_text);
  if (! stringp(text)) xlerror("not a string", text);
  csz = AvCharSize(TRUE);
  n = strlen((char *) getstring(text));
  if (n < MIN_BUTTON_STRLEN) n = MIN_BUTTON_STRLEN;

  if (pwidth != nil) *pwidth = (n + 2) * csz.h;
  if (pheight != nil) *pheight = (7 * csz.v) / 4;
}

static void InstallToggleItem(IVIEW_WINDOW theDialog, LVAL item)
{
  DialogItemData di;

  InstallBtnItem(theDialog, item, BS_CHECKBOX, TOGGLE_ITEM, &di);
  SendMessage(di.itemHandle,
	      BM_SETCHECK,
	      (slot_value(item, s_value) != NIL) ? TRUE : FALSE,
	      0);
}

void DialogToggleGetDefaultSize(LVAL item, int *pwidth, int *pheight)
{
  LVAL text;
  Point csz;
  int n;

  text = slot_value(item, s_text);
  if (! stringp(text)) xlerror("not a string", text);
  csz = AvCharSize(TRUE);
  n = strlen((char *) getstring(text));

  if (pwidth != nil) *pwidth = (n + 4) * csz.h;
  if (pheight != nil) *pheight = (3 * csz.v) / 2;
}

LVAL DialogToggleItemValue(item, set, value)
	LVAL item, value;
	int set;
{
  DialogItemData itemData;

  if (set) {
    set_slot_value(item, s_value, (value != NIL) ?  s_true : NIL);
    if (FindItemObjectData(item, &itemData)) {
      SendMessage(itemData.itemHandle,
		  BM_SETCHECK,
		  (value != NIL) ? TRUE : FALSE,
		  0);
    }
  }
  return(slot_value(item, s_value));
}

static void InstallChoiceItem(IVIEW_WINDOW theDialog, LVAL item)
{
  LVAL titles, temp;
  DialogItemData *data;
  DialogData *dialogData;
  HWND theItem;
  int itemIndex, clusterLeader, clusterSize, initial;
  char *text;
  Point loc, size;
  HANDLE hd;
  
  titles = slot_value(item, s_text);
  if (! consp(titles)) xlerror("not a list", titles);

  loc = ListToPoint(slot_value(item, s_location));
  size = ListToPoint(slot_value(item, s_size));
  clusterSize = llength(titles);
  size.v /= clusterSize;

  hd = GETDIALOGDATA(theDialog);
  dialogData = (DialogData *) GlobalLock(hd);
  check_lock(dialogData);

  clusterLeader = dialogData->count;
  for (; consp(titles); titles = cdr(titles)) {
    if (! stringp(car(titles))) xlerror("not a string", car(titles));
    text = (char *) getstring(car(titles));

    itemIndex = (dialogData->count)++;
    theItem = CreateWindow("button", text,
			   WS_CHILD | WS_VISIBLE | BS_RADIOBUTTON,
			   loc.h, loc.v, size.h, size.v,
			   (HWND) theDialog, itemIndex + ITEM_INDEX_BASE,
			   hInst, NULL);
    check_alloc(theItem);
    loc.v += size.v;

    data = GETITEMDATA(dialogData);
    data[itemIndex].type = CHOICE_ITEM;
    data[itemIndex].itemNumber = itemIndex;
    data[itemIndex].itemHandle = theItem;
    data[itemIndex].object = item;
    data[itemIndex].clusterLeader = clusterLeader;
    data[itemIndex].clusterSize = clusterSize;
  }
  temp = slot_value(item, s_value);
  initial = (fixp(temp)) ? (int) getfixnum(temp) : 0;
  if (initial < 0 || initial >= clusterSize) initial = 0;
  initial += clusterLeader;
  SendMessage(data[initial].itemHandle, BM_SETCHECK, 1, 0);
  GlobalUnlock(hd);
}

void DialogChoiceGetDefaultSize(LVAL item, int *pwidth, int *pheight)
{
  Point csz;
  LVAL text = slot_value(item, s_text);
  int n, k, h;
  
  csz = AvCharSize(TRUE);
  for (h = 0, n = 0; consp(text); text = cdr(text)) {
    k = strlen((char *) getstring(car(text)));
    if (n < k) n = k;
    h += (3 * csz.v) / 2;
  }
  if (pwidth != nil) *pwidth = (n + 4) * csz.h;
  if (pheight != nil) *pheight = h;
}

static void SetClusterValue(IVIEW_WINDOW theDialog, int index)
{
  int i, n, leader;
  HANDLE hd;
  DialogData *dialogData;
  DialogItemData *data;

  if (index < 0) return;

  hd = GETDIALOGDATA(theDialog);
  dialogData = (DialogData *) GlobalLock(hd);
  check_lock(dialogData);

  if (index < dialogData->count) {
    data = GETITEMDATA(dialogData);
    leader = data[index].clusterLeader;
    n = data[index].clusterSize + leader;

    for (i = leader; i < n; i++)
      SendMessage(data[i].itemHandle, BM_SETCHECK, 0, 0);
    SendMessage(data[index].itemHandle, BM_SETCHECK, 1, 0);
  }

  GlobalUnlock(hd);
}

LVAL DialogChoiceItemValue(LVAL item, int set, int value)
{
  DialogItemData itemData;
  LVAL dialog, textlist;

  if (set) {
    textlist = slot_value(item, s_text);
    if (! consp(textlist) || value < 0 || value >= llength(textlist))
      xlfail("Value out of range");
    set_slot_value(item, s_value, cvfixnum((FIXTYPE) value));
    if (FindItemObjectData(item, &itemData)) {
      if (value < 0 || value >= itemData.clusterSize)
	xlfail("value out of range");
      dialog = slot_value(item, s_dialog);
      SetClusterValue(GETDIALOGADDRESS(dialog),
		      itemData.clusterLeader + value);
    }
  }
  return(slot_value(item, s_value));
}

static void InstallTextItem(IVIEW_WINDOW theDialog, LVAL item)
{
  HANDLE hd;
  DialogItemData *data;
  DialogData *dialogData;
  HWND theItem;
  int editable, itemIndex;
  char *text;
  Point loc, size;
  
  if (! stringp(slot_value(item, s_text)))
    xlerror("not a string", slot_value(item, s_text));
  text = (char *) getstring(slot_value(item, s_text));
  loc = ListToPoint(slot_value(item, s_location));
  size = ListToPoint(slot_value(item, s_size));
  editable = (slot_value(item, s_editable)) ? TRUE : FALSE;

  hd = GETDIALOGDATA(theDialog);
  dialogData = (DialogData *) GlobalLock(hd);
  check_lock(dialogData);
  itemIndex = (dialogData->count)++;

  strcpy(buf, text);
  if (editable) {
    theItem = CreateWindow("edit", buf, WS_CHILD | WS_VISIBLE | WS_BORDER,
			   loc.h, loc.v, size.h, size.v,
			   (HWND) theDialog, itemIndex + ITEM_INDEX_BASE,
			   hInst, NULL);
  }
  else {
    theItem = CreateWindow("static", buf, WS_CHILD | WS_VISIBLE,
			   loc.h, loc.v, size.h, size.v,
			   (HWND) theDialog, itemIndex + ITEM_INDEX_BASE,
			   hInst, NULL);
  }
  check_alloc(theItem);

  data = GETITEMDATA(dialogData);
  data[itemIndex].type = TEXT_ITEM;
  data[itemIndex].itemNumber = itemIndex;
  data[itemIndex].itemHandle = theItem;
  data[itemIndex].object = item;
  GlobalUnlock(hd);
}

void DialogTextGetDefaultSize(LVAL item, int *pwidth, int *pheight)
{
  Point csz, tsz;
  LVAL text = slot_value(item, s_text);
  LVAL text_length = slot_value(item, xlenter("TEXT-LENGTH"));
  char *s;

  tsz.h = 0; tsz.v = 0;
  csz = AvCharSize(TRUE);
  if (stringp(text)) {
    tsz = text_size((char *) getstring(text));
  }
  if (fixp(text_length)) {
    if (tsz.h < getfixnum(text_length)) tsz.h = (int) getfixnum(text_length);
    if (tsz.v < 1) tsz.v = 1;
  }
  if (slot_value(item, s_editable) != NIL) {
    if (pwidth != nil) *pwidth = csz.h * tsz.h + EDIT_TEXT_PAD;
    if (pheight != nil) *pheight = csz.v * tsz.v + EDIT_TEXT_PAD;
  }
  else {
    if (pwidth != nil) *pwidth = csz.h * tsz.h + STATIC_TEXT_PAD;
    if (pheight != nil) *pheight = csz.v * tsz.v + STATIC_TEXT_PAD;
  }
}

LVAL DialogTextItemText(LVAL item, int set, char *text)
{
  DialogItemData di;
  int n;

  if (set) set_slot_value(item, s_text, make_string(text));
  if (FindItemObjectData(item, &di)) {
    if (set) {
      strcpy(buf, text);
      SetWindowText(di.itemHandle, buf);
    }
    n = GetWindowText(di.itemHandle, buf, STRMAX);
    buf[n] = '\0';
    set_slot_value(item, s_text, make_string(buf));
  }
  return(slot_value(item, s_text));
}

static void InstallScrollItem(IVIEW_WINDOW theDialog, LVAL item)
{
  HANDLE hd;
  DialogItemData *data;
  DialogData *dialogData;
  HWND theItem;
  int low, high, value;
  int itemIndex;
  Point loc, size;
  LVAL temp;
  DWORD flags;
  
  loc = ListToPoint(slot_value(item, s_location));
  size = ListToPoint(slot_value(item, s_size));
  flags = (size.h > size.v) ? SBS_HORZ : SBS_VERT;

  temp = slot_value(item, s_min_value);
  low = fixp(temp) ? (int) getfixnum(temp) : SCROLL_MIN;
  temp = slot_value(item, s_max_value);
  high = fixp(temp) ? (int) getfixnum(temp) : SCROLL_MAX;
  temp = slot_value(item, s_value);
  value = (fixp(temp)) ? (int) getfixnum(temp) : low;

  hd = GETDIALOGDATA(theDialog);
  dialogData = (DialogData *) GlobalLock(hd);
  check_lock(dialogData);

  itemIndex = (dialogData->count)++;
  theItem = CreateWindow("scrollbar", NULL,
			 WS_CHILD | WS_VISIBLE | WS_TABSTOP | flags,
			 loc.h, loc.v, size.h, size.v,
			 (HWND) theDialog, itemIndex + ITEM_INDEX_BASE,
			 hInst, NULL);
  check_alloc(theItem);
  SetScrollRange(theItem, SB_CTL, low, high, FALSE);
  SetScrollPos(theItem, SB_CTL, value, TRUE);

  data = GETITEMDATA(dialogData);
  data[itemIndex].type = SCROLL_ITEM;
  data[itemIndex].itemNumber = itemIndex;
  data[itemIndex].itemHandle = theItem;
  data[itemIndex].object = item;
  GlobalUnlock(hd);
}

void DialogScrollGetDefaultSize(LVAL item, int *pwidth, int *pheight)
{
  int h, w;

  h = GetSystemMetrics(SM_CYHSCROLL);
  w = SCROLL_WIDTH; // ### 10 * h
  if (pwidth != nil) *pwidth = w;
  if (pheight != nil) *pheight = h;
}

static LVAL scroll_item_value(LVAL item, int set, int value, int which)
{
  LVAL slot, temp;
  DialogItemData di;
  int low, high;

  switch (which) {
  case 'V': slot = s_value;     break;
  case 'H': slot = s_max_value; break;
  case 'L': slot = s_min_value; break;
  }

  if (set) {
    if (FindItemObjectData(item, &di)) {
      switch (which) {
      case 'V':
	SetScrollPos(di.itemHandle, SB_CTL, value, TRUE);
	break;
      case 'H':
	temp = slot_value(item, s_min_value);
	low = fixp(temp) ? (int) getfixnum(temp) : SCROLL_MIN;
	SetScrollRange(di.itemHandle, SB_CTL, low, value, TRUE);
	break;
      case 'L':
	temp = slot_value(item, s_max_value);
	high = fixp(temp) ? (int) getfixnum(temp) : SCROLL_MAX;
	SetScrollRange(di.itemHandle, SB_CTL, value, high, TRUE);
	break;
      }
    }
    set_slot_value(item, slot, cvfixnum((FIXTYPE) value));
    temp = slot_value(item, s_min_value);
    low = fixp(temp) ? (int) getfixnum(temp) : SCROLL_MIN;
    temp = slot_value(item, s_max_value);
    high = fixp(temp) ? (int) getfixnum(temp) : SCROLL_MAX;
    temp = slot_value(item, s_value);
    value = (fixp(temp)) ? (int) getfixnum(temp) : low;
    if (value < low || value > high)
      set_slot_value(item, s_value, cvfixnum((FIXTYPE) low));
  }
  return(slot_value(item, slot));
}

LVAL DialogScrollItemValue(LVAL item, int set, int value)
{
  return(scroll_item_value(item, set, value, 'V'));
}

LVAL DialogScrollItemMax(LVAL item, int set, int value)
{
  return(scroll_item_value(item, set, value, 'H'));
}

LVAL DialogScrollItemMin(LVAL item, int set, int value)
{
  return(scroll_item_value(item, set, value, 'L'));
}

static Point ListItemDims(LVAL item)
{
  LVAL listData = slot_value(item, s_list_data);
  Point sz;

  if (! listp(listData) && ! arrayp(listData)) listData = NIL;
  if (listp(listData) || simplevectorp(listData)) {
    sz.v = (listp(listData)) ? llength(listData) : getsize(listData);
    sz.h = 1;
  }
  else if (matrixp(listData)) {
    sz.v = (int) getfixnum(getelement(displacedarraydim(listData), 0));
    sz.h = (int) getfixnum(getelement(displacedarraydim(listData), 1));
  }
  else xlerror("this form of data is not yet supported", listData);
  return(sz);
}

static void InstallListItem(IVIEW_WINDOW theDialog, LVAL item)
{
  HANDLE hd;
  DialogItemData *data;
  DialogData *dialogData;
  HWND theItem;
  int itemIndex, columns, n, m, i, j, k;
  Point loc, size, csz, lsz;
  LVAL listData, next, temp;
  BOOL vscroll, hscroll;
  DWORD flags;
  char *s;
  
  csz = AvCharSize(TRUE);
  csz.v += LIST_ITEM_PAD;
  loc = ListToPoint(slot_value(item, s_location));
  size = ListToPoint(slot_value(item, s_size));

  xlsave1(listData);
  listData = slot_value(item, s_list_data);
  lsz = ListItemDims(item);
  n = lsz.v;
  m = lsz.h;
  temp = slot_value(item, s_columns);
  if (! fixp(temp) || getfixnum(temp) < 1) columns = 1;
  else columns = (int) getfixnum(temp);

  hscroll = (columns < m) ? TRUE : FALSE;
  vscroll = (n * csz.v > size.v - ((hscroll) ? GetSystemMetrics(SM_CYHSCROLL) : 0)) ? TRUE : FALSE;
  flags = WS_CHILD | WS_VISIBLE | WS_BORDER | LBS_NOTIFY;
  if (hscroll) flags |= WS_HSCROLL;
  if (vscroll) flags |= WS_VSCROLL;
  if (m > 1) flags |= LBS_MULTICOLUMN;

  hd = GETDIALOGDATA(theDialog);
  dialogData = (DialogData *) GlobalLock(hd);
  check_lock(dialogData);

  itemIndex = (dialogData->count)++;

  theItem = CreateWindow("listbox", NULL, flags,
			 loc.h, loc.v, size.h, size.v,
			 (HWND) theDialog, itemIndex + ITEM_INDEX_BASE,
			 hInst, NULL);
  check_alloc(theItem);

  SendMessage(theItem, LB_SETCOLUMNWIDTH, LIST_COL_CHARS * csz.h, 0);
  if (arrayp(listData)) listData = arraydata(listData);
  else if (listp(listData)) listData = coerce_to_vector(listData);
  SendMessage(theItem, WM_SETREDRAW, FALSE, 0);
  for (j = 0; j < m; j++) {
    for (i = 0, k = j; i < n; i++, k += m) {
      next = getelement(listData, k);
      s = (stringp(next)) ? (char *) getstring(next) : "";
      strcpy(buf, s);
      truncateListEntry(buf);
      if (SendMessage(theItem, LB_ADDSTRING, 0, buf) < 0)
	xlfail("list allocation failed");
    }
  }
  SendMessage(theItem, WM_SETREDRAW, TRUE, 0);

  data = GETITEMDATA(dialogData);
  data[itemIndex].type = LIST_ITEM;
  data[itemIndex].itemNumber = itemIndex;
  data[itemIndex].itemHandle = theItem;
  data[itemIndex].object = item;
  GlobalUnlock(hd);
  xlpop();
}

void DialogListGetDefaultSize(LVAL item, int *pwidth, int *pheight)
{
  LVAL columns = slot_value(item, s_columns);
  LVAL data = slot_value(item, s_list_data);
  Point csz, sz;

  csz = AvCharSize(TRUE);
  sz.h = LIST_COL_CHARS * (int) getfixnum(columns);
  if (listp(data)) sz.v = llength(data);
  else if (simplevectorp(data)) sz.v = getsize(data);
  else if (matrixp(data))
    sz.v = (int) getfixnum(getelement(displacedarraydim(data), 0));

  csz.v += LIST_ITEM_PAD;
  sz.h *= csz.h;
  sz.v *= csz.v;
  if (sz.v > csz.v * MAX_LIST_ROWS) {
    sz.v = csz.v * MAX_LIST_ROWS;
    sz.h += GetSystemMetrics(SM_CXVSCROLL);
  }
  if (matrixp(data)
      && getfixnum(getelement(displacedarraydim(data), 1))
	 > getfixnum(columns))
    sz.v += GetSystemMetrics(SM_CYHSCROLL);
  if (pwidth != nil) *pwidth = sz.h;
  if (pheight != nil) *pheight = sz.v;
}

LVAL DialogListItemSelection(LVAL item, int set, LVAL index)
{
  LVAL result, listData;
  DialogItemData di;
  Point lsz, cell;
  int i;
  BOOL twodim;

  lsz = ListItemDims(item);

  if (FindItemObjectData(item, &di)) {
    if (set) {
      if (index == NIL) i = -1;
      else if (fixp(index)) i = (int) getfixnum(index);
      else if (consp(index)) {
	cell = ListToPoint(index);
	i = cell.h + cell.v * lsz.h;
      }
      else xlbadtype(index);
      if (i < 0 || i >= lsz.h * lsz.v) i = -1;
      SendMessage(di.itemHandle, LB_SETCURSEL, i, 0);
    }
    twodim = matrixp(slot_value(item, s_list_data)) ? TRUE : FALSE;
    i = (int) SendMessage(di.itemHandle, LB_GETCURSEL, 0, 0);
    if (i == LB_ERR || lsz.h <= 0) result = NIL;
    else if (twodim) {
      cell.h = i % lsz.h;
      cell.v = i / lsz.h;
      result = integer_list_2(cell.h, cell.v);
    }
    else result = cvfixnum((FIXTYPE) i);
  }
  else result = NIL;
  
  return(result);
}

void DialogListItemSetText(LVAL item, LVAL index, char *text)
{
  DialogItemData di;
  Point lsz, cell;
  int i, j;

  lsz = ListItemDims(item);
  if (FindItemObjectData(item, &di)) {
    if (fixp(index)) i = (int) getfixnum(index);
    else if (consp(index)) {
      cell = ListToPoint(index);
      i = cell.h + cell.v * lsz.h;
    }
    else xlbadtype(index);
    if (i >= 0 && i < lsz.h * lsz.v) {
      j = (int) SendMessage(di.itemHandle, LB_GETCURSEL, 0, 0);
      SendMessage(di.itemHandle, LB_DELETESTRING, i, 0);
      SendMessage(di.itemHandle, LB_INSERTSTRING, i, text);
      SendMessage(di.itemHandle, LB_SETCURSEL, j, 0);
    }
  }
}

LVAL DialogGetModalItem(LVAL dialog)
{
  IVIEW_WINDOW theDialog;
  LVAL item, oldModalItem;
  BOOL oldInModalDialog;
  MSG msg;

  theDialog = GETDIALOGADDRESS(dialog);
  if (theDialog == nil) xlfail("the dialog is not visible");

  oldModalItem = ModalItem;
  oldInModalDialog = InModalDialog;
  ModalItem = NIL;
  InModalDialog = TRUE;

  while (ModalItem == NIL && GetMessage(&msg, theDialog, 0, 0)) {
    if(! TranslateMDISysAccel(hWndClient, &msg)
       && ! TranslateAccelerator(hWndFrame, hAccel, &msg)) {
      TTYFlushOutput();
      TranslateMessage(&msg);
      DispatchMessage(&msg);
    }
  }

  item = ModalItem;
  ModalItem = oldModalItem;
  InModalDialog = oldInModalDialog;

  if (item == NIL) item = slot_value(dialog, s_default_button);
  return(item);
}

void DialogSetDefaultButton(LVAL dialog, LVAL item)
{
  HANDLE hd;
  DialogData *dd;
  DialogItemData di;
  IVIEW_WINDOW theDialog;

  if (item != NIL && ! button_item_p(item))
    xlerror("not a button item", item);

  set_slot_value(dialog, s_default_button, item);
    
  theDialog = GETDIALOGADDRESS(dialog);
  if (theDialog != nil) {
    hd = GETDIALOGDATA(theDialog);
    dd = (DialogData *) GlobalLock(hd);
    check_lock(dd);

    if (FindItemData(theDialog, ITEM_INDEX_BASE + dd->dflt, &di)) {
      dd->dflt = -1;
      SendMessage(di.itemHandle, BM_SETSTYLE, BS_PUSHBUTTON, TRUE);
    }
    if (item != NIL && FindItemObjectData(item, &di)) {
      dd->dflt = di.itemNumber;
      SendMessage(di.itemHandle, BM_SETSTYLE, BS_DEFPUSHBUTTON, TRUE);
      SetFocus(di.itemHandle);
    }
    GlobalUnlock(hd);
  }
}

#ifdef DODO
Add keyboard support:
subclass all items to handle tabs
handle return as hitting default item
#endif DODO
