; dwindow.lsp                 Gordon S. Novak Jr.              04 Sep 92

; Window types and interface functions for using X windows from AKCL.

; Copyright 1992, The University of Texas at Austin (UTA).  All rights
; reserved.  By using this software the USER indicates that he or she
; has read, understood and will comply with the following:
;
; -UTA hereby grants USER nonexclusive permission to use, copy and/or
; modify this software for internal, noncommercial, research purposes only.
; Any distribution, including commercial sale or license, of this software,
; copies of the software, its associated documentation and/or modifications
; of either is strictly prohibited without the prior consent of UTA.  Title
; to copyright to this software and its associated documentation shall at
; all times remain with UTA.  Appropriate copyright notice shall be placed
; on all software copies, and a complete copy of this notice shall be
; included in all copies of the associated documentation.  No right is
; granted to use in advertising, publicity or otherwise any trademark,
; service mark, or the name of UTA.  Software and/or its associated
; documentation identified as "confidential," if any, will be protected
; from unauthorized use/disclosure with the same degree of care USER
; regularly employs to safeguard its own such information.
;
; -This software and any associated documentation is provided "as is," and
; UTA MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED, INCLUDING
; THOSE OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, OR THAT
; USE OF THE SOFTWARE, MODIFICATIONS, OR ASSOCIATED DOCUMENTATION WILL
; NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS OR OTHER INTELLECTUAL
; PROPERTY RIGHTS OF A THIRD PARTY.  UTA, the University of Texas System,
; its Regents, officers, and employees shall not be liable under any
; circumstances for any direct, indirect, special, incidental, or
; consequential damages with respect to any claim by USER or any third
; party on account of or arising from the use, or inability to use, this
; software or its associated documentation, even if UTA has been advised
; of the possibility of those damages.
;
; -Submit software operation questions to: Gordon S. Novak Jr., Department
; of Computer Sciences, UT, Austin, TX 78712, novak@cs.utexas.edu .
;
; -Submit commercialization requests to: Office of the Executive Vice
; President and Provost, UT Austin, 201 Main Bldg., Austin, TX, 78712,
; ATTN: Technology Licensing Specialist.


; These functions use the convention that positive x is upwards,
; (0 0) is the lower-left corner of a window.

; derived from {DSK}<LISPFILES>DWINDOW.CL;1  1-Mar-89 13:16:20 
; Modified for AKCL/X using Hiep Huu Nguyen's interfaces from AKCL -> C -> X.
; Parts of Nguyen's file Xinit.lsp are included.

(in-package :user)

(defvar *window-menu* nil)
(defvar *mouse-x* nil)
(defvar *mouse-y* nil)
(defvar *mouse-window* nil)

(defvar *window-fonts* '((courier-bold-12
			  "*-*-courier-bold-r-*-*-12-*-*-*-*-*-iso8859-1")
			 (8x10 "8x10")
			 (9x15 "9x15")))

(glispglobals (*window-menu* menu)
	      (*mouse-x* integer)
	      (*mouse-y* integer)
	      (*mouse-window* window))

(defvar *window-display* nil)
(defvar *window-screen* nil)
(defvar *root-window*)
(defvar *black-pixel*) 
(defvar *white-pixel*)
(defvar *default-fg-color*)
(defvar *default-bg-color*)
(defvar *default-size-hints*)
(defvar *default-GC*)
(defvar *default-colormap*)
(defvar *window-event*)
(defvar *window-default-pos-x* 10)
(defvar *window-default-pos-y* 20)
(defvar *window-default-border* 1)
(defvar *window-default-font-name* 'courier-bold-12)
(defvar *window-default-cursor* 68)
(defvar *window-save-foreground*)
(defvar *window-save-function*)
(defvar *window-attributes*)
(defvar *window-attr*)
(defvar *root-return*   (int-array 1))
(defvar *child-return*  (int-array 1))
(defvar *root-x-return* (int-array 1))
(defvar *root-y-return* (int-array 1))
(defvar *win-x-return*  (int-array 1))
(defvar *win-y-return*  (int-array 1))
(defvar *mask-return*   (int-array 1))
(defvar *x-return*      (int-array 1))
(defvar *y-return*      (int-array 1))
(defvar *width-return*  (int-array 1))
(defvar *height-return* (int-array 1))
(defvar *depth-return*  (int-array 1))
(defvar *border-width-return* (int-array 1))
(defvar *text-width-return*   (int-array 1))
(defvar *GC-Values*)

(glispobjects

(drawable anything)

(menu (listobject (menu-window     window)
		  (flat            boolean)
		  (parent-window   drawable)
		  (parent-offset-x integer)
		  (parent-offset-y integer)
		  (picture-width   integer)
		  (picture-height  integer)
		  (title           string)
		  (permanent       boolean)
		  (menu-font       symbol)
		  (item-width      integer)
		  (item-height     integer)
		  (items           (listof symbol)) )
  prop ((menuw         (menu-window or (menu-init self)) result window)
	(title-present (title and ((length title) > 0)))
	(width         ((width menuw)))
	(height        ((height menuw)))
	(base-x        ((if flat then parent-offset-x else 0)))
	(base-y        ((if flat then parent-offset-y else 0)))
	(offset        menu-offset)
	(size          menu-size) )
  msg  ((init          menu-init)
	(init?         ((menu-window and (item-height > 0)) or (init self)))
	(create        menu-create result menu)
	(clear         menu-clear)
	(select        menu-select)
	(select!       menu-select!)
	(choose        menu-choose)
	(draw          menu-draw)
	(destroy       menu-destroy)
	(moveto-xy     menu-moveto-xy)
	(box-item      menu-box-item)
	(unbox-item    menu-unbox-item)
        (display-item  menu-display-item)
	(item-value    menu-item-value   open t)
	(item-position menu-item-position result vector)
	(find-item-width    menu-find-item-width)
	(find-item-height   menu-find-item-height)
	(adjust-offset menu-adjust-offset)
        (menu-x        (glambda (m x) ((base-x m) + x)))
        (menu-y        (glambda (m y) ((base-y m) + y)))  ) )

; picture menu: a drawn object with "hot buttons" at certain points.
; note: the first 10 data items of picmenu must be the same as in menu.
(picmenu (listobject (menu-window     window)
		     (flat            boolean)
		     (parent-window   drawable)
		     (parent-offset-x integer)
		     (parent-offset-y integer)
		     (picture-width   integer)
		     (picture-height  integer)
		     (title           string)
		     (permanent       boolean)
		     (spec (transparent picmenu-spec))
		     (boxflg          boolean) )
  prop ((menuw         (menu-window or (picmenu-init self)) result window)
	(title-present (title and ((length title) > 0)))
	(width         (picture-width))
	(height        (picture-height)) )
  msg  ((init          picmenu-init)
	(init?         ((menu-window and (picture-height > 0)) or (init self)))
	(create        picmenu-create result picmenu)
	(select        picmenu-select)
	(draw          picmenu-draw)
	(box-item      picmenu-box-item)
	(unbox-item    picmenu-unbox-item)
	(item-position picmenu-item-position result vector) )
 supers (menu) )

(picmenu-spec (listobject (drawing-width   integer)
			  (drawing-height  integer)
			  (buttons         (listof picmenu-button))
			  (dotflg          boolean)
			  (drawfn          anything)
			  (menu-font       symbol) ))

(picmenu-button (list (name          symbol)
		      (offset        vector)
		      (size          vector)
		      (highlightfn   anything)
		      (unhighlightfn anything))
  msg  ((containsxy?  picmenu-button-containsxy?)) )

(window (listobject (parent drawable)
		    (gcontext anything)
		    (drawable-height integer)
		    (drawable-width integer)
		    (label string)
		    (font anything) )
default ((self nil))
prop    ((width          (drawable-width))
	 (height         (drawable-height))
	 (left           window-left open t  result integer)
	 (right          (left + width))
	 (top-neg-y      window-top-neg-y open t result integer)
	 (leftmargin     (1))
	 (rightmargin    (width - 1))
         (yposition      window-yposition result integer open t)
	 (wfunction          window-wfunction        open t)
	 (foreground         window-foreground       open t)
	 (background         window-background       open t)  )
msg     ((force-output       window-force-output     open t)
	 (set-font           window-set-font)
	 (set-foreground     window-set-foreground   open t)
	 (set-background     window-set-background   open t)
	 (set-cursor         window-set-cursor       open t)
	 (set-erase          window-set-erase        open t)
	 (set-xor            window-set-xor          open t)
	 (set-invert         window-set-invert       open t)
	 (set-copy           window-set-copy         open t)
	 (set-line-width     window-set-line-width   open t)
	 (set-line-attr      window-set-line-attr    open t)
	 (std-line-attr      window-std-line-attr    open t)
	 (unset              window-unset            open t)
	 (reset              window-reset            open t)
	 (sync               window-sync             open t)
	 (geometry           window-geometry         open t)
	 (size               window-size)
	 (get-geometry       window-get-geometry     open t)
	 (reset-geometry     window-reset-geometry   open t)
	 (query-pointer      window-query-pointer    open t)
	 (wait-exposure      window-wait-exposure)
         (clear              window-clear            open t)
	 (mapw               window-map              open t)
	 (unmap              window-unmap            open t)
	 (open               window-open             open t)
	 (close              window-close            open t)
	 (destroy            window-destroy          open t)
	 (positive-y         window-positive-y       open t)
	 (drawline           window-draw-line        open t)
	 (draw-line          window-draw-line        open t)
	 (draw-line-xy       window-draw-line-xy     open t)
	 (draw-arrow-xy      window-draw-arrow-xy    )
	 (draw-arrow2-xy     window-draw-arrow2-xy   )
	 (draw-arrowhead-xy  window-draw-arrowhead-xy )
	 (draw-box           window-draw-box         open t)
	 (draw-box-xy        window-draw-box-xy)
	 (draw-box-corners   window-draw-box-corners open t)
	 (draw-rcbox-xy      window-draw-rcbox-xy)
	 (xor-box-xy         window-xor-box-xy       open t)
	 (draw-circle        window-draw-circle      open t)
	 (draw-circle-xy     window-draw-circle-xy   open t)
	 (draw-ellipse-xy    window-draw-ellipse-xy  open t)
	 (draw-arc-xy        window-draw-arc-xy      open t)
	 (invertarea         window-invertarea       open t)
	 (invert-area        window-invert-area      open t)
	 (invert-area-xy     window-invert-area-xy   open t)
	 (copy-area-xy       window-copy-area-xy     open t)
	 (printat            window-printat          open t)
	 (printat-xy         window-printat-xy       open t)
	 (prettyprintat      window-prettyprintat    open t)
	 (prettyprintat-xy   window-prettyprintat-xy open t)
         (string-width       window-string-width     open t)
	 (erase-area         window-erase-area       open t)
	 (erase-area-xy      window-erase-area-xy    open t)
         (moveto-xy          window-moveto-xy)
         (centeroffset       window-centeroffset     open t)
	 (track-mouse        window-track-mouse)
	 (get-point          window-get-point)
	 (get-click          window-get-click)
	 (get-line-position  window-get-line-position)
	 (get-box-position   window-get-box-position)
	 (get-box-size       window-get-box-size)
	 (get-region         window-get-region)
	 (adjust-box-side    window-adjust-box-side)
	 (get-mouse-position window-get-mouse-position) ))
 ) ; glispobjects

(glispconstants                      ; used by GEV
  (windowcharwidth     9 integer)
  (windowlineyspacing 17 integer)
)

; Make something into a string.
(defun stringify (x)
  (cond ((stringp x) x)
        ((symbolp x) (symbol-name x))
	(t (princ-to-string x))))

; This function initializes variables needed by most applications.
; It uses all defaults inherited from the root window, and screen. ; H. Nguyen
(defun window-Xinit ()
  (setq *window-display* (XOpenDisplay (get-c-string "")))
  (setq *window-screen* (XdefaultScreen *window-display*))
  (setq *root-window* (XRootWindow *window-display* *window-screen*))
  (setq *black-pixel* (XBlackPixel *window-display* *window-screen*))
  (setq *white-pixel* (XWhitePixel *window-display* *window-screen*))
  (setq *default-fg-color* *black-pixel*)
  (setq *default-bg-color* *white-pixel*)
  (setq *default-GC*  (XDefaultGC *window-display* *window-screen*))
  (setq *default-colormap* (XDefaultColormap *window-display*
					     *window-screen*))
  (setq *window-attributes* (make-XsetWindowAttributes))
  (set-XsetWindowAttributes-backing_store *window-attributes* WhenMapped)
  (set-XsetWindowAttributes-save_under *window-attributes* 1) ; True
  (setq *window-attr* (make-XWindowAttributes))
  (Xflush *window-display*)
  (setq *default-size-hints* (make-XsizeHints))
  (setq *window-event* (make-XEvent))
  (setq *GC-Values* (make-XGCValues)) )

(defun window-get-mouse-position ()
  (XQueryPointer *window-display* *root-window*
		 *root-return* *child-return* *root-x-return* *root-y-return*
		 *win-x-return* *win-y-return* *mask-return*)
  (setq *mouse-x* (int-pos *root-x-return* 0))
  (setq *mouse-y* (int-pos *root-y-return* 0))
  (setq *mouse-window* (int-pos *child-return* 0)) )

; 13 Aug 91; 14 Aug 91; 06 Sep 91; 12 Sep 91; 06 Dec 91; 01 May 92; 01 Sep 92
(setf (glfnresulttype 'window-create) 'window)
(gldefun window-create (width height &optional str parentw pos-x pos-y font)
  (let (w pw fg-color bg-color)
    (or *window-display* (window-Xinit))
    (setq fg-color *default-fg-color*)
    (setq bg-color *default-bg-color*)
    (unless pos-x (pos-x \:= *window-default-pos-x*))
    (unless pos-y (pos-y \:= *window-default-pos-y*))
    (w \:= (a window with
	      drawable-width  = width
	      drawable-height = height
              label           = (if str (stringify str) else " ") ))
    (pw \:= (or parentw *root-window*))
    (window-get-geometry-b pw)
    ((parent w) \:=
       (XCreateSimpleWindow *window-display* pw
			    pos-x
			    ((int-pos *height-return* 0) - pos-y - height)
			    width height
			    *window-default-border*
			    fg-color bg-color))
    (set-Xsizehints-x      *default-size-hints* pos-x)
    (set-xsizehints-y      *default-size-hints* pos-y)
    (set-xsizehints-width  *default-size-hints* (width w))
    (set-xsizehints-height *default-size-hints* (height w))
    (set-xsizehints-flags  *default-size-hints* (+ Psize Pposition))
    (XsetStandardProperties  *window-display* (parent w)
			     (get-c-string (label w))
			     (get-c-string (label w))  ; icon name
			     none null null
			     *default-size-hints*)
    ((gcontext w) \:= (XCreateGC *window-display* (parent w) 0 null))
    (set-foreground w fg-color)
    (set-background w bg-color)
    (set-font w (or font *window-default-font-name*))
    (set-cursor w *window-default-cursor*)
    (set-line-width w 1)
    (XChangeWindowAttributes *window-display* (parent w)
			     (+ CWSaveUnder CWBackingStore)
			     *window-attributes*)
    (Xselectinput *window-display* (parent w)	
		  (+ leavewindowmask buttonpressmask buttonreleasemask
		     pointermotionmask exposuremask))
    (open w)
    w  ))

; 06 Aug 91
; Set the font for a window to the one specified by fontsymbol.
; derived from Nguyen's my-load-font.
(gldefun window-set-font (w\:window fontsymbol\:symbol)
  (let (fontstring font-info (display *window-display*))
    (fontstring \:= (or (cadr (assoc fontsymbol *window-fonts*))
			(stringify fontsymbol)))
    (font-info \:= (XloadQueryFont display (get-c-string fontstring)))
    (if (eql 0 font-info)
	then (format t "~%can't open font ~a ~a~%" fontsymbol fontstring)
	else (XsetFont display (gcontext w) (Xfontstruct-fid font-info))
	     ((font w) \:= font-info) ) ))

; 15 Oct 91
(defun window-font-info (fontsymbol)
  (XloadQueryFont *window-display*
		  (get-c-string (or (cadr (assoc fontsymbol *window-fonts*))
				    (stringify fontsymbol)))))


; Functions to allow access to window properties from plain Lisp
(gldefun window-gcontext        (w\:window) (gcontext w))
(gldefun window-parent          (w\:window) (parent w))
(gldefun window-drawable-height (w\:window) (drawable-height w))
(gldefun window-drawable-width  (w\:window) (drawable-width w))
(gldefun window-label           (w\:window) (label w))
(gldefun window-font            (w\:window) (font w))

; 07 Aug 91; 14 Aug 91
(gldefun window-foreground (w\:window)
  (XGetGCValues *window-display* (gcontext w) GCForeground *GC-Values*)
  (XGCValues-foreground  *GC-Values*) )

(gldefun window-set-foreground (w\:window fg-color\:integer)
  (XsetForeground *window-display* (gcontext w) fg-color))

(gldefun window-background (w\:window)
  (XGetGCValues *window-display* (gcontext w) GCBackground *GC-Values*)
  (XGCValues-Background  *GC-Values*) )

(gldefun window-set-background (w\:window bg-color\:integer)
  (XsetBackground *window-display* (gcontext w) bg-color))

; 08 Aug 91
(gldefun window-wfunction (w\:window)
  (XGetGCValues *window-display* (gcontext w) GCFunction *GC-Values*)
  (XGCValues-function *GC-Values*) )

; 08 Aug 91
; Get the geometry parameters of a window into global variables
(gldefun window-get-geometry (w\:window) (window-get-geometry-b (parent w)))

; 06 Dec 91
; Set cursor to a selected cursor number
(gldefun window-set-cursor (w\:window n\:integer)
  (let (c)
    (c \:= (XCreateFontCursor *window-display* n) )
    (XDefineCursor *window-display* (parent w) c) ))

(defun window-get-geometry-b (w)
  (XGetGeometry *window-display* w
		*root-return* *x-return* *y-return* *width-return* 
		*height-return* *border-width-return* *depth-return*) )

; 15 Aug 91
; clear event queue of previous motion events
(gldefun window-sync (w\:window)
  (Xsync *window-display* 1) )

; 03 Oct 91
(defun window-screen-height ()
  (window-get-geometry-b *root-window*)
  (int-pos *height-return* 0) )

; 08 Aug 91; 12 Sep 91; 28 Oct 91
; Make a list of window geometry, (x y width height border-width).
(gldefun window-geometry (w\:window)
  (let (sh)
    (sh \:= (window-screen-height))
    (get-geometry w)
  ((drawable-width w) \:= (int-pos *width-return* 0))
  ((drawable-height w) \:= (int-pos *height-return* 0))
    (list (int-pos *x-return* 0)
	  (sh - (int-pos *y-return* 0) - (int-pos *height-return* 0))
	  (int-pos *width-return* 0) 
	  (int-pos *height-return* 0)
	  (int-pos *border-width-return* 0)) ))

; 27 Nov 91
(gldefun window-size (w\:window) (result vector)
  (get-geometry w)
  (list ((drawable-width w) \:= (int-pos *width-return* 0))
	((drawable-height w) \:= (int-pos *height-return* 0)) ) )

(gldefun window-left (w\:window)
  (get-geometry w)
  (int-pos *x-return* 0))

; Get top of window in X (y increasing downwards) coordinates.
(gldefun window-top-neg-y (w\:window)
  (get-geometry w)
  (int-pos *y-return* 0))

; 08 Aug 91
; Reset the local geometry parameters of a window from its X values.
; Needed, for example, if the user resizes the window by mouse command.
(gldefun window-reset-geometry (w\:window)
  (get-geometry w)
  ((drawable-width w) \:= (int-pos *width-return* 0))
  ((drawable-height w) \:= (int-pos *height-return* 0)) )

(gldefun window-force-output (&optional w\:window) (Xflush *window-display*))

(gldefun window-query-pointer (w\:window) (window-query-pointer-b (parent w)) )

(defun window-query-pointer-b (w)
  (XQueryPointer *window-display* w
		 *root-return* *child-return* *root-x-return* *root-y-return*
		 *win-x-return* *win-y-return* *mask-return*) )

(gldefun window-positive-y (w\:\window y\:integer) ((height w) - y))

; 08 Aug 91
; Set parameters of a window for drawing by XOR, saving old values.
(gldefun window-set-xor (w\:window)
  (let ((gc (gcontext w)) )
    (setq *window-save-function*   (wfunction w))
    (XsetFunction   *window-display* gc GXxor)
    (setq *window-save-foreground* (foreground w))
    (XsetForeground *window-display* gc
		    (logxor *window-save-foreground* (background w))) ))

; 08 Aug 91
; Reset parameters of a window after change, using saved values.
(gldefun window-unset (w\:window)
  (let ((gc (gcontext w)) )
    (XsetFunction   *window-display* gc *window-save-function*)
    (XsetForeground *window-display* gc *window-save-foreground*) ))

; 04 Sep 91
; Reset parameters of a window, using default values.
(gldefun window-reset (w\:window)
  (let ((gc (gcontext w)) )
    (XsetFunction   *window-display* gc GXcopy)
    (XsetForeground *window-display* gc *default-fg-color*)
    (XsetBackground *window-display* gc *default-bg-color*)  ))

; 09 Aug 91; 03 Sep 92
; Set parameters of a window for erasing, saving old values.
(gldefun window-set-erase (w\:window)
  (let ((gc (gcontext w)) )
    (setq *window-save-function*   (wfunction w))
    (XsetFunction   *window-display* gc GXcopy)
    (setq *window-save-foreground* (foreground w))
    (XsetForeground *window-display* gc (background w)) ))

(gldefun window-set-copy (w\:window)
  (let ((gc (gcontext w)) )
    (setq *window-save-function*   (wfunction w))
    (XsetFunction   *window-display* gc GXcopy)
    (setq *window-save-foreground* (foreground w)) ))

; 12 Aug 91
; Set parameters of a window for inversion, saving old values.
(gldefun window-set-invert (w\:window)
  (let ((gc (gcontext w)) )
    (setq *window-save-function*   (wfunction w))
    (XsetFunction   *window-display* gc GXxor)
    (setq *window-save-foreground* (foreground w))
    (XsetForeground *window-display* gc
		    (logxor *window-save-foreground* (background w))) ))

; 13 Aug 91
(gldefun window-set-line-width (w\:window width\:integer)
  (set-line-attr w width nil nil nil))

; 13 Aug 91; 12 Sep 91
(gldefun window-set-line-attr
 (w\:window width &optional line-style cap-style join-style)
  (XSetLineAttributes *window-display* (gcontext w)
		      (or width 1)
		      (or line-style LineSolid)
		      (or cap-style CapButt)
		      (or join-style JoinMiter) ) )

; 13 Aug 91
; Set standard line attributes
(gldefun window-std-line-attr (w\:window)
  (XSetLineAttributes *window-display* (gcontext w)
		      1 LineSolid CapButt JoinMiter) )

; 06 Aug 91; 08 Aug 91; 12 Sep 91
(gldefun window-draw-line (w\:window from\:vector to\:vector
				     &optional linewidth)
  (window-draw-line-xy w (x from) (y from) (x to) (y to) linewidth) )

; 19 Dec 90; 07 Aug 91; 08 Aug 91; 09 Aug 91; 13 Aug 91; 12 Sep 91
(gldefun window-draw-line-xy (w\:window fromx\:integer fromy\:integer
					tox\:integer   toy\:integer
					&optional linewidth
					operation\:atom)
  (let ( (qqwheight (drawable-height w)) )
    (if (linewidth and (linewidth <> 1)) (set-line-width w linewidth))
    (case operation
      (xor (set-xor w))
      (erase (set-erase w)))
    (XDrawLine *window-display*  (parent w) (gcontext w)
	       fromx (- qqwheight fromy) tox (- qqwheight toy) )
    (case operation
      ((xor erase) (unset w) ))
    (if (linewidth and (linewidth <> 1)) (set-line-width w 1)) ))

; 09 Oct 91
(defun window-draw-arrowhead-xy (w x1 y1 x2 y2 &optional (linewidth 1) size)
  (let (th theta ysth ycth (y2dela 0) (y2delb 0) (x2dela 0) (x2delb 0))
    (or size (setq size (+ 20 (* linewidth 5))))
    (setq th (atan (- y2 y1) (- x2 x1)))
    (setq theta (* th (/ 180.0 pi)))
    (setq ysth (round (* (1+ size) (sin th))))
    (setq ycth (round (* (1+ size) (cos th))))
    (if (and (eql y1 y2) (evenp linewidth)) ; correct for even-size lines
	(if (> x2 x1) (setq y2delb 1) (setq y2dela 1)))
    (if (and (eql x1 x2) (evenp linewidth)) ; correct for even-size lines
	(if (> y2 y1) (setq x2delb 1) (setq x2dela 1)))
    (window-draw-arc-xy w (- (- x2 ysth) x2dela)
			  (+ (+ y2 ycth) y2dela) size size
			  (+ 240 theta) 30 linewidth)
    (window-draw-arc-xy w (- (+ x2 ysth) x2delb)
			  (+ (- y2 ycth) y2delb) size size
			  (+ 90 theta) 30 linewidth)   ))

(defun window-draw-arrow-xy (w x1 y1 x2 y2
			       &optional (linewidth 1) size)
  (window-draw-line-xy w x1 y1 x2 y2 linewidth)
  (window-draw-arrowhead-xy w x1 y1 x2 y2 linewidth size) )

(defun window-draw-arrow2-xy (w x1 y1 x2 y2
				&optional (linewidth 1) size)
  (window-draw-line-xy w x1 y1 x2 y2 linewidth)
  (window-draw-arrowhead-xy w x1 y1 x2 y2 linewidth size)
  (window-draw-arrowhead-xy w x2 y2 x1 y1 linewidth size) )

; 08 Aug 91; 14 Aug 91; 12 Sep 91
(gldefun window-draw-box
	 (w\:window offset\:vector size\:vector &optional linewidth)
  (window-draw-box-xy w (x offset) (y offset) (x size) (y size) linewidth) )

; 08 Aug 91; 12 Sep 91; 11 Dec 91; 01 Sep 92; 02 Sep 92
; New version avoids XDrawRectangle, which messes up when used with XOR.
; was  (XDrawRectangle *window-display* (parent w) (gcontext w)
;		       offsetx (- qqwheight (offsety + sizey)) sizex sizey)
(gldefun window-draw-box-xy
	 (w\:window offsetx\:integer offsety\:integer
		    sizex\:integer   sizey\:integer
		    &optional linewidth)
  (let ((qqwheight (drawable-height w)) miny lw lw2 lw2b (pw (parent w))
	(gc  (gcontext w)))
    (if (linewidth and (linewidth <> 1)) (set-line-width w linewidth))
    (lw \:= (or linewidth 1))
    (lw2 \:= lw / 2)
    (lw2b \:= (lw + 1) / 2)
    (miny \:= offsety - lw2b)
    (XDrawLine *window-display*  pw gc offsetx (- qqwheight miny)
	       offsetx (- qqwheight (miny + sizey + lw)))
    (XDrawLine *window-display*  pw gc (offsetx + sizex) (- qqwheight miny)
	       (offsetx + sizex) (- qqwheight (miny + sizey + lw)))
    (XDrawLine *window-display*  pw gc (offsetx + lw2b) (- qqwheight offsety)
	       (offsetx + sizex - lw2) (- qqwheight offsety) )
    (XDrawLine *window-display*  pw gc
	       (offsetx + lw2b) (- qqwheight (offsety + sizey))
	       (offsetx + sizex - lw2) (- qqwheight (offsety + sizey)) )
    (if (linewidth and (linewidth <> 1)) (set-line-width w 1)) ))

; 26 Nov 91
(gldefun window-xor-box-xy
	 (w\:window offsetx\:integer offsety\:integer
		    sizex\:integer   sizey\:integer
		    &optional linewidth)
  (window-set-xor w)
  (window-draw-box-xy w offsetx offsety sizex sizey linewidth)
  (window-unset w))

; 15 Aug 91; 12 Sep 91
; Draw a box whose corners are specified
(gldefun window-draw-box-corners (w\:window xa\:integer ya\:integer
				  xb\:integer yb\:integer
				  &optional lw)
  (draw-box-xy w (min xa xb) (min ya yb) (abs (- xa xb)) (abs (- ya yb)) lw) )

; 13 Sep 91
; Draw a box with round corners
(gldefun window-draw-rcbox-xy (w\:window x\:integer y\:integer width\:integer
					 height\:integer radius\:integer
					 &optional linewidth)
  (let (x1 x2 y1 y2 r)
    (r \:= (max 0 (min radius (truncate (abs width) 2)
			           (truncate (abs height) 2))))
    (x1 \:= x + r)
    (x2 \:= x + width - r)
    (y1 \:= y + r)
    (y2 \:= y + height - r)
    (draw-line-xy w x1 y x2 y linewidth)
    (draw-line-xy w (x + width) y1 (x + width) y2 linewidth)
    (draw-line-xy w x1 (y + height) x2 (y + height) linewidth)
    (draw-line-xy w x y1 x y2 linewidth)
    (draw-arc-xy w x1 y1 r r 180 90 linewidth)
    (draw-arc-xy w x2 y1 r r 270 90 linewidth)
    (draw-arc-xy w x2 y2 r r   0 90 linewidth)
    (draw-arc-xy w x1 y2 r r  90 90 linewidth) ))

; 13 Aug 91; 15 Aug 91; 12 Sep 91
(gldefun window-draw-arc-xy (w\:window x\:integer y\:integer
			     radiusx\:integer radiusy\:integer
			     anglea\:number angleb\:number
			     &optional linewidth)
  (if (linewidth and (linewidth <> 1)) (set-line-width w linewidth))
  (XdrawArc *window-display* (parent w) (gcontext w)
	    (x - radiusx) (positive-y w (y + radiusy))
	    (radiusx * 2) (radiusy * 2)
	    (truncate (* anglea 64)) (truncate (* angleb 64)))
  (if (linewidth and (linewidth <> 1)) (set-line-width w 1)) )

; 08 Aug 91; 12 Sep 91
(gldefun window-draw-circle-xy (w\:window x\:integer y\:integer
					  radius\:integer
					  &optional linewidth)
  (if (linewidth and (linewidth <> 1)) (set-line-width w linewidth))
  (XdrawArc *window-display* (parent w) (gcontext w)
	    (x - radius) (positive-y w (y + radius))
	    (radius * 2) (radius * 2) 0 (* 360 64))
  (if (linewidth and (linewidth <> 1)) (set-line-width w 1)) )

; 06 Aug 91; 14 Aug 91; 12 Sep 91
(gldefun window-draw-circle (w\:window pos\:vector radius\:integer
				       &optional linewidth)
  (window-draw-circle-xy w (x pos) (y pos) radius linewidth) )

; 08 Aug 91; 09 Sep 91
(gldefun window-erase-area (w\:window offset\:vector size\:vector)
  (window-erase-area-xy w (x offset) (y offset) (x size) (y size)))

; 09 Sep 91; 11 Dec 91
(gldefun window-erase-area-xy (w\:window xoff\:integer yoff\:integer
				         xsize\:integer ysize\:integer)
  (XClearArea *window-display* (parent w)
	      xoff (positive-y w (yoff + ysize - 1))
	      xsize ysize
	      0 ))     ;   exposures

; 15 Aug 91; 12 Sep 91
(gldefun window-draw-ellipse-xy (w\:window x\:integer y\:integer
			         rx\:integer ry\:integer &optional lw)
  (draw-arc-xy w x y rx ry 0 360 lw))

; 09 Aug 91
(gldefun window-copy-area-xy (w\:window fromx fromy\:integer
					tox toy\:integer width height)
  (let ((qqwheight (drawable-height w)))
    (set-copy w)
    (XCopyArea *window-display* (parent w) (parent w) (gcontext w)
	       fromx (- qqwheight (+ fromy height))
	       width height
	       tox (- qqwheight (+ toy height)))
    (unset w) ))

; 07 Dec 90; 09 Aug 91; 12 Sep 91
(gldefun window-invertarea (w\:window area\:region)
  (window-invert-area-xy w (left area) (bottom area)
			   (width area) (height area)))

; 07 Dec 90; 09 Aug 91; 12 Sep 91
(gldefun window-invert-area (w\:window offset\:vector size\:vector)
  (window-invert-area-xy w (x offset) (y offset) (x size) (y size)) )

; 12 Aug 91; 15 Aug 91; 13 Dec 91
(gldefun window-invert-area-xy (w\:window left bottom\:integer width height)
  (set-invert w)
  (XFillRectangle *window-display* (parent w) (gcontext w)
	          left (- (drawable-height w) (bottom + height - 1))
		  width height)
  (unset w) )

; 05 Dec 90; 15 Aug 91
(gldefun window-prettyprintat (w\:window s\:string pos\:vector)
  (printat w s pos) )

(gldefun window-prettyprintat-xy (w\:window s\:string x\:integer y\:integer)
  (printat-xy w s x y))

; 06 Aug 91; 08 Aug 91; 15 Aug 91
(gldefun window-printat (w\:window s\:string pos\:vector)
  (printat-xy w s (x pos) (y pos)) )

; 06 Aug 91; 08 Aug 91; 12 Aug 91
(gldefun window-printat-xy (w\:window s\:string x\:integer y\:integer)
  (let ( (sstr (stringify s)) )
    (XDrawImageString *window-display* (parent w) (gcontext w)
		      x (- (drawable-height w) y)
		      (get-c-string sstr) (length sstr)) ))

; 08 Aug 91
; Find the width of a string when printed in a given window
(gldefun window-string-width  (w\:window s\:string)
  (let ((sstr (stringify s)))
    (XTextWidth (font w) (get-c-string sstr) (length sstr)) ))

; 15 Oct 91
(gldefun window-font-string-width (font s\:string)
  (let ((sstr (stringify s)))
    (XTextWidth font (get-c-string sstr) (length sstr)) ))

(gldefun window-yposition (w\:window)
  (window-get-mouse-position)
  (positive-y w (- *mouse-y* (top-neg-y w))) )

(gldefun window-centeroffset (w\:window v\:vector)
  (a vector with x = (truncate ((width w)  - (x v)) 2)
                 y = (truncate ((height w) - (y v)) 2)))

; 18 Aug 89; 15 Aug 91
; Command to a window display manager 
(gldefun dowindowcom (w\:window)
  (let (comm)
    (comm \:= (select (window-menu)) )
  (case comm
	(close  (close w))
	(paint  (paint w))
	(clear  (clear w))
	(move   (move w))
	(t (when comm
		 (princ "This command not implemented.") (terpri))) ) ))

(gldefun window-menu ()
  (or *window-menu*
      (setq *window-menu*
	(a menu with items = '(close paint clear move)))) )

; 06 Dec 90
(gldefun window-close (w\:window)
    (unmap w)
    (force-output w))

(gldefun window-unmap (w\:window)
  (XUnMapWindow *window-display* (parent w)) )

; 06 Aug 91; 22 Aug 91
(gldefun window-open (w\:window)
  (mapw w)
  (force-output w)
  (wait-exposure w) )

(gldefun window-map (w\:window)
  (XMapWindow *window-display* (parent w))  )

; 08 Aug 91; 02 Sep 91
(gldefun window-destroy (w\:window)
  (XDestroyWindow *window-display* (parent w))
  (force-output w)
  ((parent w) \:= nil)
  (XFreeGC *window-display* (gcontext w))
  ((gcontext w) \:= nil) )

; 09 Sep 91
; Wait 3 seconds, then destroy the window where the mouse is.  Use with care.
(defun window-destroy-selected-window ()
  (prog (ww child)
    (sleep 3)
    (setq ww *root-window*)
 lp (window-query-pointer-b ww)
    (setq child (int-pos *child-return* 0))
    (if (> child 0)
	(progn (setq ww child) (go lp)))
    (if (/= ww *root-window*)
	(progn (XDestroyWindow *window-display* ww)
	       (Xflush *window-display*))) ))

; 07 Aug 91
(gldefun window-clear (w\:window)
  (XClearWindow *window-display* (parent w))
  (force-output w) )

; 08 Aug 91
(gldefun window-moveto-xy (w\:window x\:integer y\:integer)
  (XMoveWindow *window-display* (parent w) x (- (window-screen-height) y)) )

; 15 Aug 91; 05 Sep 91
; Paint in window with mouse: Left paints, Middle erases, Right quits.
(defun window-paint (window)
  (let (state)
    (window-track-mouse window
      #'(lambda (x y code)
          (if (= code 1) (if (= state 1) (setq state 0) (setq state 1))
	      (if (= code 2) (if (= state 2) (setq state 0) (setq state 2))))
          (if (= state 1) (window-draw-line-xy window x y x y 1 'paint)
	    (if (= state 2) (window-draw-line-xy window x y x y 1 'erase)))
	(= code 3))  ) ))

; 15 Aug 91
; Move a window.
(gldefun window-move (w\:w)
  (window-get-mouse-position)
  (XMoveWindow *window-display* (parent w)
	       *mouse-x* (- (window-screen-height) *mouse-y*)) )

; 13 Aug 91; 22 Aug 91; 27 Aug 91; 14 Oct 91
; Track the mouse within a window, calling function fn with args (x y event).
; event is 0 = no button, 1 = left button, 2 = middle, 3 = right button.
; Tracking continues until fn returns non-nil; result is that value.
; Partly adapted from Hiep Nguyen's code.
(defun window-track-mouse (w fn &optional outflg)
  (let (win h)
    (setq win (window-parent w))
    (setq h   (window-drawable-height w))
    (Xsync *window-display* 1) ; clear event queue of previous motion events
    (Xselectinput *window-display* win (+ ButtonPressMask PointerMotionMask))
 ;; Event processing loop: stop when function returns non-nil.
  (do ((res nil)) (res res)
    (XNextEvent *window-display* *window-event*)
    (let ((type (XAnyEvent-type *window-event*))
	  (eventwindow (XAnyEvent-window *window-event*)))
      (when (or (and (eql eventwindow win)
		     (or (eql type MotionNotify)
			 (eql type ButtonPress)))
		(and outflg (eql type ButtonPress)))
	(let ((x (XMotionEvent-x *window-event*))
	      (y (XMotionEvent-y *window-event*))
	      (code (if (eql type ButtonPress)
			(XButtonEvent-button *window-event*)
			0)))
	  (setq res (if (eql eventwindow win)
			(funcall fn x (- h y) code)
			(funcall fn -1 -1 code))) ) ) ) ) ))

; 22 Aug 91; 23 Aug 91; 27 Aug 91; 04 Sep 92
; Wait for a window to become exposed, but not more than 1 second.
(defun window-wait-exposure (w)
  (prog (win start-time max-time eventwindow type)
    (setq win (window-parent w))
    (XGetWindowAttributes *window-display* win *window-attr*)
    (unless (eql (XWindowAttributes-map_state *window-attr*) ISUnmapped)
      (return t))
    (setq start-time (get-internal-real-time))
    (setq max-time internal-time-units-per-second)
    (Xselectinput *window-display* win (+ ExposureMask))
    ; Event processing loop: stop when exposure is seen or time out
 lp (cond ((> (XPending *window-display*) 0)
	    (XNextEvent *window-display* *window-event*)
	    (setq type (XAnyEvent-type *window-event*))
	    (setq eventwindow (XAnyEvent-window *window-event*))
	    (if (and (eql eventwindow win)
		     (eql type Expose))
		(return t)))
	  ((> (- (get-internal-real-time) start-time)
	      max-time)
	    (return nil)) )
    (go lp) ))

; 14 Dec 90; 17 Dec 90; 13 Aug 91; 20 Aug 91; 30 Aug 91; 09 Sep 91; 11 Sep 91
; 15 Oct 91; 16 Oct 91; 10 Feb 92
; Initialize a menu
(gldefun menu-init (m\:menu)
  (let (maxwidth maxheight nitems)
    (or *window-display* (window-Xinit))    ; init windows if necessary
    (or (menu-font m) ((menu-font m) \:= '9x15))
    (maxwidth \:= (find-item-width m (title m)))
    (maxheight \:=  13)                      ; ***** fix for font
    (nitems \:= (if (title-present m) then 1 else 0))
    (for item in (items m) do
      (nitems _+ 1)
      (maxwidth \:= (max maxwidth (find-item-width m item)))
      (maxheight \:= (max maxheight (find-item-height m item))) )
    ((item-width m) \:= maxwidth + 6)
    ((picture-width m) \:= (item-width m) + 1)
    ((item-height m) \:=  maxheight + 2)
    ((picture-height m) \:= ((item-height m) * nitems) + 2)
    (adjust-offset m)
    (if ~ (flat m)
	((menu-window m) \:= (window-create (picture-width m)
					    (picture-height m)
					    ((title m) or "")
					    (parent-window m)
					    (parent-offset-x m)
					    (parent-offset-y m)
					    (menu-font m) )) ) ))

; 06 Sep 91; 09 Sep 91; 10 Sep 91
; Adjust a menu's offset position if necessary to keep it in parent window.
(gldefun menu-adjust-offset (m\:menu)
  (let (xbase ybase wbase hbase xoff yoff wgm width height)
    (width \:= (picture-width m))
    (height \:= (picture-height m))
    (if ~ (parent-window m)
	then (window-get-mouse-position)  ; put it where the mouse is
	     (wgm \:= t)                  ; set flag that we got mouse position
	     ((parent-window m) \:= *mouse-window*))
    (window-get-geometry-b (parent-window m))
    (setq xbase (int-pos *x-return* 0))
    (setq ybase (int-pos *y-return* 0))
    (setq wbase (int-pos *width-return* 0))
    (setq hbase (int-pos *height-return* 0))
    (if (~ (parent-offset-x m) or (parent-offset-x m) = 0)
	then (or wgm (window-get-mouse-position))
             (xoff \:= ((*mouse-x* - xbase) - (width  / 2) - 4))
             (yoff \:= ((hbase - (*mouse-y* - ybase)) - (height / 2)))
	else (xoff \:= (parent-offset-x m))
	     (yoff \:= (parent-offset-y m)))
    ((parent-offset-x m) \:= (max 0 (min xoff (wbase - width))))
    ((parent-offset-y m) \:= (max 0 (min yoff (hbase - height)))) ))

; 07 Dec 90; 14 Dec 90; 12 Aug 91; 22 Aug 91; 09 Sep 91; 10 Sep 91; 28 Jan 92;
; 10 Feb 92
(gldefun menu-draw (m\:menu)
  (let (mw xzero yzero bottom)
    (xzero \:= (menu-x m 0))
    (yzero \:= (menu-y m 0))
    (mw \:= (menu-window m))
    (open mw)
    (clear m)
    (if (flat m) (draw-box-xy mw (xzero - 1) yzero ((picture-width m) + 2)
			      ((picture-height m) + 1) 1))
    (bottom \:= (yzero + (picture-height m) + 3))
    (if (title-present m)
	then (bottom _- (item-height m))
             (printat-xy mw (stringify (title m)) (+ xzero 3) bottom)
             (invert-area-xy mw xzero (bottom - 2)
			        ((picture-width m) + 1) (item-height m)))
    (for item in (items m) do
	 (bottom _- (item-height m))
	 (display-item m item (+ xzero 3) bottom) )
    (force-output mw) ))

(gldefun menu-item-value (self item)
  (if (consp item) then (cdr item) else item))

; 06 Sep 91; 11 Sep 91; 15 Oct 91; 16 Oct 91; 23 Oct 91
(gldefun menu-find-item-width (self\:menu item)
  (let (tmp\:vector)
    (if (and (consp item)
	     (symbolp (car item))
	     (fboundp (car item)))
        then (or (and (tmp \:= (get (car item) 'display-size))
		      (x tmp))
		 40)
        else (window-font-string-width
	      (or (and (flat self)
		       (menu-window self)
		       (font (menu-window self)))
		  (window-font-info (menu-font self)))
	      (stringify (if (consp item) then (car item) else item)))) ))


; 09 Sep 91; 10 Sep 91; 11 Sep 91
(gldefun menu-find-item-height (self\:menu item)     ; ***** fix for font
  (let (tmp\:vector)
    (if (and (consp item)
	     (symbolp (car item))
	     (tmp \:= (get (car item) 'display-size)))
	then ((y tmp) + 3)
        else 15) ))

; 09 Sep 91; 10 Sep 91; 10 Feb 92
(gldefun menu-clear (m\:menu)
  (if (flat m)
      (erase-area-xy (menu-window m) ((base-x m) - 1) ((base-y m) - 1)
		     ((picture-width m) + 3) ((picture-height m) + 3))
      else (clear (menu-window m))) )

; 06 Sep 91; 04 Dec 91
(gldefun menu-display-item (self\:menu item x y)
  (let ((mw (menu-window self)))
    (if (consp item)
        then (if (and (symbolp (car item))
		      (fboundp (car item)))
 		 then (funcall (car item) mw x y)
	         elseif (or (stringp (car item)) (symbolp (car item))
			    (numberp (car item)))
	         then (printat-xy mw (car item) x y)
		 else (printat-xy mw (stringify item) x y))
        else (printat-xy mw (stringify item) x y)) ))

; 07 Dec 90; 18 Dec 90; 15 Aug 91; 27 Aug 91; 06 Sep 91; 10 Sep 91
(gldefun menu-choose (m\:menu)
  (let (mw current-item-n newn itemh itms nitems val maxx inside xzero yzero)
    (init? m)
    (mw \:= (menu-window m))
    (draw m)
    (xzero \:= (menu-x m 0))
    (yzero \:= (menu-y m 0))
    (maxx \:= (+ xzero (picture-width m)))
    (itemh \:= (item-height m))
    (itms \:= (items m))
    (nitems \:= (length itms))
    (track-mouse mw
      #'(lambda (x y code)
          (setq newn (1- (- nitems (truncate (- y (+ yzero 3)) itemh))))
	  (if ((x >= xzero) and (x <= maxx)
	       and (newn >= 0) and (newn < nitems))
	      then
	      (if current-item-n
		  then (if (/= newn current-item-n)
			   then (unbox-item m current-item-n)
			        (box-item m newn)
				(current-item-n \:= newn))
		  else (inside \:= t)
		       (box-item m newn)
		       (current-item-n \:= newn))
	      (if (and current-item-n (> code 0))
		  (unbox-item m current-item-n)
		  (val \:= current-item-n))
	      else (if current-item-n
		       then (unbox-item m current-item-n)
		            (current-item-n \:= nil))
	           (if (> code 0) or
		       (inside and ((x < xzero) or (x > maxx)
				    or (y < yzero)
				    or (y > (yzero + (picture-height m)))))
		       then (val \:= 'no-selecction-val))))
      t)
    (if (val <> 'no-selecction-val) (item-value m (nth val itms)) ) ))

; 07 Dec 90; 12 Aug 91; 10 Sep 91
(gldefun menu-box-item (m\:menu item\:integer)
  (let (itemh nitems)
    (itemh \:= (item-height m))
    (nitems \:= (length (items m)))
    (draw-box-xy (menu-window m) (menu-x m 1)
		                 (menu-y m ((nitems - item - 1) * itemh + 2))
				 ((item-width m) - 2) itemh 1) ))

; 07 Dec 90; 12 Aug 91; 14 Aug 91; 15 Aug 91
(gldefun menu-unbox-item (m\:menu item\:integer)
  (let ((mw (menuw m)))
    (set-erase mw)
    (box-item m item)
    (unset mw) ))

; 11 Sep 91
(gldefun menu-item-position (m\:menu itemname\:symbol &optional place\:symbol)
  (let ((n 0) found itms item (xsize (item-width m)) (ysize (item-height m)))
    (itms \:= (items m))
    (while itms and ~ found do
	   (n _+ 1)
	   (item -_ itms)
	   (if (or (and (symbolp item)
			(eq item itemname))
		   (and (consp item)
			(eq (cdr item) itemname)))
	       (found \:= t)))
    (if found (a vector with
		 x = ((menu-x m 0) +
		      (case place
			((center top bottom) (xsize / 2))
			(left 0)
			(right xsize)
			else 0))
		 y = ((menu-y m (((length (items m)) - n) * ysize)) +
		      (case place
			((center right left) (ysize / 2))
			(bottom 0)
			(top ysize)
			else 0)) )) ))

; 10 Dec 90; 13 Dec 90; 10 Sep 91
; Choose from menu, then close it
(gldefun menu-select (m\:menu) (menu-select-b m nil))
(gldefun menu-select! (m\:menu) (menu-select-b m t))
(gldefun menu-select-b (m\:menu flg\:boolean)
  (prog (res)
lp  (res \:= (choose m))
    (if (flg and ~res) (go lp))
    (if ~(permanent m)
	(if (flat m) then (clear m)
	                  (force-output (menu-window m))
	             else (close (menu-window m))))
    (return res)))

; 12 Aug 91
(gldefun menu-destroy (m\:menu)
  (if ~ (flat m)
      then (destroy (menu-window m))
           ((menu-window m) \:= nil) ))

; 19 Aug 91; 02 Sep 91
; Easy interface to make a menu, select from it, and destroy it.
(defun menu (items &optional title)
  (let (m res)
    (setq m (menu-create items title))
    (setq res (menu-select m))
    (menu-destroy m)
    res ))

; 12 Aug 91; 15 Aug 91; 06 Sep 91; 09 Sep 91; 12 Sep 91; 23 Oct 91
; Simple call from plain Lisp to make a menu.
(gldefun menu-create (items &optional title parentw\:window x y
			    perm\:boolean flat\:boolean font\:symbol)
  (a menu with title           = (if title (stringify title) else "")
               menu-window     = (if flat then parentw)
               items           = items
               parent-window   = (parent parentw)
	       parent-offset-x = x
	       parent-offset-y = y
	       permanent       = perm
	       flat            = flat
	       menu-font       = font ))

; 15 Oct 91; 30 Oct 91
(gldefun menu-offset (m\:menu)
  (result vector)
  (a vector with x = (base-x m) y = (base-y m)))

; 15 Oct 91; 30 Oct 91
(gldefun menu-size (m\:menu)
  (result vector)
  (a vector with x = (picture-width m) y = (picture-height m)))

; 15 Oct 91
(gldefun menu-moveto-xy (m\:menu x\:integer y\:integer)
  (if (flat m)
      then ((parent-offset-x m) \:= x)
           ((parent-offset-y m) \:= y)
	   (adjust-offset m)) )

; 09 Sep 91; 11 Sep 91; 12 Sep 91; 14 Sep 91
; Simple call from plain Lisp to make a picture menu.
(gldefun picmenu-create
  (buttons width\:integer height\:integer drawfn
         &optional title dotflg\:boolean parentw\:window x y
	           perm\:boolean flat\:boolean font\:symbol boxflg\:boolean)
  (picmenu-create-from-spec
    (picmenu-create-spec buttons width height drawfn dotflg font)
    title parentw x y perm flat boxflg))                  

; 14 Sep 91
(gldefun picmenu-create-spec (buttons width\:integer height\:integer drawfn
		              &optional dotflg\:boolean font\:symbol)
  (a picmenu-spec with drawing-width   = width
                       drawing-height  = height
		       buttons         = buttons
		       dotflg          = dotflg
		       drawfn          = drawfn
		       menu-font       = (font or '9x15)))

; 14 Sep 91
(gldefun picmenu-create-from-spec
	 (spec\:picmenu-spec &optional title parentw\:window x y
	           perm\:boolean flat\:boolean boxflg\:boolean)
  (a picmenu with title           = (if title (stringify title) else "")
                  menu-window     = (if flat then parentw)
		  parent-window   = (if parentw (parent parentw))
		  parent-offset-x = x
		  parent-offset-y = y
		  permanent       = perm
	          flat            = flat
		  spec            = spec
		  boxflg          = boxflg))

; 09 Sep 91; 10 Sep 91
; Initialize a picture menu
(gldefun picmenu-init (m\:picmenu)
  (let (maxwidth maxheight)
    (maxwidth \:= (max (if (title m) then ((* 9 (length (title m))) + 6)
		                     else 0)
		       (drawing-width m)))
    (maxheight \:= (if (title-present m) then 15 else 0) + (drawing-height m))
    ((picture-width m) \:= maxwidth)
    ((picture-height m) \:= maxheight)
    (adjust-offset m)
    (if ~ (flat m)
	((menu-window m) \:= (window-create maxwidth
					    maxheight
					    ((title m) or "")
					    (parent-window m)
					    (parent-offset-x m)
					    (parent-offset-y m)
					    (menu-font m) )) ) ))

; 09 Sep 91; 10 Sep 91; 11 Sep 91; 10 Feb 92
; Draw a picture menu
(gldefun picmenu-draw (m\:picmenu)
  (let (mw bottom xzero yzero)
    (init? m)
    (mw \:= (menu-window m))
    (open mw)
    (clear m)
    (xzero \:= (menu-x m 0))
    (yzero \:= (menu-y m 0))
    (bottom \:= yzero + (picture-height m))
    (if (title-present m)
	then (printat-xy mw (stringify (title m)) (xzero + 3) (bottom - 13))
             (invert-area-xy mw xzero (bottom - 15) (picture-width m) 16))
    (funcall (drawfn m) mw xzero yzero)
    (if (boxflg m) (draw-box-xy mw xzero yzero
				   (picture-width m) (picture-height m) 1))
    (if (dotflg m) then
	(for b in (buttons m) do
	     (draw-box-xy mw (xzero + (x (offset b)) - 2)
			     (yzero + (y (offset b)) - 2)
			     4 4 1) ) )
    (force-output mw) ))

; 09 Sep 91; 10 Sep 91; 18 Sep 91
(gldefun picmenu-select (m\:picmenu)
  (let (mw current-button\:picmenu-button item items val\:picmenu-button
	   xzero yzero inside)
    (mw \:= (menuw m))
    (if ~ (permanent m) (draw m))
    (xzero \:= (menu-x m 0))
    (yzero \:= (menu-y m 0))
    (track-mouse mw
      #'(lambda (x y code)
	  (x \:= (x - xzero))
	  (y \:= (y - yzero))
	  (if ((x >= 0) and (x <= (picture-width m))
	        and (y >= 0) and (y <= (picture-height m)))
	      then (inside \:= t))
	  (if current-button
	      (if ~ (containsxy? current-button x y)
		  then (unbox-item m current-button)
		       (current-button \:= nil)))
	  (if ~ current-button
	      then (items \:= (buttons m))
	           (while ~ current-button and (item -_ items) do
			  (if (containsxy? item x y)
			      then (box-item m item)
			           (current-button \:= item))))
	  (if ((> code 0)
	       or (inside and ((x < 0) or (x > (picture-width m))
			       or (y < 0) or (y > (picture-height m)))))
	      then (if current-button then (unbox-item m current-button))
	           (val \:= (or current-button 'no-selecction-val)) ))
      t)
    (if ~(permanent m)
	(if (flat m) then (clear m)
	                  (force-output (menu-window m))
	             else (close (menu-window m))))
    (if (val <> 'no-selecction-val) (name val)) ))

; 09 Sep 91; 10 Sep 91
(gldefun picmenu-box-item (m\:picmenu item\:picmenu-button)
  (let ((mw (menuw m)) xoff yoff siz)
    (xoff \:= (menu-x m (x (offset item))))
    (yoff \:= (menu-y m (y (offset item))))
    (if (highlightfn item)
	then (funcall (highlightfn item) (menuw m) xoff yoff)
        else (set-xor mw)
	     (if (siz \:= (size item))
	         then (draw-box-xy mw (xoff - (x siz) / 2)
			              (yoff - (y siz) / 2)
				      (x siz) (y siz) 1)
		 else (draw-box-xy mw (xoff - 6) (yoff - 6) 12 12 1))
	     (unset mw)
	     (force-output mw) ) ))

; 09 Sep 91
(gldefun picmenu-unbox-item (m\:picmenu item\:picmenu-button)
  (let ((mw (menuw m)))
    (if (unhighlightfn item)
	then (funcall (unhighlightfn m) (menuw m)
		      (x (offset item)) (y (offset item)))
             (force-output mw)
        else (box-item m item) ) ))

(defun picmenu-destroy (m) (menu-destroy m))

; 09 Sep 91; 10 Sep 91; 11 Sep 91
(gldefun picmenu-button-containsxy? (b\:picmenu-button x\:integer y\:integer)
  (let ((xsize 6) (ysize 6))
    (if (size b) then (xsize \:= (x (size b)) / 2)
                      (ysize \:= (y (size b)) / 2))
    ((x >= ((x (offset b)) - xsize)) and (x <= ((x (offset b)) + xsize)) and
     (y >= ((y (offset b)) - ysize)) and (y <= ((y (offset b)) + ysize)) ) ))

; 11 Sep 91
(gldefun picmenu-item-position (m\:picmenu itemname\:symbol
					   &optional place\:symbol)
  (let (b\:picmenu-button (xsize 0) (ysize 0))
    (if (b \:= (that (buttons m) with name = itemname))
	then (if (size b)
		 then (xsize \:= (x (size b)))
	              (ysize \:= (y (size b))))
             (a vector with
		x = ((menu-x m (x (offset b))) + 
	             (case place
		       ((center top bottom) (xsize / 2))
		       (left 0)
		       (right xsize)
		       else 0))
		y = ((menu-y m (y (offset b))) + 
	             (case place
		       ((center right left) (ysize / 2))
		       (bottom 0)
		       (top ysize)
		       else 0))) ) ))

; 15 Aug 91
; Get a point position by mouse click.  Returns (x y).
(setf (glfnresulttype 'window-get-point) 'vector)
(defun window-get-point (w)
  (let (orgx orgy)
    (window-track-mouse w                  ; get one point
	    #'(lambda (x y code)
		(when (not (zerop code))
		  (setq orgx x)
		  (setq orgy y))))
    (list orgx orgy) ))

; 23 Aug 91
; Get a point position by mouse click.  Returns (button (x y)).
(setf (glfnresulttype 'window-get-click) '(list (button integer) (pos vector)))
(defun window-get-click (w)
  (let (orgx orgy button)
    (window-track-mouse w                  ; get one point
	    #'(lambda (x y code)	
	(when (not (zerop code))
		  (setq button code)
		  (setq orgx x)
		  (setq orgy y))))
    (list button (list orgx orgy)) ))

; 13 Aug 91; 06 Aug 91
; Get a position indicated by a line from a specified origin position.
; Returns (x y) at end of line.
(setf (glfnresulttype 'window-get-line-position) 'vector)
(defun window-get-line-position (w orgx orgy)
  (window-get-icon-position w #'window-draw-line-xy (list orgx orgy 1 'paint)))

; 13 Aug 91; 15 Aug 91; 05 Sep 91
; Get a position indicated by a box of a specified size.
; (dx dy) is offset of lower-left corner of box from mouse
; Returns (x y) of lower-left corner of box.
(setf (glfnresulttype 'window-get-box-position) 'vector)
(defun window-get-box-position (w width height &optional (dx 0) (dy 0))
  (window-get-icon-position w #'window-draw-box-xy
			      (list width height 1) dx dy))

; 05 Sep 91
; Get a position indicated by an icon.
; fn is the function to draw the icon: (fn w x y . args) .
; fn must simply draw the icon, not set window parameters.
; (dx dy) is offset of lower-left corner of icon (x y) from mouse.
; Returns (x y) of mouse.
(defun window-get-icon-position (w fn args &optional (dx 0) (dy 0))
  (let (lastx lasty argl)
    (setq argl (cons w (cons 0 (cons 0 args))))   ; arg list for fn
    (window-set-xor w)
    (window-track-mouse w 
	    #'(lambda (x y code)
		(when (or (null lastx) (/= x lastx) (/= y lasty))
		  (if lastx (apply fn argl))     ; undraw
		  (rplaca (cdr argl) (+ x dx))
		  (rplaca (cddr argl) (+ y dy))
		  (apply fn argl)                ; draw
		  (setq lastx x)
		  (setq lasty y))
		(not (zerop code)) ))
    (apply fn argl)                ; undraw
    (window-unset w)
    (window-force-output w)
    (list lastx lasty) ))

; 13 Aug 91; 06 Sep 91; 06 Nov 91
; Get a box size and position.
; Click for top right, then click for bottom left, then move it.
; Returns ((x y) (width height)) where (x y) is lower-left corner of box.
(setf (glfnresulttype 'window-get-region) 'region)
(defun window-get-region (w &optional wid ht)
  (let (lastx lasty start end width height place offx offy stx sty)
    (if (and (numberp wid) (numberp ht))
	(progn (setq start (window-get-box-position w wid ht (- wid) (- ht)))
	       (setq stx (- (car start) wid))
	       (setq sty (- (cadr start) ht)) )
	(progn (setq start (window-get-point w))
	       (setq stx (car start))
	       (setq sty (cadr start))))
    (setq end (window-get-icon-position w #'window-draw-box-corners
					  (list stx sty 1)))
    (setq lastx (car end))
    (setq lasty (cadr end))
    (setq width  (abs (- stx lastx)))
    (setq height (abs (- sty lasty)))
    (setq offx (- (min stx lastx) lastx))
    (setq offy (- (min sty lasty) lasty))
    (setq place (window-get-box-position w width height offx offy))
    (list (list (+ offx (first place))
	        (+ offy (second place)))
          (list width height)) ))

; 27 Nov 91
; Get box size and echo the size in pixels.  Click for top right.
; Returns (width height) of box.
(setf (glfnresulttype 'window-get-box-size) 'vector)
(defun window-get-box-size (w offsetx offsety)
  (let (legendy lastx lasty dx dy)
    (setq offsety (min offsety 30))
    (setq legendy (- offsety 25))
    (window-erase-area-xy w offsetx legendy 70 20)
    (window-draw-box-xy w offsetx legendy 70 20)
    (window-track-mouse w 
	    #'(lambda (x y code)
		(when (or (null lastx) (/= x lastx) (/= y lasty))
		  (if lastx (window-xor-box-xy w offsetx offsety
					         (- lastx offsetx)
					         (- lasty offsety)))
		  (setq lastx nil)
		  (setq dx (- x offsetx))
		  (setq dy (- y offsety))
		  (when (and (> dx 0) (> dy 0))
		    (window-xor-box-xy w offsetx offsety dx dy)
		    (window-printat-xy w (format nil "~3D x ~3D" dx dy)
			       (+ offsetx 3) (+ legendy 5))
		    (setq lastx x)
		    (setq lasty y)))
		(not (zerop code)) ))
    (if lastx (window-xor-box-xy w offsetx offsety (- lastx offsetx)
					           (- lasty offsety)))
    (window-erase-area-xy w offsetx legendy 70 20)
    (window-force-output w)
    (list dx dy) ))

; 29 Oct 91; 30 Oct 91
; Track mouse until a button is pressed or it leaves specified region.
; Returns (x y code) or nil.  boxflg is T to box the region.
(setf (glfnresulttype 'window-track-mouse-in-region)
      '(list (code integer) (position (transparent vector))))
(defun window-track-mouse-in-region (w offsetx offsety sizex sizey
				       &optional boxflg)
  (let (res inside)
    (when boxflg
      (window-set-xor w)
      (window-draw-box-xy w (- offsetx 4) (- offsety 4)
			    (+ sizex 8) (+ sizey 8))
      (window-unset w)
      (window-force-output w) )
    (setq res (window-track-mouse w
	        #'(lambda (x y code)
		    (if (> code 0)
			(if inside (list code (list x y)) t)
			(if (or (< x offsetx)
				(> x (+ offsetx sizex))
				(< y offsety)
				(> y (+ offsety sizey)))
			    inside
			    (and (setq inside t) nil)))) ) )
    (when boxflg
      (window-set-xor w)
      (window-draw-box-xy w (- offsetx 4) (- offsety 4)
			    (+ sizex 8) (+ sizey 8))
      (window-unset w)
      (window-force-output w) )
    (if (consp res) res) ))

; 04 Nov 91
; Adjust one side of a box by mouse movement.  Returns ((x y) (width height)).
(setf (glfnresulttype 'window-adjust-box-side) 'region)
(defun window-adjust-box-side (w orgx orgy width height side)
  (let (new (xx orgx) (yy orgy) (ww width) (hh height))
    (setq new (window-get-icon-position w #'window-adj-box-xy
					(list orgx orgy width height side)))
    (case side (left (setq xx (car new))
		     (setq ww (+ width (- orgx (car new)))))
               (right (setq ww (- (car new) orgx)))
	       (top   (setq hh (- (cadr new) orgy)))
	       (bottom (setq yy (cadr new))
		       (setq hh (+ height (- orgy (cadr new))))) )
    (list (list xx yy) (list ww hh))  ))

; 04 Nov 91
(defun window-adj-box-xy (w x y orgx orgy width height side)
  (let ((xx orgx) (yy orgy) (ww width) (hh height))
    (case side (left (setq xx x) (setq ww (+ width (- orgx x))))
               (right (setq ww (- x orgx)))
	       (top   (setq hh (- y orgy)))
	       (bottom (setq yy y) (setq hh (+ height (- orgy y)))) )
    (window-draw-box-xy w xx yy ww hh) ))
          

; 14 Dec 90; 12 Aug 91; 09 Oct 91
; Compile the dwindow file into a plain Lisp file
(defun compile-dwindow ()
  (glcompfiles '("/v/ai/v0/novak/glisp/vector.lsp")       ; auxiliary files
               '("/v/ai/v0/novak/X/dwindow.lsp")      ; translated files
	       "/v/ai/v0/novak/X/dwtrans.lsp")        ; output file
  (compile-file "/v/ai/v0/novak/X/dwtrans.lsp") )
