;;; -*- Mode: LISP; Syntax: Common-lisp; Package: YY; Base: 10 -*-
;;; graphic-primitive.lisp
;;;
;;;  Copyright (C) 1989,1990,1991 Aoyama Gakuin University
;;;  graphic-primitive.lisp
;;;
;;;		All Rights Reserved
;;;
;;; This software is developed for the YY project of Aoyama Gakuin University.
;;; Permission to use, copy, modify, and distribute this software
;;; and its documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notices appear in all copies and that
;;; both that copyright notice and this permission notice appear in 
;;; supporting documentation, and that the name of Aoyama Gakuin
;;; not be used in advertising or publicity pertaining to distribution of
;;; the software without specific, written prior permission.
;;;
;;; This software is made available AS IS, and Aoyama Gakuin makes no
;;; warranty about the software, its performance or its conformity to
;;; any specification. 
;;;
;;; To make a contact: Send E-mail to ida@csrl.aoyama.ac.jp for overall
;;; issues. To ask specific questions, send to the individual authors at
;;; csrl.aoyama.ac.jp. To request a mailing list, send E-mail to 
;;; yyonx-request@csrl.aoyama.ac.jp.
;;;
;;; Authors:
;;;   version 1.0 90/06/01 by t.kosaka (kosaka@csrl.aoyama.ac.jp)
;;;   version 1.1 90/07/31 by t.kosaka
;;;   update 1.11 90/09/14 by t.kosaka
;;;   version 1.2 90/11/05 by t.kosaka

;;; $@IA2h5Z$SI=<($N$?$a$N4pK\%/%i%9$NDj5A(J
;;; 1/26 1990 $@8E:d(J

;;; Version 1.0   Coded by t.kosaka 1990-1-26
;;; Change log 1990-6-20 add get-new-position-xy 
;;;            1990-8-24 cursor-position is no operation
;;;            1990-10-4 Add work-region1 anf work-region2 to graphic-stream

(in-package :yy)

;;; $@:BI8JQ49$NAm>N4X?t(J XY$@MQ(J
(defgeneric translate-coordinate-y (object y max-height)
  (:method ((object translate-coordinate-left-bottom)
	    y max-height)
           ;;; $@:82<$9$_:BI87O$r:8>e$9$_:BI87O$KJQ49$9$k(J
	   (- max-height y))
  (:method ((object translate-coordinate-left-top)
	    y max-height)
#-:PCL
  (declare (ignore max-height))
           ;;; $@:BI87O$,:8>e$9$_$J$N$G!"$J$K$b$7$J$$(J
	   y))

;;; $@:BI8JQ49$NAm>N4X?t(J position$@MQ(J
(defgeneric translate-coordinate (object position max-height)
  (:method ((object translate-coordinate-left-bottom)
	    (position position)
	    max-height)
	   (declare (inline -))
	   (make-position :x (position-x position)
			  :y (- max-height (position-y position))))
  (:method ((object translate-coordinate-left-top)
	    (position position)
            max-height)
#-:PCL
  (declare (ignore max-height))
	   position))

;;; $@:BI8JQ49$NAm>N4X?t(J region $@MQ(J
(defgeneric translate-coordinate-region (object region max-height)
  (:method ((object translate-coordinate-left-bottom)
	    (region region)
	    max-height)
	   (declare (inline max min))
	   (let ((bottom (min (- max-height (region-bottom region))
			      (- max-height (region-top region))))
		 (top (max (- max-height (region-bottom region))
			   (- max-height (region-top region)))))
	     (make-region :left (region-left region)
			  :bottom bottom
			  :width (region-width region)
			  :top top)))
  (:method ((object translate-coordinate-left-top)
	    (region region)
	    max-height)
#-:PCL
  (declare (ignore max-height))
	   region))

;;; $@%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/%9$NHs<B9T%/%i%9%$%s%9%?%s%9(J
(defvar *non-graphic-transform-matrix* 
    (make-instance 'graphic-transform-matrix-non))

;;; $@%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/%9$r%3%T!<$9$k(J
;;; $@Hs<B9T%/%i%9(J
(defmethod copy-transform-matrix ((matrix graphic-transform-matrix-non))
  (make-instance 'graphic-transform-matrix-non))

;;; $@%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/%9%$%s%9%?%s%9I=<(%a%=%C%I(J
(defmethod print-object ((matrix  graphic-transform-matrix-mixin) stream)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((tt (matrix-theta matrix)) 
	(xx (matrix-x-time matrix)) 
	(yy (matrix-y-time matrix)))
   (format stream "~%\#<Matrix \|cos ~a ~t-sin ~a\|\|~a|~%         \|sin ~a ~t cos ~a\|\|~a\|>" 
	    tt tt xx tt tt yy)))

;;; $@%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/%9$N(Jsetf $@%a%=%C%I(J $@&H$NJQ99(J
(defmethod (setf matrix-theta) :around (new-theta
			       (matrix graphic-transform-matrix-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (if (numberp new-theta)
      (call-next-method)
  (error "The aergument ~a is an illegal value." new-theta)))

;;; $@%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/%9(J
(defmethod (setf matrix-theta) :after (new-theta
				       (matrix graphic-transform-matrix-exec))
  (declare (inline sin cos / * - round)
	   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let* ((sinx (sin (* pi (/ new-theta 180.0))))
		 (cosx (cos (* pi (/ new-theta 180.0))))
		 (list (step-slots matrix)))

    (setf (car list) cosx
		  (second list) (*  sinx -1)
		  (third list) sinx
		  (fourth list) cosx))
  )

;;; $@%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/%9$r%3%T!<$9$k(J
;;; $@<B9T%/%i%9(J
(defmethod copy-transform-matrix ((matrix graphic-transform-matrix-exec))
  (make-transform-matrix :theta (matrix-theta matrix)
						 :x-time (matrix-x-time matrix)
						 :y-time (matrix-y-time matrix)))


;;; $@%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/%9$N%"%/%;%C%5(J x-time$@$NJQ99(J
(defmethod (setf matrix-x-time) :around (new-x
				 (matrix graphic-transform-matrix-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (if (numberp new-x)
      (call-next-method)
    (error "The argument ~a is an illegal value." new-x)))

;;; $@%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/%9$N%"%/%;%C%5(J y-time$@$NJQ99(J
(defmethod (setf matrix-y-time) :around  (new-y
				 (matrix graphic-transform-matrix-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (if (numberp new-y)
      (call-next-method)
    (error "The argument ~a is an illegal value." new-y)))

;;;$@%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/%9$N@8@.4X?t(J
(defun make-transform-matrix (&key (theta 0.0) (x-time 1.0) (y-time 1.0))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
;;;$@%f!<%6!<$O!"$3$N4X?t$rMQ$$$F2sE>9TNs$r:n@.$9$k(J
  (let ((matrix (make-instance 'graphic-transform-matrix-exec
		  :step-slots (list 1 1 1 1)
			       :theta theta :x-time x-time :y-time y-time)))
    (setf (matrix-theta matrix) theta)
    matrix))
  

;;; $@%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/$NAm>N4X?t(J ($@%]%8%7%g%sMQ(J)
(defgeneric transform-by-matrix (pos matrix &key return)
  (:method ((pos position) 
	    (matrix graphic-transform-matrix-exec)
	    &key (return :multiple))
   (with-position-destructured 
    ((x y) pos)
    (let* ((list (step-slots matrix))
	   (x-time (matrix-x-time matrix))
	   (y-time (matrix-y-time matrix))
	   (new-x (round (* (+ (* x (car list)) (* y (second list))) x-time)))
	   (new-y (round (* (+ (* x (third list)) (* y (fourth list))) y-time))))
      (case return
	(:multiple (values new-x new-y))
	(:position (make-position :x new-x
				   :y new-y))
	(t (list new-x new-y)))
       )))
  (:method ((pos position)
	    (matrix graphic-transform-matrix-non)
	    &key (return :multiple))
   (with-position-destructured ((x y) pos)
       (case return
	     (:maltiple (values x y))
	     (:position pos)
	     (t (list x y)))
       )))

;;; $@%0%i%U%#%C%/%H%i%s%9%U%)!<%`%^%H%j%C%/$NAm>N4X?t(J (XY $@MQ(J)
(defgeneric transform-by-matrix-xy (x y matrix  &key return)
  (:method ((x number) (y number)
	    (matrix graphic-transform-matrix-exec)
	    &key (return :multiple))
    (let* ((list (step-slots matrix))
	   (x-time (matrix-x-time matrix))
	   (y-time (matrix-y-time matrix))
	   (new-x (round (* (+ (* x (car list)) (* y (second list))) x-time)))
	   (new-y (round (* (+ (* x (third list)) (* y (fourth list))) y-time))))
      
      (case return
	    (:multiple (values new-x new-y))
	    (:position (make-position :x new-x :y new-y))
	    (t (list new-x new-y)))
      ))
  (:method ((x number) (y number)
	    (matrix graphic-transform-matrix-non)
	    &key (return :multiple))
  (let ((new-x (round x)) (new-y (round y)))
    (case return
	  (:multiple (values new-x new-y))
	  (:position (make-position :x new-x :y new-y))
	  (t (list new-x new-y)))
    )))

;;; $@%*%Z%l!<%7%g%s$NJQ99(J
(defmethod (setf graphic-operation)  (new-value (state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((list (state-slots state)))
    (if (or (eq new-value *GCLEAR*) (eq new-value *GAND*) 
	    (eq new-value *GANDREVERSE*) (eq new-value *GCOPY*)
	    (eq new-value *GANDINTEVERTED*) (eq new-value *GNOOP*)
	    (eq new-value *GXOR*) (eq new-value *GOR*)
	    (eq new-value *GNOR*) (eq new-value *GEQIV*)
	    (eq new-value *GINVERT*) (eq new-value *GORREVERSE*)
	    (eq new-value *GCOPYINVERTED*) (eq new-value *GORINVERTED*)
	    (eq new-value *GNAND*) (eq new-value *GSET*))
	(setf (car list) new-value)
      (error "The Operation ~a is an illegal argument. Shuld be from #x00 to #xFF"
	   new-value))))

;;; $@%*%Z%l!<%7%g%s$N%"%/%;%9(J
(defmethod graphic-operation ((state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (car (state-slots state)))

;;; $@IA2h?'$NJQ99(J
(defmethod (setf graphic-color) (new-color (state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((list (state-slots state)))
    (if (eq (class-name (class-of new-color)) 'color)
	(setf (second list) new-color)
      (error "The argument ~a is not a color class instance." new-color))))

;;; $@IA2h?'$N%"%/%;%9(J
(defmethod graphic-color ((state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (second (state-slots state)))

;;; $@@~$NI}$NJQ99(J
(defmethod (setf line-width) (new-width (state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((list (state-slots state)))
    (if (integerp new-width)
	(setf (third list) new-width)
      (error "The argument ~a is an illegal value." new-width))))

;;; $@@~$NI}$N%"%/%;%9(J
(defmethod line-width ((state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (third (state-slots state)))

;;; $@@~$NC<$N7A(J
(defmethod (setf line-edge) (new-edge  (state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((list (state-slots state)))
    (if (or (eq *SQUEAR-LINE-EDGE* new-edge)
	    (eq *SQUEAR-LINE-EDGE-WITHOUT-END* new-edge)
	    (eq *ROUND-LINE-EDGE* new-edge))
	(setf (fourth list) new-edge)
    (error "The line edge style ~a is an illegal value." new-edge))))

;;; $@@~$NC<$N7A$r%"%/%;%9(J
(defmethod line-edge ((state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (fourth (state-slots state)))

;;; $@@~$N@\B3$N7A(J
(defmethod (setf line-joint-type) (new-joint (state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((list (state-slots state)))
    (if (or (eq *SHARP-JOINT* new-joint)
	    (eq *ROUND-JOINT* new-joint))
	(setf (fifth list) new-joint)
    (error "The line joint type ~a is an illegal value." new-joint))))

;;; $@@~$N@\B3$N7A$r%"%/%;%9(J
(defmethod line-joint-type ((state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (fifth (state-slots state)))

;;; $@@~$N%@%C%7%s%0$N7A>u(J
(defmethod (setf line-dashing) (new-dash (state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((list (state-slots state)))
    (if (stringp new-dash)
	(setf (sixth list) new-dash)
    (error "The dashing type ~a is an illegal value" new-dash))))

;;; $@@~$N%@%C%7%s%0$N7A>u$r%"%/%;%9(J
(defmethod line-dashing ((state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (sixth (state-slots state)))

;;; $@1_8L$N%b!<%I(J
(defmethod (setf arc-mode) (new-mode (state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((list (state-slots state)))
    (if (or (eq *ARCPIESLICE* new-mode)
	    (eq *ARCCHORD* new-mode))
	(setf (seventh list) new-mode)
    (error "The arc mode ~a is an illegal value." new-mode))))

;;; $@1_8L$N%b!<%I$r%"%/%;%9(J
(defmethod arc-mode ((state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (seventh (state-slots state)))

;;; $@EI$j$D$V$7%b!<%I(J
(defmethod (setf filled-type) (new-type  (state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((list (state-slots state)))
    (if (or (eq *FillSolid* new-type)
	    (eq *Fillednon* new-type)
	    (eq *FillTiled* new-type)
	    (eq *FillOpaqueStippled* new-type))
      (setf (eighth list) new-type)
    (error "The filled type ~a is an illigale value." new-type))))

;;; $@EI$j$D$V$7%b!<%I$r%"%/%;%9(J
(defmethod filled-type ((state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (eighth (state-slots state)))

;;; $@GX7J$N?'$^$?$O%Q%?!<%s(J
(defmethod (setf stream-default-color-pattern) 
                     (new-type  (state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((list (state-slots state)))
      (setf (ninth list) new-type)))

;;; $@GX7J$N?'$^$?$O%Q%?!<%s$r%"%/%;%9(J
(defmethod stream-default-color-pattern ((state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (ninth (state-slots state)))

;;; $@EI$j$D$V$7%k!<%k(J
(defmethod (setf filled-rule) (new-rule (state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((list (state-slots state)))
    (if (or (eq  *EvenOddRule* new-rule)
	    (eq *WindingRule* new-rule))
	(setf (tenth list) new-rule)
    (error "The filled rule ~a is an illigale value." new-rule))))

;;; $@EI$j$D$V$7%k!<%k$r%"%/%;%9(J
(defmethod filled-rule ((state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (tenth (state-slots state)))

;;; $@EI$j$D$V$7%Q%?!<%s(J
(defmethod (setf filled-pattern) 
                     (new-type  (state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((list (state-slots state)))
      (setf (nth 10 list) new-type)))

;;; $@EI$j$D$V$7%Q%?!<%s$r%"%/%;%9(J
(defmethod filled-pattern ((state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (nth 10 (state-slots state)))

;;; $@1_8L$NIA2hJ}8~$r%"%/%;%9(J
(defmethod stream-clockwize ((state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (nth 11 (state-slots state)))


;;; $@1_8L$NIA2hJ}8~$rJQ99(J
(defmethod (setf stream-clockwize) (new-clockwize 
									(state graphic-state-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (if (or (eq new-clockwize :clockwize)
		  (eq new-clockwize :counter-clockwize))
	 (setf (nth 11 (state-slots state)) new-clockwize)))


(defmethod world-height (stream)
  (if (null stream)
      0
    ))

;;; $@%F%-%9%H%3%s%H%m!<%k%/%i%9$N%3%T!<(J
(defmethod copy-graphic-text-control-mixin ((state graphic-text-control-mixin))
  (make-instance 'graphic-text-control-mixin
		 :left-margin (stream-left-margin state)
		 :right-margin (stream-right-margin state)
		 :top-margin (stream-top-margin state)
		 :bottom-margin (stream-bottom-margin state)
		 :text-font (stream-font state)
		 :line-feed (stream-line-feed state)
		 :output-direction (stream-output-direction state)))



;;; $@%"%/%;%9%a%=%C%I(J left-margin setf $@MQ(J nil $@?t;zMQ(J
(defmethod (setf stream-left-margin) :around (new-value
				      (instance graphic-text-control-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (if (or (null new-value)
	  (integerp new-value))
      (call-next-method)
    (error "The left margin ~a is an illegal value." new-value)))

;;; $@%"%/%;%9%a%=%C%I(J right-margin setf $@MQ(J nil $@?t;zMQ(J
(defmethod (setf stream-right-margin) :around (new-value
				      (instance graphic-text-control-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (if (or (null new-value)
	  (integerp new-value))
      (call-next-method)
    (error "The right margin ~a is an illegal value." new-value)))

;;; $@%"%/%;%9%a%=%C%I(J top-margin setf $@MQ(J nil 
(defmethod (setf stream-top-margin) :around (new-value 
				      (instance graphic-text-control-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (if (or (null new-value)
	  (integerp new-value))
      (call-next-method)
    (error "The top margin ~a is an illegal value." new-value)))

;;; $@%"%/%;%9%a%=%C%I(J bottom-margin setf $@MQ(J nil 
(defmethod (setf stream-bottom-margin) :around (new-value 
				      (instance graphic-text-control-mixin))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (if (or (null new-value)
	  (integerp new-value))
      (call-next-method)
    (error "The bottom margin ~a is an illegal value." new-value)))



;;; $@7?H=Dj(J
(defun graphic-stream-p (stream)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (typep stream 'graphic-stream))

;;; $@:BI8JQ49$r$7$?%j!<%8%g%s$rJV$9(J
(defmethod change-coodinate-region ((region  region)
				    (stream graphic-stream))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (translate-coordinate-region (stream-translate-coordinate stream)
			       region
			       (world-height stream))
  )
			       

#|
;;; stream-line-feed $@$N%"%/%C%;%5!<(J
(defmethod stream-line-feed ((stream graphic-stream))
  (abs (slot-value stream 'line-feed)))

;;; stream-line-feed $@$N%a%C%=%I(J
(defmethod (setf stream-line-feed) ((new-v T) (stream graphic-stream))
  (if (numberp new-v)
      (if (eq (class-name (class-of (stream-translate-coordinate stream)))
	      'translate-coordinate-left-bottom)
	  (setf (slot-value stream 'line-feed) (- new-v))
	(setf (slot-value stream 'line-feed) new-v))
    (error "The argument type is not a integer.")))

;;; graphic-stream$@$N=i4|2=%a%=%C%I(J
(defmethod initialize-instance :after ((stream graphic-stream) &rest arg)
  (let ((val (nth (+ (position :line-feed arg) 1) arg)))
    (if (eq (class-name (class-of (stream-translate-coordinate stream)))
              'translate-coordinate-left-bottom)
	(setf (slot-value stream 'line-feed) (- val))
      (setf (slot-value stream 'line-feed) val))
    ))

|#

;;;$@%+!<%=%k$N(JX$@%]%8%7%g%s$N%"%/%;%C%5(J
(defmethod stream-cursor-x-position  ((stream graphic-stream))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (position-x (slot-value stream 'cursor-position)))

;;;$@%+!<%=%k$N(JY$@%]%8%7%g%s$N%"%/%;%C%5(J
(defmethod stream-cursor-y-position ((stream graphic-stream))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (position-y (stream-cursor-position stream)))


;;;$@%+!<%=%k$N(JX$@%]%8%7%g%s$N(Jsetf$@%"%/%;%C%5(J 
(defmethod (setf stream-cursor-x-position) ((new-x integer)
    			         	    (stream graphic-stream))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (setf (position-x (slot-value stream 'cursor-position)) new-x))


;;;$@%+!<%=%k$N(JY$@%]%8%7%g%s$N(Jsetf$@%a%=%C%I(J
(defmethod (setf stream-cursor-y-position) ((new-y integer)
					   (stream graphic-stream))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (setf (position-y (stream-cursor-position stream)) new-y))
	
;;; translate-coordinate$@$NJQ99%a%=%C%I(J
(defmethod (setf translate-coordinate) ((new-value 
					 translate-coordinate-mixin)
					(stream graphic-stream))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (setf (slot-value stream 'translate-coordinate) new-value))


;;; transform-by-matrix $@$NJQ99%a%=%C%I(J
(defmethod (setf stream-transform-by-matrix) ((new-value 
					graphic-transform-matrix-mixin)
				       (stream graphic-stream))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (setf (slot-value stream 'transform-by-matrix) new-value)
)
  	
;;; $@2sE>9TNs$NJQ49$r2C$($??7$7$$(Jposition$@$rJV$9(J
;;; $@%a%=%C%I(J
(defmethod get-new-position ((pos position) (stream graphic-stream))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-transform-stream pos stream))

;;; stream graphic-stream$@0J30$O!"$=$N$^$^$rJV$9!#(J
(defmethod get-new-position ((pos position) (stream T))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  pos)


;;; $@2sE>9TNs$NJQ49$r2C$($??7$7$$(JXY$@$rJV$9(J
(defmethod get-new-position-xy ((x integer) (y integer) (stream graphic-stream))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (transform-by-matrix-xy x y (slot-value stream 'transform-by-matrix)
			  :return :multiple))

;;; stream graphic-stream$@0J30$O!"$=$N$^$^$rJV$9(J
(defmethod get-new-position-xy ((x integer) (y integer) (stream T))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (values x y))

;;; $@2sE>9TNs!":BI8JQ49$r$7$?CM$rJV$9(J($@%j%9%HMQ!K(J
(defmethod translate-transform-xy ((stream graphic-stream) 
				   (x integer)
				   (y integer))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((ret-val
	 (transform-by-matrix-xy x y (slot-value stream 'transform-by-matrix)
				 :return T)))

    (list (first ret-val) 
	  (translate-coordinate-y (slot-value stream 'translate-coordinate)
		    (second ret-val) (world-height stream)))
    ))

;;; $@2sE>9TNs!":BI8JQ49$r$7$?CM$rJV$9(J
(defmethod translate-transform-xy-valuse ((stream graphic-stream)
                                   (x integer)
                                   (y integer))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (multiple-value-bind (xx yy)
       (transform-by-matrix-xy x y (slot-value stream 'transform-by-matrix)
			       :return :multiple)
       (values xx 
	       (translate-coordinate-y (slot-value stream 
						   'translate-coordinate)
		    yy (world-height stream)))))

;;; End of file
