;;; -*- Mode: LISP; Syntax: Common-lisp; Package: YY; Base: 10 -*-
;;; graphic-method.lisp
;;;
;;;  Copyright (C) 1989,1990,1991 Aoyama Gakuin University
;;;
;;;		All Rights Reserved
;;;
;;; This software is developed for the YY project of Aoyama Gakuin University.
;;; Permission to use, copy, modify, and distribute this software
;;; and its documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notices appear in all copies and that
;;; both that copyright notice and this permission notice appear in 
;;; supporting documentation, and that the name of Aoyama Gakuin
;;; not be used in advertising or publicity pertaining to distribution of
;;; the software without specific, written prior permission.
;;;
;;; This software is made available AS IS, and Aoyama Gakuin makes no
;;; warranty about the software, its performance or its conformity to
;;; any specification. 
;;;
;;; To make a contact: Send E-mail to ida@csrl.aoyama.ac.jp for overall
;;; issues. To ask specific questions, send to the individual authors at
;;; csrl.aoyama.ac.jp. To request a mailing list, send E-mail to 
;;; yyonx-request@csrl.aoyama.ac.jp.
;;;
;;; Authors:
;;;   version 1.0 90/06/01 by t.kosaka (kosaka@csrl.aoyama.ac.jp)
;;;   version 1.1 90/07/31 by t.kosaka
;;;   update 1.11 90/09/14 by t.kosaka
;;;   version 1.2 90/11/05 by t.kosaka
;;;   version 1.3 90/12/19 by t.kosaka
;;;   version 1.3 91/01/22 by t.kosaka

;;; Drawing Method
;;; 1990/03/23 written by Yukio Ohta
;;; Update 1990/06/06 written by T.Kosaka
;;; draw-position,draw-position-xy,draw-line,draw-line-xy,
;;; draw-region,draw-region-xy,draw-polygon,draw-polygon-xy,
;;; draw-polyline,draw-poly-line-xy,draw-triange,draw-teriangle-xy,
;;; draw-ellipse,draw-ellipse-xy,draw-circle,draw-circle-xy,
;;; draw-string,draw-string-xy,draw-character,draw-character-xy,
;;; 
;;; Version 1.3 Add after method at drawing primitive method.
;;;             Add method for real-primitive function like a yy-protocol-xx.
;;              Becouse presentaion use grass territory and I can not draw
;;;             wold territory. So, I need some after drawing mehtod 
;;;             for real primitive method.
;;;             Also, Add method for image handling.
(in-package :yy)


;;; $@E@$NIA2h(J 
;;; draw-position-xy graphic-stream x y
;;; ASG. 
;;;         graphic-stream     = $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;         x y                = $@0LCV(J
(defmethod draw-position-xy ((stream graphic-stream) x y)
  (draw-position-xy-internal stream x y))

;;; draw-position$@IA2h4X?t$N%W%j%_%F%#%V(J
(defun draw-position-xy-internal (stream x y)
  (declare 
    #-CMU
    (inline + max min)
    (integer x y)
    (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-translate-transform-xy 
    ((new-x new-y) stream x  y)
    (with-temp-region-args 
      ((draw-ponit-region) (work-region1 stream)
       :left new-x :right new-x
       :top new-y :bottom new-y)
      (with-slots 
	((tno world-territory-no) presentation-instance
	 (drawing-region drawing-region)) stream
	(with-graphic-state-slots
	  (operation color) stream
         ;;; $@%o!<%k%I%j!<%8%g%s$NJQ99(J
	  (setf (world-region stream) draw-ponit-region)

       ;;; drawing-region$@$K:#$N%j!<%8%g%s$rDI2C(J
	  (set-drawing-region drawing-region draw-ponit-region)

	  ;;; $@%W%l%<%s%F!<%7%g%s$,$"$l$P!"%;%C%H$9$k(J
	  ;;; $@$J$1$l$P!"IA2h(J
	  (make-point-object presentation-instance
			     (list tno (list new-x new-y)
				   operation (color-no color))
				 stream)
	  stream)))))

;;; $@E@$NIA2h(J $@%]%8%7%g%sMQ(J
;;; draw-position graphic-stram position
;;; ARG.
;;;             graphci-stream   =  $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;             psosition        =  $@%0%i%U%#%/%9%9%H%j!<%`$G$N0LCV(J
(defmethod draw-position ((stream graphic-stream) (position position))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots (x y) position
    (draw-position-xy-internal
      stream x y)))

;;; $@@~$NIA2h(J
;;; draw-line-xy stream x1 y1 x2 y2
;;; ARG.
;;;               stream        =   $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;               x1 y1 x2 y2   =   $@@~$N:BI8(J
(defmethod draw-line-xy ((stream graphic-stream) x1 y1 x2 y2)
  (draw-line-xy-internal stream x1 y1 x2 y2))

;;; draw-line$@IA2h4X?t$N%W%j%_%#%F%V(J
(defun draw-line-xy-internal (stream x1 y1 x2 y2)
  (declare 
    #-CMU
    (number x1 x2 y1 y2)
    (inline + max min)
    (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-translate-transform-xy 
    ((new-x1 new-y1) stream x1 y1)
    (with-translate-transform-xy 
      ((new-x2 new-y2) stream x2 y2)
      (with-temp-region-args 
	((draw-line-region) (work-region1 stream)
	 :left (min new-x1 new-x2)
	 :top (max new-y1 new-y2)
	 :right (max new-x1 new-x2)
	 :bottom (min new-y1 new-y2))
	(with-slots
	  ((tno world-territory-no) presentation-instance
	   (drawing-region drawing-region)) stream
	  ;; graphic-statement
	  (with-graphic-state-slots
	    (operation line-width line-edge (color graphic-color)
		       (line-dash line-dashing)) stream

	    (let ((ll line-width))
	      
	      (with-region-slots 
		   (left bottom right top) draw-line-region
		   (decf left ll)
		   (decf bottom ll)
		   (incf right ll)
		   (incf top ll)))
	      
         ;;; $@%o!<%k%I%j!<%8%g%s$NJQ99(J
	    (setf (world-region stream) draw-line-region)

       ;;; drawing-region$@$K:#$N%j!<%8%g%s$rDI2C(J
	    (set-drawing-region drawing-region draw-line-region)
	    (make-line-object presentation-instance
			      (list tno (list new-x1 new-y1 new-x2 new-y2)
				    line-width operation line-edge
				    (color-no color) line-dash)
				  stream)
	    stream))))))


;;; $@@~$NIA2h(J $@%]%8%7%g%sMQ(J
;;; draw-line stream position1 position2
;;; ARG
;;;           stream              =   $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;           position1 position2 =   $@@~$NIA2h$9$k0LCV%$%s%9%?%s%9(J
(defmethod draw-line ((stream graphic-stream) (position1 position)
		      (position2 position))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-slots 
   ((x1 x) (y1 y)) position1
   (with-slots 
    ((x2 x) (y2 y)) position2
    (draw-line-xy-internal stream x1 y1 x2 y2))))


;;; $@6k7A$NIA2h(J
;;; draw-region-xy stream x y width height
;;; ARG.
;;;           stream           =  $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;           x y              =  $@6k7A$rIA2h$9$k3+;O0LCV(J
;;;           width height     =  $@6k7A$NBg$-$5(J
(defmethod draw-region-xy ((stream graphic-stream) x y width height)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (draw-region-xy-internal stream x y width height))

;;; draw-region-xy $@IA2h%W%j%_%F%#%V(J
(defun draw-region-xy-internal (stream x y width height)
  (declare 
   #-CMU
   (number x y width height)
   (inline + min max)
   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-translate-transform-xy 
   ((new-x new-y) stream x y)
   (with-temp-region-args 
	((draw-region-region) (work-region1 stream)
	 :left new-x :width width
	 :bottom new-y :height height)
	(with-slots 
	 ((tno world-territory-no)
	  (drawing-region drawing-region)
	  presentation-instance) stream
	  (let ((filled-pattern (if (null (filled-pattern stream))
				    0
				  (filled-pattern stream))))
		;;; graphic-statement
	    (with-graphic-state-slots
	     (color (op operation) 
		    line-width filled-type (line-dash line-dashing)) 
	     stream
	     (let ((ll (round (/ line-width 2))))
	       (with-region-slots 
		(left bottom right top) draw-region-region
		(decf left ll)
		(decf bottom ll)
		(incf right ll)
		(incf top ll)))
	     
	     ;; $@%o!<%k%I%j!<%8%g%s$NJQ99(J
	     (setf (world-region stream) draw-region-region)

	     (let ((cl-no (color-no color)))
	       ;; $@IA2h%W%j%_%F%#%V(J
	       (cond
		;; draw region without pattern
		((eql filled-type *fillednon*)
		 (make-region-object presentation-instance
				     (list tno (list x y) width height line-width
					   op cl-no line-dash)
				     stream))
			;; draw rectabglw filled color
			((null filled-pattern)
			 (make-region-object-fill presentation-instance
                                 (list tno (list x y) width height
									   op cl-no 0)
								 stream))
			;; draw rectangle filled pattern
			(t
			 (make-region-object-fill presentation-instance
									  (list tno (list x y) width height
                                       op cl-no filled-pattern)
									  stream))
			 )))
		 )
		;;; drawing-region$@$K:#$N%j!<%8%g%s$rDI2C(J
		(set-drawing-region drawing-region draw-region-region)
		stream))))

		 
;;; $@6k7A$NIA2h%]%8%7%g%sMQ(J
;;; draw-region stream position width height
;;; ARG.
;;;             stream            =   $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;             position          =   $@6k7A$rIA2h$9$k3+;O0LCV%$%s%9%?%s%9(J
;;;             width height      =   $@6k7A$NBg$-$5(J
(defmethod draw-region ((stream graphic-stream) (pos position)
			 width height)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (draw-region-xy-internal stream (position-x pos) (position-y pos)
		  width height))

;;; $@6k7A$NIA2h(J $@%j!<%8%g%sMQ(J
;;; draw-region-region stream region
;;; ARG.
;;;             stream           = $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;             region           = $@6k7A$N0LCV$HBg$-$5$r;}$D%$%s%9%?%s%9(J
(defmethod draw-region-region ((stream graphic-stream) (region region))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-region-slots 
   (left bottom right top) region
   (draw-region-xy-internal stream left bottom
		  (- right left) (- top bottom))))


;;; $@@^$l@~$NIA2h(J      
;;; draw-polyline-xy stream x xy-list
;;; ARG.
;;;          stream            = $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;          x                 = $@:G=i$N0LCV(J
;;;          xy-list           = $@0LCV(J
(defmethod draw-polyline-xy ((stream graphic-stream) x &rest xy-list)
  (draw-polyline-xy-internal stream (push x xy-list)))


;;; $@@^$l@~$NIA2h(J      
;;; draw-polyline-xy stream xy-list
;;; ARG.
;;;          stream            = $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;          xy-list           = $@0LCV$N%j%9%H(J
(defmethod draw-polyline-xy ((stream graphic-stream) (pp list) &rest args)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0))
		   (ignore args))
  (draw-polyline-xy-internal stream pp))


;;; draw-polyline-xy$@IA2h%W%j%_%F%#%V(J
(defun draw-polyline-xy-internal (stream xy-list)
  (declare 
   (inline + < > man min / car cdr)
   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-temp-region-args 
   ((draw-polyline-region) (work-region1 stream))
   (let* ((tno (world-territory-no stream))
		  (drawing-region (drawing-region stream))
		  (max-x most-negative-fixnum) (max-y most-negative-fixnum)
		  (min-x most-positive-fixnum) (min-y most-positive-fixnum)
		  ;; graphic-statement
		  (color (color-no (graphic-color stream))))
    (with-graphic-state-slots
     (operation line-width line-edge (line-joint-type joint-type)
 	 (line-dash line-dashing)) stream

	 ;; $@:BI8%j%9%H$N:BI8CMJQ99(J
	 (do ((xy xy-list (cddr xy)))
	     ((null xy))
	     (multiple-value-bind
	      (nx ny)
	      (translate-transform-xy-valuse stream (car xy)
					     (second xy))
	      (if (>= nx max-x)
		  (setf max-x nx))
	      (if (>= ny max-y)
		  (setf max-y ny))
	      (if (< nx min-x)
		  (setf min-x nx))
	      (if (< ny min-y)
		  (setf min-y ny))
	      (setf (car xy) nx 
		    (second xy) ny)
	      ))
	 
	 ;; $@IA2h%j!<%8%g%s$r@_Dj(J
	 (with-region-slots 
	  (left bottom right top) draw-polyline-region
	  (let ((ll line-width))
	    (setf left (- min-x ll)  bottom (- min-y ll)
			  right (+ max-x line-width) top (+ max-y ll))))
	 ;; $@%o!<%k%I%j!<%8%g%s$NJQ99(J
	 (setf (world-region stream) draw-polyline-region)

	 (with-slots 
	  (presentation-instance) stream
	  ;; $@@^$l@~$rIA2h(J
	 (make-polyline-object 
	  presentation-instance 
	  (list tno xy-list line-width operation line-edge
		line-joint-type color line-dash)
	  stream)
	 
	 ;; drawing-region$@$K:#$N%j!<%8%g%s$rDI2C(J
	 (set-drawing-region drawing-region draw-polyline-region)
	 stream)))))

;;; add-value-for-xy-list
;;; XY$@$N%j%9%H$KBP$7$F!"(Jdx,dy$@$r3F!92C$($k(J
(defun add-value-for-xy-list (xy-list dx dy)
  (declare 
   #-CMU
   (inline incf cddr car second)
   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (do ((xy xy-list (cddr xy)))
	  ((null xy))
	  (incf (car xy) dx)
	  (incf (second xy) dy))
  xy-list)
	    
;;; $@@^$l@~$NIA2h(J $@%]%8%7%g%sMQ(J
;;; draw-polyline-xy stream position-list
;;; ARG.
;;;          stream           =  $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;          postion          =  $@:G=i$N%]%8%7%g%s(J
;;           position-list    =  $@%]%8%7%g%s%$%s%9%?%s%974(J
(defmethod draw-polyline ((stream graphic-stream) (pos position) 
						  &rest position-list)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (push pos position-list)
  (apply #'draw-polyline-xy-internal stream (reduce #'nconc 
			    (map 'list #'(lambda (x) (list (position-x x)
				 			  (position-y x)))
					position-list))))

;;; $@B?3Q7A$NIA2h(J
;;; draw-polygon-xy stream xy-list
;;; ARG.
;;;         stream            =    $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;         xy-list           =    $@0LCV(J(x y)$@$N%j%9%H(J
(defmethod draw-polygon-xy ((stream graphic-stream) (pp list) &rest rest)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0))
		   (ignore rest))
  (draw-polygon-xy-internal stream pp))


;;; draw-polygon-xy$@IA2h%W%j%_%#%F%V(J
(defun draw-polygon-xy-internal (stream xy-list)
  (declare 
   #-CMU
   (inline + < > min max / car cdr)
   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (with-temp-region-args 
   ((draw-polygon-region) (work-region1 stream))
   (let* ((tno (world-territory-no stream))
	  (drawing-region (drawing-region stream))
	  (max-x most-negative-fixnum) (max-y most-negative-fixnum)
	  (min-x most-positive-fixnum) (min-y most-positive-fixnum)
	  (new-xy-list (copy-seq xy-list))
	  (cl-no 0))
     ;; graphic-statement
     (with-graphic-state-slots
      (color operation line-width filled-type filled-rule filled-pattern
	     (line-joint-type joint-type) (line-dash line-dashing)) stream

	     ;; $@:BI8%j%9%H$N:BI8CMJQ99(J
	     (do ((xy new-xy-list (cddr xy)))
		 ((null xy))
		 (multiple-value-bind
		  (nx ny)
		  (translate-transform-xy-valuse stream (car xy)
						 (second xy))
		  (if (> nx max-x)
		      (setf max-x nx))
		  (if (> ny max-y)
		      (setf max-y ny))
		  (if (< nx min-x)
		      (setf min-x nx))
		  (if (< ny min-y)
		      (setf min-y ny))
		  (setf (car xy) nx 
			(second xy) ny)
		  ))
      ;;; $@IA2h%j!<%8%g%s$r@_Dj(J
      (with-region-slots 
       (left bottom right top) draw-polygon-region

       (let ((ll line-width))
		 (setf left (decf min-x ll) bottom (decf min-y ll)
			   right (incf max-x ll) top (incf max-y ll))))
	  
      ;;; $@%o!<%k%I%j!<%8%g%s$NJQ99(J
      (setf (world-region stream) draw-polygon-region)
      
      (with-slots 
       (presentation-instance) stream
       ;; $@IA2h%W%j%_%F%#%V(J
       (setf cl-no (color-no color))
	  
       (cond
	;; draw polygon without filled
	((eql filled-type *fillednon*)
	 (make-polygon-object presentation-instance
			      (list tno new-xy-list line-width operation 
				    line-joint-type cl-no line-dash)
			      stream))
	;; draw ploygon filled color
	((null filled-pattern)
	 (make-polygon-object-fill presentation-instance
				   (list tno new-xy-list operation 
					 line-joint-type cl-no filled-rule 0)
				   stream))
	;; draw polygon filled pattern
	(t
	 (make-polygon-object-fill presentation-instance
				   (list tno new-xy-list operation 
					 line-joint-type cl-no 
					 filled-rule filled-pattern)
				   stream))
		)
	   )
	  )
    ;;; drawing-region$@$K:#$N%j!<%8%g%s$rDI2C(J
    (set-drawing-region drawing-region draw-polygon-region)
    stream)))

;;; $@B?3Q7A$NIA2h(J $@%]%8%7%g%sMQ(Jno uchigawa de kaku
;;; draw-polygon-inside stream position-list
;;; ARG.
;;;           stream           =  $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;           pps              =  $@:G=i$N%]%8%7%g%s(J
;;;           position-list    =  $@0LCV%$%s%9%?%s%974(J
(defmethod draw-polygon-inside ((stream graphic-stream) x-inside y-inside
				pos &rest position-list)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let* ((max-x -9999) (max-y -9999) (min-x 99999) (min-y 99999)
	 (xy-list 
	   (reduce #'nconc 
		   (map 'list #'(lambda (pos) 
				  (with-slots
				    (x y) pos
				    (if (< x min-x)
					(setf min-x x))
				    (if (< y min-y)
					(setf min-y y))
				    (if (> x max-x)
					(setf max-x x))
				    (if (> y max-y)
					(setf max-y y))
				    (list x y)))
			(push pos position-list))))
	 (center-x (float (/ (+ max-x min-x) 2)))
	 (center-y (float (/ (+ max-y min-y) 2))))
    ;;(format t "~% ~a ~a" center-x center-y)
    (do ((xy xy-list (cddr xy)))
	((null xy))
      (let ((x (car xy))
	    (y (cadr xy))
	    (xx x-inside)
	    (yy y-inside)
	    (x- (- center-x x-inside x-inside))
	    (x+ (+ center-x x-inside x-inside))
	    (y- (- center-y y-inside y-inside))
	    (y+ (+ center-y y-inside y-inside)))
	(if (< x- x x+) (setf yy (* yy 2)))
	(if (< y- y y+) (setf xx (* xx 2)))
	(if (< x x-)
	    (setf (car xy) (+ x xx))
	    (if (> x x+)
		(setf (car xy) (- x xx))))
	(if (< y y-)
	    (setf (second xy) (+ y yy))
	    (if (> y y+)
		(setf (second xy) (- y yy))))))
    (draw-polygon-xy-internal  stream xy-list)
    ))


;;; $@B?3Q7A$NIA2h(J $@%]%8%7%g%sMQ(J
;;; draw-polygon stream position-list
;;; ARG.
;;;           stream           =  $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;           pps              =  $@:G=i$N%]%8%7%g%s(J
;;;           position-list    =  $@0LCV%$%s%9%?%s%974(J
(defmethod draw-polygon ((stream graphic-stream) (pos position) 
			 &rest position-list)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (push pos position-list)
  (draw-polygon-xy-internal
   stream
   (reduce #'nconc 
	   (map 'list #'(lambda (x) (list (position-x x)
					  (position-y x)))
		position-list)))
  )

;;; $@;03Q7A$NIA2h(J   ($@FsEyJU;03Q7A(J)
;;; draw-triangle-xy stream x y width height
;;; ARG.
;;;           stream            =   $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;           x y               =   $@DcJU$NCf?40LCV(J 
;;;           width             =   $@DcJU$NI}(J
;;;           height            =   $@9b$5(J
(defmethod draw-triangle-xy ((stream graphic-stream) x y width height)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (draw-triangle-xy-internal stream x y width height))

;;; draw-triangle-xy $@IA2h%W%j%_%F%#%t(J
(defun draw-triangle-xy-internal (stream x y width height)
  (declare 
  (number x y widht height)
   #-CMU
   (inline - / + round)
   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (draw-polygon-xy-internal stream 
		   (list 
		    (- x (round (/ width 2))) y (+ x (round (/ width 2))) y
		   x (+ y height)))
  )

;;; $@;03Q7A$NIA2h(J $@%]%8%7%g%sMQ(J ($@FsEyJU;03Q7A(J)
;;; draw-triangle stream position width height
;;; ARG.
;;;              stream            = $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;              position          = $@DcJU$NCf?40LCV%$%s%9%?%s%9(J
;;;              width             = $@;03Q7A$NI}(J
;;;              height            = $@9b$5(J
(defmethod draw-triangle ((stream graphic-stream) (position position)
			  width height)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (draw-triangle-xy-internal stream (position-x position) (position-y position)
		   width height))


;;; $@BJ1_!"BJ1_8L!"BJ1_$N@p7?$NIA2h(J
;;; draw-ellipse-xy stream center-x center-y x-radius y-radius 
;;;                          &key (start-angle 0)
;;;                               (end-angle 2pi)
;;; ARG.
;;;               stream     =  $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;     center-x center-y    =  $@BJ1_$NCf?4(J
;;;               x-radius   =  $@2#<4$K$*$1$kH>7B(J
;;;               y-radius   =  $@=D<4$K$*$1$kH>7B(J
;;;          start-angle     =  $@3+;O3QEY(J($@%i%G%#%"%s(J)
;;;          end-angle       =  $@=*N;3QEY(J($@%i%G%#%"%s(J)
;;;          clockwize       =  $@IA2hJ}8~(J
(defmethod draw-ellipse-xy ((stream graphic-stream) 
			    center-x center-y
			    x-radius y-radius
			    &key (start-angle 0.0)
			    (end-angle (* 2 pi)))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (draw-ellipse-xy-internal stream center-x center-y 
			    x-radius y-radius 
			    :start-angle start-angle
			    :end-angle end-angle))

;;; draw-ellipse-xy$@IA2h%W%j%_%F%#%V(J
(defun draw-ellipse-xy-internal (stream center-x center-y x-radius y-radius
				 &key (start-angle 0)
				 (end-angle (* 2 pi)))
  (declare 
   #-CMU
   (number center-x center-y x-radius y-radius)
   (inline * + - max min)
   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (case (stream-clockwize stream)
		((:counter-clockwize))
		((:clockwize)
		 (let ((tmp start-angle))
		   (setf start-angle end-angle
				 end-angle tmp)))
		(otherwise (error "Draw-ellipse : Unkown clockwize")))
  (with-translate-transform-xy 
   ((new-x new-y) stream center-x center-y)
   (with-temp-region-args 
	((draw-ellipse-region) (work-region1 stream)
	 :left (- new-x x-radius) :right (+ new-x x-radius)
	 :top (+ new-y y-radius) :bottom (- new-y y-radius))
	(with-slots
	 ((tno world-territory-no)) stream
	 (let* ((tno (world-territory-no stream))
			(drawing-region (drawing-region stream))
			(cl-no 0))
	   ;; graphic-statement
	   (with-graphic-state-slots
		(color operation (width line-width) (dash line-dashing)
		       filled-type filled-pattern arc-mode) stream
		       ;; 1/64 $@$r$?$s$$$H$9$k!#(J
		       (setf start-angle (round (* (/ start-angle  pi) 180 64))
			     end-angle (round (* (/ end-angle pi) 180 64)))
			   
		       (let ((ll (round (/ width 2))))
			 (with-region-slots 
			  (left bottom right top) draw-ellipse-region
			  (decf left ll)
			  (decf bottom ll)
			  (incf right width)
			  (incf top ll))
			 )
		       ;; $@%o!<%k%I%j!<%8%g%s$NJQ99(J
		       (setf (world-region stream) draw-ellipse-region)

               ;;; $@0LCV$NJQ99(J
		       (with-slots
			(presentation-instance) stream
			
			(setf  cl-no (color-no color))
				 
			;; $@IA2h%W%j%_%F%#%V(J
			(cond
			 ((eql filled-type *fillednon*)
			  (make-ellipse-object 
			   presentation-instance
			   (list tno (list new-x new-y)
				 (* x-radius 2) (* y-radius 2)
				 start-angle end-angle
				 width operation cl-no dash)
			   stream))
			 ;; draw ellipse filled color
			 ((null filled-pattern)
			  (make-ellipse-object-fill 
			   presentation-instance
			   (list tno (list new-x new-y)
				 (* x-radius 2) (* y-radius 2)
				 start-angle end-angle
				 operation cl-no 0 arc-mode)
			   stream))
			 ;; draw ellipse filled pattern
			 (t
			  (make-ellipse-object-fill
			   presentation-instance
			   (list tno (list new-x new-y)
				 (* x-radius 2) (* 2 y-radius) 
				 start-angle end-angle
				 operation cl-no filled-pattern arc-mode)
			   stream))
			 )))
	   ;; drawing-region$@$K:#$N%j!<%8%g%s$rDI2C(J
	   (set-drawing-region drawing-region draw-ellipse-region)
	   stream)))))

;;; $@BJ1_!"BJ1_8L!"BJ1_$N@p7?$NIA2h(J
;;; draw-ellipse   stream posistion x-radius y-radius 
;;;                         &key (start-angle 0)
;;;                               (end-angle 2pi) (clockwize :clockwize)
;;; ARG.
;;;               stream     =  $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;               position   =  $@BJ1_$NCf?4$N0LCV%$%s%9%?%s%9(J
;;;               x-radius   =  $@2#<4$K$*$1$kH>7B(J
;;;               y-radius   =  $@=D<4$K$*$1$kH>7B(J
;;;          start-angle     =  $@3+;O3QEY(J($@%i%G%#%"%s(J)
;;;          end-angle       =  $@=*N;3QEY(J($@%i%G%#%"%s(J)
;;;          clockwize       =  $@IA2hJ}8~(J
(defmethod draw-ellipse ((stream graphic-stream) (position position)
			    x-radius y-radius &key (start-angle 0.0)
			    (end-angle (* 2 pi)) (clockwize :clockwize))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (setf (stream-clockwize stream) clockwize)
  (draw-ellipse-xy-internal
   stream (position-x position) (position-y position)
   x-radius y-radius :start-angle start-angle
   :end-angle end-angle))

;;; $@1_!"@p7?!"1_8L$NIA2h(J
;;; draw-circle-xy stream center-x center-y radius &key (start-angle 0)
;;;		(end-angle 2pi) 
;;; ARG.
;;;           stream             =  $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;       center-x center-y      =  $@1_$NCf?4(J
;;;          radius              =  $@1_$NH>7B(J
;;;          start-angle         =  $@3+;O3QEY(J($@%i%G%#%"%s(J)
;;;          end-angle           =  $@=*N;3QEY(J($@%i%G%#%"%s(J)
(defmethod draw-circle-xy ((stream graphic-stream) center-x
			   center-y radius  &key (start-angle 0.0) 
			   (end-angle (* 2 pi)))

  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (case (stream-clockwize stream)
	((:counter-clockwize))
	((:clockwize)
	 (let ((tmp start-angle))
	   (setf start-angle end-angle
		 end-angle tmp)))
	(otherwise (error "Draw-circle : Unkown clockwize")))

    ;;; draw-circle-intenal
    (draw-circle-xy-internal stream 
							 center-x center-y radius start-angle end-angle))

;;; draw-circle-xy-internal stream center-x center-y radius start-angle
;;;				 end-angle
(defun draw-circle-xy-internal (stream center-x center-y radius start-angle
				end-angle)
  (declare 
   #-CMU
   (inline + * max min)
   (optimize 
    (number center-x center-y radius start-angle end-angle)
    (compilation-speed 0) (speed 3) (safety 0)))
  (with-translate-transform-xy ((new-x new-y) stream center-x center-y)
    (with-temp-region-args ((draw-circle-region) (work-region1 stream))
       (setf draw-circle-region (get-draw-circle-region new-x new-y
		   radius (line-width stream)
		   start-angle end-angle draw-circle-region (filled-type stream)))
	   (with-slots 
		(presentation-instance  (tno world-territory-no) 
		 drawing-region) stream
	     ;; graphic-statement
		 (with-graphic-state-slots
		  (color operation (width line-width) (dash line-dashing)
		   filled-type filled-pattern arc-mode) stream
		   ;; 1/64 $@$r$?$s$$$H$9$k!#(J
		   (setf start-angle (round (* (/ start-angle pi) 180 64))
				 end-angle (round (* (/ end-angle pi) 180 64)))
		   (let ((ll (round (/ width 2))) (cl-no (color-no color)))
			 (with-region-slots 
			  (left bottom right top) draw-circle-region
			  (decf left ll) (decf bottom ll) (incf right ll) (incf top ll))
              ;;; $@%o!<%k%I%j!<%8%g%s$NJQ99(J
			 (setf (world-region stream) draw-circle-region)

              ;;; call draw-circle-primitive
			 (cond
			  ;; $@1_$NOH5Z$S1_8L$NOH(J
			  ((eql filled-type *fillednon*)
			   (if (and (= start-angle 0) (= end-angle 23040))
				   ;;; $@1_$N>l9g(J
				   (make-circle-object 
					presentation-instance
					(list tno  (list new-x new-y)
						  radius width operation cl-no dash)
					stream)
				 ;;; $@1_8L$N>l9g(J
				 (make-arc-object 
				  presentation-instance
				  (list tno (list new-x new-y) radius start-angle 
						end-angle width operation cl-no dash)
				  stream))
			   )
			  ;; $@EI$j$D$V$71_5Z$S1_8L(J ($@;XDj?'$GEI$j$D$V$7(J)
			  ((null filled-pattern)
			   (if (and (= start-angle 0) (= end-angle 23040))
				   (make-fill-circle-object 
					presentation-instance
					(list tno (list  new-x new-y) radius operation cl-no 0)
					stream)
				 ;; $@1_8L(J
				 (make-fill-arc-object 
				  presentation-instance
				  (list tno (list  new-x new-y) radius start-angle 
						end-angle operation cl-no 0 arc-mode)
				  stream)))
			  ;; $@EI$j$D$V$71_5Z$S1_8L(J ($@;XDj%Q%?!<%s$GEI$j$D$V$7(J)
			  (t
			   (if (and (= start-angle 0) (= end-angle 23040))
				   ;; $@1_(J
				   (make-fill-circle-object 
					presentation-instance
					(list tno (list  new-x new-y) radius operation cl-no
						  filled-pattern)
					stream)
				 ;; $@1_8L(J
				 (make-fill-arc-object 
				  presentation-instance
				  (list tno (list  new-x new-y) radius start-angle 
						end-angle operation cl-no filled-pattern arc-mode)
				  stream)))
			  ))
            ;;; drawing-region$@$K:#$N%j!<%8%g%s$rDI2C(J
		   (set-drawing-region drawing-region draw-circle-region)
		   stream)))))

;;; $@1_!"@p7?!"1_8L$NIA2h(J
;;; draw-circle stream position radius &key (start-angle 0)
;;;		(end-angle 2pi) clockwize
;;; ARG.
;;;           stream             =  $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;          position            =  $@1_$NCf?40LCV$N%$%s%9%?%s%9(J
;;;          radius              =  $@1_$NH>7B(J
;;;          start-angle         =  $@3+;O3QEY(J($@%i%G%#%"%s(J)
;;;          end-angle           =  $@=*N;3QEY(J($@%i%G%#%"%s(J)
;;;          clockwize           =  $@IA2hJ}8~(J
(defmethod draw-circle ((stream graphic-stream) (position position)
			(radius integer) &key (start-angle 0)
			(end-angle (* 2 pi)))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (case (stream-clockwize stream)
    ((:counter-clockwize))
    ((:clockwize)
     (let ((tmp start-angle))
       (setf start-angle end-angle
	     end-angle tmp)))
    (otherwise (error "Draw-circle : Unkown clockwize ~s")))
  (draw-circle-xy-internal
                  stream (position-x position) (position-y position) radius
                  start-angle end-angle))


;;; draw-circle $@$N%j!<%8%g%s$r5a$a$k(J
;;;              |  /
;;;              | /
;;;              |/  start
;;;      --------+----------
(defun get-draw-circle-region (x y r line-w start end region fill-mode)
  (declare 
   #-CMU
   (inline + / <= max min sin cos round)
   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let* ((rr (if fill-mode
				r
			   (+ r line-w)))
		 (start-x (abs (* (cos start) rr)))
		 (start-y (abs (* (sin start) rr)))
		 (end-x (abs (* (cos end) rr)))
		 (end-y (abs (* (sin end) rr))))
	(cond
	 ((<= 0.0 start (/ pi 2)) ;; $@3+;OE@$,Bh0l>]8=(J
	  (start-1-shougen x y r start end start-x start-y 
					   end-y end-x fill-mode line-w region))
	 ((<= (/ pi 2) start pi) ;;$@3+;OE@$,BhFs>]8=(J
	  (start-2-shougen x y r start end start-x start-y 
					   end-y end-x fill-mode line-w region))
	 ((<= pi start (+ pi (/ pi 2))) ;;$@3+;OE@$,Bh;0>]8=(J
	  (start-3-shougen x y r start end start-x start-y 
					   end-y end-x fill-mode line-w region))
	 ((<= (+ pi (/ pi 2)) start (* pi 2)) ;;$@3+;OE@$,Bh;M>]8=(J
	  (start-4-shougen x y r start end start-x start-y 
					   end-y end-x fill-mode line-w region))
	 )
	))

;;; $@3+;OE@$,Bh0l>]8=$K$"$k>l9g(J
;;; start-x start-y end-x end-y$@$O!"@dBPCM$,$/$k(J
(defun start-1-shougen (x y r start end start-x start-y 
						  end-y end-x fill-mode line-w region)

  (let ((x1 0) (y1 0) (w 0) (h 0))
	(cond
	 ((<= 0.0 end (/ pi 2)) ;; $@=*N;E@$,Bh0l>]8=(J
	  (if (< start end)
		  (setf x1 (if fill-mode
					   0
					 (round end-x))
				w (abs (- x1 (round start-x)))
				y1 (- (round end-y))
				h (if fill-mode
					  (round end-y)
						 (abs (round (- end-y start-y))))
				)
		(setf x1 (- (round r)) w (* 2 r)
			  y1 (- (round r)) h (* 2 r)))
	  )
	 ((<= (/ pi 2) end pi) ;;$@=*N;E@$,BhFs>]8=(J
	  (setf x1 (- (round end-x))
			w (abs (round (+ start-x end-x)))
			y1 (- r)
			h (if fill-mode
				  r
				(- r (round (min start-y end-y)))))
	  )
	 ((<= pi end (+ pi (/ pi 2))) ;;$@=*N;E@$,Bh;0>]8=(J
	  (setf x1 (- r)
			y1 (- r)
			w (+ r (round start-x))
			h (+ r (round end-y)))
	 )
	 ((<= (+ pi (/ pi 2)) end (* pi 2)) ;;$@=*N;E@$,Bh;M>]8=(J
	  (setf x1 (- r)
			y1 (- r)
			w (+ r (round (max start-x end-x)))
			h (* 2 r))
	  ))
	(if fill-mode
		(with-region-slots
		 (left bottom width height) region
		 (setf left (+ x1 x) bottom (+ y1 y)
			   width w height h)
		 )
		(with-region-slots
		 (left bottom width height) region
		 (setf left (+ (- x1 line-w) x) bottom (+ (- y1 line-w) y)
			   width (+ w (* 2 line-w)) height (+ (* 2 line-w) h))
		 ))
	region))

;;; $@3+;OE@$,BhFs>]8=$K$"$k>l9g(J
(defun start-2-shougen (x y r start end start-x start-y end-y end-x 
						  fill-mode line-w region)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((x1 0) (y1 0) (w 0) (h 0))
	(cond
	 ((<= (/ pi 2) end pi) ;;$@=*N;E@$,BhFs>]8=(J
	  (if (< start end)
		  (setf x1 (- (round end-x))
				y1 (- (round start-y))
				w (if fill-mode
					  (round end-x)
					(abs (round (- start-x end-x))))
				h (if fill-mode
					  (round start-y)
					(abs (round (- start-y end-y)))))
		(setf x1 (-  r) y1 (-  r)
			  w (* 2 r) h (* 2 r)))
	  )
	 ((<= pi end (+ pi (/ pi 2))) ;;$@=*N;E@$,Bh;0>]8=(J
	  (setf x1 (- r)
			y1 (- (round start-y))
			w (if fill-mode
				  r
				(abs (- r (round (min start-x end-x)))))
			h (round (+ start-x end-x)))
	 )
	 ((<= (+ pi (/ pi 2)) end (* pi 2)) ;;$@=*N;E@$,Bh;M>]8=(J
	  (setf x1 (- r)
			y1 (- (round start-y))
			w (round (+ start-x end-x))
			h (round (+ r start-y)))
	  )
	 ((<= 0.0 end (/ pi 2)) ;; $@=*N;E@$,Bh0l>]8=(J
	  (setf x1 (- r)
			y1 (- (round (max start-y end-y)))
			w (* 2 r)
			h (+ r (- y1)))
	  )
	 )
	(if fill-mode
		(with-region-slots
		 (left bottom width height) region
		 (setf left (+ x1 x) bottom (+ y1 y)
			   width w height h)
		 )
		(with-region-slots
		 (left bottom width height) region
		 (setf left (+ (- x1 line-w) x) bottom (+ (- y1 line-w) y)
			   width (+ w (* 2 line-w)) height (+ (* 2 line-w) h))
		 ))
	region))

;;; $@3+;OE@$,Bh;0>]8=$K$"$k>l9g(J
(defun start-3-shougen (x y r start end start-x start-y end-y end-x 
						  fill-mode line-w region)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((x1 0) (y1 0) (w 0) (h 0))
	(cond
	 ((<= pi end (+ pi (/ pi 2))) ;;$@=*N;E@$,Bh;0>]8=(J
	  (if (< start end)
		  (setf x1 (- (round start-x))
				y1 (if fill-mode
					   0
					   (round start-y))
				w  (if fill-mode
					   start-x
					 (abs (round (- start-x end-x))))
				h  (if fill-mode
					   end-y
					 (abs (round (- start-y end-y)))))
		(setf x1 (-  r) y1 (-  r)
			  w (* 2 r) h (* 2 r)))
	 )
	 ((<= (+ pi (/ pi 2)) end (* pi 2)) ;;$@=*N;E@$,Bh;M>]8=(J
	  (setf x1 (- (round start-x))
			y1 (if fill-mode
				   0
				   (round (min start-y end-y)))
			w (abs (round (+ start-x end-x)))
			h (if fill-mode
				  r
				(abs (- r y1))))
	  )
	 ((<= 0.0 end (/ pi 2)) ;; $@=*N;E@$,Bh0l>]8=(J
	  (setf x1 (- (round start-x))
			y1 (- (round end-y))
			w (round (+ start-x r))
			h (round (+ r end-y)))
	  )
	 ((<= (/ pi 2) end pi) ;;$@=*N;E@$,BhFs>]8=(J
	  (setf x1 (- (round (max start-x end-x)))
			y1 (- r)
			w (+ r (- x1))
			h (* r 2))
	  )
	 )
	(if fill-mode
		(with-region-slots
		 (left bottom width height) region
		 (setf left (+ x1 x) bottom (+ y1 y)
			   width w height h)
		 )
		(with-region-slots
		 (left bottom width height) region
		 (setf left (+ (- x1 line-w) x) bottom (+ (- y1 line-w) y)
			   width (+ w (* 2 line-w)) height (+ (* 2 line-w) h))
		 ))
	region))

;;; $@3+;OE@$,Bh;M>]8=$K$"$k>l9g(J
(defun start-4-shougen (x y r start end start-x start-y end-y end-x 
						  fill-mode line-w region)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((x1 0) (y1 0) (w 0) (h 0))
	(cond
	 ((<= (+ pi (/ pi 2)) end (* pi 2)) ;;$@=*N;E@$,Bh;M>]8=(J
	  (if (< start end)
		  (setf x1 (if fill-mode
					   0
					   (round start-x))
				y1 (if fill-mode
					   0
					 (round end-y))
				w (if fill-mode 
					  (round end-y)
					(abs (round (- start-x end-x))))
				h (if fill-mode
					  (round start-y)
					(abs (round (- start-y end-y)))))

		(setf x1 (-  r) y1 (-  r)
			  w (* 2 r) h (* 2 r)))
	  )
	 ((<= 0.0 end (/ pi 2)) ;; $@=*N;E@$,Bh0l>]8=(J
	  (setf x1 (if fill-mode
				   0
				   (round (min start-x end-x)))
			y1 (- (round end-y))
			w (if fill-mode
				  r
				(abs (- r x1)))
			h (round (+ start-y end-y)))
	  )
	 ((<= (/ pi 2) end pi) ;;$@=*N;E@$,BhFs>]8=(J
	  (setf x1 (round (- end-x))
			y1 (- r)
			w (round (+ r end-x))
			h (round (+ r start-y)))
	  )
	 ((<= pi end (+ pi (/ pi 2))) ;;$@=*N;E@$,Bh;0>]8=(J
	  (setf x1 (- r)
			y1 (- r)
			w (* r 2)
			h (round (+ r (max start-y end-y))))
	  )
	 )
	(if fill-mode
		(with-region-slots
		 (left bottom width height) region
		 (setf left (+ x1 x) bottom (+ y1 y)
			   width w height h)
		 )
		(with-region-slots
		 (left bottom width height) region
		 (setf left (+ (- x1 line-w) x) bottom (+ (- y1 line-w) y)
			   width (+ w (* 2 line-w)) height (+ (* 2 line-w) h))
		 ))
	region))

#|
  (let ((start-x-offset (round (* (cos start) r)))
  	(start-y-offset (round (* (sin start) r -1)))
	(end-x-offset (round (* (cos end) r)))
	(end-y-offset (round (* (sin end) r -1)))
	(left 0) (top 0) (right 0) (bottom 0))
    (with-translate-transform-xy ((new-x new-y) stream x y)
       (setf left (+ (if (<= start pi end)
    		      (- r)
		    (min start-x-offset end-x-offset))
		  new-x)
          bottom  (+ (if (<= start (+ pi (/ pi 2)) end)
	  	      (- r)
		    (min start-y-offset end-y-offset))
		  new-y)
	  right (+ (if (<= start 0 end)
	  	       r
		     (max start-x-offset end-x-offset))
		   new-x)
	  top (+ (if (<= start (/ pi 2) end)
	  		r
		      (max start-y-offset end-y-offset))
		    new-y))
       (with-region-slots
	((l1 left) (b1 bottom) (r1 right) (t1 top)) region
       (setf l1 left r1 right b1 bottom t1 top))
       region
       )
    ))


;;; $@8E:d$,JQ99(J
(defun get-draw-circle-region (stream new-x new-y r start end region)
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0))
		   (ignore stream start end))
  (with-region-slots
   ((l1 left) (b1 bottom) (r1 right) (t1 top)) region
   (setf l1 ( - new-x r) t1 (+ new-y r) r1 (+ new-x r) b1 (- new-y r)))
   region)
|#


;;; $@J8;zNs$NIA2h(J 
;;; draw-string-xy stream string x y &key start end direction
;;;            stream         = $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;            string         = $@IA2h$9$kJ8;zNs(J
;;;            x y            = $@IA2h$9$k0LCV(J
;;;            start          = $@J8;zNs$N3+;O0LCV(J(string$@$J$$$G(J
;;;            end            = $@J8;zNs$N=*N;0LCV(J(string$@$J$$$G(J)
;;;            direction      = $@IA2h$9$kJ}8~(J
;;; RET.
;;;      (string T or nil)
;;;       T --> $@2~9T$,$"$C$?(J
(defmethod draw-string-xy ((stream graphic-stream) (string string)
			x y 
			&key (start 0) (end (length string))
			(direction :horizontal))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (draw-string-xy-internal stream string x y start end direction))

;;; draw-string$@IA2h%W%j%_%F%#%V(J
(defun draw-string-xy-internal (stream string x y start end direction)
  (declare 
   (number x y start end )
   (special *root-window*) 
   #-CMU
   (inline =)
   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let* ((position (slot-value stream 'cursor-position))
	 (old-x (position-x  position))
	 (old-y (position-y  position))
	 (old-direction (stream-output-direction stream)))

    (setf (stream-cursor-x-position stream) x
	  (stream-cursor-y-position stream) y
	  (stream-output-direction stream) direction)

    ;;; $@=PNOJ}8~$rD4$Y$k(J
    (prog1 
	(if (eq direction :horizontal)
	    (progn 
	      (drawing-text-internal-yoko stream 
			  (subseq string start end))
	      (values string (if (= old-y (position-y position))
				 NIL
			       T)))
	  (progn 
	    (drawing-text-internal-tate stream (subseq string start end))
	    (values string (if (= old-x (position-x position))
			       NIL
			     T))))
      (setf (position-x position) old-x
	    (position-y position) old-y
	    (stream-output-direction stream) old-direction)
      )))
			       

;;; $@J8;zNs$NIA2h(J 
;;; draw-string stream string position &key start end direction
;;;            stream         = $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;            string         = $@IA2h$9$kJ8;zNs(J
;;;            position       = $@IA2h$9$k0LCV$N%$%s%9%?%s%9(J
;;;            start          = $@J8;zNs$N3+;O0LCV(J(string$@$J$$$G(J
;;;            end            = $@J8;zNs$N=*N;0LCV(J(string$@$J$$$G(J)
;;;            direction      = $@IA2h$9$kJ}8~(J
;;; RET.
;;;      (string T or nil)
;;;       T --> $@2~9T$,$"$C$?(J
(defmethod draw-string ((stream graphic-stream) (string string)
			(position position)
			&key (start 0) (end (length string))
			(direction :horizontal))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (draw-string-xy-internal stream string (position-x position)
		  (position-y position) start end direction))


;;; $@J8;z$NIA2h(J 
;;; draw-char-xy stream char x y &key direction
;;;            stream         = $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;            string         = $@IA2h$9$kJ8;z(J
;;;            x y            = $@IA2h$9$k0LCV(J
;;;            direction      = $@IA2h$9$kJ}8~(J
;;; RET.
;;;      (string T or nil)
;;;       T --> $@2~9T$,$"$C$?(J
(defmethod draw-character-xy ((stream graphic-stream) (char character)
			(x integer) (y integer)
			&key (direction :horizontal))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (multiple-value-bind (ret1 ret2)
	  (draw-string-xy-internal stream (string char) x y 
				   0 1 direction)
	  (declare (ignore ret1))
    
	  (values char ret2)))

;;; draw-character stream character position
;;; $@J8;z$NIA2h(J 
;;; Return $@CM(J (string T or nil)
;;; T --> $@2~9T$,$"$C$?(J

;;; $@J8;z$NIA2h(J 
;;; draw-char stream char position &key direction
;;;            stream         = $@%0%i%U%#%/%9%9%H%j!<%`(J
;;;            string         = $@IA2h$9$kJ8;z(J
;;;            position       = $@IA2h$9$k0LCV$N%$%s%9%?%s%9(J
;;;            direction      = $@IA2h$9$kJ}8~(J
;;; RET.
;;;      (string T or nil)
;;;       T --> $@2~9T$,$"$C$?(J
(defmethod draw-character ((stream graphic-stream) (char character)
			(position position)
			&key (direction :horizontal))
  (declare (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (draw-character-xy stream char (position-x position)
		     (position-y position) :direction direction))


;;; $@%W%m%s%W%H%&%#%s%I%&$X$NJ8;zNsI=<((J
;;; draw-prompt string
;;; ARG.
;;;             string        = $@I=<($9$kJ8;zNs(J
(defun draw-prompt (string)
  (declare (special *prompt-window* *white-color* *black-color* *total-send*
					*total-put-byte*)
	   (optimize (compilation-speed 0) (speed 3) (safety 0)))
  (let ((no (world-territory-no *prompt-window*))
		(c-no (color-no *white-color*))
		(b-no (color-no *black-color*)))
	(with-protocol
	 (setf *total-send* T
		   *total-put-byte* 0)
	 (yy-protocol-32 no c-no)
	 (yy-protocol-31 no 0 (font-kanji-base-line
						   (stream-font *prompt-window*))
					 #x03 b-no (font-no (stream-font *prompt-window*))
					 string)
	 (total-protocol-send-no-check))))
	 
	#| OLD definsition
  (clear-window-stream *prompt-window*)
  (draw-string-xy *prompt-window* string 0 (font-kanji-base-line 
				     (stream-font *prompt-window*)))
    |#


;;; End of file
